﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.Specifics.Behaviors;
using EffectMaker.UIDialogs.MessageDialogs;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// Custom shader options.
    /// </summary>
    [DisplayOrder(6)]
    public partial class CustomShaderOptionPane : UserControl, IOptionPane
    {
        /// <summary>
        /// Initializes the CustomShaderOptionPane instance.
        /// </summary>
        public CustomShaderOptionPane()
        {
            this.InitializeComponent();

            this.loadButton.Click += this.OnCustomShaderLoadButtonClick;
            this.resetButton.Click += this.OnCustomShaderClearButtonClick;

            this.resetButton.Enabled = false;
            this.btnCustomShaderReload.Enabled = false;

            this.filePathInput.BackColor = Color.White;
            this.filePathInput.FilePathChanged += (s, e) =>
            {
                this.resetButton.Enabled = !string.IsNullOrEmpty(this.filePathInput.FilePath);
                this.btnCustomShaderReload.Enabled = !string.IsNullOrEmpty(this.filePathInput.FilePath);
            };

            foreach (var box in this.Controls.OfType<OptionGroupBox>())
            {
                box.BorderColor = Color.FromArgb(130, 150, 185);
            }
        }

        /// <summary>
        /// オプションウィンドウのTreeViewを再描画するdelegate
        /// </summary>
        public Action DrawTreeView
        {
            get;
            set;
        }

        /// <summary>
        /// Gets the identifier.
        /// </summary>
        public string Identifier
        {
            get { return "CustomShaderOptionPane"; }
        }

        /// <summary>
        /// Gets the parent identifier.
        /// </summary>
        public string ChildOf
        {
            get { return "ProjectConfig"; }
        }

        /// <summary>
        /// Gets the display name.
        /// </summary>
        public string DisplayName
        {
            get { return Properties.Resources.OptionPaneCustomShader; }
        }

        /// <summary>
        /// Called when initialized.
        /// </summary>
        public void OnInitialize()
        {
            // カスタムシェーダ定義ファイルのドラッグ&ドロップ処理を設定
            {
                // カスタムシェーダのファイル以外は、ドラッグ&ドロップできないようにする
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowExtensions = RegexUtility.ExtractExtensions(Properties.Resources.OptionDialogFilterXmlFiles);
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    this.UpdateFilePathInputTxt(dragDropBehavior.DropResult);
                };

                this.filePathInput.Behaviors.Add(dragDropBehavior);
            }

            EffectMakerProjectConfig config = OptionStore.ProjectConfig;

            this.filePathInput.FilePath = config.CustomShaderPath;

            // カスタムシェーダが空っぽだったら、何もロードしない
            if (!string.IsNullOrEmpty(this.filePathInput.FilePath))
            {
                this.CheckAndUpdateCustomShader();
            }

            this.btnCustomShaderReload.Click += this.OnCustomShaderReloadButtonClick;

            OptionUtil.SetCommentLabelState(this.lblProjectConfigOverwrite);

            if (SpecManager.CurrentSpec.BinaryHeader == "VFXB")
            {
                // VFXBの時だけグループボックスを表示する
                this.ShowAdditionalShaderCompileSettings(config);

                OptionUtil.SetCommentLabelState(this.lblAddotionalSettings);
            }
            else
            {
                // そうでない時は存在自体を消す
                int y = this.grpAdditionalShaderSettings.Top;
                this.Controls.Remove(this.grpAdditionalShaderSettings);
                int y2 = this.fileTreeViewGroup.Top;
                this.fileTreeViewGroup.Bounds = new Rectangle(
                    this.fileTreeViewGroup.Left,
                    y,
                    this.fileTreeViewGroup.Width,
                    this.fileTreeViewGroup.Height + (y2 - y));
            }

            OptionUtil.IsNeedReloadCustomShader = !this.OnValidationCheck();
        }

        /// <summary>
        /// Called when option window is terminated.
        /// </summary>
        public void OnTerminate()
        {
        }

        /// <summary>
        /// Called when the Accept button is clicked.
        /// </summary>
        /// <returns>入力に不正がなければtrue,入力が不正であればfalse.</returns>
        public bool OnAccept()
        {
            var config = OptionStore.ProjectConfig;
            var tempConfig = OptionUtil.TempConfig;

            config.EnableShaderPreprocessOption = this.chkPreprocessIsEnabled.Checked;
            tempConfig.EnableShaderPreprocessOption = this.chkPreprocessIsEnabled.Checked;

            // カスタムシェーダ設定ファイルを変更されたとき
            if (OptionUtil.IsNeedReloadCustomShader)
            {
                // カスタムシェーダだけを変更したときはconfigを使用するが
                // プロジェクト設定を変更したときはtempConfigで上書きされるので両方書き換える
                config.CustomShaderPath = this.filePathInput.FilePath;
                tempConfig.CustomShaderPath = this.filePathInput.FilePath;

                // Load the custom shader.
                var def = UserDataHelper.Instance.LoadCustomShaderUserData(config.CustomShaderPath, true);

                // カスタムシェーダが設定されて、ロードに失敗した場合
                if (config.CustomShaderPath != string.Empty && def == null)
                {
                    this.ShowAllWarningsInPane();
                    return false;
                }

                // コマンドスタックをクリア
                CommandManager.ClearCommandStack();
            }

            return true;
        }

        /// <summary>
        /// Called when the Cancel button is clicked.
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// プロジェクトコンフィグが読み込まれた際に、その内容にUIを更新します。
        /// </summary>
        public void OnProjectSettingChanged()
        {
            this.filePathInput.FilePath = OptionUtil.TempConfig.CustomShaderPath;
            if (string.IsNullOrEmpty(this.filePathInput.FilePath))
            {
                this.OnCustomShaderClearButtonClick(null, EventArgs.Empty);
            }
            else
            {
                this.CheckAndUpdateCustomShader();
            }

            this.ShowAdditionalShaderCompileSettings(OptionUtil.TempConfig);
        }

        /// <summary>
        /// カスタムシェーダーの設定が有効か無効かを返す。
        /// </summary>
        /// <returns>
        /// 有効ならtrue、無効ならfalse.
        /// </returns>
        public bool OnValidationCheck()
        {
            return this.filePathInput.BackColor == Color.White;
        }

        /// <summary>
        /// プロジェクトコンフィグをファイル出力する際に各UIの内容を出力用インスタンスに収集します。
        /// </summary>
        public void OnExportProjectSetting()
        {
            OptionUtil.TempConfig.CustomShaderPath = this.filePathInput.FilePath;
            OptionUtil.TempConfig.EnableShaderPreprocessOption = this.chkPreprocessIsEnabled.Checked;
        }

        /// <summary>
        /// シェーダコンパイルの追加設定をUIに表示します。
        /// </summary>
        /// <param name="config">設定を表示したいプロジェクト設定</param>
        private void ShowAdditionalShaderCompileSettings(EffectMakerProjectConfig config)
        {
            this.lstDirectories.Items.Clear();
            foreach (var dir in config.AdditionalShaderIncludeDirectories)
            {
                var item = this.lstDirectories.Items.Add(dir);
                item.SubItems.Add(PathUtility.ToAbsolutePath(dir, Path.GetDirectoryName(config.ConfigFilePath)));
            }

            this.lstDirectories.AutoResizeColumns(this.lstDirectories.Items.Count > 0 ?
                ColumnHeaderAutoResizeStyle.ColumnContent :
                ColumnHeaderAutoResizeStyle.HeaderSize);

            this.lstDefinitions.Items.Clear();
            foreach (var def in config.AdditionalShaderPreprocessorDefinitions)
            {
                this.lstDefinitions.Items.Add(def);
            }

            this.lstDefinitions.AutoResizeColumns(this.lstDefinitions.Items.Count > 0 ?
                ColumnHeaderAutoResizeStyle.ColumnContent :
                ColumnHeaderAutoResizeStyle.HeaderSize);

            this.chkPreprocessIsEnabled.Checked = config.EnableShaderPreprocessOption;
        }

        /// <summary>
        /// プロジェクト設定ファイルの内容が不正だった場合のメッセージボックスを表示します.
        /// </summary>
        private void ShowWarningMessageBox()
        {
            ThreadSafeMsgBox.Show(
                    Resources.InvalidSettingFileCustomShader,
                    Resources.WarningCaption,
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Warning);
        }

        /// <summary>
        /// カスタムシェーダpane内の全警告を表示する.
        /// </summary>
        private void ShowAllWarningsInPane()
        {
            this.filePathInput.BackColor = Color.Pink;
            this.SetupShaderSettingsTree(null);
            if (this.DrawTreeView != null)
            {
                this.DrawTreeView();
            }

            // 最後のメッセージボックスを表示する.
            this.ShowWarningMessageBox();
        }

        /// <summary>
        /// カスタムシェーダをチェックして読み込む.
        /// </summary>
        private void CheckAndUpdateCustomShader()
        {
            // 指定されたカスタムシェーダが存在し、カスタムシェーダのファイルであるかチェックする.
            if (!File.Exists(this.filePathInput.FilePath) || !IOConstants.CheckRootElement(this.filePathInput.FilePath, "CustomShaderUIDefList"))
            {
                this.ShowAllWarningsInPane();
            }
            else
            {
                // カスタムシェーダのファイルであることを確認したら、セッティングツリーをリロード
                this.UpdateView(this.filePathInput.FilePath);
            }

            OptionUtil.IsNeedReloadCustomShader = true;
        }

        /// <summary>
        /// シェーダセッティングツリーを作成します。
        /// </summary>
        /// <param name="defList">The custom shader definitions.</param>
        private void SetupShaderSettingsTree(CustomShaderDefinitionList defList)
        {
            this.fileTreeView.Clear();

            // 未登録のときはツリーに専用の文言を表示
            if (defList == null)
            {
                this.fileTreeView.AddText(
                    null,
                    Properties.Resources.OptionCustomShaderUnregistered);

                return;
            }

            // シェーダ設定パスを追加
            var root = this.fileTreeView.AddFile(null, null, defList.FilePath);

            string baseFolderPath = PathUtility.GetDirectoryName(defList.BaseFolderPath);

            // 汎用バーテックスシェーダパスを追加
            var generalVshNode = this.fileTreeView.AddText(
                root,
                Properties.Resources.OptionCustomShaderGeneralVertexShaderFiles);

            foreach (string path in defList.GeneralShaderDefinition.VertexShaderFullPaths)
            {
                this.fileTreeView.AddFile(generalVshNode, null, baseFolderPath, path);
            }

            // 汎用フラグメントシェーダパスを追加
            var generalFshNode = this.fileTreeView.AddText(
                root,
                Properties.Resources.OptionCustomShaderGeneralFragmentShaderFiles);

            foreach (var path in defList.GeneralShaderDefinition.FragmentShaderFullPaths)
            {
                this.fileTreeView.AddFile(generalFshNode, null, baseFolderPath, path);
            }

            // 汎用コンピュートシェーダパスを追加
            var generalCshNode = this.fileTreeView.AddText(
                root,
                Properties.Resources.OptionCustomShaderGeneralComputeShaderFiles);

            foreach (var path in defList.GeneralShaderDefinition.ComputeShaderFullPaths)
            {
                this.fileTreeView.AddFile(generalCshNode, null, baseFolderPath, path);
            }

            // カスタムシェーダ設定
            foreach (var customShader in defList.CustomShaderDefinitions)
            {
                var customShaderNode = this.fileTreeView.AddFile(
                    root,
                    null,
                    customShader.DefinitionPath);

                // カスタムバーテックスシェーダパスを追加
                var customVshNode = this.fileTreeView.AddText(
                    customShaderNode,
                    Properties.Resources.OptionCustomShaderCustomVertexShaderFiles);

                foreach (string path in customShader.VertexShaderFullPaths)
                {
                    this.fileTreeView.AddFile(
                        customVshNode,
                        null,
                        string.Empty,
                        path);
                }

                // カスタムフラグメントシェーダパスを追加
                var customFshNode = this.fileTreeView.AddText(
                    customShaderNode,
                    Properties.Resources.OptionCustomShaderCustomFragmentShaderFiles);

                foreach (string path in customShader.FragmentShaderFullPaths)
                {
                    this.fileTreeView.AddFile(
                        customFshNode,
                        null,
                        string.Empty,
                        path);
                }

                if (customShader.UsingUdd2)
                {
                    this.fileTreeView.AddFile(
                        customShaderNode,
                        Resources.OptionCustomUserDataDefinitionFile + " : [" + customShader.Id + "] " + customShader.Name,
                        customShader.Udd2FullFilePath);
                }
                else
                {
                    // データモデルコードファイルパスを追加
                    this.fileTreeView.AddFile(
                        customShaderNode,
                        Properties.Resources.OptionCustomShaderDataModelFile,
                        string.Empty,
                        customShader.DataModelFilePath);

                    // XAMLファイルパスを追加
                    this.fileTreeView.AddFile(
                        customShaderNode,
                        Properties.Resources.OptionCustomShaderXamlFile,
                        string.Empty,
                        customShader.XamlFilePath);

                    // ConversionInfoファイルパスを追加
                    this.fileTreeView.AddFile(
                        customShaderNode,
                        Properties.Resources.OptionCustomShaderConversionInfoFile,
                        string.Empty,
                        customShader.BinaryConversionInfoPath);
                }
            }
        }

        /// <summary>
        /// Called when the CustomShaderLoad Button is clicked.
        /// </summary>
        /// <param name="sender">The CustomShaderLoad Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnCustomShaderLoadButtonClick(object sender, EventArgs e)
        {
            string initDir = null;
            if (string.IsNullOrEmpty(this.filePathInput.FilePath) == false)
            {
                string path = Path.GetDirectoryName(this.filePathInput.FilePath);
                if (Directory.Exists(path) == true)
                {
                    initDir = path;
                }
            }

            if (initDir == null)
            {
                initDir = ApplicationIOManager.GetLastAccessedDir(
                    LastAccessDirectoryTypes.CustomShaderList);
            }

            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterXmlFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
                FileName = this.filePathInput.FilePath,
                InitialDirectory = initDir,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.UpdateFilePathInputTxt(dlg.FileName);
        }

        /// <summary>
        /// filePathInput.FilePathを更新する
        /// </summary>
        /// <param name="fileName">更新するファイル名(パス)</param>
        /// <returns>filePathInput.FilePathを更新した場合はtrueを返す. 更新しなかった場合はfalseを返す.</returns>
        private bool UpdateFilePathInputTxt(string fileName)
        {
            if (!IOConstants.CheckRootElement(fileName, "CustomShaderUIDefList"))
            {
                this.ShowWarningMessageBox();
                return false;
            }

            // Update the last accessed custom shader list file path.
            ApplicationIOManager.SetLastAccessedDir(
                LastAccessDirectoryTypes.CustomShaderList,
                fileName);

            this.filePathInput.FilePath = fileName;

            // セッティングツリーをリロード
            this.UpdateView(this.filePathInput.FilePath);

            // オプションウィンドウを閉じるときにカスタムシェーダをリロードさせる
            OptionUtil.IsNeedReloadCustomShader = true;

            return true;
        }

        /// <summary>
        /// Called when the CustomShaderClear Button is clicked.
        /// </summary>
        /// <param name="sender">The CustomShaderClear Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnCustomShaderClearButtonClick(object sender, EventArgs e)
        {
            this.filePathInput.FilePath = string.Empty;
            this.SetupShaderSettingsTree(null);
            this.filePathInput.BackColor = Color.White;
            OptionUtil.IsNeedReloadCustomShader = true;
            if (this.DrawTreeView != null)
            {
                this.DrawTreeView();
            }
        }

        /// <summary>
        /// カスタムシェーダ設定をリロードします。
        /// </summary>
        /// <param name="sender">
        /// The parameter is not used.
        /// </param>
        /// <param name="e">
        /// The parameter is not used.
        /// </param>
        private void OnCustomShaderReloadButtonClick(object sender, EventArgs e)
        {
            if (!string.IsNullOrEmpty(this.filePathInput.FilePath))
            {
                this.CheckAndUpdateCustomShader();
            }
        }

        /// <summary>
        /// セッティングツリーを更新し、データ定義が取得できない場合はエラーとして色を変えます。
        /// </summary>
        /// <param name="filePath">
        /// カスタムシェーダ定義ファイルパス
        /// </param>
        private void UpdateView(string filePath)
        {
            var def = UserDataHelper.Instance.LoadCustomShaderUserData(filePath, false);
            if (!string.IsNullOrEmpty(filePath) && def == null)
            {
                this.filePathInput.BackColor = Color.Pink;
                this.ShowWarningMessageBox();
            }
            else
            {
                this.filePathInput.BackColor = Color.White;
            }

            this.SetupShaderSettingsTree(def);
            if (this.DrawTreeView != null)
            {
                this.DrawTreeView();
            }
        }
    }
}
