﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.BusinessLogic.Options;

using EffectMaker.UIDialogs.MessageDialogs;

namespace EffectMaker.Application.OptionalMessageBox
{
    /// <summary>
    /// The class holds the information for showing the optional message box.
    /// </summary>
    public class OptionalMessageBoxContext : IDisposable
    {
        /// <summary>The name of the configuration of the optional message box.</summary>
        private string configName;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="configName">The config name for the optional message box.</param>
        public OptionalMessageBoxContext(string configName)
        {
            this.configName = configName;

            EffectMaker.UIDialogs.MessageDialogs.MessageBox.MessageBoxShowing += this.OnMessageBoxShowing;
            EffectMaker.UIDialogs.MessageDialogs.MessageBox.MessageBoxClosing += this.OnMessageBoxClosing;
        }

        /// <summary>
        /// Destructor for resetting the value.
        /// </summary>
        public void Dispose()
        {
            EffectMaker.UIDialogs.MessageDialogs.MessageBox.MessageBoxShowing -= this.OnMessageBoxShowing;
            EffectMaker.UIDialogs.MessageDialogs.MessageBox.MessageBoxClosing -= this.OnMessageBoxClosing;
        }

        /// <summary>
        /// Handle MessageBoxShowing event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnMessageBoxShowing(
            object sender,
            EffectMaker.UIDialogs.MessageDialogs.MessageBox.MessageBoxShowingEventArgs e)
        {
            var msgBox = sender as EffectMaker.UIDialogs.MessageDialogs.MessageBox;
            if (msgBox == null)
            {
                return;
            }

            msgBox.IsOptionalMessageBox = true;

            DialogResult result;
            if (this.ShouldShowOptionalMessageBox(out result) == false)
            {
                e.DialogResultOnCanceling = result;
                e.Cancel = true;
            }
        }

        /// <summary>
        /// Handle MessageBoxClosing event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnMessageBoxClosing(
            object sender,
            CancelEventArgs e)
        {
            var msgBox = sender as EffectMaker.UIDialogs.MessageDialogs.MessageBox;
            if (msgBox == null)
            {
                return;
            }

            if (msgBox.IsOptionalMessageBox == false)
            {
                return;
            }

            if (msgBox.ShouldShowMessageBoxNextTime == true)
            {
                return;
            }

            var optionalMsgBoxSettings =
                OptionStore.RootOptions.Interface.OptionalMsgBoxSettings;

            OptionalMsgBoxSetting settings;
            if (optionalMsgBoxSettings.TryGetValue(this.configName, out settings) == false)
            {
                settings = new OptionalMsgBoxSetting();
                optionalMsgBoxSettings.Add(this.configName, settings);
            }

            settings.IsEnabled = msgBox.ShouldShowMessageBoxNextTime;
            settings.DialogResult = msgBox.DialogResult;
        }

        /// <summary>
        /// Determine if the optional message box should show or simply return the dialog result.
        /// </summary>
        /// <param name="dialogResult">The default dialog result for this optional message box.</param>
        /// <returns>True to show the optional message box.</returns>
        private bool ShouldShowOptionalMessageBox(out DialogResult dialogResult)
        {
            var optionalMsgBoxSettings =
                OptionStore.RootOptions.Interface.OptionalMsgBoxSettings;

            dialogResult = DialogResult.None;
            if (optionalMsgBoxSettings == null)
            {
                return true;
            }

            OptionalMsgBoxSetting setting;
            if (optionalMsgBoxSettings.TryGetValue(this.configName, out setting) == false)
            {
                return true;
            }

            dialogResult = setting.DialogResult;

            return setting.IsEnabled;
        }
    }
}
