﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.UIControls.Layout;
using Timer = System.Windows.Forms.Timer;

namespace EffectMaker.Application
{
    /// <summary>
    /// スプラッシュフォームです。
    /// 参考URL(http://dobon.net/vb/dotnet/form/splashwindow.html)
    /// </summary>
    public partial class SplashForm : Form
    {
        /// <summary>
        /// lock用のオブジェクト
        /// </summary>
        private static readonly object SyncObject = new object();

        /// <summary>
        /// Splashフォーム
        /// </summary>
        private static SplashForm splashForm = null;

        /// <summary>
        /// メインフォーム
        /// </summary>
        private static Form mainForm = null;

        /// <summary>
        /// Splashを表示するスレッド
        /// </summary>
        private static System.Threading.Thread thread = null;

        /// <summary>
        /// Splashが表示されるまで待機するための待機ハンドル
        /// </summary>
        private static System.Threading.ManualResetEvent splashShownEvent = null;

        /// <summary>
        /// メインフォームがアクティブになったときのイベントです。
        /// </summary>
        public static EventHandler MainFormActivated;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SplashForm()
        {
            this.InitializeComponent();

            // 画像の表示
            this.IconPicture.Image = Properties.Resources.Logo_EffectMaker;

            // 情報取得
            Assembly assembly = Assembly.GetExecutingAssembly();
            FileVersionInfo info = FileVersionInfo.GetVersionInfo(assembly.Location);

            string strBuiltTime = string.Format(
                "{0} {1}",
                AboutDialog.AboutDialog.VersionInformation.BuildDate.ToShortDateString(),
                AboutDialog.AboutDialog.VersionInformation.BuildDate.ToShortTimeString());

            // テキストを設定
            this.txtInfo.Font = new Font("Tahoma", 8.25F, FontStyle.Regular, GraphicsUnit.Point, 0);
            this.txtInfo.Text = string.Format(
                                            "Version {0}.{1}.{2} Build {3}\r\n{4}",
                                            info.ProductMajorPart,
                                            info.ProductMinorPart,
                                            info.ProductBuildPart,
                                            strBuiltTime,
                                            AboutDialog.AboutDialog.CopyrightString);
        }

        /// <summary>
        /// Splashフォームを表示します。
        /// </summary>
        /// <param name="argForm">メインフォーム</param>
        public static void ShowSplash(Form argForm)
        {
            lock (SyncObject)
            {
                if (splashForm != null || thread != null)
                {
                    return;
                }

                mainForm = argForm;

                // メインフォームのVisibleChangedイベントでSplashフォームを消す
                if (argForm != null)
                {
                    argForm.VisibleChanged += _mainForm_Activated;
                }

                // 待機ハンドルの作成
                splashShownEvent = new System.Threading.ManualResetEvent(false);

                // スレッドの作成
                thread = new System.Threading.Thread(StartThread)
                {
                    Name = "SplashForm",
                    IsBackground = true,
                };
                thread.SetApartmentState(System.Threading.ApartmentState.STA);

                // スレッドの開始
                thread.Start();
            }
        }

        /// <summary>
        /// Splashフォームを表示する
        /// </summary>
        public static void ShowSplash()
        {
            ShowSplash(null);
        }

        /// <summary>
        /// Splashフォームを消す
        /// </summary>
        public static void CloseSplash()
        {
            lock (SyncObject)
            {
                if (thread == null)
                {
                    return;
                }

                if (mainForm != null)
                {
                    mainForm.VisibleChanged -= _mainForm_Activated;
                }

                // Splashが表示されるまで待機する
                if (splashShownEvent != null)
                {
                    splashShownEvent.WaitOne();
                    splashShownEvent.Close();
                    splashShownEvent = null;
                }

                // Splashフォームを閉じる
                // Invokeが必要か調べる
                if (splashForm != null)
                {
                    Action closeSplashFormAction = () =>
                    {
                        if (!splashForm.IsDisposed)
                        {
                            splashForm.Close();
                        }
                    };

                    if (splashForm.InvokeRequired)
                    {
                        splashForm.Invoke(closeSplashFormAction);
                    }
                    else
                    {
                        closeSplashFormAction();
                    }
                }

                // メインフォームをアクティブにする
                if (mainForm != null)
                {
                    Action activateMainFormAction = () =>
                    {
                        if (!mainForm.IsDisposed)
                        {
                            mainForm.Activate();
                        }
                    };

                    if (mainForm.InvokeRequired)
                    {
                        mainForm.Invoke(activateMainFormAction);
                    }
                    else
                    {
                        activateMainFormAction();
                    }
                }

                splashForm = null;
                thread = null;
                mainForm = null;
            }
        }

        /// <summary>
        /// スレッドで開始するメソッド
        /// </summary>
        private static void StartThread()
        {
            // Splashフォームを作成
            splashForm = new SplashForm();

            // Splashが表示されるまでCloseSplashメソッドをブロックする
            splashForm.Activated += new EventHandler(_form_Activated);

            splashForm.TopMost = true;

            // Splashフォームを表示する
            System.Windows.Forms.Application.Run(splashForm);
        }

        /// <summary>
        /// メインフォームがアクティブになった時
        /// </summary>
        /// <param name="sender">The parameter is not used.</param>
        /// <param name="e">The parameter is not used.</param>
        private static void _mainForm_Activated(object sender, EventArgs e)
        {
            mainForm.VisibleChanged -= _mainForm_Activated;

            MainFormActivated?.Invoke(sender, e);

            if (splashForm.InvokeRequired)
            {
                splashForm.Invoke(new MethodInvoker(() => splashForm.Visible = false));
            }
            else
            {
                splashForm.Visible = false;
            }

            CloseSplash();
        }

        /// <summary>
        /// Splashフォームが表示された時
        /// </summary>
        /// <param name="sender">The parameter is not used.</param>
        /// <param name="e">The parameter is not used.</param>
        private static void _form_Activated(object sender, EventArgs e)
        {
            splashForm.Activated -= new EventHandler(_form_Activated);
            splashForm.TopMost = false;

            // CloseSplashメソッドの待機を解除
            if (splashShownEvent != null)
            {
                splashShownEvent.Set();
            }
        }
    }
}
