﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Threading;
using System.Reflection;
using System.Globalization;
using Nintendo.Foundation.IO;
using EffectMaker.EffectMakerBridge.Properties;
using EffectMaker.ProcessBridge;

namespace EffectMaker.EffectMakerBridge
{
    /// <summary>
    /// メインクラスです。
    /// </summary>
    public class Program
    {
        /// <summary>
        /// メイン関数です。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>リザルトコードを返します。</returns>
        public static int Main(string[] args)
        {
            Mutex appMutex = null;
            BridgeClient bridgeClient = null;

            // "en\EffectMakerBridge.resources.dll" があるときはカルチャを英語に設定する
            {
                string entryFolder = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                string resourceFile = Path.Combine(entryFolder, @"en\EffectMakerBridge.resources.dll");
                string cultureName = File.Exists(resourceFile) ? "en-US" : "ja-JP";

                CultureInfo cultureInfo = CultureInfo.CreateSpecificCulture(cultureName);

                Thread.CurrentThread.CurrentCulture = cultureInfo;
                Thread.CurrentThread.CurrentUICulture = cultureInfo;
            }

            try
            {
                // コマンドライン引数を解析
                CommandLineParams parameters = ParseArgs(args);

                if (parameters == null)
                {
                    return 1;
                }

                // ミューテックスを作成
                appMutex = new Mutex(false, appMutexName);

                // 多重起動防止のためミューテックスをロックする
                // すぐにロックできなければ待機するが、30秒待ってもロックできなければエラーにする
                {
                    bool locked = false;

                    try
                    {
                        locked = appMutex.WaitOne(TimeSpan.FromSeconds(30.0), false);
                    }
                    catch (AbandonedMutexException)
                    {
                        // ミューテックスが解放されずに終了されたときは例外となるがロックはされる
                        locked = true;
                    }

                    if (locked == false)
                    {
                        Console.WriteLine(Resources.ErrorAppMutexTimeout);

                        appMutex = null;
                        return 1;
                    }
                }

                // 通信クライアントを初期化
                bridgeClient = new BridgeClient();
                bool initClient = bridgeClient.Initialize();

                if (initClient == false)
                {
                    Console.WriteLine(Resources.ErrorBridgeClientInitializeFailed);

                    bridgeClient = null;
                    return 1;
                }

                // パラメータに指定された処理を呼び出す
                if (parameters.LoadFiles != null)
                {
                    bool resCall = bridgeClient.CallLoadFiles(parameters.LoadFiles.Paths);

                    if (resCall == false)
                    {
                        Console.WriteLine(Resources.ErrorCallLoadFilesFailed);
                        return 1;
                    }
                    else
                    {
                        Console.WriteLine(Resources.InfoCallLoadFilesSucceeded);
                    }
                }
            }
            finally
            {
                if (bridgeClient != null)
                {
                    bridgeClient.Release();
                    bridgeClient = null;
                }

                if (appMutex != null)
                {
                    appMutex.ReleaseMutex();
                    appMutex = null;
                }
            }

            return 0;
        }

        /// <summary>
        /// コマンドライン引数をパースします。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>パース結果を返します。</returns>
        private static CommandLineParams ParseArgs(string[] args)
        {
            // バイナリ情報を取得
            System.Reflection.Assembly assembly = System.Reflection.Assembly.GetExecutingAssembly();
            System.Diagnostics.FileVersionInfo assemblyInfo = System.Diagnostics.FileVersionInfo.GetVersionInfo(assembly.Location);

            // パーサーを作成
            StringWriter helpWriter = new StringWriter();
            StringWriter errorWriter = new StringWriter();

            var settings = new CommandLineParserSettings()
            {
                HelpWriter = helpWriter.Write,
                ErrorWriter = errorWriter.Write,
                ApplicationName = assemblyInfo.ProductName,
                Version = string.Format("{0}.{1}.{2}", assemblyInfo.ProductMajorPart, assemblyInfo.ProductMinorPart, assemblyInfo.ProductBuildPart),
                ApplicationDescription = Resources.ConsoleDescApplicationDescription
            };

            var commandLineParser = new CommandLineParser(settings);

            bool resParse = false;
            CommandLineParams parameters = null;

            // パース処理を実行
            try
            {
                //// ParseArgs()の挙動
                //// パース成功     : params = input, return = true , throw = false
                //// パースエラー   : params = null , return = false, throw = true
                //// ヘルプ表示     : params = null , return = false, throw = false
                //// バージョン表示 : params = null , return = false, throw = false
                resParse = commandLineParser.ParseArgs(args, out parameters);
            }
            catch (Exception e)
            {
                // エラーメッセージを表示
                string errorText = errorWriter.ToString();

                {
                    if (errorText.Length != 0)
                    {
                        // パースエラーを表示
                        Console.WriteLine(errorText);
                    }
                    else
                    {
                        // DLLが読み込めないときなどのアプリケーションエラーを表示
                        Console.WriteLine(e.Message);
                    }
                }

                return null;
            }

            // ヘルプ情報またはバージョン情報を表示
            if (resParse == false)
            {
                {
                    string helpText = helpWriter.ToString();
                    Console.WriteLine(helpText);
                }

                return null;
            }

            return parameters;
        }

        /// <summary>
        /// アプリケーション同期用のミューテックスの名前
        /// </summary>
        private const string appMutexName = "EffectMakerBridge_Mutex";
    }
}
