﻿using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.DataModelLogic.Utilities;
using System;
using System.Diagnostics;
using System.Reflection;
using EffectMaker.BusinessLogic.BinaryHeaders;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// バイナリコンバートのプロファイラです。
    /// </summary>
    public static class BinaryConvertProfiler
    {
        /// <summary>総時間のタイマー</summary>
        private static readonly ProfileTimer totalTimer = new ProfileTimer("TotalTime");

        /// <summary>eset 読み込みのタイマー</summary>
        private static readonly ProfileTimer readingEsetTimer = new ProfileTimer("ReadingEsetTimer");

        /// <summary>開始時間</summary>
        private static DateTime startDateTime;

        /// <summary> 終了時間</summary>
        private static DateTime finishDateTime;

        /// <summary>
        /// WriteBinaryDataSessionProfiler を取得または設定します。
        /// </summary>
        public static WriteBinaryDataSessionProfiler WriteBinaryDataSessionProfiler { get; set; }

        /// <summary>
        /// バイナリサイズを取得または設定します。
        /// </summary>
        public static long TotalSize { get; set; }

        /// <summary>
        /// eset の個数を取得または設定します。
        /// </summary>
        public static int EsetCount { get; set; }

        /// <summary>
        /// 総時間のタイマーを開始します。
        /// </summary>
        public static void StartTotalTimer()
        {
            BinaryConvertProfiler.totalTimer.Stop(false);
            BinaryConvertProfiler.totalTimer.Resume();
            startDateTime = System.DateTime.Now;
        }

        /// <summary>
        /// 総時間のタイマーを終了します。
        /// </summary>
        public static void StopTotalTimer()
        {
            BinaryConvertProfiler.totalTimer.Stop(false);
            finishDateTime = System.DateTime.Now;
        }

        /// <summary>
        /// eset 読み込みのタイマーを開始します。
        /// </summary>
        public static void StartReadingEsetTimer()
        {
            BinaryConvertProfiler.readingEsetTimer.Stop(false);
            BinaryConvertProfiler.readingEsetTimer.Resume();
        }

        /// <summary>
        /// eset 読み込みのタイマーを終了します。
        /// </summary>
        public static void StopReadingEsetTimer()
        {
            BinaryConvertProfiler.readingEsetTimer.Stop(false);
        }

        /// <summary>
        /// 結果を出力します。
        /// </summary>
        public static void OutputSummary()
        {
            var wbp = BinaryConvertProfiler.WriteBinaryDataSessionProfiler;

            Log("");
            Log("");
            string bar = " --------------------------------------------------------------";

            LogVfx(bar);
            LogVfx("");
            LogVfx(" vfx binary conversion has been successfully finished.");
            LogVfx("");
            LogVfx("         ---- basic information ----");
            LogVfx("");
            LogVfx(" start date                    :         " + startDateTime.ToShortDateString());
            LogVfx(" start time                    :           " + startDateTime.ToLongTimeString());
            LogVfx(" finish time                   :           " + finishDateTime.ToLongTimeString());
            LogVfx(" binary version                :           " + BinaryFileHeader.Version.ToString("X8"));
            LogVfx(" converter version             :              " + GetVersionString());
            LogVfx(" converter build date          :         " + GetBuildDateString());
            LogVfx("");
            LogVfx("         ---- resources ----");
            LogVfx("                                number  binary size");
            LogVfx("                                            (bytes)");
            LogVfx(" total                         :" + $"{TotalSize,19}");
            LogVfx("   eset                        :" + $"{EsetCount,6}");
            LogVfx("   texture                     :" + ToCountSizeString(wbp.TextureBlockInfo));
            LogVfx("   primitive                   :" + ToCountSizeString(wbp.PrimitiveBlockInfo));
            LogVfx("   g3d primitive               :" + ToCountSizeString(wbp.G3dPrimitiveBlockInfo));
            LogVfx("   shader                      :" + ToCountSizeString(wbp.ShaderBinBlockInfo));
            LogVfx("   compute shader              :" + ToCountSizeString(wbp.ComputeShaderBinBlockInfo));
            LogVfx("");
            LogVfx("         ---- time (seconds) ----");
            LogVfx("");
            LogVfx(" total                         :" + ToLogString(totalTimer));
            LogVfx("   eset reading                :" + ToLogString(readingEsetTimer));
            LogVfx("   texture conversion          :" + ToTimeString(wbp.TextureBlockInfo));
            LogVfx("   primitive conversion        :" + ToTimeString(wbp.PrimitiveBlockInfo));
            LogVfx("   g3d Primitive conversion    :" + ToTimeString(wbp.G3dPrimitiveBlockInfo));
            LogVfx("   shader conversion           :" + ToTimeString(wbp.ShaderBinBlockInfo));
            LogVfx("   compute Shader conversion   :" + ToTimeString(wbp.ComputeShaderBinBlockInfo));

            LogVfx("");
            LogVfx(bar);
            Log("");
            Log("");
        }

        /// <summary>
        /// EffectMaker のバージョン文字列を取得します。
        /// </summary>
        /// <returns></returns>
        private static string GetVersionString()
        {
            Assembly assembly = Assembly.GetExecutingAssembly();
            FileVersionInfo info = FileVersionInfo.GetVersionInfo(assembly.Location);
            return string.Format("{0}.{1}.{2}",
                info.ProductMajorPart,
                info.ProductMinorPart,
                info.ProductBuildPart);
        }

        /// <summary>
        /// EffectMaker のビルド日時を取得します。
        /// </summary>
        /// <returns></returns>
        private static string GetBuildDateString()
        {
            return AboutDialog.AboutDialog.VersionInformation.BuildDate.ToShortDateString();
        }

        /// <summary>
        /// 個数とサイズの文字列に変換します。
        /// </summary>
        /// <param name="info">ProfileInfo</param>
        /// <returns>変換後の文字列</returns>
        private static string ToCountSizeString(WriteBinaryDataSessionProfiler.BinaryBlockInfo info)
        {
            return $"{info.Count,6} {info.Size,12}";
        }

        /// <summary>
        /// 時間の文字列に変換します。
        /// </summary>
        /// <param name="info">ProfileInfo</param>
        /// <returns>変換後の文字列</returns>
        private static string ToTimeString(WriteBinaryDataSessionProfiler.BinaryBlockInfo info)
        {
            return string.Format("{0,19:0.00}", info.Time);
        }

        /// <summary>
        /// 時間の文字列に変換します。
        /// </summary>
        /// <param name="timer">ProfileTimer</param>
        /// <returns>変換後の文字列</returns>
        private static string ToLogString(ProfileTimer timer)
        {
            return string.Format("{0,19:0.00}", timer.ElapsedTime.TotalSeconds);
        }

        /// <summary>
        /// ログ出力します。
        /// </summary>
        /// <param name="format">フォーマット</param>
        /// <param name="args">引数</param>
        private static void LogVfx(string format, params object[] args)
        {
            Log("[VfxConv] " + format, args);
        }

        /// <summary>
        /// ログ出力します。
        /// </summary>
        /// <param name="format">フォーマット</param>
        /// <param name="args">引数</param>
        private static void Log(string format, params object[] args)
        {
            // Console への出力
            Logger.Log("Console", LogLevels.Information, format, args);
#if !DEBUG
            // Relase 実行時でも VisualStudio 出力ウィンドウでログを確認できるようにするため
            System.Diagnostics.Trace.WriteLine(string.Format(format, args));
#endif
        }
    }
}
