﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Security;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Data;
using System.Xml;

namespace CombinerNodeEditor.Utility
{
    class NodeInfoHelper
    {
        private static readonly string xmlHeader = "<?xml version=\"1.0\"?>";
        private static readonly string xmlRootTagName = "CombinerNodeDefinitions";
        private static readonly string xmlRootTagOpen = string.Format("<{0}>", xmlRootTagName);
        private static readonly string xmlRootTagClose = string.Format("</{0}>", xmlRootTagName);

        public static string PreprocessGlslToNodeInfoXml(string text)
        {
            //-----------------------------------------------------------------
            // 先頭が /// なら /// だけ削除
            // 先頭が // なら行ごと削除
            // 先頭が /// でも // でもなければ、行全体の文字を XML 用にエスケープ
            //-----------------------------------------------------------------
            string result = "";
            string[] lines = text.Split('\n');
            foreach (var l in lines)
            {
                if (l.StartsWith("///"))
                {
                    var s = l.Substring(3);
                    result += s + "\n";
                }
                else if (l.StartsWith("//"))
                {
                    result += l + "\n";
                }
                else
                {
                    // MEMO: XML パーサーを通すのでエスケープする
                    //       <, > などの比較演算子がそのままだと引っかかる
                    result += SecurityElement.Escape(l) + "\n";
                }
            }

            // ルートタグで囲んでおく
            return xmlHeader + xmlRootTagOpen + result + xmlRootTagClose;
        }

        public static string SaveNodeDescToString(NodeDesc node)
        {
            // ノード定義からアノテーション付き glsl を吐き出す
            // とりあえず雑に xml 関係なくテキストテンプレート的に出力
            string ioDesc = "";
            string funcIoDesc = "";
            string blockDesc = string.Empty;
            string uniformDesc = string.Empty;
            string previewDesc = string.Empty;

            // TODO: ここは後で整理した方が見やすい
            {
                // MEMO: func タグは順番を維持する必要があるので、
                //       return -> out -> in -> return の順で吐く

                // 出力線
                var outputs = node.OutputDescs;
                for (int i = 0; i < outputs.Count; ++i)
                {
                    string name = outputs[i].DisplayName;
                    string type = outputs[i].Type.ToStringExt();
                    string target = outputs[i].ReturnTarget;
                    string semantics = outputs[i].Semantics;

                    // 関数の出力情報（void のときだけ）
                    if (node.ReturnType == ValueType.Void)
                    {
                        // ここは name タグだが引数名を入れるので target
                        funcIoDesc += string.Format(
                            "///   <out name=\"{0}\" type=\"{1}\"/>\n",
                            target, type);
                    }

                    // 出力ノード情報
                    // MEMO: semantic 属性はあるときだけ付加
                    string semanticsDesc = "";
                    if (semantics.Length > 0)
                    {
                        // MEMO: 前段と繋げるために半角スペースを頭に付加している
                        semanticsDesc = string.Format(" semantic=\"{0}\"", semantics);
                    }

                    ioDesc += string.Format(
                        "///   <out displayname=\"{0}\" type=\"{1}\" target=\"{2}\"{3}/>\n",
                        name, type, target, semanticsDesc);
                }

                // 入力線
                var input = node.InputDescs;
                for (int i = 0; i < input.Count; ++i)
                {
                    string name = input[i].DisplayName;
                    string type = input[i].Type.ToStringExt();
                    string target = input[i].ArgTarget;
                    string semantics = input[i].Semantics;

                    // 関数の入力情報
                    // ここは name タグだが引数名を入れるので target
                    funcIoDesc += string.Format(
                        "///   <in name=\"{0}\" type=\"{1}\"/>\n",
                        target, type);

                    // 入力ノード情報
                    // MEMO: semantic 属性はあるときだけ付加
                    string semanticsDesc = "";
                    if (semantics.Length > 0)
                    {
                        // MEMO: 前段と繋げるために半角スペースを頭に付加している
                        semanticsDesc = string.Format(" semantic=\"{0}\"", semantics);
                    }

                    ioDesc += string.Format(
                        "///   <in displayname=\"{0}\" type=\"{1}\" target=\"{2}\"{3}/>\n",
                        name, type, target, semanticsDesc);
                }

                // void でないなら return タグを追加
                if (node.ReturnType != ValueType.Void)
                {
                    funcIoDesc += string.Format(
                        "///   <return type=\"{0}\"/>\n", node.ReturnType.ToStringExt());
                }

                var uniform = node.Uniform;
                if (uniform.Type != UniformType.None)
                {
                    if (uniform.Type != UniformType.File)
                    {
                        uniformDesc = string.Format(
                            "///   <uniform name=\"{0}\" displayname=\"{1}\" type=\"{2}\" />\n",
                            node.Uniform.Name, node.Uniform.DisplayName, node.Uniform.Type.ToStringExt());
                    }
                    else
                    {
                        uniformDesc = string.Format(
                            "///   <uniform name=\"{0}\" displayname=\"{1}\" type=\"{2}\" extension=\"{3}\"/>\n",
                                node.Uniform.Name, node.Uniform.DisplayName, node.Uniform.Type.ToStringExt(), node.Uniform.ActiveExtensions.Extension.ToStringExt());
                    }
                }

                var preview = node.Preview;
                if (preview.IsEnabled)
                {
                    previewDesc = string.Format(
                        "///   <preview enabled=\"{0}\" width=\"{1}\" height=\"{2}\"/>\n",
                                node.Preview.IsEnabled.ToString().ToLower(), node.Preview.Width, node.Preview.Height);
                }

                var block = node.Block;
                if (block.Group == string.Empty)
                {
                    blockDesc = string.Format(
                        "/// <block guid=\"{0}\" displayname=\"{1}\">\n",
                        "{" + node.Block.Guid.ToString().ToUpper() + "}",
                        node.Block.DisplayName);
                }
                else
                {
                    blockDesc = string.Format(
                        "/// <block guid=\"{0}\" displayname=\"{1}\" group=\"{2}\">\n",
                        "{" + node.Block.Guid.ToString().ToUpper() + "}",
                        node.Block.DisplayName,
                        node.Block.Group);
                }
            }

            // マクロかどうか
            string isMacro = "";
            if (node.IsMacro)
            {
                isMacro = " is_macro=\"true\"";
            }

            // アノテーション付き glsl のテンプレートを埋める
            string result = string.Format((
                "/// <tag name=\"{0}\">\n" +
                "///   <func id=\"{1}\"/>\n" +
                "/// </tag>\n" +
                "/// <func id=\"{1}\" name=\"{1}\"{10}>\n" +
                "{6}" +
                "///   <description>{3}</description>\n" +
                "/// </func>\n" +
                "{4}" +
                "///   <func id=\"{1}\"/>\n" +
                "{5}" +
                "{7}" +
                "{8}" +
                "/// </block>\n" +
                "/// <code>\n" +
                "{9}\n" +
                "/// </code>\n"),
                node.TagName,
                node.FunctionName,
                node.ReturnType.ToStringExt(),
                node.Description,
                blockDesc,
                ioDesc,
                funcIoDesc,
                previewDesc,
                uniformDesc,
                node.ShaderCode,
                isMacro);

            // 改行コードを "\r\n" にしておく
            // 既にある分を一旦戻してから全置換
            // TODO: ポリシー決める
            result = result.Replace("\r\n", "\n").Replace("\n", "\r\n");

            return result;
        }

        public static ObservableCollection<NodeDesc> LoadNodeDesc(string xmlText, ref string noConvertCode)
        {
            ObservableCollection<NodeDesc> desc = new ObservableCollection<NodeDesc>();

            XmlDocument doc = new XmlDocument();
            doc.LoadXml(xmlText);

            var trimmed = new char[] { '\r', '\n', '\t', ' ' };
            XmlNode node = doc.DocumentElement.FirstChild;
            while (node != null)
            {
                // NoConvert は読み飛ばす
                if (node.Name == "tag")
                {
                    // MEMO: ノード定義一つ一つを今は XML タグで囲んでいない。
                    //       tag, func, block, code で1セット、という事でパースしておく
                    var tagNode = node;
                    var funcNode = tagNode.NextSibling;
                    var blockNode = funcNode.NextSibling;
                    var codeNode = blockNode.NextSibling;

                    NodeDesc newNode = new NodeDesc();
                    newNode.ReturnType = ValueType.Void; // <return> タグが無ければ void になる

                    // <tag>
                    {
                        var elem = (XmlElement)tagNode;
                        newNode.TagName = elem.GetAttribute("name");
                        elem = (XmlElement)tagNode.FirstChild;  // <func>
                        newNode.FunctionName = elem.GetAttribute("id"); // <func id="***">
                    }

                    // <func>
                    {
                        // マクロかどうか
                        // MEMO: 今のところ、CombinerNodeEditor のテンプレート生成の補助にしか使わない
                        var elem = (XmlElement)funcNode;
                        newNode.IsMacro = (elem.GetAttribute("is_macro").ToLower() == "true");

                        // 返り値と説明
                        var n = funcNode.FirstChild;
                        while (n != null)
                        {
                            // <description> <return> 以外は <block> の方でまとめて取得できるので無視
                            if (n.Name == "description")
                            {
                                newNode.Description = n.InnerText;
                            }
                            else if (n.Name == "return")
                            {
                                var e = (XmlElement)n;
                                newNode.ReturnType = ValueTypeExt.FromString(e.GetAttribute("type"));
                            }
                            n = n.NextSibling;
                        }
                    }

                    // <block>
                    {
                        var elem = (XmlElement)blockNode;
                        newNode.Block.DisplayName = elem.GetAttribute("displayname");
                        newNode.Block.Group = elem.GetAttribute("group");
                        Guid guid = Guid.Empty;
                        Guid.TryParse(elem.GetAttribute("guid"), out guid);

                        if (guid != Guid.Empty)
                        {
                            newNode.Block.Guid = guid;
                        }
                        var n = blockNode.FirstChild;
                        while (n != null)
                        {
                            // <func> は <tag> で取得済みなので無視
                            switch (n.Name)
                            {
                                case "in":
                                    {
                                        InputDesc input = new InputDesc(newNode);
                                        var e = (XmlElement)n;
                                        input.DisplayName = e.GetAttribute("displayname");
                                        input.Type = ValueTypeExt.FromString(e.GetAttribute("type"));
                                        input.ArgTarget = e.GetAttribute("target");
                                        input.Semantics = e.GetAttribute("semantic");
                                        newNode.AddInputDesc(input);
                                    }
                                    break;
                                case "out":
                                    {
                                        OutputDesc output = new OutputDesc(newNode);
                                        var e = (XmlElement)n;
                                        output.DisplayName = e.GetAttribute("displayname");
                                        output.Type = ValueTypeExt.FromString(e.GetAttribute("type"));
                                        output.ReturnTarget = e.GetAttribute("target");
                                        output.Semantics = e.GetAttribute("semantic");
                                        newNode.AddOutputDesc(output);
                                    }
                                    break;
                                case "preview":
                                    {
                                        Preview preview = new Preview();
                                        var e = (XmlElement)n;
                                        preview.IsEnabled = bool.Parse(e.GetAttribute("enabled"));
                                        preview.Width = float.Parse(e.GetAttribute("width"));
                                        preview.Height = float.Parse(e.GetAttribute("height"));
                                        newNode.Preview = preview;
                                    }
                                    break;
                                case "uniform":
                                    {
                                        Uniform uniform = new Uniform();
                                        var e = (XmlElement)n;
                                        uniform.Name = e.GetAttribute("name");
                                        uniform.Type = UniformTypeExt.FromString(e.GetAttribute("type"));
                                        uniform.DisplayName = e.GetAttribute("displayname");
                                        var extStr = e.GetAttribute("extension");
                                        if (!string.IsNullOrEmpty(extStr))
                                        {
                                            uniform.SetExtensionEnable(UniformExtTypeExt.FromString(extStr), true);
                                        }

                                        newNode.Uniform = uniform;
                                    }
                                    break;
                            }
                            n = n.NextSibling;
                        }
                    }

                    // <code>
                    {
                        newNode.ShaderCode = codeNode.InnerText.TrimStart(trimmed).TrimEnd(trimmed);
                    }
                    desc.Add(newNode);
                    node = codeNode;
                }
                else if (node.Name == "NoConvert")
                {
                    // ノード定義以外の glsl コード。
                    noConvertCode += node.InnerText.TrimStart(trimmed).TrimEnd(trimmed);
                }
                node = node.NextSibling;
            }

            return desc;
        }

        public static string SaveNodeDesc(ObservableCollection<NodeDesc> desc)
        {
            return "";
        }

    }

    public class FilePathToFileNameConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            if (value is string)
            {
                FileInfo info = new FileInfo((string)value);
                return info.Name;
            }
            // MEMO: 初期化時などは value に null が流れてくるので、その時に処理を止めずに UI を空欄にする
            return value;
        }

        public object ConvertBack(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            throw new NotImplementedException();
        }
    }

    public class ShaderCodePreviewConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            if (value is string)
            {
                string[] s = ((string)value).Split(new string[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);
                string result = "";
                for (int i = 0; i < Math.Min(3, s.Length); ++i)
                {
                    result += s[i] + "\n";
                }
                result = result.TrimStart(' ').TrimEnd('\n');
                return result;
            }
            // MEMO: 初期化時などは value に null が流れてくるので、その時に処理を止めずに UI を空欄にする
            return value;
        }

        public object ConvertBack(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            throw new NotImplementedException();
        }
    }

}
