﻿/// <tag name="Utility">
///  <func id="3ColorInterporation"/>
/// </tag>
/// <func id="3ColorInterporation" name="Get3ColorInterporation">
///  <in name="color0" type="vec3"/>
///  <in name="color1" type="vec3"/>
///  <in name="color2" type="vec3"/>
///  <in name="pos" type="float"/>
///  <return type="vec3"/>
///  <description>3色補間を行います</description>
/// </func>
/// <block guid="{48922680-4DFC-4e94-95E5-161A93743A9F}" displayname="3ColorInterporation">
///  <func id="3ColorInterporation"/>
///  <in displayname="color0[0.0]" type="vec3" target="color0"/>
///  <in displayname="color1[0.5]" type="vec3" target="color1"/>
///  <in displayname="color2[1.0]" type="vec3" target="color2"/>
///  <in displayname="pos" type="float" target="pos"/>
///  <out displayname="RGB" type="vec3" target="return"/>
/// </block>
/// <code>
vec3 Get3ColorInterporation( vec3 color0, vec3 color1, vec3 color2, float pos )
{
  // 三色補間
  float t0 = clamp( 1 - pos * 2, 0.0, 1.0 );
  float t1 = 1 - abs( pos * 2 - 1 );
  float t2 = clamp( pos * 2 - 1, 0.0, 1.0 );
  return color0 * t0 + color1 * t1 + color2 * t2;
}
/// </code>

/// <tag name="Utility">
///  <func id="3ValueInterporation"/>
/// </tag>
/// <func id="3ValueInterporation" name="Get3ValueInterporation">
///  <in name="value0" type="float"/>
///  <in name="value1" type="float"/>
///  <in name="value2" type="float"/>
///  <in name="pos" type="float"/>
///  <return type="float"/>
///  <description>3要素の補間を行います</description>
/// </func>
/// <block guid="{CBE4DFCC-2362-47BA-A0A0-4108D1D937D7}" displayname="3ValueInterporation">
///  <func id="3ValueInterporation"/>
///  <in displayname="value0[0.0]" type="float" target="value0"/>
///  <in displayname="value1[0.5]" type="float" target="value1"/>
///  <in displayname="value2[1.0]" type="float" target="value2"/>
///  <in displayname="pos" type="float" target="pos"/>
///  <out displayname="return" type="float" target="return"/>
/// </block>
/// <code>
float Get3ValueInterporation( float value0, float value1, float value2, float pos )
{
  // 3要素補間
  float t0 = clamp( 1 - pos * 2, 0.0, 1.0 );
  float t1 = 1 - abs( pos * 2 - 1 );
  float t2 = clamp( pos * 2 - 1, 0.0, 1.0 );
  return value0 * t0 + value1 * t1 + value2 * t2;
}
/// </code>


/// <tag name="Utility">
///  <func id="Brightness"/>
/// </tag>
/// <func id="Brightness" name="Brightness">
///  <in name="color" type="vec3"/>
///  <return type="float"/>
///  <description>輝度成分を出力</description>
/// </func>
/// <block guid="{48922680-4DFC-4e94-85E5-150A93743A9F}" displayname="Brightness">
///  <func id="Brightness"/>
///  <in displayname="RGB" type="vec3" target="color"/>
///  <out displayname="return[0,1]" type="float" target="return"/>
/// </block>
/// <code>
float Brightness( vec3 color )
{
  return  (color.r * 0.298912) +
          (color.g * 0.586611) +
          (color.b * 0.114478) ;
}
/// </code>


/// <tag name="Utility">
///  <func id="get_f0"/>
/// </tag>
/// <func id="get_f0" name="getF0">
///  <return type="float"/>
///  <description>フレネル係数を出力</description>
/// </func>
/// <block guid="{EA871B8B-7912-4afd-B01C-48DFF00B0CE2}" displayname="Fresnel coeff">
///  <func id="get_f0"/>
///  <out displayname="value" type="float" target="return"/>
/// </block>
/// <code>
float getF0()
{
  return 0.02;
}
/// </code>


/// <tag name="Utility">
///  <func id="get_fresnel_term"/>
/// </tag>
/// <func id="get_fresnel_term" name="getFresnelTerm">
///  <in name="F0" type="float"/>
///  <in name="H" type="vec3"/>
///  <in name="V" type="vec3"/>
///  <return type="float"/>
///  <description>フレネル項を計算</description>
/// </func>
/// <block guid="{3F6C93B3-B9A7-409a-94C8-5DCA7F4A38A2}" displayname="Fresnel term">
///  <func id="get_fresnel_term"/>
///  <in displayname="F0" type="float" target="F0"/>
///  <in displayname="H" type="vec3" target="H"/>
///  <in displayname="V" type="vec3" target="V"/>
///  <out displayname="value" type="float" target="return"/>
/// </block>
/// <code>
float getFresnelTerm(float F0, vec3 H, vec3 V)
{
  return F0 + (1. - F0) * pow(1. - dot(H, V), 5.);
}
/// </code>


/// <tag name="Utility">
///   <func id="GetEM_Fresnel" />
/// </tag>
/// <func id="GetEM_Fresnel" name="GetEM_Fresnel">
///   <return type="float" />
///   <in name="dir" type="vec3" />
///   <in name="normal" type="vec3" />
///   <description>EffectMaker と同等のフレネルアルファです。（厳密なフレネル実装とは異なります）</description>
/// </func>
/// <block guid="{0A72096D-4C01-4CBA-822D-8FA255A3306B}" displayname="EM_Fresnel">
///   <func id="GetEM_Fresnel"/>
///   <in displayname="dir" type="vec3" target="dir" />
///   <in displayname="normal" type="vec3" target="normal" />
///   <out displayname="alpha" type="float" target="return" />
/// </block>
/// <code>
float GetEM_Fresnel(vec3 dir, vec3 normal)
{
  return CalcFresnelRate( dir, normal );
}
/// </code>


/// <tag name="Utility">
///  <func id="LanbertReflectionModel"/>
/// </tag>
/// <func id="LanbertReflectionModel" name="LanbertReflectionModel">
///  <in name="lightColor" type="vec3"/>
///  <in name="lightDir" type="vec3"/>
///  <in name="albedo" type="vec3"/>
///  <in name="norm" type="vec3"/>
///  <return type="vec3"/>
///  <description>ランバート拡散反射を計算</description>
/// </func>
/// <block guid="{3F6C93B3-BCC9-419a-95C8-5DCA7F4A38A2}" displayname="LanbertReflection">
///  <func id="LanbertReflectionModel"/>
///  <in displayname="Light Color" type="vec3" target="lightColor"/>
///  <in displayname="Light Dir" type="vec3" target="lightDir"/>
///  <in displayname="Albedo" type="vec3" target="albedo"/>
///  <in displayname="Normal" type="vec3" target="norm"/>
///  <out displayname="RGB" type="vec3" target="return"/>
/// </block>
/// <code>
vec3 LanbertReflectionModel(vec3 lightColor, vec3 lightDir, vec3 albedo, vec3 norm)
{
  // Lanbert Reflection Model
  lightDir = normalize( -lightDir );
  norm = normalize( norm );

  float dLanbert = clamp( dot( lightDir, norm ), 0.0, 1.0 );
  vec3 cLanbert = vec3(
    albedo.x * lightColor.x,
    albedo.y * lightColor.y,
    albedo.z * lightColor.z );

  return dLanbert * cLanbert;
}
/// </code>


/// <tag name="Utility">
///  <func id="PhongReflectionModel"/>
/// </tag>
/// <func id="PhongReflectionModel" name="PhongReflectionModel">
///  <in name="lightColor" type="vec3"/>
///  <in name="reflectColor" type="vec3"/>
///  <in name="lightDir" type="vec3"/>
///  <in name="lookDir" type="vec3"/>
///  <in name="norm" type="vec3"/>
///  <in name="phongCoeff" type="float"/>
///  <return type="vec3"/>
///  <description>Phongの鏡面反射を計算</description>
/// </func>
/// <block guid="{3F6C93B3-BBB9-409a-94C8-5DCA7F4A38A2}" displayname="PhongReflection">
///  <func id="PhongReflectionModel"/>
///  <in displayname="Light Color" type="vec3" target="lightColor"/>
///  <in displayname="Reflect Color" type="vec3" target="reflectColor"/>
///  <in displayname="Light Dir" type="vec3" target="lightDir"/>
///  <in displayname="Look Dir" type="vec3" target="lookDir"/>
///  <in displayname="Normal" type="vec3" target="norm"/>
///  <in displayname="Phong Coeff" type="float" target="phongCoeff"/>
///  <out displayname="RGB" type="vec3" target="return"/>
/// </block>
/// <code>
vec3 PhongReflectionModel(vec3 lightColor, vec3 reflectColor, vec3 lightDir, vec3 lookDir, vec3 norm, float phongCoeff)
{
  // Phong Reflection Model

  lightDir = normalize( lightDir );
  lookDir = normalize( -lookDir );
  vec3 reflection;
  {
    // I - 2.0 * dot(N, I) * N
    float d = dot( norm, lightDir );
    // Ignore reverse side
    if( d < 0 ){ reflection = lightDir - 2 * d * norm; }
    else{ reflection = vec3( 0, 0, 0 ); }
  }

  float dPhong = clamp( dot( reflection, lookDir ), 0, 1 );
  dPhong = pow( dPhong, phongCoeff );

  return lightColor * reflectColor * dPhong;
}
/// </code>


/// <tag name="Utility">
///  <func id="Blinn_PhongReflectionModel"/>
/// </tag>
/// <func id="Blinn_PhongReflectionModel" name="Blinn_PhongReflectionModel">
///  <in name="lightColor" type="vec3"/>
///  <in name="reflectColor" type="vec3"/>
///  <in name="lightDir" type="vec3"/>
///  <in name="lookDir" type="vec3"/>
///  <in name="phongCoeff" type="float"/>
///  <return type="vec3"/>
///  <description>Blinn-Phongの鏡面反射を計算</description>
/// </func>
/// <block guid="{3F6C93B3-BCB9-510b-95C8-5DCA7F4A38A2}" displayname="Blinn_PhongReflection">
///  <func id="Blinn_PhongReflectionModel"/>
///  <in displayname="Light Color" type="vec3" target="lightColor"/>
///  <in displayname="Reflect Color" type="vec3" target="reflectColor"/>
///  <in displayname="Light Dir" type="vec3" target="lightDir"/>
///  <in displayname="Look Dir" type="vec3" target="lookDir"/>
///  <in displayname="Phong Coeff" type="float" target="phongCoeff"/>
///  <out displayname="RGB" type="vec3" target="return"/>
/// </block>
/// <code>
vec3 Blinn_PhongReflectionModel(vec3 lightColor, vec3 reflectColor, vec3 lightDir, vec3 lookDir, float phongCoeff)
{
  // Blinn-Phong Reflection Model

  lightDir = normalize( lightDir );
  lookDir = normalize( lookDir );
  vec3 v = lightDir + lookDir;
  vec3 h = v / length( v );
  float dPhong = clamp( dot( h, lookDir ), 0, 1 );
  dPhong = pow( dPhong, phongCoeff );

  return lightColor * reflectColor * dPhong;
}
/// </code>
