﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Workflow.Core
{
    public static class Utility
    {
        public static readonly IEqualityComparer<IUniquelyIdentifiable> UniquelyIdentifiableEqualityComparer = new UniquelyIdentifiableEqualityComparer();

        /// <summary>
        /// Throws an InputPlugException to tell that the InputPlug is already 'attached' to a WorkflowItem.
        /// </summary>
        /// <param name="inputPlug">The InputPlug that is already 'attached' to a WorkflowItem.</param>
        public static void ThrowInputAlreadyAttachedException<T>(InputPlug<T> inputPlug)
        {
            var message = string.Format(Messages.EXCEPTION_INPUT_PLUG_ALREADY_ATTACHED,
                inputPlug.Identifier, inputPlug.Name, inputPlug.WorkflowItem.Identifier, inputPlug.WorkflowItem.Name);

            throw new InputPlugException<T>(inputPlug, message);
        }

        /// <summary>
        /// Throws an OutputPlugException to tell that the OutputPlug is already 'attached' to a WorkflowItem.
        /// </summary>
        /// <param name="outputPlug">The OutputPlug that is already 'attached' to a WorkflowItem.</param>
        public static void ThrowOutputAlreadyAttachedException<T>(OutputPlug<T> outputPlug)
        {
            var message = string.Format(Messages.EXCEPTION_OUTPUT_PLUG_ALREADY_ATTACHED,
                outputPlug.Identifier, outputPlug.Name, outputPlug.WorkflowItem.Identifier, outputPlug.WorkflowItem.Name);

            throw new OutputPlugException<T>(outputPlug, message);
        }

        /// <summary>
        /// Throws an InputPlugException to tell that the InputPlug is not 'attached' to any WorkflowItem.
        /// </summary>
        /// <param name="inputPlug">The InputPlug that is not 'attached' to any WorkflowItem.</param>
        public static void ThrowInputNotAttachedException<T>(InputPlug<T> inputPlug)
        {
            var message = string.Format(Messages.EXCEPTION_INPUT_PLUG_NOT_ATTACHED,
                inputPlug.Identifier, inputPlug.Name);

            throw new InputPlugException<T>(inputPlug, message);
        }

        /// <summary>
        /// Throws an OutputPlugException to tell that the OutputPlug is not 'attached' to any WorkflowItem.
        /// </summary>
        /// <param name="outputPlug">The OutputPlug that is not 'attached' to any WorkflowItem.</param>
        public static void ThrowOutputNotAttachedException<T>(OutputPlug<T> outputPlug)
        {
            var message = string.Format(Messages.EXCEPTION_OUTPUT_PLUG_NOT_ATTACHED,
                outputPlug.Identifier, outputPlug.Name);

            throw new OutputPlugException<T>(outputPlug, message);
        }

        /// <summary>
        /// Throws an InvalidOperationException to tell that both InputPlug and OutputPlug cannot be connected together.
        /// The reason is that they are both connected to the same WorkflowItem, and thus create a implicit circular reference.
        /// </summary>
        /// <param name="inputPlug">The InputPlug that is connected </param>
        /// <param name="outputPlug"></param>
        public static void ThrowSelfConnectionException<T>(InputPlug<T> inputPlug, OutputPlug<T> outputPlug)
        {
            var message = string.Format(Messages.EXCEPTION_SELF_CONNECTION,
                inputPlug.Identifier,
                inputPlug.Name,
                outputPlug.Identifier,
                outputPlug.Name,
                inputPlug.WorkflowItem.Identifier,
                inputPlug.WorkflowItem.Name);

            throw new InvalidOperationException(message);
        }

        /// <summary>
        /// Throws an InputPlugException to tell that the InputPlug is already 'plugged' to an OutputPlug.
        /// </summary>
        /// <param name="inputPlug">The InputPlug that is already 'plugged' to an OutputPlug.</param>
        public static void ThrowInputAlreadyPluggedException<T>(InputPlug<T> inputPlug)
        {
            var message = string.Format(Messages.EXCEPTION_INPUT_ALREADY_PLUGGED_TO_OUTPUT,
                inputPlug.Identifier, inputPlug.Name, inputPlug.RemoteOutputPlug.Identifier, inputPlug.RemoteOutputPlug.Name);

            throw new InputPlugException<T>(inputPlug, message);
        }
    }
}
