﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using ShaderTyping;
using ShaderGenerator.GLSL;

namespace ShaderGeneratorTests
{
    /*
    [TestClass]
    public class GeneratorTests
    {
        private static void CheckOutputs(PlugValue[] outputs, params PlugValue[] expectedOutputs)
        {
            if (expectedOutputs.Length != outputs.Length)
                throw new Exception(string.Format("ProduceOutputs should return {0} plugs for block but returned: {1}", expectedOutputs.Length, outputs.Length));

            for (int i = 0; i < expectedOutputs.Length; ++i)
            {
                if (expectedOutputs[i].Type != outputs[i].Type ||
                    expectedOutputs[i].Name != outputs[i].Name)
                    throw new Exception(string.Format("ProduceOutputs returned element #{0} be [{1}] but is [{2}]", i, expectedOutputs[i], outputs[i]));
            }
        }

        private static void CheckOutputs(string id, PlugValue[] outputs, params PlugValue[] expectedOutputs)
        {
            if (expectedOutputs.Length != outputs.Length)
                throw new Exception(string.Format("ProduceOutputs should return {1} plugs for block {0} but returned: {2}", id, expectedOutputs.Length, outputs.Length));

            for (int i = 0; i < expectedOutputs.Length; ++i)
            {
                if (expectedOutputs[i].Type != outputs[i].Type ||
                    expectedOutputs[i].Name != outputs[i].Name)
                    throw new Exception(string.Format("ProduceOutputs returned element #{0} be [{1}] but is [{2}]", i, expectedOutputs[i], outputs[i]));
            }
        }

        #region GetPrototype

        [TestMethod]
        public void GetPrototypeNoParamReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");

            var function = new FunctionDefinition("f", voidType);

            var code = function.GetPrototype();
            var expectedCode = "void f();";
            if (code != expectedCode)
                throw new Exception(string.Format("GetPrototype should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetPrototypeNoParamReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");

            var function = new FunctionDefinition("f", intType);

            var code = function.GetPrototype();
            var expectedCode = "int f();";
            if (code != expectedCode)
                throw new Exception(string.Format("GetPrototype should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetPrototypeOneParamReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input)
            };
            var function = new FunctionDefinition("f", intType, parameters);

            var code = function.GetPrototype();
            var expectedCode = "int f(in vec3 foo);";
            if (code != expectedCode)
                throw new Exception(string.Format("GetPrototype should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetPrototypeThreeParamsReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input),
                new ParameterDefinition("bar", vecType, ParameterDirection.Input),
                new ParameterDefinition("baz", vecType, ParameterDirection.Input)
            };
            var function = new FunctionDefinition("f", intType, parameters);

            var code = function.GetPrototype();
            var expectedCode = "int f(in vec3 foo, in vec3 bar, in vec3 baz);";
            if (code != expectedCode)
                throw new Exception(string.Format("GetPrototype should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetPrototypeOneOutputParamsReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Output),
            };
            var function = new FunctionDefinition("f", voidType, parameters);

            var code = function.GetPrototype();
            var expectedCode = "void f(out vec3 foo);";

            if (code != expectedCode)
                throw new Exception(string.Format("GetPrototype should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetPrototypeOneReferenceParamsReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Reference),
            };
            var function = new FunctionDefinition("f", voidType, parameters);

            var code = function.GetPrototype();
            var expectedCode = "void f(inout vec3 foo);";

            if (code != expectedCode)
                throw new Exception(string.Format("GetPrototype should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetPrototypeInputOuputAndReferenceParamsReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var floatType = new EffectDefinitions.ShaderTypeDefinition("float");
            var vec2Type = new EffectDefinitions.ShaderTypeDefinition("vec2");
            var vec3Type = new EffectDefinitions.ShaderTypeDefinition("vec3");
            var vec4Type = new EffectDefinitions.ShaderTypeDefinition("vec4");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vec2Type, ParameterDirection.Input),
                new ParameterDefinition("bar", vec3Type, ParameterDirection.Output),
                new ParameterDefinition("baz", vec4Type, ParameterDirection.Reference),
                new ParameterDefinition("bbq", floatType, ParameterDirection.Input),
            };
            var function = new FunctionDefinition("f", intType, parameters);

            var code = function.GetPrototype();
            var expectedCode = "int f(in vec2 foo, out vec3 bar, inout vec4 baz, in float bbq);";

            if (code != expectedCode)
                throw new Exception(string.Format("GetPrototype should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        #endregion

        #region ProduceOutputs

        [TestMethod]
        public void ProduceOutputsNoParamReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");

            var function = new FunctionDefinition("f", voidType);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);
            var outputs = generator.ProduceOutputs(function, new PlugValue[0]);
            context.EndGeneration();

            CheckOutputs(outputs);

            var code = context.FullShader;
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nvoid f();\r\n\r\nvoid main()\r\n{\r\n    f();\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void ProduceOutputsNoParamReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");

            var function = new FunctionDefinition("f", intType);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);
            var outputs = generator.ProduceOutputs(function, new PlugValue[0]);
            context.EndGeneration();

            CheckOutputs(outputs, new PlugValue(intType, "f_retval_1"));

            var code = context.FullShader;
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nint f();\r\n\r\nvoid main()\r\n{\r\n    int f_retval_1 = f();\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void ProduceOutputsOneParamReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input)
            };
            var function = new FunctionDefinition("f", intType, parameters);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);
            var inputs = new PlugValue[]
            {
                new PlugValue(vecType, "v"),
            };
            var outputs = generator.ProduceOutputs(function, inputs);
            context.EndGeneration();

            CheckOutputs(outputs, new PlugValue(intType, "f_retval_1"));

            var code = context.FullShader;
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nint f(in vec3 foo);\r\n\r\nvoid main()\r\n{\r\n    int f_retval_1 = f(v);\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void ProduceOutputsThreeParamsReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input),
                new ParameterDefinition("bar", vecType, ParameterDirection.Input),
                new ParameterDefinition("baz", vecType, ParameterDirection.Input)
            };
            var function = new FunctionDefinition("f", intType, parameters);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);
            var inputs = new PlugValue[]
            {
                new PlugValue(vecType, "u"),
                new PlugValue(vecType, "v"),
                new PlugValue(vecType, "w"),
            };
            var outputs = generator.ProduceOutputs(function, inputs);
            context.EndGeneration();

            CheckOutputs(outputs, new PlugValue(intType, "f_retval_1"));

            var code = context.FullShader;
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nint f(in vec3 foo, in vec3 bar, in vec3 baz);\r\n\r\nvoid main()\r\n{\r\n    int f_retval_1 = f(u, v, w);\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void ProduceOutputsOneOutputParamsReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Output),
            };
            var function = new FunctionDefinition("f", voidType, parameters);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);
            var outputs = generator.ProduceOutputs(function, new PlugValue[0]);
            context.EndGeneration();

            CheckOutputs(outputs, new PlugValue(vecType, "f_foo_1"));

            var code = context.FullShader;
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nvoid f(out vec3 foo);\r\n\r\nvoid main()\r\n{\r\n    vec3 f_foo_1;\r\n    f(f_foo_1);\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void ProduceOutputsOneReferenceParamsReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Reference),
            };
            var function = new FunctionDefinition("f", voidType, parameters);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);
            var inputs = new PlugValue[]
            {
                new PlugValue(vecType, "v"),
            };
            var outputs = generator.ProduceOutputs(function, inputs);
            context.EndGeneration();

            CheckOutputs(outputs, new PlugValue(vecType, "f_foo_1"));

            var code = context.FullShader;
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nvoid f(inout vec3 foo);\r\n\r\nvoid main()\r\n{\r\n    vec3 f_foo_1 = v;\r\n    f(f_foo_1);\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void ProduceOutputsInputOuputAndReferenceParamsReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var floatType = new EffectDefinitions.ShaderTypeDefinition("float");
            var vec2Type = new EffectDefinitions.ShaderTypeDefinition("vec2");
            var vec3Type = new EffectDefinitions.ShaderTypeDefinition("vec3");
            var vec4Type = new EffectDefinitions.ShaderTypeDefinition("vec4");

            var parameters = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vec2Type, ParameterDirection.Input),
                new ParameterDefinition("bar", vec3Type, ParameterDirection.Output),
                new ParameterDefinition("baz", vec4Type, ParameterDirection.Reference),
                new ParameterDefinition("bbq", floatType, ParameterDirection.Input),
            };
            var function = new FunctionDefinition("f", intType, parameters);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);
            var inputs = new PlugValue[]
            {
                new PlugValue(vec2Type, "uv"),
                new PlugValue(vec4Type, "color"),
                new PlugValue(floatType, "x"),
            };
            var outputs = generator.ProduceOutputs(function, inputs);
            context.EndGeneration();

            CheckOutputs(outputs,
                         new PlugValue(intType, "f_retval_1"),
                         new PlugValue(vec3Type, "f_bar_1"),
                         new PlugValue(vec4Type, "f_baz_1"));

            var code = context.FullShader;
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nint f(in vec2 foo, out vec3 bar, inout vec4 baz, in float bbq);\r\n\r\nvoid main()\r\n{\r\n    vec3 f_bar_1;\r\n    vec4 f_baz_1 = color;\r\n    int f_retval_1 = f(uv, f_bar_1, f_baz_1, x);\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        #endregion

        #region Various blocks

        [TestMethod]
        public void ProduceOutputsNoParamReturnIntAndOneIntParamReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec4");

            var parametersF = new ParameterDefinition[0];
            var parametersG = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input)
            };
            var functionF = new FunctionDefinition("f", vecType, parametersF);
            var functionG = new FunctionDefinition("g", voidType, parametersG);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);
            var outputs1 = generator.ProduceOutputs(functionF, new PlugValue[0]);
            var outputs2 = generator.ProduceOutputs(functionG, outputs1);
            context.EndGeneration();

            CheckOutputs("F", outputs1, new PlugValue(vecType, "f_retval_1"));
            CheckOutputs("G", outputs2);

            var code = context.FullShader;
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nvec4 f();\r\nvoid g(in vec4 foo);\r\n\r\nvoid main()\r\n{\r\n    vec4 f_retval_1 = f();\r\n    g(f_retval_1);\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void ProduceOutputsThreeCascadedBlocks()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec4");

            var parametersF = new ParameterDefinition[0];
            var parametersG = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input)
            };
            var parametersH = new ParameterDefinition[]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input)
            };
            var functionF = new FunctionDefinition("f", vecType, parametersF);
            var functionG = new FunctionDefinition("g", vecType, parametersG);
            var functionH = new FunctionDefinition("h", voidType, parametersH);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);
            var outputs1 = generator.ProduceOutputs(functionF, new PlugValue[0]);
            var outputs2 = generator.ProduceOutputs(functionG, outputs1);
            var outputs3 = generator.ProduceOutputs(functionH, outputs2);
            context.EndGeneration();

            CheckOutputs("F", outputs1, new PlugValue(vecType, "f_retval_1"));
            CheckOutputs("G", outputs2, new PlugValue(vecType, "g_retval_1"));
            CheckOutputs("H", outputs3);

            var code = context.FullShader;
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nvec4 f();\r\nvec4 g(in vec4 foo);\r\nvoid h(in vec4 foo);\r\n\r\nvoid main()\r\n{\r\n    vec4 f_retval_1 = f();\r\n    vec4 g_retval_1 = g(f_retval_1);\r\n    h(g_retval_1);\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void ProduceOutputsCascadedBlocks()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec4");

            var parametersF = new ParameterDefinition[0];
            var parametersG = new ParameterDefinition[1]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input)
            };
            var parametersH = new ParameterDefinition[1]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input)
            };
            var functionF = new FunctionDefinition("f", vecType, parametersF);
            var functionG = new FunctionDefinition("g", vecType, parametersG);
            var functionH = new FunctionDefinition("h", voidType, parametersH);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);

            var outputs1 = generator.ProduceOutputs(functionF, new PlugValue[0]);
            CheckOutputs("F", outputs1, new PlugValue(vecType, "f_retval_1"));

            PlugValue[] outputsG = outputs1;
            var numberOfCalls = 100;
            for (int i = 1; i <= numberOfCalls; ++i)
            {
                outputsG = generator.ProduceOutputs(functionG, outputsG);
                CheckOutputs("G" + i, outputsG, new PlugValue(vecType, "g_retval_" + i));
            }
            var outputs3 = generator.ProduceOutputs(functionH, outputsG);
            CheckOutputs("H", outputs3);

            context.EndGeneration();

            var code = context.FullShader;
            var gCallBlocks = "    vec4 g_retval_1 = g(f_retval_1);\r\n";
            for (int i = 1; i < numberOfCalls; ++i)
            {
                gCallBlocks += "    vec4 g_retval_" + (i + 1) + " = g(g_retval_" + i + ");\r\n";
            }
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nvec4 f();\r\nvec4 g(in vec4 foo);\r\nvoid h(in vec4 foo);\r\n\r\nvoid main()\r\n{\r\n    vec4 f_retval_1 = f();\r\n" + gCallBlocks + "    h(g_retval_" + numberOfCalls + ");\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void ProduceOutputsCrossCascadedBlocks()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec4");
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var floatType = new EffectDefinitions.ShaderTypeDefinition("float");

            var parametersF = new ParameterDefinition[]
            {
                new ParameterDefinition("v", vecType, ParameterDirection.Output),
                new ParameterDefinition("x", floatType, ParameterDirection.Output),
                new ParameterDefinition("i", intType, ParameterDirection.Output),
            };
            var parametersG1 = new ParameterDefinition[]
            {
                new ParameterDefinition("v", vecType, ParameterDirection.Reference),
                new ParameterDefinition("x", floatType, ParameterDirection.Input),
            };
            var parametersG2 = new ParameterDefinition[]
            {
                new ParameterDefinition("v", vecType, ParameterDirection.Reference),
                new ParameterDefinition("i", intType, ParameterDirection.Input),
            };
            var parametersH = new ParameterDefinition[]
            {
                new ParameterDefinition("u", vecType, ParameterDirection.Input),
                new ParameterDefinition("v", vecType, ParameterDirection.Input),
            };
            var functionF = new FunctionDefinition("f", voidType, parametersF);
            var functionG1 = new FunctionDefinition("g1", intType, parametersG1);
            var functionG2 = new FunctionDefinition("g2", floatType, parametersG2);
            var functionH = new FunctionDefinition("h", voidType, parametersH);

            var context = new GlslGenerationContext();
            context.BeginGeneration();
            var generator = new GlslGenerator(context);

            var outputs1 = generator.ProduceOutputs(functionF, new PlugValue[0]);
            CheckOutputs("F", outputs1, new PlugValue(vecType, "f_v_1"), new PlugValue(floatType, "f_x_1"), new PlugValue(intType, "f_i_1"));

            PlugValue[] inputsG1 = new PlugValue[] { outputs1[0], outputs1[1] };
            PlugValue[] inputsG2 = new PlugValue[] { outputs1[0], outputs1[2] };
            PlugValue[] outputsG1 = new PlugValue[2];
            PlugValue[] outputsG2 = new PlugValue[2];
            var numberOfCalls = 100;
            for (int i = 1; i <= numberOfCalls; ++i)
            {
                outputsG1 = generator.ProduceOutputs(functionG1, inputsG1);
                outputsG2 = generator.ProduceOutputs(functionG2, inputsG2);
                CheckOutputs("G1-" + i, outputsG1, new PlugValue(intType, "g1_retval_" + i), new PlugValue(vecType, "g1_v_" + i));
                CheckOutputs("G2-" + i, outputsG2, new PlugValue(floatType, "g2_retval_" + i), new PlugValue(vecType, "g2_v_" + i));
                inputsG1 = new PlugValue[] { outputsG1[1], outputsG2[0] };
                inputsG2 = new PlugValue[] { outputsG2[1], outputsG1[0] };
            }
            var outputs3 = generator.ProduceOutputs(functionH, new PlugValue[] { outputsG1[1], outputsG2[1] });
            CheckOutputs("H", outputs3);

            context.EndGeneration();

            var code = context.FullShader;
            var gCallBlocks = string.Empty;
            string vec1val = "f_v_1";
            string vec2val = "f_v_1";
            string intVal = "f_i_1";
            string floatVal = "f_x_1";
            for (int i = 1; i <= numberOfCalls; ++i)
            {
                var g1retVal = "g1_retval_" + i;
                var g2retVal = "g2_retval_" + i;
                var newVec1val = "g1_v_" + i;
                var newVec2val = "g2_v_" + i;
                gCallBlocks += "    vec4 " + newVec1val + " = " + vec1val + ";\r\n    int " + g1retVal + " = g1(" + newVec1val + ", " + floatVal + ");\r\n";
                gCallBlocks += "    vec4 " + newVec2val + " = " + vec2val + ";\r\n    float " + g2retVal + " = g2(" + newVec2val + ", " + intVal + ");\r\n";
                intVal = g1retVal;
                floatVal = g2retVal;
                vec1val = newVec1val;
                vec2val = newVec2val;
            }
            var expectedCode = "//\r\n// Generated shader code\r\n//\r\n\r\nvoid f(out vec4 v, out float x, out int i);\r\nint g1(inout vec4 v, in float x);\r\nfloat g2(inout vec4 v, in int i);\r\nvoid h(in vec4 u, in vec4 v);\r\n\r\nvoid main()\r\n{\r\n    vec4 f_v_1;\r\n    float f_x_1;\r\n    int f_i_1;\r\n    f(f_v_1, f_x_1, f_i_1);\r\n" + gCallBlocks + "    h(g1_v_" + numberOfCalls + ", g2_v_" + numberOfCalls + ");\r\n}\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("FullShader should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        #endregion
    }
    */
}
