﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using ShaderTyping;
using ShaderGenerator.GLSL;

namespace ShaderGeneratorTests
{
    [TestClass]
    public class FunctionCallTest
    {
        [TestMethod]
        public void GetCallCodeNoParamReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");

            var function = new FunctionDefinition("f", voidType);

            var functionCall = new FunctionCall(function, new PlugValue[0].GroupBy(x => x.BaseTarget).ToArray(), new PlugValue[0]);

            var code = functionCall.GetCallCode();
            var expectedCode = "f();\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("GetCallCode should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetCallCodeNoParamReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");

            var function = new FunctionDefinition("f", intType);

            var outputs = new PlugValue[1] { new PlugValue(intType, "i", "i") };
            var functionCall = new FunctionCall(function, new PlugValue[0].GroupBy(x => x.BaseTarget).ToArray(), outputs);

            var code = functionCall.GetCallCode();
            var expectedCode = "int i = f();\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("GetCallCode should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetCallCodeOneParamReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[1] { new ParameterDefinition("foo", vecType, ParameterDirection.Input) };
            var function = new FunctionDefinition("f", intType, parameters);

            var inputs = new PlugValue[1] { new PlugValue(vecType, "v", "v") }.GroupBy(x => x.BaseTarget).ToArray();
            var outputs = new PlugValue[1] { new PlugValue(intType, "i", "i") };
            var functionCall = new FunctionCall(function, inputs, outputs);

            var code = functionCall.GetCallCode();
            var expectedCode = "int i = f(v);\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("GetCallCode should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetCallCodeThreeParamsReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[3]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Input),
                new ParameterDefinition("bar", vecType, ParameterDirection.Input),
                new ParameterDefinition("baz", vecType, ParameterDirection.Input)
            };
            var function = new FunctionDefinition("f", intType, parameters);

            var inputs = new PlugValue[3]
            {
                new PlugValue(vecType, "u", "u"),
                new PlugValue(vecType, "v", "v"),
                new PlugValue(vecType, "w", "w")
            }.GroupBy(x => x.BaseTarget).ToArray();
            var outputs = new PlugValue[1] { new PlugValue(intType, "i", "i") };
            var functionCall = new FunctionCall(function, inputs, outputs);

            var code = functionCall.GetCallCode();
            var expectedCode = "int i = f(u, v, w);\r\n";
            if (code != expectedCode)
                throw new Exception(string.Format("GetCallCode should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetCallCodeOneOutputParamsReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[1]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Output),
            };
            var function = new FunctionDefinition("f", voidType, parameters);

            var outputs = new PlugValue[1]
            {
                new PlugValue(vecType, "v", "v"),
            };
            var functionCall = new FunctionCall(function, new PlugValue[0].GroupBy(x => x.BaseTarget).ToArray(), outputs);

            var code = functionCall.GetCallCode();
            var expectedCode = "vec3 v;\r\nf(v);\r\n";

            if (code != expectedCode)
                throw new Exception(string.Format("GetCallCode should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetCallCodeOneReferenceParamsReturnVoid()
        {
            var voidType = new EffectDefinitions.ShaderTypeDefinition("void");
            var vecType = new EffectDefinitions.ShaderTypeDefinition("vec3");

            var parameters = new ParameterDefinition[1]
            {
                new ParameterDefinition("foo", vecType, ParameterDirection.Reference),
            };
            var function = new FunctionDefinition("f", voidType, parameters);

            var inputs = new PlugValue[1]
            {
                new PlugValue(vecType, "u", "u"),
            }.GroupBy(x => x.BaseTarget).ToArray();
            var outputs = new PlugValue[1]
            {
                new PlugValue(vecType, "v", "v"),
            };
            var functionCall = new FunctionCall(function, inputs, outputs);

            var code = functionCall.GetCallCode();
            var expectedCode = "vec3 v = u;\r\nf(v);\r\n";

            if (code != expectedCode)
                throw new Exception(string.Format("GetCallCode should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }

        [TestMethod]
        public void GetCallCodeInputOuputAndReferenceParamsReturnInt()
        {
            var intType = new EffectDefinitions.ShaderTypeDefinition("int");
            var floatType = new EffectDefinitions.ShaderTypeDefinition("float");
            var vec2Type = new EffectDefinitions.ShaderTypeDefinition("vec2");
            var vec3Type = new EffectDefinitions.ShaderTypeDefinition("vec3");
            var vec4Type = new EffectDefinitions.ShaderTypeDefinition("vec4");

            var parameters = new ParameterDefinition[4]
            {
                new ParameterDefinition("foo", vec2Type, ParameterDirection.Input),
                new ParameterDefinition("bar", vec3Type, ParameterDirection.Output),
                new ParameterDefinition("baz", vec4Type, ParameterDirection.Reference),
                new ParameterDefinition("bbq", floatType, ParameterDirection.Input),
            };
            var function = new FunctionDefinition("f", intType, parameters);

            var inputs = new PlugValue[3]
            {
                new PlugValue(vec2Type, "uv", "uv"),
                new PlugValue(vec4Type, "color", "color"),
                new PlugValue(floatType, "x", "x"),
            }.GroupBy(x => x.BaseTarget).ToArray();
            var outputs = new PlugValue[3]
            {
                new PlugValue(intType, "i", "i"),
                new PlugValue(vec3Type, "pos", "pos"),
                new PlugValue(vec4Type, "newColor", "newColor"),
            };
            var functionCall = new FunctionCall(function, inputs, outputs);

            var code = functionCall.GetCallCode();
            var expectedCode = "vec3 pos;\r\nvec4 newColor = color;\r\nint i = f(uv, pos, newColor, x);\r\n";

            if (code != expectedCode)
                throw new Exception(string.Format("GetCallCode should return:\r\n{0}\r\nbut returned:\r\n{1}", expectedCode, code));
        }
    }
}
