﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SharpDX.Direct2D1;
using SharpDX.DirectWrite;
using SharpDX;

namespace Renderer2D.Direct2D1
{
    public static class Extensions
    {
        public static Color4 ToD2D1Color(this Core.IColor color)
        {
            if (color == null)
                throw new ArgumentNullException("color");

            return new Color4(color.R / 255.0f, color.G / 255.0f, color.B / 255.0f, color.A / 255.0f);
        }

        public static Vector2 ToPoint2F(this Core.IPoint point)
        {
            if (point == null)
                throw new ArgumentNullException("point");

            return new Vector2((float)point.X, (float)point.Y);
        }

        public static RectangleF ToD2D1Rect(this Core.IRectangle rectangle)
        {
            if (rectangle == null)
                throw new ArgumentNullException("rectangle");

            return new RectangleF((float)rectangle.X, (float)rectangle.Y,
                (float)(rectangle.X + rectangle.Width), (float)(rectangle.Y + rectangle.Height));
        }

        public static DrawTextOptions ToD2D1DrawTextOptions(this Core.DrawTextOptions drawTextOptions)
        {
            switch (drawTextOptions)
            {
                case Core.DrawTextOptions.Clip: return DrawTextOptions.Clip;
                case Core.DrawTextOptions.NoSnap: return DrawTextOptions.NoSnap;
            }
            return DrawTextOptions.None;
        }

        public static FontStretch ToD2D1FontStretch(this Core.FontStretch stretch)
        {
            switch (stretch)
            {
                case Core.FontStretch.Condensed: return FontStretch.Condensed;
                case Core.FontStretch.Expanded: return FontStretch.Expanded;
                case Core.FontStretch.ExtraCondensed: return FontStretch.ExtraCondensed;
                case Core.FontStretch.ExtraExpanded: return FontStretch.ExtraExpanded;
                case Core.FontStretch.Medium: return FontStretch.Medium;
                case Core.FontStretch.SemiCondensed: return FontStretch.SemiCondensed;
                case Core.FontStretch.SemiExpanded: return FontStretch.SemiExpanded;
                case Core.FontStretch.UltraCondensed: return FontStretch.UltraCondensed;
                case Core.FontStretch.UltraExpanded: return FontStretch.UltraExpanded;
            }
            return FontStretch.Normal;
        }

        public static FontStyle ToD2D1Style(this Core.FontStyle style)
        {
            switch (style)
            {
                case Core.FontStyle.Italic: return FontStyle.Italic;
                case Core.FontStyle.Oblique: return FontStyle.Oblique;
            }
            return FontStyle.Normal;
        }

        public static FontWeight ToD2D1Weight(this Core.FontWeight weight)
        {
            switch (weight)
            {
                case Core.FontWeight.Black: return FontWeight.Black;
                case Core.FontWeight.Bold: return FontWeight.Bold;
                case Core.FontWeight.DemiBold: return FontWeight.DemiBold;
                case Core.FontWeight.ExtraBlack: return FontWeight.ExtraBlack;
                case Core.FontWeight.ExtraBold: return FontWeight.ExtraBold;
                case Core.FontWeight.ExtraLight: return FontWeight.ExtraLight;
                case Core.FontWeight.Heavy: return FontWeight.Heavy;
                case Core.FontWeight.Light: return FontWeight.Light;
                case Core.FontWeight.Medium: return FontWeight.Medium;
                case Core.FontWeight.Regular: return FontWeight.Regular;
                case Core.FontWeight.SemiBold: return FontWeight.SemiBold;
                case Core.FontWeight.Thin: return FontWeight.Thin;
                case Core.FontWeight.UltraBlack: return FontWeight.UltraBlack;
                case Core.FontWeight.UltraBold: return FontWeight.UltraBold;
                case Core.FontWeight.UltraLight: return FontWeight.UltraLight;
            }
            return FontWeight.Normal;
        }

        public static bool Equals(this Core.ISize source, Core.ISize other)
        {
            if (source == null)
                throw new ArgumentNullException("source");
            if (other == null)
                throw new ArgumentNullException("other");

            return Math.Abs(source.Width - other.Width) <= double.Epsilon &&
                Math.Abs(source.Height - other.Height) <= double.Epsilon;
        }

        public static Core.IRectangle ToRectangle(this Core.IEllipse ellipse)
        {
            if (ellipse == null)
                throw new ArgumentNullException("ellipse");

            return new Core.Rectangle(
                ellipse.Center.X - ellipse.RadiusX,
                ellipse.Center.Y - ellipse.RadiusY,
                ellipse.RadiusX * 2.0,
                ellipse.RadiusY * 2.0);
        }

        public static Ellipse ToD2D1Ellipse(this Core.IEllipse ellipse)
        {
            if (ellipse == null)
                throw new ArgumentNullException("ellipse");

            return new Ellipse(ellipse.Center.ToPoint2F(),
                (float)ellipse.RadiusX,
                (float)ellipse.RadiusY);
        }

        public static RoundedRectangle ToD2D1RoundedRect(this Core.IRoundedRectangle roundedRectangle)
        {
            if (roundedRectangle == null)
                throw new ArgumentNullException("roundedRectangle");

            return new RoundedRectangle
            {
                Rect = roundedRectangle.ToD2D1Rect(),
                RadiusX = (float)roundedRectangle.RadiusX,
                RadiusY = (float)roundedRectangle.RadiusY,
            };
        }

        public static Core.IMatrix ToGenericMatrix(this Matrix3x2 matrix)
        {
            return new Core.Matrix(
                matrix.M11,
                matrix.M12,
                matrix.M21,
                matrix.M22,
                matrix.M31,
                matrix.M32);
        }

        public static Matrix3x2 ToSpecificMatrix(this Core.IMatrix matrix)
        {
            return new Matrix3x2
            {
                M11 = (float)matrix.M11,
                M12 = (float)matrix.M12,
                M21 = (float)matrix.M21,
                M22 = (float)matrix.M22,
                M31 = (float)matrix.M31,
                M32 = (float)matrix.M32,
            };
        }
    }
}
