﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Collections.ObjectModel;

namespace OperationManager.Core
{
    /// <summary>
    /// アンドゥ・リドゥ可能な履歴のストレージを表すクラスです。
    /// <remarks>このクラスのインスタンスはスレッドセーフです。</remarks>
    /// </summary>
    /// <typeparam name="T">履歴単位に属する型</typeparam>
    public class OperationManager<T> : IOperationManager<T> where T : Operation
    {
        /// <summary>
        /// デフォルトの履歴最大保持数
        /// </summary>
        private const int DefaultMaxOperationsCount = 200;

        /// <summary>
        /// 同期用オブジェクト
        /// </summary>
        private readonly object syncRoot = new object();

        /// <summary>
        /// 履歴の最大保持数
        /// </summary>
        private readonly int maxOperationsCount;

        /// <summary>
        /// 履歴リスト
        /// </summary>
        private readonly List<T> operations;

        /// <summary>
        /// 履歴リストの読み取り用コレクション
        /// </summary>
        private readonly ReadOnlyCollection<T> _readOnlyOperations;

        /// <summary>
        /// 履歴の現在位置を示すインデックス
        /// </summary>
        private int index;

        /// <summary>
        /// アンドゥ可能か
        /// </summary>
        private bool canUndo;

        /// <summary>
        /// リドゥ可能か
        /// </summary>
        private bool canRedo;

        /// <summary>
        /// デフォルトコンストラクタです。
        /// <see cref="DefaultMaxOperationsCount"/>
        /// </summary>
        public OperationManager()
            : this(null)
        {
        }

        /// <summary>
        /// 履歴の最大保持数を指定するコンストラクタです。
        /// <param name="maxCapacity">履歴の最大保持数</param>
        /// </summary>
        public OperationManager(int maxCapacity)
            : this(maxCapacity, null)
        {
        }

        /// <summary>
        /// 既存の履歴を引き継ぐコンストラクタです。
        /// 最大保持数はデフォルト値です。
        /// </summary>
        /// <param name="originalOperations">既存の履歴コレクション</param>
        public OperationManager(IEnumerable<T> originalOperations)
            : this(DefaultMaxOperationsCount, originalOperations)
        {
        }

        /// <summary>
        /// 履歴の最大保持数を指定し、既存の履歴を引き継ぐコンストラクタです。
        /// </summary>
        /// <param name="maxCapacity">履歴の最大保持数</param>
        /// <param name="originalOperaions">既存の履歴コレクション</param>
        public OperationManager(int maxCapacity, IEnumerable<T> originalOperaions)
        {
            if (maxCapacity <= 0)
            {
                throw new ArgumentOutOfRangeException("maxCapacity", string.Format(Messages.EXCEPTION_GREATER_THAN_ZERO, "maxCapacity"));
            }

            this.maxOperationsCount = maxCapacity;
            this.operations = new List<T>(originalOperaions ?? new T[0]);
            this.UpdateIndex();
            this._readOnlyOperations = new ReadOnlyCollection<T>(operations);
        }

        /// <summary>
        /// 履歴のコレクションを取得します。
        /// </summary>
        public IEnumerable<T> Operations
        {
            get { return this._readOnlyOperations; }
        }

        /// <summary>
        /// アンドゥ可能か否かを取得します。
        /// </summary>
        public bool CanUndo
        {
            get { return this.canUndo; }

            private set
            {
                if (this.canUndo == value)
                {
                    return;
                }

                this.canUndo = value;
                this.OnCanUndoChanged();
            }
        }

        /// <summary>
        /// リドゥ可能か否かを取得します。
        /// </summary>
        public bool CanRedo
        {
            get { return this.canRedo; }

            private set
            {
                if (this.canRedo == value)
                {
                    return;
                }

                this.canRedo = value;
                this.OnCanRedoChanged();
            }
        }

        /// <summary>
        /// 履歴リスト内での現在位置を示すインデックスを取得します。
        /// </summary>
        protected int Index
        {
            get { return this.index; }

            private set
            {
                if (this.index == value)
                {
                    return;
                }

                this.index = value;

                // アンドゥ・リドゥの可能状態を更新
                this.EvaluateCanDoAccessors();
            }
        }

        /// <summary>
        /// 履歴が追加された時に発生します。
        /// </summary>
        public event EventHandler<OperationsEventArgs<T>> OperationsAdded;

        /// <summary>
        /// 履歴が破棄された時に発生します。
        /// </summary>
        public event EventHandler<DiscardedOperationsEventArgs<T>> OperationsDiscarded;

        /// <summary>
        /// 履歴がクリアされた時に発生します。
        /// </summary>
        public event EventHandler OperationsCleared;

        /// <summary>
        /// アンドゥが完了した時に発生します。
        /// </summary>
        public event EventHandler<OperationEventArgs<T>> Undone;

        /// <summary>
        /// リドゥが完了した時に発生します。
        /// </summary>
        public event EventHandler<OperationEventArgs<T>> Redone;

        /// <summary>
        /// アンドゥ可能状態が変更された時に発生します。
        /// </summary>
        public event EventHandler CanRedoChanged;

        /// <summary>
        /// リドゥ可能状態が変更された時に発生します。
        /// </summary>
        public event EventHandler CanUndoChanged;

        /// <summary>
        /// 履歴の現在位置を示すGUIDを発行した時に発生します。
        /// </summary>
        public event EventHandler<CurrentGuidAcquiredEventArgs> CurrentGuidAcquired;

        /// <summary>
        /// 履歴を追加します。
        /// </summary>
        /// <param name="unit">追加する履歴単位</param>
        public void Add(T unit)
        {
            this.Add(unit, false);
        }

        /// <summary>
        /// 履歴を追加します。
        /// </summary>
        /// <param name="unit">追加する履歴単位</param>
        /// <param name="runRedo">追加時にリドゥ操作を実行するか否か</param>
        public void Add(T unit, bool runRedo)
        {
            if (unit == null)
            {
                throw new ArgumentNullException("unit");
            }

            this.AddRange(new[] { unit });

            if (runRedo)
            {
                unit.Execute();
            }
        }

        /// <summary>
        /// 複数の履歴を一括で追加します。
        /// </summary>
        /// <param name="operations">追加する履歴単位のコレクション</param>
        public void AddRange(IEnumerable<T> operations)
        {
            if (operations == null)
            {
                throw new ArgumentNullException("operations");
            }

            var cachedUnits = operations.ToArray();
            if (cachedUnits.Length == 0)
            {
                return;
            }

            this.AddRange(cachedUnits);
        }

        /// <summary>
        /// 履歴をクリアします。
        /// </summary>
        public void Clear()
        {
            this.InternalClear();
            this.OnOperationsCleared();
        }

        /// <summary>
        /// アンドゥを実行します。
        /// </summary>
        /// <returns>アンドゥが完了したらtrue,そうでなければfalse.</returns>
        public bool Undo()
        {
            T localOperation;

            lock (this.syncRoot)
            {
                if (this.CanUndo == false)
                {
                    return false;
                }

                // インデックスを戻し
                this.Index--;

                // アンドゥ対象の履歴を取り出し
                var operation = this.operations[this.Index];

                // スレッドセーフ化のためにローカルに退避させて
                localOperation = operation;

                // アンドゥ実行
                operation.Rollback();
            }

            // ロックの外側でイベントを発生させる
            this.OnUndone(new OperationEventArgs<T>(localOperation));

            return true;
        }

        /// <summary>
        /// リドゥを実行します。
        /// </summary>
        /// <returns>リドゥが完了したらtrue,そうでなければfalse.</returns>
        public bool Redo()
        {
            T localOperation;

            lock (this.syncRoot)
            {
                if (this.CanRedo == false)
                {
                    return false;
                }

                // リドゥ対象の履歴を取り出し
                var operation = this.operations[this.Index];

                // スレッドセーフ化のためにローカルに退避させて
                localOperation = operation;

                // インデックスを進めて
                this.Index++;

                // リドゥ実行
                operation.Execute();
            }

            // ロックの外側でイベントを発生させる
            this.OnRedone(new OperationEventArgs<T>(localOperation));

            return true;
        }

        /// <summary>
        /// 履歴の現在位置を示すGUIDを発行します。
        /// </summary>
        /// <returns>履歴の現在位置を示すGUID</returns>
        /// <remarks>空のGUIDを返した場合は、現在位置が履歴の先頭であることを示します。</remarks>
        public Guid AcquireCurrentGuid()
        {
            Guid identifier;

            if (this.CanUndo)
            {
                // アンドゥ可能な場合は現在位置のGUIDを発行する
                lock (this.syncRoot)
                {
                    identifier = this.operations[this.Index - 1].Identifier;
                }
            }
            else
            {
                // 履歴の先頭ならば空のGUIDを発行する
                identifier = Guid.Empty;
            }

            // GUID発行イベントを発生
            this.OnCurrentGuidAcquired(new CurrentGuidAcquiredEventArgs(identifier));

            return identifier;
        }

        /// <summary>
        /// 履歴の現在位置が引数に与えられたGUIDが指す履歴と一致するか否かを返します。
        /// </summary>
        /// <param name="identifier">現在位置に対して一致しているか否か判定したいGUID</param>
        /// <returns>引数が現在位置を指していたらtrue,そうでなければfalse</returns>
        public bool IsCurrentGuid(Guid identifier)
        {
            if (identifier == null)
            {
                throw new ArgumentNullException("identifier");
            }

            if (this.CanUndo)
            {
                Guid guid;
                lock (this.syncRoot)
                {
                    guid = this.operations[this.Index - 1].Identifier;
                }

                return identifier == guid;
            }

            return identifier == Guid.Empty;
        }

        /// <summary>
        /// GUID発行イベントを発生させます。
        /// </summary>
        /// <param name="e">GUID発行イベント</param>
        protected virtual void OnCurrentGuidAcquired(CurrentGuidAcquiredEventArgs e)
        {
            var handler = this.CurrentGuidAcquired;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// 履歴破棄イベントを発生させます。
        /// </summary>
        /// <param name="e"></param>
        protected virtual void OnOperationsDiscarded(DiscardedOperationsEventArgs<T> e)
        {
            var handler = this.OperationsDiscarded;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// 履歴追加イベントを発生させます。
        /// </summary>
        /// <param name="e"></param>
        protected virtual void OnOperationsAdded(OperationsEventArgs<T> e)
        {
            var handler = this.OperationsAdded;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// 履歴クリアイベントを発生させます。
        /// </summary>
        protected virtual void OnOperationsCleared()
        {
            var handler = this.OperationsCleared;
            if (handler != null)
            {
                handler(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// アンドゥ可能状態変更イベントを発生させます。
        /// </summary>
        protected virtual void OnCanUndoChanged()
        {
            var handler = this.CanUndoChanged;
            if (handler != null)
            {
                handler(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// リドゥ可能状態変更イベントを発生させます。
        /// </summary>
        protected virtual void OnCanRedoChanged()
        {
            var handler = this.CanRedoChanged;
            if (handler != null)
            {
                handler(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// アンドゥ完了イベントを発生させます。
        /// </summary>
        /// <param name="e"></param>
        protected void OnUndone(OperationEventArgs<T> e)
        {
            var handler = this.Undone;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// リドゥ完了イベントを発生させます。
        /// </summary>
        /// <param name="e"></param>
        protected void OnRedone(OperationEventArgs<T> e)
        {
            var handler = this.Redone;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// インデックスを更新します。
        /// </summary>
        private void UpdateIndex()
        {
            this.Index = this.operations.Count;
        }

        /// <summary>
        /// アンドゥ・リドゥの可能状態を更新します。
        /// </summary>
        private void EvaluateCanDoAccessors()
        {
            this.CanUndo = this.Index > 0;
            this.CanRedo = this.Index < this.operations.Count;
        }

        /// <summary>
        /// クリア処理の本体です。
        /// </summary>
        private void InternalClear()
        {
            lock (this.syncRoot)
            {
                this.operations.Clear();
                this.UpdateIndex();
                this.OnOperationsCleared();
            }
        }

        /// <summary>
        /// 履歴追加処理の本体です。
        /// </summary>
        /// <param name="addingOperations">追加する履歴単位のコレクション</param>
        private void AddRange(T[] addingOperations)
        {
            lock (this.syncRoot)
            {
                // 履歴追加にあたって削除すべき履歴があれば削除
                this.UnsafeDisbranchedCleanup();

                // 履歴をリストに追加
                this.operations.AddRange(addingOperations);

                // 最大保持数を超えたものを破棄
                T[] discarded = null;
                if (this.operations.Count > this.maxOperationsCount)
                {
                    // 破棄する履歴を抽出
                    discarded = this.operations
                        .Take(this.operations.Count - this.maxOperationsCount)
                        .ToArray();

                    // リストから削除
                    this.operations.RemoveRange(0, this.operations.Count - this.maxOperationsCount);
                }

                // インデックスを更新
                this.UpdateIndex();

                // もし破棄した履歴があれば
                if (discarded != null)
                {
                    // 破棄イベントを発生
                    this.OnOperationsDiscarded(new DiscardedOperationsEventArgs<T>(
                        OperationDiscardType.Overflowed,
                        discarded));
                }

                var added = addingOperations;
                if (addingOperations.Length > this.maxOperationsCount)
                {
                    // 最大保持数を超える数が追加されていたら最大保持数分だけのコレクションを作りなおして
                    added = addingOperations.Skip(addingOperations.Length - this.maxOperationsCount).ToArray();
                }

                // 履歴の追加イベントを発生
                this.OnOperationsAdded(new OperationsEventArgs<T>(added));
            }
        }

        /// <summary>
        /// 履歴の追加時に削除すべき履歴を処理します。
        /// </summary>
        private void UnsafeDisbranchedCleanup()
        {
            if (this.Index < 0)
            {
                // 履歴の先頭までアンドゥされきっている場合は完全なクリアを行う

                // リストの全データをキャッシュし
                var discarded = this.operations.ToArray();

                // クリア処理を行い
                this.InternalClear();

                // キャッシュした配列を使ってイベントを発生
                this.OnOperationsDiscarded(new DiscardedOperationsEventArgs<T>(
                    OperationDiscardType.Disbranched,
                    discarded));
            }
            else if (this.operations.Count - this.Index > 0)
            {
                // 先頭から末尾の間までアンドゥされてる場合は、対象となる区間のみを削除する

                var discardedUnits = this.operations
                    .Skip(this.Index) // 現在位置から
                    .Take(this.operations.Count - this.Index) // 末尾までの個数分を取り出し
                    .ToArray(); // 配列化してキャッシュする

                // リストから削除し
                this.operations.RemoveRange(this.Index, this.operations.Count - this.Index);

                // インデックスを更新して
                this.UpdateIndex();

                // キャッシュした配列を使ってイベントを発生
                this.OnOperationsDiscarded(new DiscardedOperationsEventArgs<T>(
                    OperationDiscardType.Disbranched,
                    discardedUnits));
            }
        }
    }
}
