﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace EffectDefinitions
{
    /// <summary>
    /// Meant to expose the function parameters: it has a name, a type, and optional semantic and default values.
    /// </summary>
    public class Plug : IEquatable<Plug>
    {
        public Plug(string name,
            ShaderTypeDefinition type,
            string target,
            bool isInput,
            string semantic,
            string[] defaultValues)
        {
            if (name == null)
                throw new ArgumentNullException("name");
            if (string.IsNullOrWhiteSpace(name))
                throw new ArgumentException("name");
            if (type.IsVoid)
                throw new ArgumentException(Messages.EXCEPTION_ILLEGAL_PARAMETER_TYPE_VOID, "type");
            if (target == null)
                throw new ArgumentNullException("target");
            if (string.IsNullOrWhiteSpace(target))
                throw new ArgumentException("target");

            Name = name;
            Type = type;
            Target = target;
            BaseTarget = GetTargetBase(target);
            IsInput = isInput;

            if (string.IsNullOrWhiteSpace(semantic) == false)
                Semantic = semantic.Trim();

            DefaultValues = defaultValues;
        }

        public string Name { get; private set; }
        public ShaderTypeDefinition Type { get; private set; }
        public string Target { get; private set; }

        /// <summary>
        /// Target without the fields.
        /// </summary>
        public string BaseTarget { get; private set; }
        public string Semantic { get; private set; }
        public string[] DefaultValues { get; private set; }

        private static string GetTargetBase(string target)
        {
            var i = target.IndexOf('.');
            return i < 0 ? target : target.Substring(0, i);
        }

        public bool IsInput { get; private set; }

        public virtual bool CanPlugTo(Plug dest)
        {
            return (IsInput ^ dest.IsInput  &&
                    this.Type.Equals(dest.Type));
        }

        public virtual bool IsMatchingParameter(ShaderTyping.ParameterDefinition parameter)
        {
            bool typeAreMatching = (Type == parameter.Type);
            bool nameAreSame = (Target == parameter.Name);
            bool nameAreMatching = (BaseTarget == parameter.Name);
            return nameAreMatching && (typeAreMatching || !nameAreSame);
        }

        public bool IsMachingReturnValue(ShaderTyping.ShaderTypeDefinition type)
        {
            return IsMatchingParameter(new ShaderTyping.ParameterDefinition(ParameterDefinition.ReturnParameterString, type, ShaderTyping.ParameterDirection.Output));
        }

        public override string ToString()
        {
            return string.Format("{0} {1} -> {2}{3}", Type, Name, Target, string.IsNullOrEmpty(Semantic) ? string.Empty : " (" + Semantic + ")");
        }

        #region Equals

        public override bool Equals(object obj)
        {
            if (obj is Plug)
                return Equals((Plug)obj);

            return false;
        }

        public virtual bool Equals(Plug other)
        {
            if ((object)other == null)
                return false;

            var defaultValuesEqual = DefaultValues == null && other.DefaultValues == null;
            if (defaultValuesEqual == false)
            {
                if (DefaultValues != null && other.DefaultValues != null)
                    defaultValuesEqual = DefaultValues.SequenceEqual(other.DefaultValues);
            }

            return Name == other.Name &&
                Type == other.Type &&
                Target == other.Target &&
                Semantic == other.Semantic &&
                defaultValuesEqual;
        }

        public override int GetHashCode()
        {
            return (
                Name +
                Type.GetHashCode() +
                Target +
                Semantic +
                (DefaultValues != null
                    ? string.Join(";", DefaultValues)
                    : string.Empty)
                ).GetHashCode();
        }

        public static bool operator ==(Plug t1, Plug t2)
        {
            if ((object)t1 == null)
                return (object)t2 == null;

            return t1.Equals(t2);
        }

        public static bool operator !=(Plug t1, Plug t2)
        {
            return !(t1 == t2);
        }

        #endregion
    }
}
