﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace EffectDefinitions
{
    /// <summary>
    /// Definition of a shader language function in a definitions file: it has a source, a unique id, and some optional documentation.
    /// </summary>
    public class FunctionDefinition : ShaderTyping.FunctionDefinition
    {
        public FunctionDefinition(DefinitionLocation source,
            string id,
            string name,
            ShaderTypeDefinition returnType,
            string returnUsage,
            string returnDescription,
            string description,
            params ParameterDefinition[] parameters)
            : base(name, returnType, parameters)
        {
            if (id == null)
                throw new ArgumentNullException("id");
            if (!Regex.IsMatch(name, Utility.ValidIdentifierRegEx))
                throw new ArgumentException(string.Format(Messages.NOT_VALID_IDENTIFIER, "name"), "name");
            if (string.IsNullOrEmpty(id))
                throw new ArgumentException("id");

            this.Source = source;

            this.Id = id;
            this.ReturnUsage = returnUsage;
            this.ReturnDescription = returnDescription;
            this.Description = description;
        }

        public DefinitionLocation Source { get; internal set; }

        internal delegate IEnumerable<string> TagDelegate(FunctionDefinition function);
        private TagDelegate tagDelegate = delegate { return Enumerable.Empty<string>(); };
        internal void SetTagDelegate(TagDelegate f) { tagDelegate = f; }
        public IEnumerable<string> Tags { get { return tagDelegate(this); } }

        public string Id { get; private set; }
        public string ReturnUsage { get; private set; }
        public string Description { get; private set; }
        public string ReturnDescription { get; private set; }

        public override string ToString()
        {
            return string.Format("{0}: {1}", Id, base.ToString());
        }

        #region Equals

        public override bool Equals(object obj)
        {
            var other = obj as FunctionDefinition;
            return other != null && Equals(other);
        }

        public override bool Equals(ShaderTyping.FunctionDefinition other)
        {
            if ((object)other == null)
                return false;

            if (!(other is FunctionDefinition))
                return false;

            var f = (FunctionDefinition)other;

            return base.Equals(other) &&
                Id == f.Id &&
                ReturnUsage == f.ReturnUsage &&
                Description == f.Description &&
                ReturnDescription == f.ReturnDescription;
        }

        public override int GetHashCode()
        {
            return (base.GetHashCode().ToString() +
                Id +
                ReturnUsage +
                Description +
                ReturnDescription).GetHashCode();
        }

        #endregion
    }
}
