﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Globalization;
using System.Threading;
using System.IO;

namespace EffectCombiner.Core
{
    public class LocalizationManager
    {
        private readonly object syncRoot = new object();
        private readonly List<LocalizationContext> localizationContexts = new List<LocalizationContext>();

        public LocalizationManager()
        {
            CurrentUICulture = CultureInfo.InvariantCulture;
        }

        public IDisposable RegisterLocalization(Action localizeUserInterface)
        {
            if (localizeUserInterface == null)
                throw new ArgumentNullException("localizeUserInterface");

            lock (syncRoot)
            {
                var context = new LocalizationContext
                {
                    SynchronizationContext = SynchronizationContext.Current,
                    Method = localizeUserInterface,
                };

                localizationContexts.Add(context);

                localizeUserInterface();

                return new AnonymousDisposer(() =>
                    {
                        lock (syncRoot)
                            localizationContexts.Remove(context);
                    });
            }
        }

        public CultureInfo CurrentUICulture { get; private set; }

        public bool ChangeLanguage(string cultureName)
        {
            try
            {
                var culture = CultureInfo.GetCultureInfo(cultureName);
                ChangeLanguage(culture);
                return true;
            }
            catch
            {
                return false;
            }
        }

        public void ChangeLanguage(CultureInfo culture)
        {
            if (culture == null)
                throw new ArgumentNullException("culture");

            CurrentUICulture = culture;
            Thread.CurrentThread.CurrentUICulture = CurrentUICulture;

            lock (syncRoot)
            {
                foreach (var context in localizationContexts)
                {
                    var method = context.Method;
                    context.SynchronizationContext.Post(_ => method(), null);
                }
            }

            OnCultureChanged();
        }

        public event EventHandler CultureChanged;

        protected virtual void OnCultureChanged()
        {
            var handler = CultureChanged;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        public CultureInfo[] DetectAvailableLanguages()
        {
            var path = AppDomain.CurrentDomain.BaseDirectory;

            return Directory.GetDirectories(path)
                .Where(d => Directory.GetFiles(d, "*.resources.dll").Any())
                .Select(Path.GetFileName)
                .Select(TryGetCultureInfo)
                .ToArray();
        }

        public static CultureInfo TryGetCultureInfo(string cultureName)
        {
            try
            {
                return CultureInfo.GetCultureInfo(cultureName);
            }
            catch
            {
                return null;
            }
        }

        private class LocalizationContext
        {
            public SynchronizationContext SynchronizationContext;
            public Action Method;
        }
    }
}
