﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Threading;
using System.IO;

namespace EffectCombiner.Core
{
    [Flags]
    public enum DumpType
    {
        Normal = 0x00000000,
        WithDataSegs = 0x00000001,
        WithFullMemory = 0x00000002,
        WithHandleData = 0x00000004,
        FilterMemory = 0x00000008,
        ScanMemory = 0x00000010,
        WithUnloadedModules = 0x00000020,
        WithIndirectlyReferencedMemory = 0x00000040,
        FilterModulePaths = 0x00000080,
        WithProcessThreadData = 0x00000100,
        WithPrivateReadWriteMemory = 0x00000200,
        WithoutOptionalData = 0x00000400,
        WithFullMemoryInfo = 0x00000800,
        WithThreadInfo = 0x00001000,
        WithCodeSegs = 0x00002000,
        WithoutManagedState = 0x00004000,
    };

    public enum ExceptionInfo
    {
        None,
        Present
    }

    //typedef struct _MINIDUMP_EXCEPTION_INFORMATION {
    //    DWORD ThreadId;
    //    PEXCEPTION_POINTERS ExceptionPointers;
    //    BOOL ClientPointers;
    //} MINIDUMP_EXCEPTION_INFORMATION, *PMINIDUMP_EXCEPTION_INFORMATION;
    [StructLayout(LayoutKind.Sequential, Pack = 4)]  // Pack=4 is important! So it works also for x64!
    public struct MiniDumpExceptionInformation
    {
        public uint ThreadId;
        public IntPtr ExceptionPointers;
        [MarshalAs(UnmanagedType.Bool)]
        public bool ClientPointers;
    }

    //BOOL
    //WINAPI
    //MiniDumpWriteDump(
    //    __in HANDLE hProcess,
    //    __in DWORD ProcessId,
    //    __in HANDLE hFile,
    //    __in MINIDUMP_TYPE DumpType,
    //    __in_opt PMINIDUMP_EXCEPTION_INFORMATION ExceptionParam,
    //    __in_opt PMINIDUMP_USER_STREAM_INFORMATION UserStreamParam,
    //    __in_opt PMINIDUMP_CALLBACK_INFORMATION CallbackParam
    //    );

    public static class CoreDump
    {
        // Overload requiring MiniDumpExceptionInformation
        [DllImport("dbghelp.dll", EntryPoint = "MiniDumpWriteDump", CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, ExactSpelling = true, SetLastError = true)]
        private static extern bool MiniDumpWriteDump(IntPtr hProcess, uint processId, SafeHandle hFile, uint dumpType, ref MiniDumpExceptionInformation expParam, IntPtr userStreamParam, IntPtr callbackParam);

        // Overload supporting MiniDumpExceptionInformation == NULL
        [DllImport("dbghelp.dll", EntryPoint = "MiniDumpWriteDump", CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, ExactSpelling = true, SetLastError = true)]
        static extern bool MiniDumpWriteDump(IntPtr hProcess, uint processId, SafeHandle hFile, uint dumpType, IntPtr expParam, IntPtr userStreamParam, IntPtr callbackParam);

        [DllImport("kernel32.dll", EntryPoint = "GetCurrentThreadId", ExactSpelling = true)]
        static extern uint GetCurrentThreadId();

        private static bool Write(SafeHandle fileHandle, DumpType dumpType, ExceptionInfo exceptionInfo)
        {
            var currentProcess = Process.GetCurrentProcess();

            var currentProcessHandle = currentProcess.Handle;
            var currentProcessId = (uint)currentProcess.Id;

            var exp = new MiniDumpExceptionInformation
            {
                ThreadId = GetCurrentThreadId(),
                ClientPointers = false,
                ExceptionPointers = IntPtr.Zero,
            };

            if (exceptionInfo == ExceptionInfo.Present)
                exp.ExceptionPointers = Marshal.GetExceptionPointers();

            if (exp.ExceptionPointers == IntPtr.Zero)
                return MiniDumpWriteDump(currentProcessHandle, currentProcessId, fileHandle, (uint)dumpType, IntPtr.Zero, IntPtr.Zero, IntPtr.Zero);

            return MiniDumpWriteDump(currentProcessHandle, currentProcessId, fileHandle, (uint)dumpType, ref exp, IntPtr.Zero, IntPtr.Zero);
        }

        public static bool Dump(string filename)
        {
            return Dump(filename, DumpType.WithFullMemory);
        }

        public static bool Dump(string filename, DumpType dumpType)
        {
            using (var fs = new FileStream(filename, FileMode.Create, FileAccess.Write, FileShare.Write))
            {
                return Write(fs.SafeFileHandle, dumpType, ExceptionInfo.Present);
            }
        }
    }
}
