﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using EffectDefinitions;

namespace Blocks.Core
{
    public class BlockElementEventArgs : EventArgs
    {
        public BlockElementBase BlockElement { get; private set; }

        public BlockElementEventArgs(BlockElementBase blockElement)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            BlockElement = blockElement;
        }
    }

    public class BlockElementsEventArgs : EventArgs
    {
        public BlockElementBase[] BlockElements { get; private set; }

        /*
        public BlockElementsEventArgs(BlockElement blockElement)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            BlockElements = new[] { blockElement };
        }
        */

        public BlockElementsEventArgs(BlockElementBase[] blockElements)
        {
            if (blockElements == null)
                throw new ArgumentNullException("blockElements");

            BlockElements = blockElements;
        }
    }

    public class BlockElementPositionChangedEventArgs : PositionChangedEventArgs
    {
        public BlockElementBase BlockElement { get; private set; }

        public BlockElementPositionChangedEventArgs(
            BlockElementBase blockElement,
            PositionChangedEventArgs e)
            : this(blockElement, e.OldLeft, e.OldTop, e.NewLeft, e.NewTop)
        {
        }

        public BlockElementPositionChangedEventArgs(
            BlockElementBase blockElement,
            double oldLeft, double oldTop, double newLeft, double newTop)
            : base(oldLeft, oldTop, newLeft, newTop)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            BlockElement = blockElement;
        }
    }

    public class PositionChangedEventArgs : EventArgs
    {
        public double OldLeft { get; private set; }
        public double OldTop { get; private set; }
        public double NewLeft { get; private set; }
        public double NewTop { get; private set; }

        public PositionChangedEventArgs(double oldLeft, double oldTop, double newLeft, double newTop)
        {
            OldLeft = oldLeft;
            OldTop = oldTop;
            NewLeft = newLeft;
            NewTop = newTop;
        }
    }

    public class BlockElementSizeChangedEventArgs : SizeChangedEventArgs
    {
        public BlockElementBase BlockElement { get; private set; }

        public BlockElementSizeChangedEventArgs(
            BlockElementBase blockElement,
            SizeChangedEventArgs e)
            : this(blockElement, e.OldWidth, e.OldHeight, e.NewWidth, e.NewHeight)
        {
        }

        public BlockElementSizeChangedEventArgs(
            BlockElementBase blockElement,
            double oldWidth, double oldHeight, double newWidth, double newHeight)
            : base(oldWidth, oldHeight, newWidth, newHeight)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            BlockElement = blockElement;
        }
    }

    public class SizeChangedEventArgs : EventArgs
    {
        public double OldWidth { get; private set; }
        public double OldHeight { get; private set; }
        public double NewWidth { get; private set; }
        public double NewHeight { get; private set; }

        public SizeChangedEventArgs(double oldWidth, double oldHeight, double newWidth, double newHeight)
        {
            OldWidth = oldWidth;
            OldHeight = oldHeight;
            NewWidth = newWidth;
            NewHeight = newHeight;
        }
    }

    public class BlockElementZOrderChangedEventArgs : ZOrderChangedEventArgs
    {
        public BlockElementBase BlockElement { get; private set; }

        public BlockElementZOrderChangedEventArgs(BlockElementBase blockElement, ZOrderChangedEventArgs e)
            : this(blockElement, e.OldZOrder, e.NewZOrder)
        {
        }

        public BlockElementZOrderChangedEventArgs(BlockElementBase blockElement, int oldZOrder, int newZOrder)
            : base(oldZOrder, newZOrder)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            BlockElement = blockElement;
        }
    }

    public class ZOrderChangedEventArgs : EventArgs
    {
        public int OldZOrder { get; private set; }
        public int NewZOrder { get; private set; }

        public ZOrderChangedEventArgs(int oldZOrder, int newZOrder)
        {
            OldZOrder = oldZOrder;
            NewZOrder = newZOrder;
        }
    }

    /// <summary>
    /// ブロック定義データが変わったときに発生するイベントの情報です。
    /// </summary>
    public class BlockDefinitionChangedEventArgs : EventArgs
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="oldDef">元の定義データ</param>
        /// <param name="newDef">新しい定義データ</param>
        public BlockDefinitionChangedEventArgs(BlockDefinition oldDef, BlockDefinition newDef)
        {
            OldDef = oldDef;
            NewDef = newDef;
        }

        /// <summary>
        /// ブロックエレメントを取得します。
        /// </summary>
        public BlockElementBase BlockElement { get; private set; }

        /// <summary>
        /// 元の定義データを取得します。
        /// </summary>
        public BlockDefinition OldDef { get; private set; }

        /// <summary>
        /// 新しい定義データを取得します。
        /// </summary>
        public BlockDefinition NewDef { get; private set; }
    }

    /// <summary>
    /// ブロックエレメントが置き換わったときに発生するイベントの情報です。
    /// </summary>
    public class BlockElementReplacedEventArgs : EventArgs
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="oldBlockElement">元のブロックエレメント</param>
        /// <param name="newBlockElement">新しいブロックエレメント</param>
        public BlockElementReplacedEventArgs(BlockElementBase oldBlockElement, BlockElementBase newBlockElement)
        {
            this.OldBlockElement = oldBlockElement;
            this.NewBlockElement = newBlockElement;
        }

        /// <summary>
        /// 元のブロックエレメントを取得します。
        /// </summary>
        public BlockElementBase OldBlockElement { get; private set; }

        /// <summary>
        /// 新しいブロックエレメントを取得します。
        /// </summary>
        public BlockElementBase NewBlockElement { get; private set; }
    }
}
