﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Blocks.Core;
using EffectCombiner.Core;
using EffectCombiner.Primitives.Generation;

namespace EffectCombiner.Primitives.Operations
{
    /// <summary>
    /// Uniform変数の値変更コマンドです。
    /// </summary>
    public class UniformValueChangeOperation : OperationBase
    {
        /// <summary>
        /// Uniform変数名です。
        /// </summary>
        private readonly string uniformName;

        /// <summary>
        /// 新しい値です。
        /// </summary>
        private readonly UniformData newValue;

        /// <summary>
        /// 元の値です。
        /// </summary>
        private readonly UniformData oldValue;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="blockManager">ブロックマネージャー</param>
        /// <param name="uniformName">ユーザ定義ブロック</param>
        /// <param name="oldValues">元の値</param>
        /// <param name="newValues">新しい値</param>
        public UniformValueChangeOperation(
            BlockManagerBase blockManager,
            string uniformName,
            UniformData oldValue,
            UniformData newValue)
            : base(OperationType.UniformValueChange, blockManager)
        {
            if (blockManager == null)
                throw new ArgumentNullException("blockManager");
            if (uniformName == null)
                throw new ArgumentNullException("uniformName");
            if (oldValue == null)
                throw new ArgumentNullException("oldValue");
            if (newValue == null)
                throw new ArgumentNullException("newValue");

            this.uniformName = uniformName;
            this.oldValue = (UniformData)oldValue.Clone();
            this.newValue = (UniformData)newValue.Clone();
        }

        /// <summary>
        /// 値の変更をロールバックします。
        /// </summary>
        public override void Rollback()
        {
            Generation.Globals.UniformManager.SetUniformParameter(this.uniformName, this.oldValue);
        }

        /// <summary>
        /// コマンド処理を実行します。
        /// </summary>
        public override void Execute()
        {
            Generation.Globals.UniformManager.SetUniformParameter(this.uniformName, this.newValue);
        }
    }
}
