﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using System.Reflection;
using System.Globalization;
using EffectCombiner.Core;
using EffectCombiner.Primitives.Generation;

namespace EffectCombiner.Primitives.Constant
{
    public static class DataTypeDescriptors
    {
        public static IDataTypeDescriptor[] Descriptors { get; private set; }

        static DataTypeDescriptors()
        {
            var assembly = Assembly.GetExecutingAssembly();

            var itfType = typeof(IDataTypeDescriptor);

            Descriptors = assembly.GetTypes()
                .Where(itfType.IsAssignableFrom)
                .Where(t => t != itfType)
                .Where(t => t.IsGenericType == false)
                .Select(t => (IDataTypeDescriptor)Activator.CreateInstance(t))
                .ToArray();
        }
    }

    public class BooleanTypeDescriptor : IDataTypeDescriptor
    {
        public PrimitiveShaderType DataType
        {
            get { return PrimitiveShaderType.Boolean; }
        }

        public Control ProduceControl()
        {
            return new CheckBox
            {
                AutoSize = true,
                MinimumSize = new System.Drawing.Size(16, 0),
            };
        }

        public string GetValue(Control control)
        {
            return ((CheckBox)control).Checked ? "true" : "false";
        }

        public void SetValue(Control control, string value)
        {
            var v = value == "true";

            ((CheckBox)control).Checked = v;
        }

        public IDisposable AttachHandlers(Control control, EventHandler onChanged, EventHandler onValidated)
        {
            var ctrl = (CheckBox)control;
            ctrl.CheckedChanged += onChanged;
            ctrl.CheckedChanged += onValidated;

            return new AnonymousDisposer(() =>
                {
                    ctrl.CheckedChanged -= onChanged;
                    ctrl.CheckedChanged -= onValidated;
                });
        }
    }

    public class UnsignedIntegerTypeDescriptor : IDataTypeDescriptor
    {
        public PrimitiveShaderType DataType
        {
            get { return PrimitiveShaderType.UnsignedInteger; }
        }

        public Control ProduceControl()
        {
            return new PrimitiveNumericUpDown
            {
                DecimalPlaces = 0,
                Maximum = uint.MaxValue,
                Minimum = uint.MinValue,
                MinimumSize = new System.Drawing.Size(50, 0),
                MaximumSize = new System.Drawing.Size(250, 0),
            };
        }

        public string GetValue(Control control)
        {
            return decimal.ToUInt32(((NumericUpDown)control).Value).ToString(CultureInfo.InvariantCulture);
        }

        public void SetValue(Control control, string value)
        {
            uint v;
            if (uint.TryParse(value, NumberStyles.Integer, CultureInfo.InvariantCulture, out v) == false)
            {
                v = 0;
            }

            ((NumericUpDown)control).Value = new decimal(v);
        }

        public IDisposable AttachHandlers(Control control, EventHandler onChanged, EventHandler onValidated)
        {
            var ctrl = (NumericUpDown)control;
            ctrl.ValueChanged += onChanged;
            ctrl.ValueChanged += onChanged;

            return new AnonymousDisposer(() =>
                {
                    ((NumericUpDown)control).ValueChanged -= onChanged;
                });
        }
    }

    public class IntegerTypeDescriptor : IDataTypeDescriptor
    {
        public PrimitiveShaderType DataType
        {
            get { return PrimitiveShaderType.Integer; }
        }

        public Control ProduceControl()
        {
            return new PrimitiveNumericUpDown
            {
                DecimalPlaces = 0,
                Maximum = int.MaxValue,
                Minimum = int.MinValue,
                MinimumSize = new System.Drawing.Size(50, 0),
                MaximumSize = new System.Drawing.Size(250, 0),
            };
        }

        public string GetValue(Control control)
        {
            return decimal.ToInt32(((NumericUpDown)control).Value).ToString(CultureInfo.InvariantCulture);
        }

        public void SetValue(Control control, string value)
        {
            int v;
            if (int.TryParse(value, NumberStyles.Integer, CultureInfo.InvariantCulture, out v) == false)
            {
                v = 0;
            }

            ((NumericUpDown)control).Value = new decimal(v);
        }

        public IDisposable AttachHandlers(Control control, EventHandler onChanged, EventHandler onValidated)
        {
            var ctrl = (NumericUpDown)control;
            ctrl.ValueChanged += onChanged;
            ctrl.Validated += onValidated;

            return new AnonymousDisposer(() =>
                {
                    ctrl.ValueChanged -= onChanged;
                    ctrl.Validated -= onValidated;
                });
        }
    }

    public class FloatTypeDescriptor : IDataTypeDescriptor
    {
        public PrimitiveShaderType DataType
        {
            get { return PrimitiveShaderType.Float; }
        }

        public Control ProduceControl()
        {
            return new PrimitiveNumericUpDown
            {
                DecimalPlaces = 8,
                Minimum = decimal.MinValue,
                Maximum = decimal.MaxValue,
                MinimumSize = new System.Drawing.Size(50, 0),
                MaximumSize = new System.Drawing.Size(250, 0),
            };
        }

        public string GetValue(Control control)
        {
            var result = decimal.ToSingle(((NumericUpDown)control).Value).ToString(CultureInfo.InvariantCulture);
            if (!result.Contains('.'))
            {
                result += ".0";
            }

            return result;
        }

        public void SetValue(Control control, string value)
        {
            float v;
            if (float.TryParse(value, NumberStyles.Float, CultureInfo.InvariantCulture, out v) == false)
            {
                v = 0.0f;
            }

            ((NumericUpDown)control).Value = new decimal(v);
        }

        public IDisposable AttachHandlers(Control control, EventHandler onChanged, EventHandler onValidated)
        {
            var ctrl = (NumericUpDown)control;
            ctrl.ValueChanged += onChanged;
            ctrl.Validated += onValidated;

            return new AnonymousDisposer(() =>
                {
                    ctrl.ValueChanged -= onChanged;
                    ctrl.Validated -= onValidated;
                });
        }
    }

    public class DoubleTypeDescriptor : IDataTypeDescriptor
    {
        public PrimitiveShaderType DataType
        {
            get { return PrimitiveShaderType.Double; }
        }

        public Control ProduceControl()
        {
            return new PrimitiveNumericUpDown
            {
                DecimalPlaces = 8,
                Minimum = decimal.MinValue,
                Maximum = decimal.MaxValue,
                MinimumSize = new System.Drawing.Size(50, 0),
                MaximumSize = new System.Drawing.Size(250, 0),
            };
        }

        public string GetValue(Control control)
        {
            var result = decimal.ToDouble(((NumericUpDown)control).Value).ToString(CultureInfo.InvariantCulture);
            if (!result.Contains('.'))
            {
                result += ".0";
            }

            return result;
        }

        public void SetValue(Control control, string value)
        {
            double v;
            if (double.TryParse(value, NumberStyles.Float, CultureInfo.InvariantCulture, out v) == false)
            {
                v = 0.0;
            }

            ((NumericUpDown)control).Value = new decimal(v);
        }

        public IDisposable AttachHandlers(Control control, EventHandler onChanged, EventHandler onValidated)
        {
            var ctrl = (NumericUpDown)control;
            ctrl.ValueChanged += onChanged;
            ctrl.Validated += onValidated;

            return new AnonymousDisposer(() =>
                {
                    ctrl.ValueChanged -= onChanged;
                    ctrl.Validated -= onValidated;
                });
        }
    }

    /// <summary>
    /// プリミティブデータ用の NumericUpDown コントロールです。
    /// </summary>
    class PrimitiveNumericUpDown : NumericUpDown
    {
        /// <summary>
        /// 内部に作成されるボタンです。
        /// </summary>
        private Control internalButtons;

        /// <summary>
        /// 内部に作成されるテキストボックスです。
        /// </summary>
        private TextBox internalTextBox;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PrimitiveNumericUpDown()
        {
            internalButtons = this.Controls[0];
            internalTextBox = this.Controls.OfType<TextBox>().First();
        }

        #if false
        /// <summary>
        /// コントロールをクリックしたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnClick(EventArgs e)
        {
            base.OnClick(e);

            Point mousePosition = internalButtons.PointToClient(Control.MousePosition);

            // ボタン以外がクリックされたとき、文字列を全選択状態にする
            if (internalButtons.DisplayRectangle.Contains(MousePosition) == false)
            {
                this.internalTextBox.SelectAll();
            }
        }
        #endif

        /// <summary>
        /// コントロールにエンターしたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            // TAB キーを押してフォーカスが移ったときのため、文字列を全選択状態にする
            this.internalTextBox.SelectAll();
        }
    }
}
