﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Blocks.Core;
using Workflow.Core;
using EffectDefinitions;
using ShaderGenerator.GLSL;

namespace EffectCombiner.Primitives.Generation.Semantic
{
    public static class SemanticHelper
    {
        public static event EventHandler<InputPlugEventArgs<PlugValue>> SemanticCorrespondanceNotFound;
        public static event EventHandler<InputPlugEventArgs<PlugValue>> SemanticOverDefined;

        /// <summary>
        /// 指定されたブロックエレメントのセマンティックを展開します。
        /// </summary>
        /// <remarks>
        /// 指定されたブロックエレメントについて input プラグのセマンティックと同じ
        /// セマンティックの output プラグを持つブロック定義を検索し、見つかったときは
        /// そのゴーストブロックエレメントを作成し、それらのプラグを接続します。
        /// </remarks>
        /// <param name="blockGenerator">BlockGenerator</param>
        /// <param name="availableBlockDefinitions">有効なブロック定義</param>
        /// <param name="blockElementToExpand">処理するブロックエレメント</param>
        /// <returns>作成したゴーストブロックエレメントを返します。</returns>
        public static BlockElementBase[] ExpandSemantics(
            IBlockElementGenerator blockGenerator,
            IEnumerable<BlockDefinition> availableBlockDefinitions,
            EffectBlockElementBase blockElementToExpand)
        {
            if (blockGenerator == null)
                throw new ArgumentNullException("blockGenerator");
            if (blockElementToExpand == null)
                throw new ArgumentNullException("blockElementToExpand");

            var inputPlugsWithSemantic = blockElementToExpand.WorkflowItem.InputPlugs
                .Cast<EffectInputPlug>()
                .Where(p => p.RemoteOutputPlug == null)
                .Where(p => p.BlockDefinitionPlug.Semantic != null);

            var list = new List<BlockElementBase>();

            foreach (var ip in inputPlugsWithSemantic)
            {
                var foundBlockDefs = availableBlockDefinitions
                    .Where(bd => bd.OutputPlugs.Count(p => p.Semantic == ip.BlockDefinitionPlug.Semantic) > 0)
                    .ToArray();

                if (foundBlockDefs.Length == 0)
                {
                    var handler = SemanticCorrespondanceNotFound;
                    if (handler != null)
                        handler(null, new InputPlugEventArgs<PlugValue>(ip));
                    continue;
                }
                if (foundBlockDefs.Length > 1)
                {
                    var handler = SemanticOverDefined;
                    if (handler != null)
                        handler(null, new InputPlugEventArgs<PlugValue>(ip));
                    continue;
                }

                var sourceBlockDef = foundBlockDefs[0];

                var ghostBlockElement = (GhostBlockElement)blockGenerator.CreateBlockElement(new GhostBlockDefinition(sourceBlockDef));
                ghostBlockElement.SetGhostedBlockElement(blockElementToExpand);
                list.Add(ghostBlockElement);

                var index = Array.FindIndex(sourceBlockDef.OutputPlugs,
                    p => p.Semantic == ip.BlockDefinitionPlug.Semantic);

                var outputPlug = (EffectOutputPlug)ghostBlockElement.WorkflowItem.OutputPlugs[index];
                ConnectionManager.Connect(outputPlug, ip);

                // 作成したゴーストブロックエレメントについてもセマンティックの展開を行う
                list.AddRange(ExpandSemantics(blockGenerator, availableBlockDefinitions, ghostBlockElement));
            }

            return list.ToArray();
        }
    }
}
