﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;
using EffectCombiner.Primitives.Generation.Extensions;
using EffectDefinitions;
using Workflow.Core;

namespace EffectCombiner.Primitives.Generation
{
    public class PolymorphicBlockElement : RegularEffectBlockElement
    {
        public PolymorphicBlockElement(BlockDefinition blockDefinition)
            : base(blockDefinition)
        {
            foreach (var inputPlug in WorkflowItem.InputPlugs)
            {
                inputPlug.OutputPlugChanged += InputPlug_OutputConnectionChanged;
            }

            foreach (var outputPlug in WorkflowItem.OutputPlugs)
            {
                outputPlug.InputPlugAdded += OutputPlug_InputConnectionAdded;
                outputPlug.InputPlugRemoved += OutputPlug_InputConnectionRemoved;
            }

            ResetUsableFunctions();
        }

        private void InputPlug_OutputConnectionChanged(object sender, OutputPlugChangedEventArgs<ShaderGenerator.GLSL.PlugValue> e)
        {
            if (e.NewOutputPlug != null)
            {
                if (e.OldOutputPlug == null)
                    EvaluateUsableFunctions(new HashSet<PolymorphicBlockElement>(), true);
                else
                {
                    EvaluateUsableFunctions(new HashSet<PolymorphicBlockElement>(), false);
                    EvaluateUsableFunctions(new HashSet<PolymorphicBlockElement>(), true);
                }
            }
            else
            {
                // no need to test old value because both new and old set to null is an impossible case
                EvaluateUsableFunctions(new HashSet<PolymorphicBlockElement>(), false);
            }
        }

        private void OutputPlug_InputConnectionAdded(object sender, InputPlugEventArgs<ShaderGenerator.GLSL.PlugValue> e)
        {
            EvaluateUsableFunctions(new HashSet<PolymorphicBlockElement>(), true);
        }

        private void OutputPlug_InputConnectionRemoved(object sender, InputPlugEventArgs<ShaderGenerator.GLSL.PlugValue> e)
        {
            EvaluateUsableFunctions(new HashSet<PolymorphicBlockElement>(), false);
        }

        public FunctionDefinition[] UsableFunctionDefinitions { get; private set; }

        internal void ResetUsableFunctions()
        {
            UsableFunctionDefinitions = BlockDefinition.GetBoundFunctionDefinitions().ToArray();
        }

        public void EvaluateUsableFunctions(ISet<PolymorphicBlockElement> knownBlocks, bool isConnecting)
        {
            if (knownBlocks.Add(this) == false)
                return;

            // reset the possibly usable function list to all functions
            ResetUsableFunctions();

            // get all the input remote polymorphic blocks of the current block
            var remoteInputs = WorkflowItem.GetInputWorkflowItems()
                .Where(wi => wi != null)
                .Cast<EffectWorkflowItem>()
                .Select(w => w.BlockElement)
                .OfType<PolymorphicBlockElement>()
                .ToArray();

            // get all the output remote polymorphic blocks of the current block
            var remoteOutputs = WorkflowItem.GetOutputWorkflowItems()
                .Where(wi => wi != null)
                .Cast<EffectWorkflowItem>()
                .Select(w => w.BlockElement)
                .OfType<PolymorphicBlockElement>()
                .ToArray();

            // when disconnecting, evaluate children before self evaluation
            if (isConnecting == false)
            {
                foreach (var remoteInputBlock in remoteInputs)
                    remoteInputBlock.EvaluateUsableFunctions(knownBlocks, isConnecting);

                foreach (var remoteOutputBlock in remoteOutputs)
                    remoteOutputBlock.EvaluateUsableFunctions(knownBlocks, isConnecting);
            }

            // retrieve information about the type of the connected plugs
            var connectionTypeInfo = ConnectionsTypeInfo.FromWorkflowItem(WorkflowItem);

            // retrieve available function definitions
            var functions = BlockDefinition.GetBoundFunctionDefinitions().ToArray();

            // ===========================================================================================
            var i = 0;

            // iterate over input types
            foreach (var t in connectionTypeInfo.InputTypes)
            {
                // we care only non-null polymorphic types
                if (t != null && t.Equals(Globals.PolymorphicType))
                {
                    // retrieve remote output plug for each processed input plug
                    var remotePlug = WorkflowItem.InputPlugs[i].RemoteOutputPlug;
                    if (remotePlug != null)
                    {
                        var remoteIndex = remotePlug.Index;
                        var remoteBlock = (PolymorphicBlockElement)((EffectWorkflowItem)remotePlug.WorkflowItem).BlockElement;

                        // keep all functions where type of input parameter matches the type of output parameter
                        functions = functions.Where(fd =>
                            {
                                // get type of input parameter
                                var inputParameterType = fd.GetInputParameterTypes()[i];
                                return remoteBlock.UsableFunctionDefinitions.Any(rfd =>
                                    {
                                        // get type of output parameter
                                        var outputParameterType = rfd.GetOutputParameterTypes()[remoteIndex];
                                        // tells whether input and output types are matching or not
                                        return outputParameterType.Equals(inputParameterType);
                                    });
                            })
                            .ToArray(); // run and cache it now
                    }
                }
                // increment input parameter index
                i++;
            }

            // reset index
            i = 0;

            // iterate over output types
            foreach (var t in connectionTypeInfo.OutputTypes)
            {
                // we care only non-null polymorphic types
                if (t != null && t.Equals(Globals.PolymorphicType))
                {
                    if (WorkflowItem.OutputPlugs[i].RemoteInputPlugs.Any())
                    {
                        // retrieve remote input plug for each processed output plug
                        // the first input is enough because all other inputs must be of same type
                        var remotePlug = WorkflowItem.OutputPlugs[i].RemoteInputPlugs[0];
                        if (remotePlug != null)
                        {
                            var remoteIndex = remotePlug.Index;
                            var remoteBlock = (PolymorphicBlockElement)((EffectWorkflowItem)remotePlug.WorkflowItem).BlockElement;

                            // keep all functions where type of output parameter matches the type of input parameter
                            functions = functions.Where(fd =>
                                {
                                    // get type of output parameter
                                    var outputParameterType = fd.GetOutputParameterTypes()[i];
                                    return remoteBlock.UsableFunctionDefinitions.Any(rfd =>
                                        {
                                            // get type of input parameter
                                            var inputParameterType = rfd.GetInputParameterTypes()[remoteIndex];
                                            // tells whether output and input types are matching or not
                                            return inputParameterType.Equals(outputParameterType);
                                        });
                                })
                            .ToArray(); // run and cache it now
                        }
                    }
                }
                // increment output parameter index
                i++;
            }
            // ===========================================================================================

            // final filtering with function where known types (non-polymorphic) match
            var availableFunctionDefinitions = functions
                .Where(fd => ConnectionsTypeInfoMatch(fd, connectionTypeInfo))
                .ToArray();

            // set the publically known usable function definitions
            UsableFunctionDefinitions = availableFunctionDefinitions;

            // when connecting, evaluate children after self evaluation
            if (isConnecting)
            {
                foreach (var remoteInputBlock in remoteInputs)
                    remoteInputBlock.EvaluateUsableFunctions(knownBlocks, isConnecting);

                foreach (var remoteOutputBlock in remoteOutputs)
                    remoteOutputBlock.EvaluateUsableFunctions(knownBlocks, isConnecting);
            }
        }

        public ShaderTyping.ShaderTypeDefinition[] GetPossibleTypes(EffectWorkflowItem workflowItem, bool input, int index)
        {
            var list = new List<ShaderTyping.ShaderTypeDefinition>();

            foreach (var func in UsableFunctionDefinitions)
            {
                ShaderTyping.ShaderTypeDefinition[] prms;

                if (input)
                {
                    prms = func.Parameters
                        .Where(p => p.Direction == ShaderTyping.ParameterDirection.Input)
                        .Select(p => p.Type)
                        .ToArray();
                }
                else
                {
                    prms = func.ReturnType.IsVoid
                        ? new ShaderTyping.ShaderTypeDefinition[0]
                        : new[] { func.ReturnType };
                    prms = prms.Concat(func.Parameters
                        .Where(p => p.Direction == ShaderTyping.ParameterDirection.Output)
                        .Select(p => p.Type))
                        .ToArray();
                }

                list.Add(prms[index]);
            }

            return list.ToArray();
        }

        private static bool ConnectionsTypeInfoMatch(FunctionDefinition funcDef, ConnectionsTypeInfo connTypeInfo)
        {
            var i = 0;

            foreach (var paramIn in funcDef.Parameters.Where(p => p.Direction == ShaderTyping.ParameterDirection.Input))
            {
                if (connTypeInfo.InputTypes[i] != null)
                {
                    if (connTypeInfo.InputTypes[i].Equals(Globals.PolymorphicType) == false)
                    {
                        if (connTypeInfo.InputTypes[i].Equals(paramIn.Type) == false)
                            return false;
                    }
                }
                i++;
            }

            i = 0;

            if (funcDef.ReturnType.IsVoid == false)
            {
                if (connTypeInfo.OutputTypes[i] != null)
                {
                    if (connTypeInfo.OutputTypes[i].Equals(Globals.PolymorphicType) == false)
                    {
                        if (connTypeInfo.OutputTypes[i].Equals(funcDef.ReturnType) == false)
                            return false;
                    }
                }
                i++;
            }

            foreach (var paramOut in funcDef.Parameters.Where(p => p.Direction == ShaderTyping.ParameterDirection.Output))
            {
                if (connTypeInfo.OutputTypes[i] != null)
                {
                    if (connTypeInfo.OutputTypes[i].Equals(Globals.PolymorphicType) == false)
                    {
                        if (connTypeInfo.OutputTypes[i].Equals(paramOut.Type) == false)
                            return false;
                    }
                }
                i++;
            }

            return true;
        }

    }

    public class ConnectionsTypeInfo
    {
        public ShaderTyping.ShaderTypeDefinition[] InputTypes { get; set; }
        public ShaderTyping.ShaderTypeDefinition[] OutputTypes { get; set; }

        public static ConnectionsTypeInfo FromWorkflowItem(EffectWorkflowItem workflowItem)
        {
            var result = new ConnectionsTypeInfo
            {
                InputTypes = new ShaderTyping.ShaderTypeDefinition[workflowItem.InputPlugs.Length],
                OutputTypes = new ShaderTyping.ShaderTypeDefinition[workflowItem.OutputPlugs.Length],
            };

            foreach (var inputPlug in workflowItem.InputPlugs)
            {
                if (inputPlug.RemoteOutputPlug == null)
                    continue;

                var remoteItem = inputPlug.RemoteOutputPlug.WorkflowItem as EffectWorkflowItem;
                if (remoteItem == null)
                    continue;

                var remoteIndex = inputPlug.RemoteOutputPlug.Index;
                var type = remoteItem.BlockElement.BlockDefinition.OutputPlugs[remoteIndex].Type;

                result.InputTypes[inputPlug.Index] = type;
            }

            foreach (var outputPlug in workflowItem.OutputPlugs)
            {
                if (outputPlug.RemoteInputPlugs.Any() == false)
                    continue;

                var remoteItem = outputPlug.RemoteInputPlugs[0].WorkflowItem as EffectWorkflowItem;
                if (remoteItem == null)
                    continue;

                var remoteIndex = outputPlug.RemoteInputPlugs[0].Index;
                var type = remoteItem.BlockElement.BlockDefinition.InputPlugs[remoteIndex].Type;

                result.OutputTypes[outputPlug.Index] = type;
            }

            return result;
        }
    }
}
