﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;

namespace EffectCombiner.Primitives.Generation
{
    public enum PrimitiveShaderType
    {
        Boolean,
        UnsignedInteger,
        Integer,
        Float,
        Double,
    }

    public enum ShaderType
    {
        Boolean,
        Integer,
        UnsignedInteger,
        Float,
        Double,

        BooleanVector2,
        BooleanVector3,
        BooleanVector4,

        IntegerVector2,
        IntegerVector3,
        IntegerVector4,

        UnsignedIntegerVector2,
        UnsignedIntegerVector3,
        UnsignedIntegerVector4,

        FloatVector2,
        FloatVector3,
        FloatVector4,

        DoubleVector2,
        DoubleVector3,
        DoubleVector4,

        Matrix2By2,
        Matrix3By2,
        Matrix4By2,

        Matrix2By3,
        Matrix3By3,
        Matrix4By3,

        Matrix2By4,
        Matrix3By4,
        Matrix4By4,
    }

    public static class GlslTypingUtility
    {
        public static string FormatNumericValue(PrimitiveShaderType shaderType, int baseValue)
        {
            switch (shaderType)
            {
                case PrimitiveShaderType.Float:
                    return string.Format("{0}.0f", baseValue);
                case PrimitiveShaderType.Double:
                    return string.Format("{0}.0", baseValue);
            }
            return baseValue.ToString(CultureInfo.InvariantCulture);
        }

        /// <summary>
        /// 文字列で記述された値を適切な記述になるように修正します。
        /// </summary>
        /// <param name="shaderType">値のプリミティブ型</param>
        /// <param name="valueText">文字列で記述された値</param>
        /// <returns>修正した結果の文字列を返します。</returns>
        public static string FixValueFormat(PrimitiveShaderType shaderType, string valueText)
        {
            string fixedText = valueText;

            switch (shaderType)
            {
                case PrimitiveShaderType.Integer:
                {
                    int intValue;
                    bool resParse = int.TryParse(valueText, out intValue);

                    // ここでパースに失敗するのは、.ecmb ファイルに非数が書き込まれているような、
                    // 異常事態なのでこの時点で例外を吐く。
                    if (resParse == false)
                    {
                        throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "valueText"), "valueText");
                    }

                    fixedText = intValue.ToString();
                } break;

                case PrimitiveShaderType.UnsignedInteger:
                {
                    uint uintValue;
                    bool resParse = uint.TryParse(valueText, out uintValue);

                    if (resParse == false)
                    {
                        throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "valueText"), "valueText");
                    }

                    fixedText = uintValue.ToString();
                } break;

                case PrimitiveShaderType.Float:
                {
                    //-----------------------------------------------------------------
                    //  string 型の value を float 型にパースして、
                    //  float の精度的に十分な桁数で出力する。
                    //  {0:f} だと、"0" の整数値が入った時に "0.00" が返ってくるので、
                    //  "0.0" を返すようにフォーマット指定に少し手を入れています。
                    //-----------------------------------------------------------------

                    float floatValue = 0.0f;
                    bool resParse = float.TryParse(valueText, out floatValue);

                    if (resParse == false)
                    {
                        throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "valueText"), "valueText");
                    }

                    fixedText = string.Format("{0:0.0#########}", floatValue);
                } break;

                case PrimitiveShaderType.Double:
                {
                    // double 型の値はとりあえず float と同じ処理をしておく
                    double doubleValue = 0.0;
                    bool resParse = double.TryParse(valueText, out doubleValue);

                    if (resParse == false)
                    {
                        throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "valueText"), "valueText");
                    }

                    fixedText = string.Format("{0:0.0#########}", doubleValue);
                } break;
            }

            return fixedText;
        }

        public static string GetDefaultValueAsString(PrimitiveShaderType primitiveShaderType)
        {
            switch (primitiveShaderType)
            {
                case PrimitiveShaderType.Boolean: return "false";
                case PrimitiveShaderType.Integer: return "0";
                case PrimitiveShaderType.UnsignedInteger: return "0";
                case PrimitiveShaderType.Float: return "0.0";
                case PrimitiveShaderType.Double: return "0.0";
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "primitiveShaderType"), "primitiveShaderType");
        }

        /// <summary>
        /// シェーダタイプの値に変更変換できない場合はデフォルト値に設定します。
        /// </summary>
        /// <param name="array">変換する配列</param>
        /// <param name="shaderType">シェーダタイプ</param>
        public static void ConvertShaderTypeValue(string[,] array, ShaderType shaderType)
        {
            PrimitiveShaderType type = GetPrimitiveShaderType(shaderType);
            for (var i = 0; i < array.GetLength(0); i++)
            {
                for (int j = 0; j < array.GetLength(1); j++)
                {
                    switch (type)
                    {
                        case PrimitiveShaderType.Boolean:
                            bool bParam;
                            if (!bool.TryParse(array[i, j], out bParam))
                            {
                                array[i, j] = GetDefaultValueAsString(type);
                            }

                            continue;
                        case PrimitiveShaderType.Integer:
                            int iParam;
                            if (!int.TryParse(array[i, j], out iParam))
                            {
                                array[i, j] = GetDefaultValueAsString(type);
                            }

                            continue;
                        case PrimitiveShaderType.UnsignedInteger:
                            uint uParam;
                            if (!uint.TryParse(array[i, j], out uParam))
                            {
                                array[i, j] = GetDefaultValueAsString(type);
                            }

                            continue;
                        case PrimitiveShaderType.Float:
                            float fParam;
                            if (!float.TryParse(array[i, j], out fParam))
                            {
                                array[i, j] = GetDefaultValueAsString(type);
                            }

                            continue;
                        case PrimitiveShaderType.Double:
                            double dParam;
                            if (!double.TryParse(array[i, j], out dParam))
                            {
                                array[i, j] = GetDefaultValueAsString(type);
                            }

                            continue;
                    }
                }
            }
        }

        public static string[,] CreateDefaultValue(ShaderType shaderType)
        {
            var dimmensionX = GetDimensionX(shaderType);
            var dimmensionY = GetDimensionY(shaderType);
            var defaultValue = GetDefaultValueAsString(GetPrimitiveShaderType(shaderType));

            var array = new string[dimmensionX, dimmensionY];
            for (var y = 0; y < dimmensionY; y++)
            {
                for (var x = 0; x < dimmensionX; x++)
                {
                    array[x, y] = defaultValue;
                }
            }

            return array;
        }

        public static string GetValueExpression(ShaderType shaderType, object[,] values)
        {
            var dimensionX = GetDimensionX(shaderType);
            var dimensionY = GetDimensionY(shaderType);
            var typeName = GetGlslTypeNameFromShaderType(shaderType);
            var primitiveType = GetPrimitiveShaderType(shaderType);

            if (dimensionX == 1 && dimensionY == 1)
            {
                return FixValueFormat(primitiveType, values[0, 0].ToString());
            }

            var list = new List<string>();
            for (var y = 0; y < dimensionY; y++)
            {
                for (var x = 0; x < dimensionX; x++)
                {
                    list.Add(FixValueFormat(primitiveType, values[x, y].ToString()));
                }
            }

            return string.Format("{0}({1})", typeName, string.Join(", ", list));
        }

        public static string GetValueExpression(ShaderType shaderType, string[] values)
        {
            // MEMO: 現状、CreateInputValue() でデフォルト値を設定する際にしか呼ばれない。
            //       基本的な処理は上段の GetValueExpression(ShaderType shaderType, object[,] values) の方を通る。

            if (values == null)
            {
                return "?";
            }

            var dimensionX = GetDimensionX(shaderType);
            var dimensionY = GetDimensionY(shaderType);
            var typeName = GetGlslTypeNameFromShaderType(shaderType);

            if (dimensionX == 1 && dimensionY == 1)
            {
                return values[0];
            }

            return string.Format("{0}({1})", typeName, string.Join(", ", values));
        }

        public static int GetDimensionX(ShaderType shaderType)
        {
            switch (shaderType)
            {
                case ShaderType.Boolean:
                case ShaderType.Integer:
                case ShaderType.UnsignedInteger:
                case ShaderType.Float:
                case ShaderType.Double:
                    return 1;

                case ShaderType.BooleanVector2:
                case ShaderType.IntegerVector2:
                case ShaderType.UnsignedIntegerVector2:
                case ShaderType.FloatVector2:
                case ShaderType.DoubleVector2:
                    return 2;

                case ShaderType.BooleanVector3:
                case ShaderType.IntegerVector3:
                case ShaderType.UnsignedIntegerVector3:
                case ShaderType.FloatVector3:
                case ShaderType.DoubleVector3:
                    return 3;

                case ShaderType.BooleanVector4:
                case ShaderType.IntegerVector4:
                case ShaderType.UnsignedIntegerVector4:
                case ShaderType.FloatVector4:
                case ShaderType.DoubleVector4:
                    return 4;

                case ShaderType.Matrix2By2:
                case ShaderType.Matrix2By3:
                case ShaderType.Matrix2By4:
                    return 2;

                case ShaderType.Matrix3By2:
                case ShaderType.Matrix3By3:
                case ShaderType.Matrix3By4:
                    return 3;

                case ShaderType.Matrix4By2:
                case ShaderType.Matrix4By3:
                case ShaderType.Matrix4By4:
                    return 4;
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
        }

        public static int GetDimensionY(ShaderType shaderType)
        {
            switch (shaderType)
            {
                case ShaderType.Boolean:
                case ShaderType.Integer:
                case ShaderType.UnsignedInteger:
                case ShaderType.Float:
                case ShaderType.Double:
                    return 1;

                case ShaderType.BooleanVector2:
                case ShaderType.IntegerVector2:
                case ShaderType.UnsignedIntegerVector2:
                case ShaderType.FloatVector2:
                case ShaderType.DoubleVector2:
                    return 1;

                case ShaderType.BooleanVector3:
                case ShaderType.IntegerVector3:
                case ShaderType.UnsignedIntegerVector3:
                case ShaderType.FloatVector3:
                case ShaderType.DoubleVector3:
                    return 1;

                case ShaderType.BooleanVector4:
                case ShaderType.IntegerVector4:
                case ShaderType.UnsignedIntegerVector4:
                case ShaderType.FloatVector4:
                case ShaderType.DoubleVector4:
                    return 1;

                case ShaderType.Matrix2By2:
                case ShaderType.Matrix3By2:
                case ShaderType.Matrix4By2:
                    return 2;

                case ShaderType.Matrix2By3:
                case ShaderType.Matrix3By3:
                case ShaderType.Matrix4By3:
                    return 3;

                case ShaderType.Matrix2By4:
                case ShaderType.Matrix3By4:
                case ShaderType.Matrix4By4:
                    return 4;
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
        }

        public static PrimitiveShaderType GetPrimitiveShaderType(ShaderType shaderType)
        {
            switch (shaderType)
            {
                case ShaderType.Boolean: return PrimitiveShaderType.Boolean;
                case ShaderType.Integer: return PrimitiveShaderType.Integer;
                case ShaderType.UnsignedInteger: return PrimitiveShaderType.UnsignedInteger;
                case ShaderType.Float: return PrimitiveShaderType.Float;
                case ShaderType.Double: return PrimitiveShaderType.Double;

                case ShaderType.BooleanVector2:
                case ShaderType.BooleanVector3:
                case ShaderType.BooleanVector4:
                    return PrimitiveShaderType.Boolean;

                case ShaderType.IntegerVector2:
                case ShaderType.IntegerVector3:
                case ShaderType.IntegerVector4:
                    return PrimitiveShaderType.Integer;

                case ShaderType.UnsignedIntegerVector2:
                case ShaderType.UnsignedIntegerVector3:
                case ShaderType.UnsignedIntegerVector4:
                    return PrimitiveShaderType.UnsignedInteger;

                case ShaderType.FloatVector2:
                case ShaderType.FloatVector3:
                case ShaderType.FloatVector4:
                    return PrimitiveShaderType.Float;

                case ShaderType.DoubleVector2:
                case ShaderType.DoubleVector3:
                case ShaderType.DoubleVector4:
                    return PrimitiveShaderType.Double;

                case ShaderType.Matrix2By2:
                case ShaderType.Matrix3By2:
                case ShaderType.Matrix4By2:
                case ShaderType.Matrix2By3:
                case ShaderType.Matrix3By3:
                case ShaderType.Matrix4By3:
                case ShaderType.Matrix2By4:
                case ShaderType.Matrix3By4:
                case ShaderType.Matrix4By4:
                    return PrimitiveShaderType.Float;
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
        }

        public static string GetGlslTypeNameFromShaderType(ShaderType shaderType)
        {
            switch (shaderType)
            {
                case ShaderType.Boolean: return "bool";
                case ShaderType.Integer: return "int";
                case ShaderType.UnsignedInteger: return "uint";
                case ShaderType.Float: return "float";
                case ShaderType.Double: return "double";

                case ShaderType.BooleanVector2: return "bvec2";
                case ShaderType.BooleanVector3: return "bvec3";
                case ShaderType.BooleanVector4: return "bvec4";

                case ShaderType.IntegerVector2: return "ivec2";
                case ShaderType.IntegerVector3: return "ivec3";
                case ShaderType.IntegerVector4: return "ivec4";

                case ShaderType.UnsignedIntegerVector2: return "uvec2";
                case ShaderType.UnsignedIntegerVector3: return "uvec3";
                case ShaderType.UnsignedIntegerVector4: return "uvec4";

                case ShaderType.FloatVector2: return "vec2";
                case ShaderType.FloatVector3: return "vec3";
                case ShaderType.FloatVector4: return "vec4";

                case ShaderType.DoubleVector2: return "dvec2";
                case ShaderType.DoubleVector3: return "dvec3";
                case ShaderType.DoubleVector4: return "dvec4";

                case ShaderType.Matrix2By2: return "mat2";
                case ShaderType.Matrix3By2: return "mat3x2";
                case ShaderType.Matrix4By2: return "mat4x2";
                case ShaderType.Matrix2By3: return "mat2x3";
                case ShaderType.Matrix3By3: return "mat3";
                case ShaderType.Matrix4By3: return "mat4x3";
                case ShaderType.Matrix2By4: return "mat2x4";
                case ShaderType.Matrix3By4: return "mat3x4";
                case ShaderType.Matrix4By4: return "mat4";
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
        }

        /// <summary>
        /// Constantノード表記用の文字列を取得します。
        /// </summary>
        /// <param name="shaderType">シェーダタイプ</param>
        /// <param name="parameter">パラメータ</param>
        /// <param name="name">値の名前</param>
        /// <returns>文字列です。</returns>
        public static string GetShaderParamFromShaderType(ShaderType shaderType, string[,] parameter, string[] name)
        {
            var parameterNames = GetShaderNameFromShaderType(shaderType, name);

            switch (shaderType)
            {
                case ShaderType.Boolean:
                case ShaderType.Integer:
                case ShaderType.UnsignedInteger:
                case ShaderType.Float:
                case ShaderType.Double:
                    return string.Format("{0}:({1}) ", parameterNames[0], parameter[0, 0]);
                case ShaderType.BooleanVector2:
                case ShaderType.IntegerVector2:
                case ShaderType.UnsignedIntegerVector2:
                case ShaderType.FloatVector2:
                case ShaderType.DoubleVector2:
                    return string.Format("{0}:({1},{2}) ", parameterNames[0], parameter[0, 0], parameter[1, 0]);
                case ShaderType.IntegerVector3:
                case ShaderType.BooleanVector3:
                case ShaderType.UnsignedIntegerVector3:
                case ShaderType.FloatVector3:
                case ShaderType.DoubleVector3:
                    return string.Format("{0}:({1},{2},{3}) ", parameterNames[0], parameter[0, 0], parameter[1, 0], parameter[2, 0]);
                case ShaderType.BooleanVector4:
                case ShaderType.IntegerVector4:
                case ShaderType.UnsignedIntegerVector4:
                case ShaderType.FloatVector4:
                case ShaderType.DoubleVector4:
                    return string.Format("{0}:({1},{2},{3},{4}) ", parameterNames[0], parameter[0, 0], parameter[1, 0], parameter[2, 0], parameter[3, 0]);
                case ShaderType.Matrix2By2:
                    return string.Format("{0}:({1},{2}) \r\n{3}:({4},{5}) ",
                                         parameterNames[0], parameter[0, 0], parameter[1, 0],
                                         parameterNames[1], parameter[0, 1], parameter[1, 1]);
                case ShaderType.Matrix3By2:
                    return string.Format("{0}:({1},{2},{3}) \r\n{4}:({5},{6},{7}) ",
                                         parameterNames[0], parameter[0, 0], parameter[1, 0], parameter[2, 0],
                                         parameterNames[1], parameter[0, 1], parameter[1, 1], parameter[2, 1]);
                case ShaderType.Matrix4By2:
                    return string.Format("{0}:({1},{2},{3},{4}) \r\n{5}:({6},{7},{8},{9}) ",
                                         parameterNames[0], parameter[0, 0], parameter[1, 0], parameter[2, 0], parameter[3, 0],
                                         parameterNames[1], parameter[0, 1], parameter[1, 1], parameter[2, 1], parameter[3, 1]);
                case ShaderType.Matrix2By3:
                    return string.Format("{0}:({1},{2}) \r\n{3}:({4},{5}) \r\n{6}:({7},{8}) ",
                                         parameterNames[0], parameter[0, 0], parameter[1, 0],
                                         parameterNames[1], parameter[0, 1], parameter[1, 1],
                                         parameterNames[2], parameter[0, 2], parameter[1, 2]);
                case ShaderType.Matrix3By3:
                    return string.Format("{0}:({1},{2},{3}) \r\n{4}:({5},{6},{7}) \r\n{8}:({9},{10},{11}) ",
                                         parameterNames[0], parameter[0, 0], parameter[1, 0], parameter[2, 0],
                                         parameterNames[1], parameter[0, 1], parameter[1, 1], parameter[2, 1],
                                         parameterNames[2], parameter[0, 2], parameter[1, 2], parameter[2, 2]);
                case ShaderType.Matrix4By3:
                    return string.Format("{0}:({1},{2},{3},{4}) \r\n{5}:({6},{7},{8},{9}) \r\n{10}:({11},{12},{13},{14}) ",
                                         parameterNames[0], parameter[0, 0], parameter[1, 0], parameter[2, 0], parameter[3, 0],
                                         parameterNames[1], parameter[0, 1], parameter[1, 1], parameter[2, 1], parameter[3, 1],
                                         parameterNames[2], parameter[0, 2], parameter[1, 2], parameter[2, 2], parameter[3, 2]);
                case ShaderType.Matrix2By4:
                    return string.Format("{0}:({1},{2}) \r\n{3}:({4},{5}) \r\n{6}:({7},{8}) \r\n{9}:({10},{11}) ",
                                         parameterNames[0], parameter[0, 0], parameter[1, 0],
                                         parameterNames[1], parameter[0, 1], parameter[1, 1],
                                         parameterNames[2], parameter[0, 2], parameter[1, 2],
                                         parameterNames[3], parameter[0, 3], parameter[1, 3]);
                case ShaderType.Matrix3By4:
                    return string.Format("{0}:({1},{2},{3}) \r\n{4}:({5},{6},{7}) \r\n{8}:({9},{10},{11}) \r\n{12}:({13},{14},{15}) ",
                                         parameterNames[0], parameter[0, 0], parameter[1, 0], parameter[2, 0],
                                         parameterNames[1], parameter[0, 1], parameter[1, 1], parameter[2, 1],
                                         parameterNames[2], parameter[0, 2], parameter[1, 2], parameter[2, 2],
                                         parameterNames[3], parameter[0, 3], parameter[1, 3], parameter[2, 3]);
                case ShaderType.Matrix4By4:
                    return string.Format("{0}:({1},{2},{3},{4}) \r\n{5}:({6},{7},{8},{9}) \r\n{10}:({11},{12},{13},{14}) \r\n{15}:({16},{17},{18},{19}) ",
                                         parameterNames[0], parameter[0, 0], parameter[1, 0], parameter[2, 0], parameter[3, 0],
                                         parameterNames[1], parameter[0, 1], parameter[1, 1], parameter[2, 1], parameter[3, 1],
                                         parameterNames[2], parameter[0, 2], parameter[1, 2], parameter[2, 2], parameter[3, 2],
                                         parameterNames[3], parameter[0, 3], parameter[1, 3], parameter[2, 3], parameter[3, 3]);
                default:
                    throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
            }
        }

        public static ShaderType GetShaderTypeFromGlslTypeName(string glslShaderTypeName)
        {
            switch (glslShaderTypeName)
            {
                case "bool": return ShaderType.Boolean;
                case "int": return ShaderType.Integer;
                case "uint": return ShaderType.UnsignedInteger;
                case "float": return ShaderType.Float;
                case "double": return ShaderType.Double;

                case "bvec2": return ShaderType.BooleanVector2;
                case "bvec3": return ShaderType.BooleanVector3;
                case "bvec4": return ShaderType.BooleanVector4;

                case "ivec2": return ShaderType.IntegerVector2;
                case "ivec3": return ShaderType.IntegerVector3;
                case "ivec4": return ShaderType.IntegerVector4;

                case "uvec2": return ShaderType.UnsignedIntegerVector2;
                case "uvec3": return ShaderType.UnsignedIntegerVector3;
                case "uvec4": return ShaderType.UnsignedIntegerVector4;

                case "vec2": return ShaderType.FloatVector2;
                case "vec3": return ShaderType.FloatVector3;
                case "vec4": return ShaderType.FloatVector4;

                case "dvec2": return ShaderType.DoubleVector2;
                case "dvec3": return ShaderType.DoubleVector3;
                case "dvec4": return ShaderType.DoubleVector4;

                case "mat2":
                case "mat2x2": return ShaderType.Matrix2By2;
                case "mat3x2": return ShaderType.Matrix3By2;
                case "mat4x2": return ShaderType.Matrix4By2;
                case "mat2x3": return ShaderType.Matrix2By3;
                case "mat3":
                case "mat3x3": return ShaderType.Matrix3By3;
                case "mat4x3": return ShaderType.Matrix4By3;
                case "mat2x4": return ShaderType.Matrix2By4;
                case "mat3x4": return ShaderType.Matrix3By4;
                case "mat4":
                case "mat4x4": return ShaderType.Matrix4By4;
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "glslShaderTypeName"), "glslShaderTypeName");
        }

        /// <summary>
        /// テキスト表示をする際の変数名を取得します。
        /// </summary>
        /// <param name="shaderType">変数の型</param>
        /// <param name="name">オリジナルの名前</param>
        /// <returns>テキスト表示時の変数名です。</returns>
        private static string[] GetShaderNameFromShaderType(ShaderType shaderType, string[] name)
        {
            string[] defaultName = null;
            switch (shaderType)
            {
                case ShaderType.Boolean:
                case ShaderType.Integer:
                case ShaderType.UnsignedInteger:
                case ShaderType.Float:
                case ShaderType.Double:
                    defaultName = new string[] { "value" };
                    break;
                case ShaderType.BooleanVector2:
                case ShaderType.IntegerVector2:
                case ShaderType.UnsignedIntegerVector2:
                case ShaderType.FloatVector2:
                case ShaderType.DoubleVector2:
                case ShaderType.IntegerVector3:
                case ShaderType.BooleanVector3:
                case ShaderType.UnsignedIntegerVector3:
                case ShaderType.FloatVector3:
                case ShaderType.DoubleVector3:
                case ShaderType.BooleanVector4:
                case ShaderType.IntegerVector4:
                case ShaderType.UnsignedIntegerVector4:
                case ShaderType.FloatVector4:
                case ShaderType.DoubleVector4:
                    defaultName = new string[] { "vector" };
                    break;
                case ShaderType.Matrix2By2:
                case ShaderType.Matrix3By2:
                case ShaderType.Matrix4By2:
                    defaultName = new string[] { "row1", "row2" };
                    break;
                case ShaderType.Matrix2By3:
                case ShaderType.Matrix3By3:
                case ShaderType.Matrix4By3:
                    defaultName = new string[] { "row1", "row2", "row3" };
                    break;
                case ShaderType.Matrix2By4:
                case ShaderType.Matrix3By4:
                case ShaderType.Matrix4By4:
                    defaultName = new string[] { "row1", "row2", "row3", "row4" };
                    break;
                default:
                    throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "ShaderType"), "ShaderType");
            }

            if (name != null)
            {
                for (int i = 0; i < defaultName.Length; i++)
                {
                    if (i < name.Length)
                    {
                        if (!string.IsNullOrWhiteSpace(name[i]) && !string.IsNullOrEmpty(name[i]))
                        {
                            defaultName[i] = name[i];
                        }
                    }
                }
            }

            return defaultName;
        }
    }
}
