﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectCombiner.Primitives.Generation.Extensions;
using EffectDefinitions;

namespace EffectCombiner.Primitives.Generation.AutoGen
{
    public class PolymorphicBlockDefinition : BlockDefinition
    {
        public PolymorphicBlockDefinition(
            string name,
            string guid,
            params FunctionDefinition[] functions)
            : base(null, name, guid,
            string.Empty,
            GenerateInputs(functions),
            GenerateOutputs(functions),
            null,
            new PreviewDefinition(),
            GenerateFunctionBindings(functions))
        {
            if (functions == null)
                throw new ArgumentNullException("functions");
            if (functions.Length == 0)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "functions"), "functions");

            if (functions.SelectMany(f => f.Parameters).Any(pd => pd.Direction == ShaderTyping.ParameterDirection.Reference))
                throw new ArgumentException(Messages.EXCEPTION_REF_PARAM_NOT_SUPPORT_WITH_AUTOGEN);

            var inParams = GetInputCount(functions);
            var outParams = GetOutputCount(functions);

            foreach (var f in functions.Skip(1))
            {
                var localInParams = GetInputCount(f);
                if (localInParams != inParams)
                {
                    var message = string.Format(Messages.EXCEPTION_FUNC_HAS_INVALID_INPUT_PARAM_COUNT,
                        f.Name, f.Id, inParams, localInParams);
                    throw new ArgumentException(message);
                }

                var localOutParams = GetOutputCount(f);
                if (localOutParams != outParams)
                {
                    var message = string.Format(Messages.EXCEPTION_FUNC_HAS_INVALID_OUTPUT_PARAM_COUNT,
                        f.Name, f.Id, outParams, localOutParams);
                    throw new ArgumentException(message);
                }
            }

            FunctionDefinitions = functions;

            foreach (var plug in InputPlugs)
                plug.SetBlockDefinition(this);

            foreach (var plug in OutputPlugs)
                plug.SetBlockDefinition(this);
        }

        public FunctionDefinition[] FunctionDefinitions { get; private set; }

        public new PolymorphicPlug[] InputPlugs
        {
            get { return base.InputPlugs.Cast<PolymorphicPlug>().ToArray(); }
        }

        public new PolymorphicPlug[] OutputPlugs
        {
            get { return base.OutputPlugs.Cast<PolymorphicPlug>().ToArray(); }
        }

        private static int GetInputCount(FunctionDefinition[] funcs)
        {
            if (funcs == null || funcs.Length == 0)
                return 0;

            return GetInputCount(funcs[0]);
        }

        private static int GetInputCount(FunctionDefinition func)
        {
            if (func == null)
                return 0;

            return func.Parameters.Count(p => p.Direction == ShaderTyping.ParameterDirection.Input);
        }

        private static int GetOutputCount(FunctionDefinition[] funcs)
        {
            if (funcs == null || funcs.Length == 0)
                return 0;

            return GetOutputCount(funcs[0]);
        }

        private static int GetOutputCount(FunctionDefinition func)
        {
            if (func == null)
                return 0;

            var outParams = func.Parameters.Count(p => p.Direction == ShaderTyping.ParameterDirection.Output);
            if (func.ReturnType.IsVoid == false)
                outParams++;
            return outParams;
        }

        private static ICollection<Plug> GenerateInputs(FunctionDefinition[] functions)
        {
            var prms = functions[0].Parameters.Where(pd => pd.Direction == ShaderTyping.ParameterDirection.Input).ToArray();
            var count = prms.Length;

            var plugs = new Plug[count];

            for (var i = 0; i < count; i++)
            {
                var pname = prms[i].Name;
                plugs[i] = new PolymorphicPlug(i, pname, pname, true, null);
            }

            return plugs;
        }

        private static ICollection<Plug> GenerateOutputs(FunctionDefinition[] functions)
        {
            var offset = 0;
            PolymorphicPlug[] plugs;

            var prms = functions[0].Parameters.Where(pd => pd.Direction == ShaderTyping.ParameterDirection.Output).ToArray();
            var count = prms.Length;

            if (functions[0].ReturnType.IsVoid == false)
            {
                offset = 1;
                count++;

                plugs = new PolymorphicPlug[count];
                plugs[0] = new PolymorphicPlug(0, "return", ParameterDefinition.ReturnParameterString, false, null);
            }
            else
                plugs = new PolymorphicPlug[count];

            for (var i = offset; i < count; i++)
            {
                var pname = prms[i].Name;
                plugs[i] = new PolymorphicPlug(i, pname, pname, false, null);
            }

            return plugs;
        }

        private static FunctionBinding[] GenerateFunctionBindings(IEnumerable<FunctionDefinition> functions)
        {
            if (functions == null)
                throw new ArgumentNullException("functions");

            return functions.Select(f => new FunctionBinding(null, null, f.Id, null, f.Name)).ToArray();
        }
    }

    public class PolymorphicPlug : Plug
    {
        public int Index { get; private set; }
        public PolymorphicBlockDefinition BlockDefinition { get; private set; }

        public PolymorphicPlug(
            int index,
            string name,
            string target,
            bool isInput,
            string[] defaultValues)
            : base(
            name,
            new ShaderTypeDefinition(Globals.PolymorphicTypeString),
            target,
            isInput,
            null,
            defaultValues)
        {
            Index = index;
        }

        public override bool CanPlugTo(Plug dest)
        {
            if (IsInput == dest.IsInput)
                return false;

            if (Index < 0)
                return false;

            if (dest is PolymorphicPlug)
                return true;

            foreach (var fd in BlockDefinition.GetBoundFunctionDefinitions())
            {
                var types = IsInput
                    ? fd.GetInputParameterTypes()
                    : fd.GetOutputParameterTypes();

                if (Index >= types.Length)
                    return false;

                if (types[Index].Equals(dest.Type))
                    return true;
            }

            return false;
        }

        public override bool IsMatchingParameter(ShaderTyping.ParameterDefinition parameter)
        {
            var typeAreMatching = Type.Equals(Globals.PolymorphicType) || Type == parameter.Type;
            var targetNameAreSame = (Target == parameter.Name);
            var baseTargetNameAreMatching = (BaseTarget == parameter.Name);

            return baseTargetNameAreMatching && (typeAreMatching || !targetNameAreSame);
        }

        internal void SetBlockDefinition(PolymorphicBlockDefinition parentBlock)
        {
            if (parentBlock == null)
                throw new ArgumentNullException("parentBlock");

            BlockDefinition = parentBlock;
        }
    }
}
