﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using EffectCombiner.Core;
using EffectCombiner.Primitives.Generation;
using ShaderGenerator.GLSL;

namespace EffectCombiner.Generator
{
    public static class CodeGenerator
    {
        public static string[] GetShaderCodes(
            IEnumerable<EffectWorkflowItem> workflowItems,
            IContentRepository contentRepository,
            ShaderGenSettings settings,
            string shaderMainName)
        {
            if (contentRepository == null)
                throw new ArgumentNullException("contentRepository");

            var query = from workflowItem in workflowItems
                        where workflowItem.IsGraphOutput
                        let connectedGroup = workflowItem.GetConnectedGroup(false)
                        select GetConnectedGroupShaderCode(
                                workflowItem,
                                connectedGroup.Cast<EffectWorkflowItem>().ToArray(),
                                contentRepository,
                                settings,
                                shaderMainName)
                            into code
                            where string.IsNullOrWhiteSpace(code) == false
                            select code;

            return query.ToArray();
        }

        private static string GetConnectedGroupShaderCode(
            EffectWorkflowItem outputWorkflowItem,
            EffectWorkflowItem[] connectedGroup,
            IContentRepository contentRepository,
            ShaderGenSettings settings,
            string shaderMainName)
        {
            if (connectedGroup.Any(w => w.BlockElement.BlockDefinition.IsBound == false))
                return null;

            // See ticket 2310 for bellow check
            if (EffectWorkflowItem.IsGraphProperlyConnected(connectedGroup) == false)
                return null;

            var sourceRepository = new ResourceLocator(connectedGroup
                .SelectMany(w => w.BlockElement.BlockDefinition.FunctionBindings
                    .Select(fb => fb.Definition)
                    .Where(fd => fd != null))
                .ToArray());

            var glslGenerationContext = new GlslGenerationContext(
                sourceRepository,
                contentRepository,
                shaderMainName)
            {
                DefinitionMode = settings.GeneratedCodeDependencyMode,
            };

            foreach (var workflowItem in connectedGroup)
                workflowItem.SetGenerationContext(glslGenerationContext);

            glslGenerationContext.BeginGeneration();
            outputWorkflowItem.PollInputValues();
            glslGenerationContext.EndGeneration();

            return glslGenerationContext.FullShader;
        }
    }
}
