﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Blocks.Core;
using EffectCombiner.Core;
using EffectCombiner.Data;
using EffectCombiner.Data.Processors.Version2;
using EffectCombiner.Primitives;
using EffectCombiner.Primitives.Blocks;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Semantic;
using EffectDefinitions;
using Renderer2D.Core;
using Workflow.Core;
using Globals = EffectCombiner.Primitives.Globals;

namespace EffectCombiner.Editor
{
    public class ProjectManager : FileManagerBase
    {
        public static readonly ushort CurrentVersion = 2;

        private readonly MRUManager<string> mruManager = new MRUManager<string>();
        private readonly IWorkflowDataEventReporter reporter;
        private readonly IWorkspaceManager workspaceManager;

        public ProjectManager(IWorkspaceManager workspaceManager, IWorkflowDataEventReporter reporter)
        {
            if (workspaceManager == null)
                throw new ArgumentNullException("workspaceManager");
            if (reporter == null)
                throw new ArgumentNullException("reporter");

            if (Globals.Options.EnvironmentSettings != null && Globals.Options.RecentProjects != null)
                mruManager.Initialize(Globals.Options.RecentProjects);

            this.reporter = reporter;
            this.workspaceManager = workspaceManager;
        }

        public IEnumerable<string> MostRecentlyUsed { get { return mruManager.RecentlyUsed; } }

        protected override string CloseDialogMessage
        {
            get
            {
                return Localization.Messages.DLG_PROJECT_SAVE_LAST_MODIFICATIONS;
            }
        }

        protected override string CloseDialogTitle
        {
            get
            {
                return Localization.Messages.DLG_PROJECT_SAVE_LAST_MODIFICATIONS_TITLE;
            }
        }

        protected override string DialogBoxFilters
        {
            get
            {
                return string.Format("{0}|*.ecmb|{1}|*.*",
                    Localization.Messages.DLG_FILTER_ECMB_FILES,
                    Localization.Messages.DLG_FILTER_ALL_FILES);
            }
        }

        protected override string OpenDialogTitle
        {
            get
            {
                return Localization.Messages.DLG_PROJECT_LOAD_FILE_TITLE;
            }
        }

        protected override string SaveDialogTitle
        {
            get
            {
                return Localization.Messages.DLG_PROJECT_SAVE_FILE_TITLE;
            }
        }

        protected override void OnClose()
        {
            workspaceManager.UpdateWorkspacePosition(new Point());
            workspaceManager.BlockManager.ClearBlocks();
            Primitives.Generation.Globals.UniformManager.InitializeValues();
        }

        protected override void OnLoad(string filename)
        {
            int cmdResult = EventCommandExecuter.Execute(
                EventCommand.PreOpen,
                new List<string> {filename});
            if (cmdResult != 0)
            {
                return;
            }

            try
            {
                Load(filename);
            }
            catch (Exception ex)
            {
                reporter.Report(new WorkflowDataEventReport
                {
                    Message = () => ex.Message,
                    IsError = true,
                    Exception = ex,
                });
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_ERROR_OCCURED, false, System.Drawing.Color.Red);
            }
        }

        private void Load(string filename)
        {
            ushort version;
            if (WorkflowDataHelper.FindVersion(filename, out version) == false)
            {
                reporter.Report(new WorkflowDataEventReport
                {
                    Message = () => Localization.Messages.CANNOT_DETERMINE_FILE_VERSION,
                    IsError = true
                });
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_ERROR_OCCURED, false, System.Drawing.Color.Red);
                return;
            }

            var processor = WorkflowDataHelper.GetProcessor(version);
            if (processor == null)
            {
                reporter.Report(new WorkflowDataEventReport
                {
                    Message = () => string.Format(Localization.Messages.NO_PROCESSOR_FOR_FILE_VERSION_N, version),
                    IsError = true
                });
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_ERROR_OCCURED, false, System.Drawing.Color.Red);
                return;
            }

            object data;
            processor.SetReporter(reporter);
            using (var fs = File.OpenRead(filename))
                data = processor.Load(fs);

            if (data == null)
            {
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_ERROR_OCCURED, false, System.Drawing.Color.Red);
                return; // here processor.Load log the errors
            }

            if (version < CurrentVersion)
            {
                var converter = WorkflowDataHelper.GetUpgradeConverter(version, CurrentVersion);
                data = converter.Convert(data);
            }
            else if (version > CurrentVersion)
            {
                var converter = WorkflowDataHelper.GetDowngradeConverter(version, CurrentVersion);
                data = converter.Convert(data);
            }

            if (data == null)
            {
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_ERROR_OCCURED, false, System.Drawing.Color.Red);
                reporter.Report(new WorkflowDataEventReport
                {
                    Message = () => string.Format(Localization.Messages.CONVERSION_FAILED, version, CurrentVersion),
                    IsError = true
                });
                return;
            }

            mruManager.IndicateAsUsed(filename);

            var blockManager = workspaceManager.BlockManager;

            blockManager.BeginShaderCodeGenerationLock();
            try
            {
                SetupEnvironment(filename, data);
            }
            finally
            {
                blockManager.EndShaderCodeGenerationLock();
            }

            blockManager.RunShaderGenerationProcessAsync();
        }

        protected override void OnSave(string filename)
        {
            if (EventCommandExecuter.Execute(
                EventCommand.PreSave,
                new List<string>() { filename }) != 0)
            {
                return;
            }

            try
            {
                Save(filename);
            }
            catch (Exception ex)
            {
                reporter.Report(new WorkflowDataEventReport
                {
                    Message = () => ex.Message,
                    IsError = true,
                    Exception = ex,
                });
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_ERROR_OCCURED, false, System.Drawing.Color.Red);
            }

            if (EventCommandExecuter.Execute(
                EventCommand.PostSave,
                new List<string>() { filename }) != 0)
            {
                return;
            }
        }

        private void Save(string filename)
        {
            mruManager.IndicateAsUsed(filename);

            var data = AcquireEnvironment(filename);
            var processor = WorkflowDataHelper.GetProcessor(CurrentVersion);
            using (var fs = File.Open(filename, FileMode.Create, FileAccess.Write))
                processor.Save(fs, data);
        }

        public override void ApplicationClose(out bool cancel)
        {
            base.ApplicationClose(out cancel);

            Globals.Options.RecentProjects = mruManager.RecentlyUsed.ToArray();
        }

        private object AcquireEnvironment(string filename)
        {
            var blockManager = workspaceManager.BlockManager;

            var blocks = from b in blockManager.BlockElements
                         where (b is GhostBlockElement) == false
                         select CreateBlock(b);

            var inputPlugs = blockManager.BlockElements
                .Cast<EffectBlockElementBase>()
                .SelectMany(b => b.WorkflowItem.InputPlugs.Cast<EffectInputPlug>())
                .Where(p => p.RemoteOutputPlug != null)
                .Where(p => (((EffectWorkflowItem)p.RemoteWorkflowItem).BlockElement is GhostBlockElement) == false)
                .ToArray();

            var connections = from p in inputPlugs
                              select new ConnectionV2
                              {
                                  Source = new BlockReferenceV2
                                  {
                                      BlockIdentifier = ((EffectOutputPlug)p.RemoteOutputPlug).BlockElement.InstanceIdentifier.ToString("x8"),
                                      PlugIndex = p.RemoteOutputPlug.Index,
                                  },
                                  Target = new BlockReferenceV2
                                  {
                                      BlockIdentifier = p.BlockElement.InstanceIdentifier.ToString("x8"),
                                      PlugIndex = p.Index,
                                  }
                              };

            // 配置されているブロックの uniform 変数名を列挙
            var usingUniformNames = blockManager.BlockElements
                              .OfType<UserDefinitionBlockElement>()
                              .Where(p => p.BlockDefinition.Uniform != null)
                              .Select(p => p.BlockDefinition.Uniform.Name)
                              .Distinct()
                              .ToArray();

            List<UniformV2> uniforms = new List<UniformV2>();

            // 配置されているブロックの uniform 変数値を保存用にまとめる
            foreach (var uniformItem in Primitives.Generation.Globals.UniformManager.Uniforms)
            {
                if (usingUniformNames.Contains(uniformItem.Key))
                {
                    var uniformData = uniformItem.Value;

                    // ファイルパス情報は相対パスに変換して保存する
                    if (uniformItem.Value is UniformDataFile)
                    {
                        var uniformDataFile = (UniformDataFile)uniformItem.Value.Clone();
                        uniformDataFile.Value = PathUtility.GetRelativePath(filename, uniformDataFile.Value);

                        uniformData = uniformDataFile;
                    }

                    uniforms.Add(new UniformV2
                    {
                        Name = uniformItem.Key,
                        Type = uniformData.Type,
                        Values = uniformData.ToString()
                    });
                }
            }

            return new ProjectV2
            {
                Version = CurrentVersion,
                WorkspacePositionX = (float)workspaceManager.WorkspacePosition.X,
                WorkspacePositionY = (float)workspaceManager.WorkspacePosition.Y,
                Blocks = new BlocksV2 { Block = blocks.ToArray() },
                Connections = new ConnectionsV2 { Connection = connections.ToArray() },
                Uniforms = new UniformsV2 { Uniform = uniforms.ToArray() },
            };
        }

        private void SetupEnvironment(string filename, object genericData)
        {
            var blockManager = workspaceManager.BlockManager;

            var data = genericData as ProjectV2;
            if (data == null)
            {
                var message = string.Format(Localization.Messages.EXCEPTION_UNKNOWN_DATA_STRUCTURE,
                    genericData.GetType().FullName, typeof(ProjectV2).FullName);
                throw new InvalidOperationException(message);
            }

            var wspos = new Point(data.WorkspacePositionX, data.WorkspacePositionY);
            workspaceManager.UpdateWorkspacePosition(wspos);

            blockManager.ClearBlocks();

            if (data.Blocks != null && data.Blocks.Block != null)
            {
                var blockInstances = new Dictionary<uint, EffectBlockElementBase>();

                this.SetupBlocks(data, blockInstances);
                this.SetupConnections(data, blockInstances);
                this.SetupUniforms(filename, data);
            }
        }

        /// <summary>
        /// uniform 変数をセットアップします。
        /// </summary>
        /// <param name="filename">ファイル名</param>
        /// <param name="data">プロジェクトデータ</param>
        private void SetupUniforms(string filename, ProjectV2 data)
        {
            var uniformManager = Primitives.Generation.Globals.UniformManager;

            // uniform 変数の設定がないときは全てデフォルト値で初期化する
            if (data.Uniforms?.Uniform == null)
            {
                uniformManager.InitializeValues();
                return;
            }

            var uniformDataList = new List<KeyValuePair<string, UniformData>>();

            // 文字列で指定された値を UniformData に変換する
            foreach (var uniform in data.Uniforms.Uniform)
            {
                // 定義ファイルに uniform 変数名が定義されているかチェック
                if (uniformManager.Uniforms.Keys.Contains(uniform.Name) == false)
                {
                    Reporting.Report(new EventReport(() => string.Format(Localization.Messages.UNIFORM_NAME_NOT_DEFINITION, uniform.Name),
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                UniformData uniformData;

                // 文字列を値に変換
                bool resParse = UniformDataHelper.TryParse(uniform.Type, uniform.Values, out uniformData);

                if (resParse == false)
                {
                    continue;
                }

                // ファイルパス情報は相対パスから絶対パスに変換する
                if (uniformData is UniformDataFile)
                {
                    var uniformDataFile = (UniformDataFile)uniformData;

                    if (string.IsNullOrEmpty(uniformDataFile.Value) == false)
                    {
                        var dirPath = PathUtility.GetDirectoryName(filename);
                        uniformDataFile.Value = PathUtility.GetRootedPath(dirPath, uniformDataFile.Value);

                        // ファイルがなければ警告を出しておく
                        if (File.Exists(uniformDataFile.Value) == false)
                        {
                            Reporting.Report(new EventReport(() => string.Format(Localization.Messages.UNIFORM_FILE_NOT_FOUND, uniformDataFile.Value),
                                ReportLevel.Warning, ReportCategory.Project, null, null, null));
                        }
                    }
                }

                uniformDataList.Add(new KeyValuePair<string, UniformData>(uniform.Name, uniformData));
            }

            // 指定された値で uniform 変数を初期化する
            uniformManager.InitializeValues(uniformDataList);
        }

        private void SetupBlocks(ProjectV2 data, IDictionary<uint, EffectBlockElementBase> blockInstances)
        {
            foreach (var b in data.Blocks.Block.Where(b => b != null))
            {
                var block = b;
                uint id;

                if (string.IsNullOrWhiteSpace(block.Identifier))
                {
                    Reporting.Report(new EventReport(() => Localization.Messages.BLOCK_IDENTIFIER_NOT_SET,
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                try
                {
                    id = Convert.ToUInt32(block.Identifier, 16);
                }
                catch (Exception ex)
                {
                    Reporting.Report(new EventReport(() => string.Format(Localization.Messages.IMPOSSIBLE_TO_PARSE_IDENTIFIER, block.Identifier),
                        ReportLevel.Error, ReportCategory.Project, null, () => ex.Message, ex));
                    continue;
                }

                EffectBlockElementBase element = null;

                if (string.IsNullOrWhiteSpace(block.BlockGuid))
                {
                    Reporting.Report(new EventReport(() => Localization.Messages.BLOCK_GUID_NOT_SET,
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                BlockDefinition definition;
                if (Primitives.Generation.Globals.BlockDefinitions.TryGetValue(block.BlockGuid, out definition))
                {
                    bool createError;
                    Exception createException = null;

                    try
                    {
                        element = (EffectBlockElementBase)workspaceManager.BlockManager.CreateBlockElement(definition);
                        var expandedBlocks = SemanticHelper.ExpandSemantics(workspaceManager.BlockManager, Primitives.Generation.Globals.BlockDefinitions.Values, element);
                        foreach (var eb in expandedBlocks)
                            workspaceManager.BlockManager.AddBlock(eb);

                        createError = element == null;
                    }
                    catch (Exception ex)
                    {
                        createError = true;
                        createException = ex;
                    }

                    if (createError)
                    {
                        Reporting.Report(
                            new EventReport(
                                () => string.Format(Localization.Messages.IMPOSSIBLE_TO_CREATE_BLOCK_ELEMENT, definition.Guid),
                                ReportLevel.Error, ReportCategory.Project, null,
                                () => createException != null ? createException.Message : null, createException));
                        continue;
                    }

                    var constant = element as ConstantBlockElement;
                    if (constant != null)
                    {
                        if (block.ConstInfo == null)
                        {
                            Reporting.Report(new EventReport(
                                () => string.Format(Localization.Messages.CONST_BLOCK_INFO_NOT_AVAILABLE, definition.Guid),
                                ReportLevel.Error, ReportCategory.Project, null, null, null));
                            continue;
                        }

                        try
                        {
                            constant.SetupConstantBlockElement(block.ConstInfo.Type, block.ConstInfo.Values);
                        }
                        catch
                        {
                            Reporting.Report(new EventReport(
                                () => string.Format(Localization.Messages.IMPOSSIBLE_TO_SETUP_CONST_BLOCK_ELEMENT, block.Identifier, definition.Guid),
                                ReportLevel.Error, ReportCategory.Project, null, null, null));
                            continue;
                        }
                    }

                    // データロード時にデシリアライズしたデータから作業用に復元
                    var comment = element as CommentBlockElement;
                    if (comment != null)
                    {
                        if (block.ConstInfo == null)
                        {
                            Reporting.Report(new EventReport(
                                () => string.Format(Localization.Messages.CONST_BLOCK_INFO_NOT_AVAILABLE, definition.Guid),
                                ReportLevel.Error, ReportCategory.Project, null, null, null));
                            continue;
                        }

                        try
                        {
                            comment.SetupCommentBlockElement(block.ConstInfo.Values);
                        }
                        catch
                        {
                            Reporting.Report(new EventReport(
                                () => string.Format(Localization.Messages.COMMENT_BLOCK_INFO_NOT_AVAILABLE, block.Identifier, definition.Guid),
                                ReportLevel.Error, ReportCategory.Project, null, null, null));
                            continue;
                        }
                    }
                }
                else
                {
                    bool createError;
                    Exception createException = null;

                    try
                    {
                        element = CreateFakeBlockElement(block, data.Connections.Connection);
                        createError = element == null;
                    }
                    catch (Exception ex)
                    {
                        createError = true;
                        createException = ex;
                    }

                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.UNKNOWN_GUID, block.BlockGuid),
                        ReportLevel.Error, ReportCategory.Project, null, // FIXME: would be nice to have the source
                        () => createException != null ? createException.Message : null,
                        createException));

                    if (createError)
                    {
                        Reporting.Report(new EventReport(
                            () => string.Format(Localization.Messages.IMPOSSIBLE_TO_CREATE_FAKE_BLOCK_ELEMENT, block.BlockGuid),
                            ReportLevel.Error, ReportCategory.Project, null,
                            () => createException != null ? createException.Message : null,
                            createException));
                        continue;
                    }
                }

                blockInstances.Add(id, element);

                element.SetPosition(block.PositionX, block.PositionY);

                // コメントブロックに関してはサイズもセーブデータから復元する
                if (element is CommentBlockElement)
                {
                    // 大きい方優先
                    element.SetComputedSize(
                        Math.Max(block.SizeW, element.Width),
                        Math.Max(block.SizeH, element.Height));
                }

                workspaceManager.BlockManager.AddBlock(element);
            }
        }

        private void SetupConnections(ProjectV2 data, IDictionary<uint, EffectBlockElementBase> blockInstances)
        {
            if (data.Connections == null || data.Connections.Connection == null)
                return;

            foreach (var ci in data.Connections.Connection.Where(c => c != null))
            {
                var connectionInfo = ci;

                var sourceBlockInfo = connectionInfo.Source;
                if (sourceBlockInfo == null)
                {
                    Reporting.Report(new EventReport(() => Localization.Messages.CONNECTION_SOURCE_NOT_AVAILABLE,
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                var targetBlockInfo = connectionInfo.Target;
                if (targetBlockInfo == null)
                {
                    Reporting.Report(new EventReport(() => Localization.Messages.CONNECTION_TARGET_NOT_AVAILABLE,
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                if (string.IsNullOrWhiteSpace(sourceBlockInfo.BlockIdentifier))
                {
                    Reporting.Report(new EventReport(() => Localization.Messages.BLOCK_ID_NOT_AVAILABLE_IN_SOURCE_CONNECTION,
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                if (string.IsNullOrWhiteSpace(targetBlockInfo.BlockIdentifier))
                {
                    Reporting.Report(new EventReport(() => Localization.Messages.BLOCK_ID_NOT_AVAILABLE_IN_TARGET_CONNECTION,
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                uint sourceId;
                try
                {
                    sourceId = Convert.ToUInt32(sourceBlockInfo.BlockIdentifier, 16);
                }
                catch (Exception ex)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.CANNOT_PARSE_BLOCK_ID_IN_SOURCE_CONNECTION, sourceBlockInfo.BlockIdentifier),
                        ReportLevel.Error, ReportCategory.Project, null, () => ex.Message, ex));
                    continue;
                }

                uint targetId;

                try
                {
                    targetId = Convert.ToUInt32(targetBlockInfo.BlockIdentifier, 16);
                }
                catch (Exception ex)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.CANNOT_PARSE_BLOCK_ID_IN_TARGET_CONNECTION, targetBlockInfo.BlockIdentifier),
                        ReportLevel.Error, ReportCategory.Project, null, () => ex.Message, ex));
                    continue;
                }

                EffectBlockElementBase sourceBlock;
                if (blockInstances.TryGetValue(sourceId, out sourceBlock) == false)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.IMPOSSIBLE_TO_FIND_SOURCE_BLOCK, sourceId),
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                EffectBlockElementBase targetBlock;
                if (blockInstances.TryGetValue(targetId, out targetBlock) == false)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.IMPOSSIBLE_TO_FIND_TARGET_BLOCK, targetId),
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                if (sourceBlock.WorkflowItem == null)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.WORKFLOW_ITEM_SOURCE_NOT_CONSTRUCTED,
                        sourceBlock.BlockDefinition.Name),
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                if (targetBlock.WorkflowItem == null)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.WORKFLOW_ITEM_TARGET_NOT_CONSTRUCTED,
                        targetBlock.BlockDefinition.Name),
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                if (connectionInfo.Source.PlugIndex < 0 ||
                    connectionInfo.Source.PlugIndex >= sourceBlock.WorkflowItem.OutputPlugs.Length)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.INVALID_SOURCE_PLUG_INDEX,
                            connectionInfo.Source.PlugIndex, sourceBlock.BlockDefinition.Name),
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                if (connectionInfo.Target.PlugIndex < 0 || connectionInfo.Target.PlugIndex >= targetBlock.WorkflowItem.InputPlugs.Length)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.INVALID_TARGET_PLUG_INDEX,
                            connectionInfo.Target.PlugIndex, targetBlock.BlockDefinition.Name),
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                var po = sourceBlock.BlockDefinition.OutputPlugs[connectionInfo.Source.PlugIndex];
                var pi = targetBlock.BlockDefinition.InputPlugs[connectionInfo.Target.PlugIndex];

                if (pi.CanPlugTo(po) == false && po.CanPlugTo(pi) == false)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.IMPOSSIBLE_TO_CONNECT_BLOCK,
                            sourceBlock.BlockDefinition.Name, connectionInfo.Source.PlugIndex, po.Type.TypeString,
                            targetBlock.BlockDefinition.Name, connectionInfo.Target.PlugIndex, pi.Type.TypeString),
                        ReportLevel.Error, ReportCategory.Project, null, null, null));
                    continue;
                }

                try
                {
                    ConnectionManager.Connect(
                        sourceBlock.WorkflowItem.OutputPlugs[connectionInfo.Source.PlugIndex],
                        targetBlock.WorkflowItem.InputPlugs[connectionInfo.Target.PlugIndex]);
                }
                catch (Exception ex)
                {
                    Reporting.Report(new EventReport(
                        () => string.Format(Localization.Messages.IMPOSSIBLE_TO_CONNECT_BLOCK,
                            sourceBlock.BlockDefinition.Name, connectionInfo.Source.PlugIndex, po.Type.TypeString,
                            targetBlock.BlockDefinition.Name, connectionInfo.Target.PlugIndex, pi.Type.TypeString),
                        ReportLevel.Error, ReportCategory.Project, null, () => ex.Message, ex));
                }
            }
        }

        private EffectBlockElementBase CreateFakeBlockElement(BlockV2 blockInfo, ConnectionV2[] connections)
        {
            var inputCount = 0;
            var outputCount = 0;

            if (connections != null)
            {
                try
                {
                    inputCount = connections.Where(c =>
                        string.Equals(c.Target.BlockIdentifier, blockInfo.Identifier, StringComparison.InvariantCultureIgnoreCase))
                        .Max(c => c.Target.PlugIndex) + 1;
                }
                catch (InvalidOperationException)
                {
                }

                try
                {
                    outputCount = connections.Where(c =>
                        string.Equals(c.Source.BlockIdentifier, blockInfo.Identifier, StringComparison.InvariantCultureIgnoreCase))
                        .Max(c => c.Source.PlugIndex) + 1;
                }
                catch (InvalidOperationException)
                {
                }
            }

            if (inputCount == 0 && outputCount == 0)
            {
                inputCount = 1;
                outputCount = 1;
            }

            var fakeBlockDefinition = new FakeBlockDefinition(blockInfo.BlockGuid, inputCount, outputCount);

            return (EffectBlockElementBase)workspaceManager.BlockManager.CreateBlockElement(fakeBlockDefinition);
        }

        private static BlockV2 CreateBlock(BlockElementBase blockElement)
        {
            var constant = blockElement as ConstantBlockElement;

            ConstInfoV2 constInfo = null;
            if (constant != null)
            {
                var type = constant.BlockDefinition.OutputPlugs[0].Type.TypeString;
                var values = CoreUtility.MultiDimArrayToString(constant.Values);
                constInfo = new ConstInfoV2 { Type = type, Values = values };
            }

            // データセーブ時にシリアライズ用のデータにパッキング
            var comment = blockElement as CommentBlockElement;
            if (comment != null)
            {
                var values = CoreUtility.MultiDimArrayToString(comment.Values);
                constInfo = new ConstInfoV2 { Values = values };
            }

            return new BlockV2
            {
                Identifier = blockElement.InstanceIdentifier.ToString("x8"),
                PositionX = (float)blockElement.Left,
                PositionY = (float)blockElement.Top,
                SizeW = (float)blockElement.Width,
                SizeH = (float)blockElement.Height,
                BlockGuid = blockElement.BlockDefinition.Guid,
                ConstInfo = constInfo,
            };
        }
    }
}
