﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using EffectCombiner.Primitives;

namespace EffectCombiner.Editor.OptionPanes
{
    public partial class EnvironmentSettingsPane : UserControl, IOptionPane
    {
        private static readonly int minPlugSnappingDistance = (int)(Globals.VisualResources.PlugRadius * 1.5);
        private static readonly int maxPlugSnappingDistance = (int)(Globals.VisualResources.PlugRadius * 10.0);

        public EnvironmentSettingsPane()
        {
            InitializeComponent();

            numMinSnapDistance.DecimalPlaces = 0;
            numMinSnapDistance.Minimum = minPlugSnappingDistance;
            numMinSnapDistance.Maximum = maxPlugSnappingDistance;
            trkMinSnapDistance.Minimum = minPlugSnappingDistance;
            trkMinSnapDistance.Maximum = maxPlugSnappingDistance;
            lblMinSnapDistanceMin.Text = minPlugSnappingDistance.ToString();
            lblMinSnapDistanceMax.Text = maxPlugSnappingDistance.ToString();

            UpdatePlugSnappingAvailability();

            chkSnappingActive.CheckedChanged += chkSnappingActive_CheckedChanged;
            numMinSnapDistance.ValueChanged += numMinSnapDistance_ValueChanged;
            trkMinSnapDistance.ValueChanged += trkMinSnapDistance_ValueChanged;

            btnBrowsePreSaveCommandPath.Click += BrowsePreSaveCommandPathClick;
            btnBrowsePostSaveCommandPath.Click += BrowsePostSaveCommandPathClick;
            btnBrowsePreOpenCommandPath.Click += BrowsePreOpenCommandPathClick;

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    grpBlockSelection.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_BLOCK_SELECTION_GROUP;
                    chkCompletelyIncludeToSelect.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_BLOCK_SELECTION_COMPLETLY_INCLUDED;
                    grpBlockDisplay.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_BLOCK_DISPLAY_GROUP;
                    chkShowTextShadow.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_BLOCK_DISPLAY_SHOW_TEXT_SHADOW;
                    grpPlugSnapping.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_PLUG_SNAPPING_GROUP;
                    chkSnappingActive.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_PLUG_SNAPPING_ACTIVATED;
                    lblMinSnapDistance.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_PLUG_SNAPPING_MIN_DISTANCE;
                    numMinSnapDistance.Left = lblMinSnapDistance.Right + 6;
                    lblSaveGroup.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_SAVE_COMMAND_GROUP;
                    lblPreSaveCommand.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_PRE_SAVE_COMMAND;
                    lblPostSaveCommand.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_POST_SAVE_COMMAND;
                    lblOpenGroup.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_OPEN_COMMAND_GROUP;
                    lblPreOpenCommand.Text = Localization.Controls.OPTION_PANE_ENVIRONMENT_PRE_OPEN_COMMNAD;
                });

            Disposed += (ss, ee) => localizationSubscription.Dispose();
        }

        private void numMinSnapDistance_ValueChanged(object sender, EventArgs e)
        {
            trkMinSnapDistance.Value = (int)numMinSnapDistance.Value;
        }

        private void trkMinSnapDistance_ValueChanged(object sender, EventArgs e)
        {
            numMinSnapDistance.Value = trkMinSnapDistance.Value;
        }

        private void chkSnappingActive_CheckedChanged(object sender, EventArgs e)
        {
            UpdatePlugSnappingAvailability();
        }

        private void UpdatePlugSnappingAvailability()
        {
            var snapEnabled = chkSnappingActive.Checked;
            lblMinSnapDistance.Enabled = snapEnabled;
            numMinSnapDistance.Enabled = snapEnabled;
            trkMinSnapDistance.Enabled = snapEnabled;
        }

        public string Identifier
        {
            get { return "EnvironmentSettingsPane"; }
        }

        public string ChildOf
        {
            get { return null; }
        }

        public string DisplayName
        {
            get { return Localization.Controls.OPTION_PANE_NAME_ENVIRONMENT; }
        }

        public void OnInitialize()
        {
            if (Globals.Options.EnvironmentSettings == null)
                return;

            chkCompletelyIncludeToSelect.Checked = Globals.Options.EnvironmentSettings.CompletelyIncludeToSelect;

            chkShowTextShadow.Checked = Globals.Options.EnvironmentSettings.DrawTextShadow;

            chkSnappingActive.Checked = Globals.Options.EnvironmentSettings.PlugSnapping.IsActive;

            var minDist = (int)Globals.Options.EnvironmentSettings.PlugSnapping.MinimumDistance;
            numMinSnapDistance.Value = Math.Max(minPlugSnappingDistance, Math.Min(minDist, maxPlugSnappingDistance));

            txtPreSaveCommandPath.Text = Globals.Options.EnvironmentSettings.PreSaveCommandPath;
            txtPostSaveCommandPath.Text = Globals.Options.EnvironmentSettings.PostSaveCommandPath;
            txtPreOpenCommandPath.Text = Globals.Options.EnvironmentSettings.PreOpenCommandPath;
        }

        public void OnTerminate()
        {
        }

        public void OnAccept()
        {
            if (Globals.Options.EnvironmentSettings == null)
                Globals.Options.EnvironmentSettings = new EnvironmentSettings();

            Globals.Options.EnvironmentSettings.CompletelyIncludeToSelect = chkCompletelyIncludeToSelect.Checked;

            Globals.Options.EnvironmentSettings.DrawTextShadow = chkShowTextShadow.Checked;

            Globals.Options.EnvironmentSettings.PlugSnapping.IsActive = chkSnappingActive.Checked;
            Globals.Options.EnvironmentSettings.PlugSnapping.MinimumDistance = (double)numMinSnapDistance.Value;

            Globals.Options.EnvironmentSettings.PreSaveCommandPath = txtPreSaveCommandPath.Text;
            Globals.Options.EnvironmentSettings.PostSaveCommandPath = txtPostSaveCommandPath.Text;
            Globals.Options.EnvironmentSettings.PreOpenCommandPath = txtPreOpenCommandPath.Text;
        }

        public void OnCancel()
        {
        }

        /// <summary>
        /// 保存前コマンドの選択ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BrowsePreSaveCommandPathClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                CheckFileExists = true,
                CheckPathExists = true,
                ShowHelp = true,
                Filter = string.Format("{0}|*.bat|{1}|*.*",
                Localization.Messages.DLG_FILTER_BATCH_FILES,
                Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_PRE_SAVE_COMMAND_TITLE,
            };

            if (dlg.ShowDialog() == DialogResult.OK)
            {
                this.txtPreSaveCommandPath.Text = dlg.FileName;
            }
        }

        /// <summary>
        /// 保存後コマンドの選択ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BrowsePostSaveCommandPathClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                CheckFileExists = true,
                CheckPathExists = true,
                ShowHelp = true,
                Filter = string.Format("{0}|*.bat|{1}|*.*",
                Localization.Messages.DLG_FILTER_BATCH_FILES,
                Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_POST_SAVE_COMMAND_TITLE,
            };

            if (dlg.ShowDialog() == DialogResult.OK)
            {
                this.txtPostSaveCommandPath.Text = dlg.FileName;
            }
        }

        /// <summary>
        /// オープン前コマンドの選択ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BrowsePreOpenCommandPathClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                CheckFileExists = true,
                CheckPathExists = true,
                ShowHelp = true,
                Filter = string.Format("{0}|*.bat|{1}|*.*",
                Localization.Messages.DLG_FILTER_BATCH_FILES,
                Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_PRE_OPEN_COMMAND_TITLE,
            };

            if (dlg.ShowDialog() == DialogResult.OK)
            {
                this.txtPreOpenCommandPath.Text = dlg.FileName;
            }
        }
    }
}
