﻿using System;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Timers;
using System.Windows;
using System.Windows.Input;
using System.Windows.Threading;
using EffectCombiner.Primitives.Blocks;
using Nintendo.ToolFoundation.Windows.Input;

namespace EffectCombiner.Editor
{
    public class HwndPresenterViewModel : INotifyPropertyChanged
    {
        /// <summary>
        /// 接続プロパティの変更イベントです。
        /// </summary>
        private static readonly PropertyChangedEventArgs IsConnectedChangedEventArgs = new PropertyChangedEventArgs(nameof(IsConnected));

        /// <summary>
        /// 接続中かどうか
        /// </summary>
        private bool isConnected = false;

        /// <summary>
        /// 接続状態を確認します。
        /// </summary>
        private DispatcherTimer connectCheckTimer;

        public event PropertyChangedEventHandler PropertyChanged;

        public ICommand BindingCommand { get; }

        public ICommand StartCommand { get; }

        public ICommand SetHwndCommand { get; }

        public ICommand UnsetHwndCommand { get; }

        public ICommand ExitCommand { get; }

        public ICommand ContextMenuItemCommand { get; }

        #region public IntPtr WindowHandle { get; set; } = IntPtr.Zero;

        private static readonly PropertyChangedEventArgs WindowHandlePropertyChangedEventArgs = new PropertyChangedEventArgs(nameof(WindowHandle));

        private IntPtr previewWindowHandle = IntPtr.Zero;

        private Process previewProcess = null;

        private const string previewPath = @"StartPreviewer.bat";

        public IntPtr WindowHandle
        {
            get { return this.previewWindowHandle; }
            set
            {
                if (this.previewWindowHandle == value) { return; }
                this.previewWindowHandle = value;
                this.PropertyChanged?.Invoke(this, WindowHandlePropertyChangedEventArgs);
            }
        }
        #endregion

        public HwndPresenterViewModel()
        {
            this.BindingCommand = new DelegateCommand(o =>
            {
                Debug.WriteLine($"Command : {nameof(FrameworkElement.InputBindings)}");
            },
            o => true);

            this.ContextMenuItemCommand = new DelegateCommand(o =>
            {
                Debug.WriteLine($"Invoke : {nameof(HwndPresenterViewModel)}.{nameof(ContextMenuItemCommand)}");
            },
            o => true);

            this.StartCommand = new DelegateCommand(o =>
            {
                this.StartAndSetHwnd();
            },
            o => this.previewProcess == null || this.previewProcess.HasExited);

            this.SetHwndCommand = new DelegateCommand(o =>
            {
                this.SetHwnd();
            },
            o => true);

            this.UnsetHwndCommand = new DelegateCommand(o =>
            {
                this.UnsetHwnd();
            },
            o => this.WindowHandle != IntPtr.Zero);

            this.ExitCommand = new DelegateCommand(o =>
            {
                this.ExitProcess();
            },
            o => this.previewProcess != null && !this.previewProcess.HasExited);

            this.connectCheckTimer = new DispatcherTimer();
            this.connectCheckTimer.Tick += new EventHandler(this.ConnectionCheker);
            this.connectCheckTimer.Interval = new TimeSpan(0, 0, 1);
            this.connectCheckTimer.Start();
        }

        /// <summary>
        /// 接続中かどうか
        /// </summary>
        public bool IsConnected
        {
            get
            {
                return this.isConnected;
            }

            private set
            {
                if (this.isConnected != value)
                {
                    this.isConnected = value;
                    this.PropertyChanged?.Invoke(this, IsConnectedChangedEventArgs);
                }
            }
        }

        /// <summary>
        /// 定期的に、CombinerViewerと通信を行っているか判定します。
        /// </summary>
        /// <param name="sender">送信元</param>
        /// <param name="e">イベントハンドラ</param>
        private void ConnectionCheker(object sender, EventArgs e)
        {
            this.UpdatePreviewProcess();
            this.IsConnected = (this.previewProcess != null && !this.previewProcess.HasExited);
        }

        /// <summary>
        /// プレビューウィンドウのプロセスを更新します。
        /// </summary>
        /// <returns>プレビューウィンドウのプロセスが有効なときtrue、無効なときfalseを返します。</returns>
        private bool UpdatePreviewProcess()
        {
            if (this.previewProcess == null || this.previewProcess.HasExited)
            {
                this.previewProcess = Process.GetProcessesByName(PreviewController.CombinerViewerProcessName).FirstOrDefault();
            }

            if (this.previewProcess == null)
            {
                return false;
            }

            return true;
        }

        private void StartProcess()
        {
            this.ExitProcess();

            string file = this.GetViewerPath();

            if (!System.IO.File.Exists(file))
            {
                return;
            }

            this.previewProcess = Process.Start(file);
        }

        private string GetViewerPath()
        {
            var path = Path.Combine(
                this.ExecutableFolderPath,
                HwndPresenterViewModel.previewPath);
            return path;
        }

        private string ExecutableFolderPath
        {
            get
            {
                return Path.GetDirectoryName(this.ExecutablePath);
            }
        }

        private string ExecutablePath
        {
            get
            {
                var asm = Assembly.GetEntryAssembly();
                return asm != null ?
                    Path.GetFullPath(asm.Location) :
                    Path.GetFullPath(Assembly.GetExecutingAssembly().Location);
            }
        }

        private void StartAndSetHwnd()
        {
            this.StartProcess();
            Mouse.OverrideCursor = System.Windows.Input.Cursors.Wait;
            System.Threading.Thread.Sleep(500);
            this.SetHwnd();
            Mouse.OverrideCursor = System.Windows.Input.Cursors.Arrow;
        }

        private void SetHwnd()
        {
            for (int i = 0; i < 50; i++)
            {
                bool isPreviewFind = this.FindAndUpdatePreviewProcess();
                if (isPreviewFind)
                {
                    // 埋め込むと MainWindowHandle が取得できなくなるため記憶しておく
                    PreviewController.EmbededHandle = this.WindowHandle;
                    PreviewController.EmbededProcess = this.previewProcess;
                    return;
                }
                System.Threading.Thread.Sleep(TimeSpan.FromSeconds(0.01));
            }
        }

        /// <summary>
        /// プレビューのウィンドウを探し更新します。
        /// </summary>
        /// <returns>プレビューのウィンドウハンドルを返します。</returns>
        private bool FindAndUpdatePreviewProcess()
        {
            IntPtr handle = IntPtr.Zero;

            var p = Process.GetProcessesByName(PreviewController.CombinerViewerProcessName).FirstOrDefault();

            if (p != null)
            {
                this.previewProcess = p;

                // プロパティ経由で更新しないと変更通知が発生しない
                this.WindowHandle = p.MainWindowHandle;
                return true;
            }

            return false;
        }

        private void UnsetHwnd()
        {
            this.WindowHandle = IntPtr.Zero;
        }

        private void ExitProcess()
        {
            if (this.previewProcess == null)
            {
                return;
            }

            if (this.previewProcess.HasExited)
            {
                return;
            }

            this.WindowHandle = IntPtr.Zero;

            this.previewProcess?.Kill();

            this.previewProcess.WaitForExit();
            this.previewProcess.Dispose();
            this.previewProcess = null;
        }
    }
}
