﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Windows.Forms;
using Blocks.Core;
using EffectCombiner.Primitives;
using EffectCombiner.Primitives.Blocks;
using Renderer2D.Core;
using Renderer2D.Core.WinForms;

namespace EffectCombiner.Editor.Controls
{
    public class RenderSurface : RenderControl
    {
        public IWorkspaceManager WorkspaceManager { get; private set; }

        public void SetWorkspaceManager(IWorkspaceManager workspaceManager)
        {
            if (workspaceManager == null)
                throw new ArgumentNullException("workspaceManager");

            if (WorkspaceManager != null)
                throw new InvalidOperationException(Localization.Messages.EXCEPTION_LOCKED);

            WorkspaceManager = workspaceManager;
        }

        #region Implements Drag & Drop

        /// <summary>
        /// 現在のコントロールがドラッグ先のターゲットになったときの処理を行います。
        /// </summary>
        /// <param name="drgevent">ドラッグイベント情報</param>
        protected override void OnDragEnter(DragEventArgs drgevent)
        {
            base.OnDragEnter(drgevent);

            MouseInputManager mouseInputManager = this.WorkspaceManager.MouseInputManager;

            object draggingData = null;
            DragDropDataType draggingDataType = DragDropDataType.File;

            // ドラッグ中のデータを取得
            if (drgevent.Data.GetDataPresent(DataFormats.FileDrop))
            {
                draggingData = drgevent.Data.GetData(DataFormats.FileDrop);
                draggingDataType = DragDropDataType.File;
            }
            else
            {
                foreach (Type type in mouseInputManager.DraggableTypes)
                {
                    if (drgevent.Data.GetDataPresent(type))
                    {
                        draggingData = drgevent.Data.GetData(type);
                        draggingDataType = DragDropDataType.ObjectData;

                        break;
                    }
                }
            }

            if (draggingData == null)
            {
                return;
            }

            IPoint mousePos = null;

            // カーソル位置をワークスペース上の座標に変換
            {
                var controlMousePos = this.PointToClient(new System.Drawing.Point(drgevent.X, drgevent.Y));

                IPoint p = new Point(controlMousePos.X, controlMousePos.Y);

                this.WorkspaceManager.UpdateTransform();
                var mtx = this.Renderer.Transform.Inverse();

                mousePos = mtx.TransformPoint(p);
            }

            // MouseInputManager に処理を渡す
            DragDropState state = this.WorkspaceManager.MouseInputManager.OnDragEnter(mousePos.X, mousePos.Y, draggingDataType, draggingData);

            // ステータスを更新
            drgevent.Effect = ConvertDragDropState(state, drgevent.Effect);

            this.Invalidate();
        }

        /// <summary>
        /// 現在のコントロールがドラッグ先のターゲットから外れたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnDragLeave(EventArgs e)
        {
            base.OnDragLeave(e);

            // MouseInputManager に処理を渡す
            this.WorkspaceManager.MouseInputManager.OnDragLeave();

            this.Invalidate();
        }

        /// <summary>
        /// ドラッグ中にマウスを移動したときの処理を行います。
        /// </summary>
        /// <param name="drgevent">ドラッグイベント情報</param>
        protected override void OnDragOver(DragEventArgs drgevent)
        {
            base.OnDragOver(drgevent);

            MouseInputManager mouseInputManager = this.WorkspaceManager.MouseInputManager;

            object draggingData = null;
            DragDropDataType draggingDataType = DragDropDataType.File;

            // ドラッグ中のデータを取得
            if (drgevent.Data.GetDataPresent(DataFormats.FileDrop))
            {
                draggingData = drgevent.Data.GetData(DataFormats.FileDrop);
                draggingDataType = DragDropDataType.File;

                // *.ecmb ファイルは MainForm で処理してここでは扱わない
                if (Path.GetExtension(((string[])draggingData)[0]) == FileExtensions.ExtEcmbFile)
                {
                    drgevent.Effect = DragDropEffects.Copy;
                    draggingData = null;
                }
            }
            else
            {
                foreach (Type type in mouseInputManager.DraggableTypes)
                {
                    if (drgevent.Data.GetDataPresent(type))
                    {
                        draggingData = drgevent.Data.GetData(type);
                        draggingDataType = DragDropDataType.ObjectData;

                        break;
                    }
                }
            }

            if (draggingData == null)
            {
                return;
            }

            IPoint mousePos = null;

            // カーソル位置をワークスペース上の座標に変換
            {
                var controlMousePos = this.PointToClient(new System.Drawing.Point(drgevent.X, drgevent.Y));

                IPoint p = new Point(controlMousePos.X, controlMousePos.Y);

                this.WorkspaceManager.UpdateTransform();
                var mtx = this.Renderer.Transform.Inverse();

                mousePos = mtx.TransformPoint(p);
            }

            // MouseInputManager に処理を渡す
            DragDropState state = this.WorkspaceManager.MouseInputManager.OnDragOver(mousePos.X, mousePos.Y, draggingDataType, draggingData);

            // ステータスを更新
            drgevent.Effect = ConvertDragDropState(state, drgevent.Effect);

            this.Invalidate();
        }

        /// <summary>
        /// データがドロップされたときの処理を行います。
        /// </summary>
        /// <param name="drgevent">ドラッグイベント情報</param>
        protected override void OnDragDrop(DragEventArgs drgevent)
        {
            base.OnDragDrop(drgevent);

            MouseInputManager mouseInputManager = this.WorkspaceManager.MouseInputManager;

            object draggingData = null;
            DragDropDataType draggingDataType = DragDropDataType.File;

            // ドラッグ中のデータを取得
            if (drgevent.Data.GetDataPresent(DataFormats.FileDrop))
            {
                draggingData = drgevent.Data.GetData(DataFormats.FileDrop);
                draggingDataType = DragDropDataType.File;

                // *.ecmb ファイルは MainForm で処理してここでは扱わない
                if (Path.GetExtension(((string[])draggingData)[0]) == FileExtensions.ExtEcmbFile)
                {
                    drgevent.Effect = DragDropEffects.Copy;
                    draggingData = null;
                }
            }
            else
            {
                foreach (Type type in mouseInputManager.DraggableTypes)
                {
                    if (drgevent.Data.GetDataPresent(type))
                    {
                        draggingData = drgevent.Data.GetData(type);
                        draggingDataType = DragDropDataType.ObjectData;

                        break;
                    }
                }
            }

            if (draggingData == null)
            {
                return;
            }

            IPoint mousePos = null;

            // カーソル位置をワークスペース上の座標に変換
            {
                var controlMousePos = this.PointToClient(new System.Drawing.Point(drgevent.X, drgevent.Y));

                IPoint p = new Point(controlMousePos.X, controlMousePos.Y);

                this.WorkspaceManager.UpdateTransform();
                var mtx = this.Renderer.Transform.Inverse();

                mousePos = mtx.TransformPoint(p);
            }

            DragDropState state = this.WorkspaceManager.MouseInputManager.OnDragDrop(mousePos.X, mousePos.Y, draggingDataType, draggingData);

            // ステータスを更新
            drgevent.Effect = ConvertDragDropState(state, drgevent.Effect);

            this.Invalidate();
        }

        /// <summary>
        /// DragDropState を DragDropEffects にコンバートします。
        /// </summary>
        /// <param name="state">DragDropState の値</param>
        /// <param name="defaultValue">DragDropState が Default のときに返す DragDropEffects の値</param>
        /// <returns>DragDropEffects の値を返します。</returns>
        private DragDropEffects ConvertDragDropState(DragDropState state, DragDropEffects defaultValue)
        {
            DragDropEffects result = 0;

            if (state == DragDropState.Default)
            {
                result = defaultValue;
            }
            else if (state == DragDropState.None)
            {
                result = DragDropEffects.None;
            }
            else
            {
                result |= ((state & DragDropState.Copy) != 0 ? DragDropEffects.Copy : 0);
                result |= ((state & DragDropState.Link) != 0 ? DragDropEffects.Link : 0);
                result |= ((state & DragDropState.Move) != 0 ? DragDropEffects.Move : 0);
                result |= ((state & DragDropState.Scroll) != 0 ? DragDropEffects.Scroll : 0);
            }

            return result;
        }

        #endregion

        protected override void OnRender()
        {
            this.Renderer.Clear(this.BackColor.ToRendererColor());

            this.WorkspaceManager.UpdateTransform();

            this.WorkspaceManager.WorkspaceRendering.RenderGrid(this.Renderer);
            this.WorkspaceManager.WorkspaceRendering.CalculateBlockSize(this.Renderer);
            this.WorkspaceManager.WorkspaceRendering.RenderLinks(this.Renderer);
            this.WorkspaceManager.WorkspaceRendering.RenderBlocks(this.Renderer);
            this.WorkspaceManager.WorkspaceRendering.RenderConnectingLink(this.Renderer);
            this.WorkspaceManager.WorkspaceRendering.RenderSideIndicators(this.Renderer);
            this.WorkspaceManager.WorkspaceRendering.RenderRectangularSelection(this.Renderer);
            this.WorkspaceManager.WorkspaceRendering.RenderDraggingBlocks(this.Renderer);
        }

        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            var minDeltaZoom = 0.4 / WorkspaceManager.Zoom;
            var maxDeltaZoom = 4.0 / WorkspaceManager.Zoom;

            var deltaZoom = 1.0 + (e.Delta / 1200.0);
            if (deltaZoom < minDeltaZoom) deltaZoom = minDeltaZoom;
            if (deltaZoom > maxDeltaZoom) deltaZoom = maxDeltaZoom;

            WorkspaceManager.ZoomAt(deltaZoom, e.X, e.Y);

            Invalidate();
        }

        private Point prevMousePos = new Point();
        private IMatrix mouseDownMatrix;

        private MouseButtonState CheckMouseButton(MouseButtons buttons, MouseButtons button)
        {
            return (buttons & button) == button ?
                MouseButtonState.Pressed :
                MouseButtonState.Released;
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            IPoint p = new Point(e.X, e.Y);
            prevMousePos = new Point(e.X, e.Y);

            if (Renderer.Transform != null)
            {
                mouseDownMatrix = Renderer.Transform.Inverse();
                p = mouseDownMatrix.TransformPoint(p);
            }

            WorkspaceManager.MouseInputManager.OnMouseDown(new MouseInput(p.X, p.Y, e.Delta,
                CheckMouseButton(e.Button, MouseButtons.Left),
                CheckMouseButton(e.Button, MouseButtons.Middle),
                CheckMouseButton(e.Button, MouseButtons.Right)));
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            // フォーカスを得たときにマウスを動かしていなくてもイベントが発生するのを無視する
            if (e.X == prevMousePos.X && e.Y == prevMousePos.Y)
            {
                return;
            }

            IPoint p = new Point(e.X, e.Y);
            prevMousePos = new Point(e.X, e.Y);

            var mtx = mouseDownMatrix ?? Renderer.Transform;
            if (mtx != null)
                p = mtx.TransformPoint(p);

            WorkspaceManager.MouseInputManager.OnMouseMove(new MouseInput(p.X, p.Y, e.Delta,
                CheckMouseButton(e.Button, MouseButtons.Left),
                CheckMouseButton(e.Button, MouseButtons.Middle),
                CheckMouseButton(e.Button, MouseButtons.Right)));
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);

            IPoint p = new Point(e.X, e.Y);
            prevMousePos = new Point(e.X, e.Y);

            var mtx = mouseDownMatrix ?? Renderer.Transform;
            if (mtx != null)
                p = mtx.TransformPoint(p);

            WorkspaceManager.MouseInputManager.OnMouseUp(new MouseInput(p.X, p.Y, e.Delta,
                CheckMouseButton(e.Button, MouseButtons.Left),
                CheckMouseButton(e.Button, MouseButtons.Middle),
                CheckMouseButton(e.Button, MouseButtons.Right)));
        }

        protected override void OnKeyDown(KeyEventArgs e)
        {
            base.OnKeyDown(e);

            WorkspaceManager.KeyboardInputManager.OnKeyDown(new KeyboardInput(e.KeyValue, e.Alt, e.Control, e.Shift));
        }

        protected override void OnKeyPress(KeyPressEventArgs e)
        {
            base.OnKeyPress(e);

            //if (BlockManager == null)
            //    return;

            //BlockManager.OnKeyDown(new Core.KeyboardInput(e.KeyValue, e.Alt, e.Control, e.Shift));
        }

        protected override void OnKeyUp(KeyEventArgs e)
        {
            base.OnKeyUp(e);

            WorkspaceManager.KeyboardInputManager.OnKeyUp(new KeyboardInput(e.KeyValue, e.Alt, e.Control, e.Shift));
        }

        protected override void OnGotFocus(EventArgs e)
        {
            base.OnGotFocus(e);

            var keyInput = new KeyboardInput(
                (int)Keys.None,
                (Control.ModifierKeys & Keys.Alt) == Keys.Alt,
                (Control.ModifierKeys & Keys.Control) == Keys.Control,
                (Control.ModifierKeys & Keys.Shift) == Keys.Shift
            );

            WorkspaceManager.KeyboardInputManager.OnGotFocus(keyInput);
        }

        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);

            WorkspaceManager.KeyboardInputManager.OnLostFocus(new KeyboardInput((int)Keys.None, false, false, false));
        }
    }
}
