﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using Blocks.Core;
using EffectCombiner.Primitives.Blocks;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Operations;
using EffectDefinitions;
using Globals = EffectCombiner.Primitives.Globals;

namespace EffectCombiner.Editor.Controls
{
    /// <summary>
    /// ユーザデータブロックの値を入力するパネルです。
    /// </summary>
    public partial class EffectBlockValuePanel : UserControl, IBlockElementEditPanel
    {
        /// <summary>
        /// 左マウス押下中かどうか。
        /// </summary>
        private bool leftMouseDown;

        /// <summary>
        /// マウスのスクリーン座標押下位置です。
        /// </summary>
        private Point screenMouseDownPosition;

        /// <summary>
        /// 編集中のブロックエレメントです。
        /// </summary>
        private EffectBlockElementBase blockElement;

        /// <summary>
        /// パネルの非表示設定を無視するかどうか。
        /// ブロックエレメントを置き換えたときにパネルの表示がちらつかくのを防ぐため使用します。
        /// </summary>
        private bool ignoreHideOperation;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EffectBlockValuePanel()
        {
            this.InitializeComponent();

            this.SetupSizersLayout();

            uniformValueControl.DataChanged += this.UniformValueControl_DataChanged;
            Primitives.Generation.Globals.UniformManager.UniformDataChanged += this.UniformData_ValuesChanged;
        }

        /// <summary>
        /// ブロック定義情報を取得または設定します。
        /// </summary>
        public EffectDefinitionsContainer DefinitionContainer { get; set; }

        /// <summary>
        /// 指定されたブロックエレメントがこのパネルで編集可能かどうか取得します。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        /// <returns>編集可能なときはtrue、それ以外はfalseを返します。</returns>
        public bool CanEditBlockElement(BlockElementBase blockElement)
        {
            return (blockElement is EffectBlockElementBase);
        }

        /// <summary>
        /// 編集するブロックエレメントを設定します。
        /// </summary>
        /// <param name="blockElement">編集するブロックエレメント</param>
        public void SetBlockElement(BlockElementBase blockElement)
        {
            this.blockElement = blockElement as EffectBlockElementBase;

            // ブロックエレメントが null ならパネルを非表示にする
            if (this.blockElement == null)
            {
                if (this.ignoreHideOperation == false)
                {
                    this.Visible = false;
                }

                return;
            }

            // Visible状態が即時に変わらないため、一時変数で対応
            bool comboBoxVisible = false;
            bool valueControlVisible = false;

            // ブロックタイプ選択コンボボックスを更新
            {
                cmbBlockType.Items.Clear();

                var group = this.blockElement.BlockDefinition.Group;

                if (string.IsNullOrEmpty(group) == false)
                {
                    var groupBlocks = this.DefinitionContainer.Blocks.Where(p => p.Group == group);
                    int counter = 0;

                    // グループが同じブロックを選択項目に追加する
                    foreach (var block in groupBlocks)
                    {
                        cmbBlockType.Items.Add(block.Name);

                        // 現在のブロックタイプを選択しておく
                        if (block == this.blockElement.BlockDefinition)
                        {
                            cmbBlockType.SelectedIndexChanged -= this.BlockType_SelectedChanged;
                            cmbBlockType.SelectedIndex = counter;
                            cmbBlockType.SelectedIndexChanged += this.BlockType_SelectedChanged;
                        }

                        counter++;
                    }

                    comboBoxVisible = true;
                }
                else
                {
                    comboBoxVisible = false;
                }

                this.lblShaderType.Visible = comboBoxVisible;
                this.cmbBlockType.Visible = comboBoxVisible;
            }

            // ユニフォーム値を変更するパネルを設定
            if (this.blockElement is UserDefinitionBlockElement)
            {
                var uniformBlock = this.blockElement as UserDefinitionBlockElement;

                if (((UserDefinition)uniformBlock.BlockDefinition).CanEditValue)
                {
                    var typeName = uniformBlock.BlockDefinition.Uniform.Type;
                    var shaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(typeName);

                    var dimmensionX = GlslTypingUtility.GetDimensionX(shaderType);
                    var dimmensionY = GlslTypingUtility.GetDimensionY(shaderType);
                    var dataType = GlslTypingUtility.GetPrimitiveShaderType(shaderType);

                    var strArray = UniformDataHelper.UniformDataToStringArray(uniformBlock.UniformValue);
                    uniformValueControl.SetInfo(dataType, dimmensionX, dimmensionY, strArray);

                    valueControlVisible = true;
                }
                else
                {
                    valueControlVisible = false;
                }

                this.uniformValueControl.Visible = valueControlVisible;

                if (!comboBoxVisible)
                {
                    this.uniformValueControl.Top = 15;
                }
                else
                {
                    this.uniformValueControl.Top = 41;
                }
            }

            this.Visible = (comboBoxVisible || valueControlVisible);
        }

        /// <summary>
        /// 編集中のブロックエレメントを取得します。
        /// </summary>
        /// <returns>編集中のブロックエレメントを返します。</returns>
        public BlockElementBase GetBlockElement()
        {
            return this.blockElement;
        }

        /// <summary>
        /// ユーザがUniformのパネルに値を打ち込むと呼ばれるメソッド
        /// </summary>
        /// <param name="sender">コントロール</param>
        /// <param name="e">イベントパラメータ</param>
        private void UniformValueControl_DataChanged(object sender, EventArgs e)
        {
            if (this.blockElement != null)
            {
                // ユーザが値を変更する度に、履歴を積む
                var newValue = uniformValueControl.Values;
                var oldValue = uniformValueControl.OldValues;

                bool modified = !Core.CoreUtility.ArrayEquals(newValue, oldValue);

                if (modified)
                {
                    var uniformName = this.blockElement.BlockDefinition.Uniform.Name;
                    var uniformType = this.blockElement.BlockDefinition.Uniform.Type;
                    var oldUniformData = UniformDataHelper.StringArrayToUniformData(uniformType, oldValue);
                    var newUniformData = UniformDataHelper.StringArrayToUniformData(uniformType, newValue);
                    var operation = new UniformValueChangeOperation(this.blockElement.BlockManager, uniformName, oldUniformData, newUniformData);
                    Globals.MainOperationManager.Add(operation, true);
                }

                Globals.WorkspaceManager.InvalidateRender();
            }
        }

        /// <summary>
        /// レイアウト定義です。
        /// </summary>
        private void SetupSizersLayout()
        {
            const int SizerThickness = 4;

            this.pnlAngleSizer.Left = 0;
            this.pnlAngleSizer.Top = 0;
            this.pnlAngleSizer.Width = SizerThickness;
            this.pnlAngleSizer.Height = SizerThickness;
            this.pnlAngleSizer.Cursor = Cursors.SizeNWSE;
            this.pnlAngleSizer.MouseDown += this.PnlAnySizer_MouseDown;
            this.pnlAngleSizer.MouseMove += this.PnlAngleSizer_MouseMove;
            this.pnlAngleSizer.MouseUp += this.PnlAnySizer_MouseUp;

            this.pnlHorizontalSizer.Left = 0;
            this.pnlHorizontalSizer.Top = SizerThickness;
            this.pnlHorizontalSizer.Width = SizerThickness;
            this.pnlHorizontalSizer.Height = this.Height - SizerThickness;
            this.pnlHorizontalSizer.Cursor = Cursors.SizeWE;
            this.pnlHorizontalSizer.MouseDown += this.PnlAnySizer_MouseDown;
            this.pnlHorizontalSizer.MouseMove += this.PnlHorizontalSizer_MouseMove;
            this.pnlHorizontalSizer.MouseUp += this.PnlAnySizer_MouseUp;

            this.pnlVerticalSizer.Left = SizerThickness;
            this.pnlVerticalSizer.Top = 0;
            this.pnlVerticalSizer.Width = this.Width - SizerThickness;
            this.pnlVerticalSizer.Height = SizerThickness;
            this.pnlVerticalSizer.Cursor = Cursors.SizeNS;
            this.pnlVerticalSizer.MouseDown += this.PnlAnySizer_MouseDown;
            this.pnlVerticalSizer.MouseMove += this.PnlVerticalSizer_MouseMove;
            this.pnlVerticalSizer.MouseUp += this.PnlAnySizer_MouseUp;
        }

        #region[mouse operation]

        /// <summary>
        /// サイズ変更UIのマウス押した時のイベントです。
        /// </summary>
        /// <param name="sender">コントロール</param>
        /// <param name="e">イベントパラメータ</param>
        void PnlAnySizer_MouseDown(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.leftMouseDown = true;
                this.screenMouseDownPosition = UserControl.MousePosition;
            }
        }

        /// <summary>
        /// サイズ変更UIのマウスを離した時のイベントです。
        /// </summary>
        /// <param name="sender">コントロール</param>
        /// <param name="e">イベントパラメータ</param>
        void PnlAnySizer_MouseUp(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                this.leftMouseDown = false;
            }
        }

        /// <summary>
        /// 斜め方向のサイズ変更イベントです。
        /// </summary>
        /// <param name="sender">コントロール</param>
        /// <param name="e">イベントパラメータ</param>
        void PnlAngleSizer_MouseMove(object sender, MouseEventArgs e)
        {
            if (this.leftMouseDown)
            {
                var dx = MousePosition.X - this.screenMouseDownPosition.X;
                var dy = MousePosition.Y - this.screenMouseDownPosition.Y;

                this.MouseMoveX(dx);
                this.MouseMoveY(dy);

                this.screenMouseDownPosition = UserControl.MousePosition;
            }
        }

        /// <summary>
        /// 横方向のサイズ変更イベントです。
        /// </summary>
        /// <param name="sender">コントロール</param>
        /// <param name="e">イベントパラメータ</param>
        void PnlHorizontalSizer_MouseMove(object sender, MouseEventArgs e)
        {
            if (this.leftMouseDown)
            {
                var dx = UserControl.MousePosition.X - this.screenMouseDownPosition.X;

                this.MouseMoveX(dx);

                this.screenMouseDownPosition = UserControl.MousePosition;
            }
        }

        /// <summary>
        /// 縦方向のサイズ変更イベントです。
        /// </summary>
        /// <param name="sender">コントロール</param>
        /// <param name="e">イベントパラメータ</param>
        void PnlVerticalSizer_MouseMove(object sender, MouseEventArgs e)
        {
            if (this.leftMouseDown)
            {
                var dy = UserControl.MousePosition.Y - this.screenMouseDownPosition.Y;

                this.MouseMoveY(dy);

                this.screenMouseDownPosition = UserControl.MousePosition;
            }
        }

        /// <summary>
        /// マウスのX軸移動です。
        /// </summary>
        /// <param name="deltaX">移動量</param>
        private void MouseMoveX(int deltaX)
        {
            this.Width -= deltaX;
            this.Left += deltaX;
        }

        /// <summary>
        /// マウスのY軸移動です。
        /// </summary>
        /// <param name="deltaY">移動量</param>
        private void MouseMoveY(int deltaY)
        {
            this.Height -= deltaY;
            this.Top += deltaY;
        }

        #endregion

        /// <summary>
        /// Uniform値の変更イベントです。
        /// </summary>
        /// <param name="sender">コントロール</param>
        /// <param name="e">イベントパラメータ</param>
        private void UniformData_ValuesChanged(object sender, UniformDataChangedEventArgs e)
        {
            if (this.blockElement == null)
            {
                return;
            }

            if (this.blockElement is UserDefinitionBlockElement)
            {
                var uniformBlock = this.blockElement as UserDefinitionBlockElement;

                if (!((UserDefinition)uniformBlock.BlockDefinition).CanEditValue)
                {
                    return;
                }
            }

            var typeName = this.blockElement.BlockDefinition.OutputPlugs[0].Type.TypeString;
            var shaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(typeName);
            var functionDefinition = this.blockElement.BlockDefinition.CurrentFunctionDefinition;
            var dimmensionX = GlslTypingUtility.GetDimensionX(shaderType);
            var dimmensionY = GlslTypingUtility.GetDimensionY(shaderType);
            var dataType = GlslTypingUtility.GetPrimitiveShaderType(shaderType);
            var strArray = UniformDataHelper.UniformDataToStringArray(e.NewValue);

            uniformValueControl.SetInfo(dataType, dimmensionX, dimmensionY, strArray);
        }

        /// <summary>
        /// ブロック種類の変更イベントです。
        /// </summary>
        /// <param name="sender">送信元</param>
        /// <param name="e">イベント</param>
        private void BlockType_SelectedChanged(object sender, EventArgs e)
        {
            ComboBox cmb = sender as ComboBox;
            var tag = this.blockElement.BlockDefinition.Tags.FirstOrDefault();
            var blockDefinition = this.DefinitionContainer.Blocks
                .Where(p => tag == p.Tags.FirstOrDefault())
                .Where(p => p.Name == cmb.SelectedItem.ToString()).FirstOrDefault();

            // ブロックエレメントの置き換えでパネルがちらつかないように、一時的にパネルの非表示設定を無視させる
            this.ignoreHideOperation = true;

            var operation =
                new ReplaceBlockOperation(
                    this.blockElement.BlockManager,
                    this.blockElement,
                    blockDefinition);
            Globals.MainOperationManager.Add(operation);
            operation.Execute();

            this.ignoreHideOperation = false;

            Globals.WorkspaceManager.InvalidateRender();
        }
    }
}
