﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Windows.Forms;
using EffectCombiner.Primitives.Blocks;
using EffectCombiner.Primitives.Operations;
using EffectCombiner.Primitives.Generation;
using Renderer2D.Core;
using Blocks.Core;

using Globals = EffectCombiner.Primitives.Globals;
using Point = System.Drawing.Point;

namespace EffectCombiner.Editor.Controls
{
    /// <summary>
    /// コメントを入力するパネル
    /// </summary>
    public partial class CommentTextPanel : UserControl, IBlockElementEditPanel
    {
        /// <summary>
        /// 編集前の値
        /// </summary>
        private string oldText;
        private ISize oldSize;

        /// <summary>
        /// 編集中のブロックエレメントです。
        /// </summary>
        private CommentBlockElement blockElement;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public CommentTextPanel()
        {
            InitializeComponent();

            SetupSizersLayout();

            // テキスト編集中もリアルタイムに反映
            textBox1.TextChanged += text_DataChanged;

            // フォーカスを受けたら値をキープ
            textBox1.GotFocus += (s, e) =>
            {
                oldText = textBox1.Text;
                oldSize = new Size(blockElement.Width, blockElement.Height);
            };

            // フォーカスが外れたら履歴に積む
            textBox1.LostFocus += text_Validated;

            // Enterキーが押されたら履歴に積む
            textBox1.KeyDown += (s, e) =>
            {
                if (e.KeyCode == Keys.Enter)
                {
                    text_Validated(null, null);
                }
            };

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    lblShaderType.Text = "Comment:";
                    textBox1.Left = lblShaderType.Right + 6;
                });

            Disposed += (ss, ee) => localizationSubscription.Dispose();
        }

        private void text_Validated(object sender, EventArgs e)
        {
            // 同値は処理せず
            if (this.oldText == textBox1.Text)
            {
                return;
            }

            var newSize = new Size(blockElement.Width, blockElement.Height);

            // ユーザが値を変更する度に、履歴を積む
            Globals.MainOperationManager.Add(new CommentTextChangeOperation(
                Globals.WorkspaceManager.BlockManager, blockElement,
                this.oldText, oldSize, textBox1.Text, newSize));

            // 編集前の値を更新
            this.oldText = textBox1.Text;
            oldSize = newSize;
        }

        /// <summary>
        /// ユーザがConstantのパネルに値を打ち込むと呼ばれるメソッド
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e">イベントデータ</param>
        private void text_DataChanged(object sender, EventArgs e)
        {
            if (blockElement != null)
            {
                blockElement.Text = textBox1.Text;
                Globals.WorkspaceManager.InvalidateRender();
            }
        }

        private void SetupSizersLayout()
        {
            const int sizerThickness = 4;

            pnlAngleSizer.Left = 0;
            pnlAngleSizer.Top = 0;
            pnlAngleSizer.Width = sizerThickness;
            pnlAngleSizer.Height = sizerThickness;
            pnlAngleSizer.Cursor = Cursors.SizeNWSE;
            pnlAngleSizer.MouseDown += pnlAnySizer_MouseDown;
            pnlAngleSizer.MouseMove += pnlAngleSizer_MouseMove;
            pnlAngleSizer.MouseUp += pnlAnySizer_MouseUp;

            pnlHorizontalSizer.Left = 0;
            pnlHorizontalSizer.Top = sizerThickness;
            pnlHorizontalSizer.Width = sizerThickness;
            pnlHorizontalSizer.Height = Height - sizerThickness;
            pnlHorizontalSizer.Cursor = Cursors.SizeWE;
            pnlHorizontalSizer.MouseDown += pnlAnySizer_MouseDown;
            pnlHorizontalSizer.MouseMove += pnlHorizontalSizer_MouseMove;
            pnlHorizontalSizer.MouseUp += pnlAnySizer_MouseUp;

            pnlVerticalSizer.Left = sizerThickness;
            pnlVerticalSizer.Top = 0;
            pnlVerticalSizer.Width = Width - sizerThickness;
            pnlVerticalSizer.Height = sizerThickness;
            pnlVerticalSizer.Cursor = Cursors.SizeNS;
            pnlVerticalSizer.MouseDown += pnlAnySizer_MouseDown;
            pnlVerticalSizer.MouseMove += pnlVerticalSizer_MouseMove;
            pnlVerticalSizer.MouseUp += pnlAnySizer_MouseUp;
        }

        private bool leftMouseDown;
        private Point screenMouseDownPosition;

        void pnlAnySizer_MouseDown(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                leftMouseDown = true;
                screenMouseDownPosition = MousePosition;
            }
        }

        void pnlAnySizer_MouseUp(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
                leftMouseDown = false;
        }

        void pnlAngleSizer_MouseMove(object sender, MouseEventArgs e)
        {
            if (leftMouseDown)
            {
                var dx = MousePosition.X - screenMouseDownPosition.X;
                var dy = MousePosition.Y - screenMouseDownPosition.Y;

                MouseMoveX(dx);
                MouseMoveY(dy);

                screenMouseDownPosition = MousePosition;
            }
        }

        void pnlHorizontalSizer_MouseMove(object sender, MouseEventArgs e)
        {
            if (leftMouseDown)
            {
                var dx = MousePosition.X - screenMouseDownPosition.X;

                MouseMoveX(dx);

                screenMouseDownPosition = MousePosition;
            }
        }

        void pnlVerticalSizer_MouseMove(object sender, MouseEventArgs e)
        {
            if (leftMouseDown)
            {
                var dy = MousePosition.Y - screenMouseDownPosition.Y;

                MouseMoveY(dy);

                screenMouseDownPosition = MousePosition;
            }
        }

        private void MouseMoveX(int deltaX)
        {
            this.Width -= deltaX;
            this.Left += deltaX;
        }

        private void MouseMoveY(int deltaY)
        {
            this.Height -= deltaY;
            this.Top += deltaY;
        }

        /// <summary>
        /// 指定されたブロックエレメントがこのパネルで編集可能かどうか取得します。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        /// <returns>編集可能なときはtrue、それ以外はfalseを返します。</returns>
        public bool CanEditBlockElement(BlockElementBase blockElement)
        {
            return (blockElement is CommentBlockElement);
        }

        /// <summary>
        /// 編集するブロックエレメントを設定します。
        /// </summary>
        /// <param name="blockElement">編集するブロックエレメント</param>
        public void SetBlockElement(BlockElementBase blockElement)
        {
            // 前のブロックエレメントに登録していたイベントを解除する
            if (this.blockElement != null)
            {
                this.blockElement.ValuesChanged -= commentBlockElement_ValuesChanged;
            }

            this.blockElement = blockElement as CommentBlockElement;

            // ブロックエレメントが null ならパネルを非表示にする
            if (this.blockElement == null)
            {
                this.Visible = false;
                return;
            }

            this.blockElement.ValuesChanged += commentBlockElement_ValuesChanged;

            UpdateData();

            this.Visible = true;
        }

        /// <summary>
        /// 編集中のブロックエレメントを取得します。
        /// </summary>
        /// <returns>編集中のブロックエレメントを返します。</returns>
        public BlockElementBase GetBlockElement()
        {
            return this.blockElement;
        }

        private void commentBlockElement_ValuesChanged(object sender, EventArgs e)
        {
            if (blockElement.Values.LongLength != 1) return;
            textBox1.Text = blockElement.Values[0, 0];
        }

        private void UpdateData()
        {
            blockElement.UpdateData();
            if (blockElement.Values.LongLength != 1) return;
            textBox1.Text = blockElement.Values[0, 0];
        }
    }
}
