﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace EffectCombiner.Editor.Log
{
    /// <summary>
    /// コンソール画面へのログ出力を行うクラスです。
    /// </summary>
    public class ConsoleDisplay : ILogHandler
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ConsoleDisplay()
        {
        }

        /// <summary>
        /// ファイナライザです。
        /// </summary>
        ~ConsoleDisplay()
        {
            Debug.Assert(this.Enabled == false);
        }

        /// <summary>
        /// ログハンドラの名前を取得します。
        /// </summary>
        public string LogHandlerName
        {
            get { return "Console"; }
        }

        #if false
        /// <summary>
        /// ログハンドラの同期コンテキストを取得します。
        /// </summary>
        public SynchronizationContext SynchronizationContext
        {
            get { return null; }
        }
        #endif

        /// <summary>
        /// コンソール画面が有効かどうか取得します。
        /// </summary>
        public bool Enabled { get; private set; }

        /// <summary>
        /// コンソール画面がアタッチされたかどうかを取得します。
        /// コンソールランチャーの起動時は常にfalseです。
        /// </summary>
        public bool Attached { get; private set; }

        /// <summary>
        /// コンソールの自動割り当てが有効かどうか取得します。
        /// </summary>
        public bool AllocConsoleEnabled { get; private set; }

        /// <summary>
        /// サイレントモードかどうか取得または設定します。
        /// </summary>
        public bool IsSilentMode { get; set; }

        /// <summary>
        /// コンソール画面を初期化します。
        /// </summary>
        /// <param name="enableAllocConsole">コンソールの自動割り当てを行うかどうか</param>
        public void Initialize(bool enableAllocConsole)
        {
            // 表示中のコンソール画面を取得
            if (AttachConsole(AttachParentProcess) == true)
            {
                Console.WriteLine(string.Empty);  // 先頭を改行
                this.Attached = true;
            }
            else if (enableAllocConsole)
            {
                // コンソール画面を表示していないとき、自分でコンソール画面を表示させる
                this.Attached = AllocConsole();
            }

            this.AllocConsoleEnabled = enableAllocConsole;

            // 初期化完了！
            this.Enabled = true;
        }

        /// <summary>
        /// コンソール画面を破棄します。
        /// </summary>
        public void Release()
        {
            if (this.Attached)
            {
                FreeConsole();
                this.Attached = false;
            }

            this.Enabled = false;
        }

        /// <summary>
        /// ロガーからメッセージが送られたときの処理を行います。
        /// </summary>
        /// <param name="destinations">ログ出力先</param>
        /// <param name="level">ログレベル</param>
        /// <param name="message">ログメッセージ</param>
        /// <param name="callStack">コールスタック</param>
        public void Log(IEnumerable<string> destinations, LogLevels level, string message, StackFrame callStack)
        {
            if (this.Enabled == false) return;

            // リリースビルドのときはデバッグメッセージを表示しないようにする
            #if !DEBUG
            if (level == LogLevels.Debug)
            {
                return;
            }
            #endif

            // サイレントモードのときはエラーメッセージ以外を表示しないようにする
            if (this.IsSilentMode && level != LogLevels.Error && level != LogLevels.Fatal)
            {
                return;
            }

            // 出力先として明示的に"Console"を指定しているエラーは、
            // コンソールが無効なときでもメッセージボックスを出してエラー表示する
            if (destinations.Any(dest => dest == "Console") && level == LogLevels.Error)
            {
                // ただし、コンソールランチャーからの起動時はダイアログは出さない（出せない）
                if (this.AllocConsoleEnabled && this.Attached == false && Program.SyncContext != null)
                {
                    MessageBox.Show(
                        message,
                        Properties.Resources.ErrorCaption,
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Error);

                    return;
                }
            }

            var originalColor = ConsoleColor.White;

            // 文字色を設定
            try
            {
                // 現在の文字色を取得
                originalColor = Console.ForegroundColor;

                // 文字色をログレベルに対応した色に変更
                switch (level)
                {
                    case LogLevels.Profile:
                        Console.ForegroundColor = ConsoleColor.Cyan;
                        break;

                    case LogLevels.Information:
                        Console.ForegroundColor = ConsoleColor.Gray;
                        break;

                    case LogLevels.Warning:
                        Console.ForegroundColor = ConsoleColor.Yellow;
                        break;

                    case LogLevels.Error:
                        Console.ForegroundColor = ConsoleColor.Red;
                        break;

                    default:
                        Console.ForegroundColor = ConsoleColor.White;
                        break;
                }
            }
            catch
            {
            }

            TextWriter output = (level == LogLevels.Error ? Console.Error : Console.Out);

            // ログメッセージを表示
            output.WriteLine(message);

            // 文字色を元に戻す
            try
            {
                Console.ForegroundColor = originalColor;
            }
            catch
            {
            }
        }

        /// <summary>
        /// Win32 API Constant value: ATTACH_PARENT_PROCESS
        /// </summary>
        private const int AttachParentProcess = -1;

        //// <summary>
        //// Win32 API Function: AttachConsole(DWORD)
        //// </summary>
        [DllImport("kernel32.dll")]
        private static extern bool AttachConsole(int processId);

        //// <summary>
        //// Win32 API Function: AttachConsole()
        //// </summary>
        [DllImport("kernel32.dll")]
        private static extern bool AllocConsole();

        //// <summary>
        //// Win32 API Function: FreeConsole()
        //// </summary>
        [DllImport("kernel32.dll")]
        private static extern bool FreeConsole();
    }
}
