﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.IO;
using EffectDefinitions;
using EffectCombiner.Primitives;

namespace EffectCombiner.BlockAssistant
{
    public static class XmlGeneration
    {
        public static string FromFunctionDefinition(FunctionDefinition funcDef, bool full = false)
        {
            if (funcDef == null)
                throw new ArgumentNullException("funcDef");

            var ms = new MemoryStream();
            using (var writer = new XmlTextWriter(ms, Encoding.UTF8))
            {
                writer.Indentation = 2;
                writer.IndentChar = ' ';
                writer.Formatting = Formatting.Indented;

                writer.WriteStartElement("func");
                {
                    writer.WriteAttributeString("id", funcDef.Id);
                    writer.WriteAttributeString("name", funcDef.Name);

                    if (full)
                    {
                        writer.WriteStartElement("description");
                        writer.WriteString("Function description goes here");
                        writer.WriteEndElement();

                        writer.WriteStartElement("tag");
                        writer.WriteString("A tag goes here");
                        writer.WriteEndElement();

                        writer.WriteStartElement("tag");
                        writer.WriteString("Another tag goes here");
                        writer.WriteEndElement();
                    }

                    foreach (var p in funcDef.Parameters)
                    {
                        var elementName = "in";
                        if (p.Direction == ShaderTyping.ParameterDirection.Output)
                            elementName = "out";
                        else if (p.Direction == ShaderTyping.ParameterDirection.Reference)
                            elementName = "ref";
                        WriteParam(writer, elementName, p.Name, p.Type.ToString(), full);
                    }

                    if (funcDef.ReturnType.IsVoid == false)
                        WriteParam(writer, "return", null, funcDef.ReturnType.ToString(), full);
                }
                writer.WriteEndElement();
            }

            return Encoding.UTF8.GetString(ms.ToArray());
        }

        private static void WriteParam(XmlWriter writer, string elementName, string name, string type, bool full)
        {
            writer.WriteStartElement(elementName);
            if (name != null)
                writer.WriteAttributeString("name", name);
            writer.WriteAttributeString("type", type);

            if (full)
            {
                writer.WriteAttributeString("usage", "usage?");

                if (elementName == "return")
                    writer.WriteString("Description of return value goes here");
                else
                    writer.WriteString(string.Format("Description of parameter {0} goes here", elementName));
            }

            writer.WriteEndElement();
        }

        public static string FromBlockDefinition(BlockDefinition blockDef, bool full = false)
        {
            if (blockDef == null)
                throw new ArgumentNullException("blockDef");

            var ms = new MemoryStream();
            using (var writer = new XmlTextWriter(ms, Encoding.UTF8))
            {
                writer.Indentation = 2;
                writer.IndentChar = ' ';
                writer.Formatting = Formatting.Indented;

                writer.WriteStartElement("block");
                {
                    writer.WriteAttributeString("guid", blockDef.Guid);
                    writer.WriteAttributeString("displayname", blockDef.Name);

                    if (full)
                    {
                        writer.WriteStartElement("description");
                        writer.WriteString("Block description goes here");
                        writer.WriteEndElement();
                    }

                    writer.WriteStartElement("func");
                    writer.WriteAttributeString("id", blockDef.CurrentFunctionDefinition.Id);
                    writer.WriteEndElement();

                    foreach (var plug in blockDef.InputPlugs)
                    {
                        writer.WriteStartElement("in");
                        writer.WriteAttributeString("displayname", plug.Name);
                        writer.WriteAttributeString("type", plug.Type.ToString());
                        writer.WriteAttributeString("target", plug.Target);
                        if (full)
                        {
                            writer.WriteAttributeString("semantic", "semantic?");
                            try
                            {
                                var shaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(plug.Type.TypeString);
                                var dimX = GlslTypingUtility.GetDimensionX(shaderType);
                                var dimY = GlslTypingUtility.GetDimensionY(shaderType);
                                writer.WriteAttributeString("defaultvalue", string.Join(",", Enumerable.Repeat("0", dimX * dimY)));
                            }
                            catch
                            {
                                writer.WriteAttributeString("defaultvalue", "?");
                            }
                            writer.WriteString("Input plug description goes here");
                        }
                        writer.WriteEndElement();
                    }

                    foreach (var plug in blockDef.OutputPlugs)
                    {
                        writer.WriteStartElement("out");
                        writer.WriteAttributeString("displayname", plug.Name);
                        writer.WriteAttributeString("type", plug.Type.ToString());
                        writer.WriteAttributeString("target", plug.Target);
                        if (full)
                            writer.WriteString("Output plug description goes here");
                        writer.WriteEndElement();
                    }
                }
                writer.WriteEndElement();
            }

            return Encoding.UTF8.GetString(ms.ToArray());
        }
    }
}
