﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RGetSliceInfos ParseSliceResource6 ParseSliceResource7
// AppendSliceInfo GetVisibleSliceRects
// SkipDescriptorItem SkipReferenceItem

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//-----------------------------------------------------------------------------
//! @brief Photoshop リソースから Unicode 文字列を取得し、ポインタをインクリメントします。
//-----------------------------------------------------------------------------
static std::wstring GetMemPSUnicodeString(const uint8_t*& pBuf)
{
    const int len = RGetMemBigInt(pBuf);
    pBuf += sizeof(int);
    std::wstring str;
    if (len > 0)
    {
        wchar_t* pStr = new wchar_t[len];
        for (int ic = 0; ic < len; ++ic)
        {
            pStr[ic] = RGetMemBigUShort(pBuf);
            pBuf += sizeof(unsigned short);
        }
        str = std::wstring(pStr, len);
    }
    return str;
}

//-----------------------------------------------------------------------------
//! @brief 矩形のデバッグ用文字列を返します。
//-----------------------------------------------------------------------------
inline std::string GetRectString(const VRect& rect)
{
    return "(" +
        RGetNumberString(rect.left  , "%2d") + ", " +
        RGetNumberString(rect.top   , "%2d") + ") - (" +
        RGetNumberString(rect.right , "%2d") + ", " +
        RGetNumberString(rect.bottom, "%2d") + ")";
}

//-----------------------------------------------------------------------------
//! @brief スライス情報のデバッグ用文字列を返します。
//-----------------------------------------------------------------------------
std::string RSliceInfo::GetString(void) const
{
    return "(" +
        RGetNumberString(m_X, "%2d") + ", " +
        RGetNumberString(m_Y, "%2d") + ") (" +
        RGetNumberString(m_W, "%2d") + " x " +
        RGetNumberString(m_H, "%2d") + ") " +
        RGetNumberString(m_IsUser ? 1 : 0) + " " +
        RGetNumberString(m_IsImage ? 1 : 0) + " " +
        RGetNumberString(m_IsDivided ? 1 : 0) + " \"" +
        m_IndexName + "\" \"" +
        m_ZeroBasedName + "\" \"" +
        m_Name + "\"";
}

//-----------------------------------------------------------------------------
//! @brief 選択されたスライスの ID を取得します。
//!        CS6 SDK の PIGetSelectedSliceID では
//!        int32 型へのポインタを intptr_t 型へのポインタに変換していて危険なので
//!        替わりにこの関数を使用します。
//!
//! @return 選択されたスライスの ID を返します。選択されていなければ -1 を返します。
//-----------------------------------------------------------------------------
static int GetSelectedSliceID(void)
{
    intptr_t sliceID = 0;
    sPSProperty->getPropertyProc(kPhotoshopSignature, propSelectedSliceID, 0, &sliceID, 0);
    return static_cast<int>(sliceID);
}

//-----------------------------------------------------------------------------
//! @brief 選択されたスライスのインデックスを取得します。
//!        CS6 SDK の PIGetSelectedSliceIndex では
//!        int32 型へのポインタを intptr_t 型へのポインタに変換していて危険なので
//!        替わりにこの関数を使用します。
//!
//! @return 選択されたスライスのインデックスを返します。選択されていなければ 0 を返します。
//-----------------------------------------------------------------------------
static int GetSelectedSliceIndex(void)
{
    intptr_t sliceIndex = 0;
    sPSProperty->getPropertyProc(kPhotoshopSignature, propSelectedSliceIndex, 0, &sliceIndex, 0);
    return static_cast<int>(sliceIndex);
}

//-----------------------------------------------------------------------------
//! @brief スライス ID からスライスインデックスを取得します。
//!
//! @param[in] id スライス ID です。
//!
//! @return スライスインデックスを返します。
//-----------------------------------------------------------------------------
static int GetSliceIndexFromId(const int id)
{
    int index = 0;
    if (id > 0)
    {
        // スライス ID を指定して選択し、選択されたスライスのインデックスを取得します。
        const int idBak = GetSelectedSliceID();
        PISetSelectedSliceID(id);
        index = GetSelectedSliceIndex();
        PISetSelectedSliceID(idBak); // 選択をリストアします。
    }
    return index;
}

//-----------------------------------------------------------------------------
//! @brief ポイントがスライスの範囲内（端の 1 ピクセルを除く）にあれば true を返します。
//-----------------------------------------------------------------------------
static bool IsPointInnerSliceBounds(const VRect& bounds, const int x, const int y)
{
    return (
        bounds.left + 1 <= x && x <= bounds.right  - 2 &&
        bounds.top  + 1 <= y && y <= bounds.bottom - 2);
}

//-----------------------------------------------------------------------------
//! @brief 可視ピクセルから可視のスライスの矩形配列を取得します。
//!
//! @param[out] visRects 可視のスライスの矩形配列を格納します。
//! @param[in] srcRect 対象スライスの矩形です。
//! @param[in] frontRects 前面にあるスライスの矩形配列です。
//-----------------------------------------------------------------------------
static void GetVisibleSliceRectsByPixel(
    std::vector<VRect>& visRects,
    const VRect& srcRect,
    const std::vector<VRect>& frontRects
)
{
    //-----------------------------------------------------------------------------
    // 可視フラグのピクセルマップを確保します。
    const int srcW = srcRect.right  - srcRect.left;
    const int srcH = srcRect.bottom - srcRect.top;
    const int pixelCount = srcW * srcH;
    if (pixelCount == 0)
    {
        return;
    }
    uint8_t* pPixels = new uint8_t[pixelCount];
    memset(pPixels, 0x01, pixelCount); // 可視なら非 0

    //-----------------------------------------------------------------------------
    // 前面にあるスライスの範囲の可視フラグを 0 にします。
    for (size_t iFront = 0; iFront < frontRects.size(); ++iFront)
    {
        const VRect& front = frontRects[iFront];
        const int minBottom = RMin(front.bottom, srcRect.bottom);
        const int minRight  = RMin(front.right , srcRect.right );
        for (int iy = front.top; iy < minBottom; ++iy)
        {
            if (srcRect.top <= iy && iy < srcRect.bottom)
            {
                uint8_t* pDst = pPixels + (iy - srcRect.top) * srcW;
                for (int ix = front.left; ix < minRight; ++ix)
                {
                    if (srcRect.left <= ix && ix < srcRect.right)
                    {
                        pDst[ix - srcRect.left] = 0;
                    }
                }
            }
        }
    }
    //for (int iy = 0; iy < srcH; ++iy)
    //{
    //  std::string line;
    //  const uint8_t* p = pPixels + iy * srcW;
    //  for (int ix = 0; ix < srcW; ++ix) line += (*p++ != 0) ? "1" : "0";
    //  RNoteTrace(line.c_str()); // 現在最大 1024 文字までなので注意
    //}

    //-----------------------------------------------------------------------------
    // 可視フラグが非 0 の矩形配列を取得します。
    for (;;)
    {
        //-----------------------------------------------------------------------------
        // 矩形の左上を求めます。
        int iLeftTop = -1;
        for (int iPix = 0; iPix < pixelCount; ++iPix)
        {
            if (pPixels[iPix] != 0)
            {
                iLeftTop = iPix;
                break;
            }
        }
        if (iLeftTop == -1)
        {
            break;
        }

        //-----------------------------------------------------------------------------
        // 矩形を取得します。
        VRect rect;
        rect.left   = iLeftTop % srcW;
        rect.top    = iLeftTop / srcW;
        rect.right  = rect.left + 1;
        rect.bottom = rect.top  + 1;

        const uint8_t* pLine = pPixels + rect.top * srcW;
        while (rect.right < srcW && pLine[rect.right] != 0)
        {
            ++rect.right;
        }
        const int rectW = rect.right - rect.left;

        while (rect.bottom < srcH)
        {
            const uint8_t* pSrc = pPixels + rect.bottom * srcW + rect.left;
            bool zeroExists = false;
            for (int ix = 0; ix < rectW; ++ix)
            {
                if (*pSrc++ == 0)
                {
                    zeroExists = true;
                    break;
                }
            }
            if (zeroExists)
            {
                break;
            }
            ++rect.bottom;
        }
        const int rectH = rect.bottom - rect.top;
        //RNoteTrace("vis rect: %s", GetRectString(rect).c_str());

        //-----------------------------------------------------------------------------
        // 矩形の可視フラグを 0 にします。
        for (int iy = 0; iy < rectH; ++iy)
        {
            uint8_t* pDst = pPixels + (rect.top + iy) * srcW + rect.left;
            memset(pDst, 0x00, rectW);
        }

        //-----------------------------------------------------------------------------
        // 矩形を追加します。
        rect.left   += srcRect.left;
        rect.right  += srcRect.left;
        rect.top    += srcRect.top;
        rect.bottom += srcRect.top;
        visRects.push_back(rect);
    }

    //-----------------------------------------------------------------------------
    // 可視フラグのピクセルマップを解放します。
    delete[] pPixels;
}

//-----------------------------------------------------------------------------
//! @brief 可視のスライスの矩形配列を取得します。
//!
//! @param[out] visRects 可視のスライスの矩形配列を格納します。
//! @param[in] srcRect 対象スライスの矩形です。
//! @param[in] frontRects 前面にあるスライスの矩形配列です。
//-----------------------------------------------------------------------------
static void GetVisibleSliceRects(
    std::vector<VRect>& visRects,
    const VRect& srcRect,
    const std::vector<VRect>& frontRects
)
{
    //-----------------------------------------------------------------------------
    // 前面にある他のスライスに覆われている矩形をカットします。

    // 対象スライスが前面にある他のスライスによって分割されている場合があります。
    //   SS        FF      FFSS
    // FFFFFF    SSFFSS    SSSS
    //   SS        FF

    VRect curRect = srcRect;
    bool isDivided = false;
    for (size_t iFront = 0; iFront < frontRects.size(); ++iFront)
    {
        const VRect& front = frontRects[iFront]; // 最前面に近いスライスからチェック
        //const VRect& front = frontRects[frontRects.size() - 1 - iFront]; // 対象スライスに近いスライスからチェック

        //-----------------------------------------------------------------------------
        // 完全に覆われているかチェックします。
        const bool overH = (front.left <= curRect.left && curRect.right  <= front.right );
        const bool overV = (front.top  <= curRect.top  && curRect.bottom <= front.bottom);
        if (overH && overV)
        {
            curRect.left = curRect.top = curRect.right = curRect.bottom = 0;
            break;
        }

        //-----------------------------------------------------------------------------
        // 部分的に覆われているかチェックします。
        if (overH)
        {
            if (curRect.top < front.top && front.top < curRect.bottom) // 下側が覆われている
            {
                if (front.bottom < curRect.bottom)
                {
                    isDivided = true;
                }
                //RNoteTrace("over bottom: %d: %s <- %s", isDivided, GetRectString(curRect).c_str(), GetRectString(front).c_str());
                curRect.bottom = front.top;
            }
            else if (curRect.top < front.bottom && front.bottom < curRect.bottom) // 上側が覆われている
            {
                if (front.top > curRect.top)
                {
                    isDivided = true;
                }
                //RNoteTrace("over top: %d: %s <- %s", isDivided, GetRectString(curRect).c_str(), GetRectString(front).c_str());
                curRect.top = front.bottom;
            }
        }
        if (overV)
        {
            if (curRect.left < front.left && front.left < curRect.right) // 右側が覆われている
            {
                if (front.right < curRect.right)
                {
                    isDivided = true;
                }
                //RNoteTrace("over right: %d: %s <- %s", isDivided, GetRectString(curRect).c_str(), GetRectString(front).c_str());
                curRect.right = front.left;
            }
            else if (curRect.left < front.right && front.right < curRect.right)
            {
                // 左側が覆われている
                if (front.left > curRect.left)
                {
                    isDivided = true;
                }
                //RNoteTrace("over left: %d: %s <- %s", isDivided, GetRectString(curRect).c_str(), GetRectString(front).c_str());
                curRect.left = front.right;
            }
        }

        //-----------------------------------------------------------------------------
        // 対象スライス内に前面スライスの 4 隅のポイントが存在するかチェックします。
        if (IsPointInnerSliceBounds(curRect, front.left     , front.top       ) ||
            IsPointInnerSliceBounds(curRect, front.right - 1, front.top       ) ||
            IsPointInnerSliceBounds(curRect, front.left     , front.bottom - 1) ||
            IsPointInnerSliceBounds(curRect, front.right - 1, front.bottom - 1))
        {
            isDivided = true;
            //RNoteTrace("inner: %d: %s <- %s", isDivided, GetRectString(curRect).c_str(), GetRectString(front).c_str());
        }
    }

    //-----------------------------------------------------------------------------
    // 分割されていなければカットした矩形を返します。
    // 分割されていれば可視ピクセルから可視の矩形配列を取得します。
    if (!isDivided)
    {
        visRects.push_back(curRect);
    }
    else
    {
        GetVisibleSliceRectsByPixel(visRects, srcRect, frontRects);
    }
}

//-----------------------------------------------------------------------------
//! @brief スライス情報を配列に追加します。
//!
//! @param[in,out] infos スライス情報配列です。
//! @param[in] id スライス ID です。
//! @param[in] groupId スライスのグループ ID です。
//! @param[in] isImageSlice 画像のあるスライスなら true を指定します。
//! @param[in] srcRect 対象スライスの矩形です。
//! @param[in] frontRects 前面にあるスライスの矩形配列です。
//! @param[in] name スライス名です。
//! @param[in] cutPrefix スライス名の先頭から削除する文字列です。
//-----------------------------------------------------------------------------
static void AppendSliceInfo(
    RSliceInfoArray& infos,
    const int id,
    const int groupId,
    const bool isImageSlice,
    const VRect& srcRect,
    const std::vector<VRect>& frontRects,
    const std::wstring& name,
    const std::string& cutPrefix
)
{
    if (id != 0 || infos.empty())
    {
        //-----------------------------------------------------------------------------
        // 可視のスライスの矩形配列を取得します。
        std::vector<VRect> visRects;
        GetVisibleSliceRects(visRects, srcRect, frontRects);
        const bool isDivided = (visRects.size() >= 2);

        //-----------------------------------------------------------------------------
        // スライス情報配列に追加します。
        const int index = RMax(GetSliceIndexFromId(id), 1);
        std::string sname = RTrimString(RGetShiftJisFromUnicode(name));
        if (!cutPrefix.empty() && sname.find(cutPrefix) == 0)
        {
            sname = sname.substr(cutPrefix.length(), sname.length());
        }
        for (size_t iRect = 0; iRect < visRects.size(); ++iRect)
        {
            const VRect& visRect = visRects[iRect];
            const int isValidRect = (
                visRect.left >= 0 &&
                visRect.top  >= 0 &&
                visRect.right  > visRect.left &&
                visRect.bottom > visRect.top);
            if (isValidRect)
            {
                RSliceInfo info(index, visRect.left, visRect.top,
                    visRect.right - visRect.left, visRect.bottom - visRect.top, sname.c_str());
                info.m_IsUser = (groupId != 0);
                info.m_IsImage = isImageSlice;
                info.m_IsDivided = isDivided;
                infos.push_back(info);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief スライスリソース Ver6 を解析します。
//!
//! @param[in,out] infos スライス情報配列を格納します。
//! @param[in] cutPrefix スライス名の先頭から削除する文字列です。
//! @param[in] pSlices スライスリソースへのポインタです。
//-----------------------------------------------------------------------------
static void ParseSliceResource6(
    RSliceInfoArray& infos,
    const std::string& cutPrefix,
    const Ptr pSlices
)
{
    const uint8_t* pBuf = reinterpret_cast<const uint8_t*>(pSlices);
    pBuf += 4; // skip version (00 00 00 06)
    pBuf += 4 * 4; // skip total rect
    const std::wstring groupName = GetMemPSUnicodeString(pBuf);
    //RNoteTrace("slice: groupName: %s", RGetShiftJisFromUnicode(groupName).c_str());
    const int sliceCount = RGetMemBigInt(pBuf);
    pBuf += 4;
    //RNoteTrace("slice: count: %d", sliceCount);
        // ユーザー定義スライスなしなら 1（画像全体スライスのみ）
        // ユーザー定義スライス 1 つなら 2

    //-----------------------------------------------------------------------------
    // スライスについてループします。
    std::vector<VRect> frontRects;
    for (int iSlice = 0; iSlice < sliceCount; ++iSlice)
    {
        // 前面にあるスライスが先
        // 他のスライスに完全に覆われているスライスも含まれる
        // ユーザー定義スライスの後に自動スライス、その後に画像全体スライス

        //-----------------------------------------------------------------------------
        // スライスリソースから値を取得します。
        const int id = RGetMemBigInt(pBuf); // 0:デフォルトで存在する画像全体スライス, 1以上:ユーザー定義スライス or 自動スライス
        pBuf += 4;
        const int groupId = RGetMemBigInt(pBuf); // 0:自動スライス or 画像全体スライス, 1以上:ユーザー定義スライス
        pBuf += 4;
        const int origin = RGetMemBigInt(pBuf); // 0:自動, 1:レイヤーに関連付け, 2:通常の矩形指定
        pBuf += 4;
        int layerId = -1;
        if (origin == 1)
        {
            layerId = RGetMemBigInt(pBuf);
            pBuf += 4;
        }
        const std::wstring name = GetMemPSUnicodeString(pBuf);
        const int type = RGetMemBigInt(pBuf); // 0:画像なし, 1:画像
        pBuf += 4;
        VRect bounds = { 0 };
        bounds.left = RGetMemBigInt(pBuf);
        pBuf += 4;
        bounds.top = RGetMemBigInt(pBuf);
        pBuf += 4;
        bounds.right = RGetMemBigInt(pBuf);
        pBuf += 4;
        bounds.bottom = RGetMemBigInt(pBuf);
        pBuf += 4;
        const std::wstring url     = GetMemPSUnicodeString(pBuf);
        const std::wstring target  = GetMemPSUnicodeString(pBuf);
        const std::wstring message = GetMemPSUnicodeString(pBuf);
        const std::wstring altTag  = GetMemPSUnicodeString(pBuf);
        const uint8_t isCellTextHtml = *pBuf++; // 0:画像なし？, 1:画像？
        const std::wstring cellText = GetMemPSUnicodeString(pBuf); // 画像なしの場合の HTML テキスト
        const int alignH = RGetMemBigInt(pBuf);
        pBuf += 4;
        const int alignV = RGetMemBigInt(pBuf);
        pBuf += 4;
        const uint8_t colA = *pBuf++;
        const uint8_t colR = *pBuf++;
        const uint8_t colG = *pBuf++;
        const uint8_t colB = *pBuf++;

        //RNoteTrace("v6 slice%d: %2d: %3d %d %d %d: [%-4s] (%2d, %2d) - (%2d, %2d) %d %d %d %02x%02x%02x%02x",
        //  iSlice, GetSliceIndexFromId(id), id, groupId, origin, type,
        //  RGetShiftJisFromUnicode(name).c_str(),
        //  bounds.left, bounds.top, bounds.right, bounds.bottom,
        //  isCellTextHtml, alignH, alignV, colR, colG, colB, colA);

        //-----------------------------------------------------------------------------
        // スライス情報を配列に追加します。
        const bool isImageSlice = (type == 1);
        AppendSliceInfo(infos, id, groupId, isImageSlice, bounds, frontRects, name, cutPrefix);
        frontRects.push_back(bounds);

        R_UNUSED_VARIABLE(isCellTextHtml);
        R_UNUSED_VARIABLE(alignH);
        R_UNUSED_VARIABLE(alignV);
        R_UNUSED_VARIABLE(colA);
        R_UNUSED_VARIABLE(colR);
        R_UNUSED_VARIABLE(colG);
        R_UNUSED_VARIABLE(colB);
    }
}

//-----------------------------------------------------------------------------
//! @brief Descriptor の項目数を取得します。
//!
//! @param[out] itemCount 項目数を格納します。
//! @param[in] pBuf Descriptor へのポインタです。
//!
//! @brief 最初の項目へのポインタ返します。
//-----------------------------------------------------------------------------
static const uint8_t* GetDescriptorItemCount(int& itemCount, const uint8_t* pBuf)
{
    const std::wstring nameFromClassId = GetMemPSUnicodeString(pBuf);
    const int classIdLen = RGetMemBigInt(pBuf);
    const int classIdDataLen = (classIdLen == 0) ? 4 : classIdLen;
    pBuf += 4;
    const std::string classId = std::string(reinterpret_cast<const char*>(pBuf), classIdDataLen);
    pBuf += classIdDataLen;
    itemCount = RGetMemBigInt(pBuf);
    pBuf += 4;
    //RNoteTrace("item count: [%s] [%s] %d", RGetShiftJisFromUnicode(nameFromClassId).c_str(), classId.c_str(), itemCount);
    return pBuf;
}

//-----------------------------------------------------------------------------
//! @brief Descriptor の項目の名前とタイプを取得します。
//!
//! @param[out] itemName 項目の名前を格納します。
//! @param[out] itemType 項目のタイプを格納します。
//! @param[in] pBuf 項目へのポインタです。
//!
//! @brief 項目の値へのポインタ返します。
//-----------------------------------------------------------------------------
static const uint8_t* GetDescriptorItemName(
    std::string& itemName,
    std::string& itemType,
    const uint8_t* pBuf
)
{
    const int itemNameLen = RGetMemBigInt(pBuf);
    const int itemNameDataLen = (itemNameLen == 0) ? 4 : itemNameLen;
    pBuf += 4;
    itemName = std::string(reinterpret_cast<const char*>(pBuf), itemNameDataLen);
    pBuf += itemNameDataLen;
    itemType = std::string(reinterpret_cast<const char*>(pBuf), 4);
    pBuf += 4;
    return pBuf;
}

//-----------------------------------------------------------------------------
//! @brief Reference の項目をスキップします。
//!
//! @param[in] pBuf 項目の値へのポインタです。
//! @param[in] type 項目のタイプです。
//!
//! @return 次の項目の値へのポインタを返します。
//-----------------------------------------------------------------------------
static const uint8_t* SkipReferenceItem(const uint8_t* pBuf, const std::string& type)
{
    if (type == "prop") // Property
    {
        GetMemPSUnicodeString(pBuf);
        const int classIdLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((classIdLen == 0) ? 4 : classIdLen);
        const int keyIdLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((keyIdLen == 0) ? 4 : keyIdLen);
    }
    else if (type == "Clss") // Class
    {
        GetMemPSUnicodeString(pBuf);
        const int classIdLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((classIdLen == 0) ? 4 : classIdLen);
    }
    else if (type == "Enmr") // Enumerated Reference
    {
        GetMemPSUnicodeString(pBuf);
        const int classIdLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((classIdLen == 0) ? 4 : classIdLen);
        const int typeLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((typeLen == 0) ? 4 : typeLen);
        const int enumLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((enumLen == 0) ? 4 : enumLen);
    }
    else if (type == "rele") // Offset
    {
        GetMemPSUnicodeString(pBuf);
        const int classIdLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((classIdLen == 0) ? 4 : classIdLen);
        pBuf += 4; // Value of the offset
    }
    else if (type == "Idnt") // Identifier
    {
        GetMemPSUnicodeString(pBuf);
        const int classIdLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((classIdLen == 0) ? 4 : classIdLen);
        pBuf += 4; // Integer Value（通常、Identifier は実行時に割り当てられるので毎回値が異なる）
    }
    else if (type == "indx") // Index
    {
        GetMemPSUnicodeString(pBuf);
        const int classIdLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((classIdLen == 0) ? 4 : classIdLen);
        pBuf += 4; // Integer Value
    }
    else if (type == "name") // Name
    {
        GetMemPSUnicodeString(pBuf);
        const int classIdLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((classIdLen == 0) ? 4 : classIdLen);
        GetMemPSUnicodeString(pBuf); // String Value
    }
    else
    {
        pBuf = NULL;
    }
    return pBuf;
}

//-----------------------------------------------------------------------------
//! @brief Descriptor の項目をスキップします。再帰的に呼ばれます。
//!
//! @param[in] pBuf 項目の値へのポインタです。
//! @param[in] type 項目のタイプです。
//!
//! @return 次の項目の値へのポインタを返します。
//-----------------------------------------------------------------------------
static const uint8_t* SkipDescriptorItem(const uint8_t* pBuf, const std::string& type)
{
    if (type == "Obj ") // Reference
    {
        const int itemCount = RGetMemBigInt(pBuf);
        pBuf += 4;
        for (int iItem = 0; iItem < itemCount; ++iItem)
        {
            const std::string itemType = std::string(reinterpret_cast<const char*>(pBuf), 4);
            pBuf += 4;
            pBuf = SkipReferenceItem(pBuf, itemType);
            if (pBuf == NULL)
            {
                break;
            }
        }
    }
    else if (type == "Objc" || type == "GlbO") // Descriptor or GlobalObject
    {
        int itemCount;
        pBuf = GetDescriptorItemCount(itemCount, pBuf);
        for (int iItem = 0; iItem < itemCount; ++iItem)
        {
            std::string itemName;
            std::string itemType;
            pBuf = GetDescriptorItemName(itemName, itemType, pBuf);
            //RNoteTrace("skip item: %d/%d: %s (%s)", iItem, itemCount, itemName.c_str(), itemType.c_str());
            pBuf = SkipDescriptorItem(pBuf, itemType);
            if (pBuf == NULL)
            {
                break;
            }
        }
    }
    else if (type == "VlLs") // List
    {
        const int valueCount = RGetMemBigInt(pBuf);
        pBuf += 4;
        for (int iValue = 0; iValue < valueCount; ++iValue)
        {
            const std::string itemType = std::string(reinterpret_cast<const char*>(pBuf), 4);
            //RNoteTrace("skip list value: %d/%d: %s", iValue, valueCount, itemType.c_str());
            pBuf += 4;
            pBuf = SkipDescriptorItem(pBuf, itemType);
            if (pBuf == NULL)
            {
                break;
            }
        }
    }
    else if (type == "doub") // Double
    {
        pBuf += 8;
    }
    else if (type == "UntF") // Unit float
    {
        pBuf += 4 + 8;
    }
    else if (type == "TEXT") // String
    {
        const int len = RGetMemBigInt(pBuf);
        pBuf += 4 + len * 2;
    }
    else if (type == "enum") // Enumerated
    {
        const int typeLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((typeLen == 0) ? 4 : typeLen);
        const int enumLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((enumLen == 0) ? 4 : enumLen);
    }
    else if (type == "long") // Integer
    {
        pBuf += 4;
    }
    else if (type == "bool") // Boolean
    {
        pBuf += 1;
    }
    else if (type == "type" || type == "GlbC") // Class
    {
        GetMemPSUnicodeString(pBuf);
        const int classIdLen = RGetMemBigInt(pBuf);
        pBuf += 4 + ((classIdLen == 0) ? 4 : classIdLen);
    }
    else if (type == "alis") // Alias
    {
        const int dataLen = RGetMemBigInt(pBuf);
        pBuf += 4 + dataLen;
    }
    else if (type == "tdta") // Raw Data
    {
        const int dataLen = RGetMemBigInt(pBuf);
        pBuf += 4 + dataLen;
    }
    else
    {
        pBuf = NULL;
    }
    return pBuf;
}

//-----------------------------------------------------------------------------
//! @brief Descriptor の項目を名前で検索します。
//!
//! @param[in] pBuf Descriptor へのポインタです。
//! @param[in] name 項目の名前です。
//!
//! @return 項目の値へのポインタを返します。見つからなければ NULL を返します。
//-----------------------------------------------------------------------------
static const uint8_t* FindDescriptorItem(const uint8_t* pBuf, const std::string& name)
{
    int itemCount;
    pBuf = GetDescriptorItemCount(itemCount, pBuf);
    for (int iItem = 0; iItem < itemCount; ++iItem)
    {
        std::string itemName;
        std::string itemType;
        pBuf = GetDescriptorItemName(itemName, itemType, pBuf);
        //RNoteTrace("desc item: %d/%d: %s (%s)", iItem, itemCount, itemName.c_str(), itemType.c_str());
        if (itemName == name)
        {
            return pBuf;
        }
        pBuf = SkipDescriptorItem(pBuf, itemType);
        //RNoteTrace("desc next: %p", pBuf);
        if (pBuf == NULL)
        {
            break;
        }
    }
    return NULL;
}

//-----------------------------------------------------------------------------
//! @brief Descriptor の項目を名前で検索して整数値を取得します。
//!
//! @param[in] pBuf Descriptor へのポインタです。
//! @param[in] name 項目の名前です。
//! @param[in] defaultValue 項目が見つからなかった場合に返すデフォルト値です。
//!
//! @return 整数値を返します。
//-----------------------------------------------------------------------------
static int FindGetIntegerDescriptorItem(
    const uint8_t* pBuf,
    const std::string& name,
    const int defaultValue = 0
)
{
    pBuf = FindDescriptorItem(pBuf, name);
    return (pBuf != NULL) ? RGetMemBigInt(pBuf) : defaultValue;
}

//-----------------------------------------------------------------------------
//! @brief Descriptor の項目から列挙型の値を取得します。
//!
//! @param[in] pBuf Descriptor の項目の値へのポインタです。
//!
//! @return 列挙型の値を文字列で返します。
//-----------------------------------------------------------------------------
static std::string GetEnumFromDescriptorItem(const uint8_t* pBuf)
{
    const int typeLen = RGetMemBigInt(pBuf);
    pBuf += 4 + ((typeLen == 0) ? 4 : typeLen);
    const int enumLen = RGetMemBigInt(pBuf);
    pBuf += 4;
    return std::string(reinterpret_cast<const char*>(pBuf), ((enumLen == 0) ? 4 : enumLen));
}

//-----------------------------------------------------------------------------
//! @brief Descriptor の項目から範囲を取得します。
//!
//! @param[in] pBuf Descriptor の項目の値へのポインタです。
//!
//! @return 範囲を返します。
//-----------------------------------------------------------------------------
static VRect GetBoundsFromDescriptorItem(const uint8_t* pBuf)
{
    VRect bounds;
    bounds.top    = FindGetIntegerDescriptorItem(pBuf, "Top ");
    bounds.left   = FindGetIntegerDescriptorItem(pBuf, "Left");
    bounds.bottom = FindGetIntegerDescriptorItem(pBuf, "Btom");
    bounds.right  = FindGetIntegerDescriptorItem(pBuf, "Rght");
    return bounds;
}

//-----------------------------------------------------------------------------
//! @brief スライスリソース Ver7 以降を解析します。
//!
//! @param[in,out] infos スライス情報配列を格納します。
//! @param[in] cutPrefix スライス名の先頭から削除する文字列です。
//! @param[in] pSlices スライスリソースへのポインタです。
//-----------------------------------------------------------------------------
static void ParseSliceResource7(
    RSliceInfoArray& infos,
    const std::string& cutPrefix,
    const Ptr pSlices
)
{
    //-----------------------------------------------------------------------------
    // Descriptor までポインタを進めます。
    const uint8_t* pBuf = reinterpret_cast<const uint8_t*>(pSlices);
    pBuf += 4; // skip version (00 00 00 07) or (00 00 00 08)
    pBuf += 4; // skip descriptor version (00 00 00 10)
    // baseName (TEXT) "ユーザ"
    // bounds (Objc)
    // slices (VlLs)

    //-----------------------------------------------------------------------------
    // スライスリストを取得します。
    pBuf = FindDescriptorItem(pBuf, "slices");
    if (pBuf == NULL)
    {
        return;
    }
    //RNoteTrace("slices list: %p", pBuf);
    const int sliceCount = RGetMemBigInt(pBuf);
    pBuf += 4;

    //-----------------------------------------------------------------------------
    // スライスについてループします。
    std::vector<VRect> frontRects;
    for (int iSlice = 0; iSlice < sliceCount && pBuf != NULL; ++iSlice)
    {
        // 前面にあるスライスが先
        // 他のスライスに完全に覆われているスライスも含まれる
        // ユーザー定義スライスの後に自動スライス、その後に画像全体スライス

        //-----------------------------------------------------------------------------
        // スライスの Descriptor から値を取得します。
        int id = 0;
        int groupId = 0;
        std::string origin;
        std::string sliceType;
        std::wstring name;
        VRect bounds = { 0 };

        const std::string valueType = std::string(reinterpret_cast<const char*>(pBuf), 4); // "Objc"
        pBuf += 4;
        int itemCount;
        pBuf = GetDescriptorItemCount(itemCount, pBuf);
        for (int iItem = 0; iItem < itemCount; ++iItem)
        {
            std::string itemName;
            std::string itemType;
            pBuf = GetDescriptorItemName(itemName, itemType, pBuf);
            if (itemName == "sliceID")
            {
                id = RGetMemBigInt(pBuf); // 0:デフォルトで存在する画像全体スライス, 1以上:ユーザー定義スライス or 自動スライス
            }
            else if (itemName == "groupID")
            {
                groupId = RGetMemBigInt(pBuf); // 0:自動スライス or 画像全体スライス, 1以上:ユーザー定義スライス
            }
            else if (itemName == "origin")
            {
                origin = GetEnumFromDescriptorItem(pBuf); // userGenerated / layerGenerated / autoGenerated
            }
            else if (itemName == "Type")
            {
                sliceType = GetEnumFromDescriptorItem(pBuf); // noImage / "Img "
            }
            else if (itemName == "Nm  ") // デフォルトでない場合のみ存在
            {
                const uint8_t* pName = pBuf; // pBuf をインクリメントしないためにコピーします。
                name = GetMemPSUnicodeString(pName);
            }
            else if (itemName == "bounds")
            {
                bounds = GetBoundsFromDescriptorItem(pBuf);
            }
            // url (TEXT)
            // null (TEXT)
            // Msge (TEXT)
            // altTag (TEXT)
            // cellTextIsHTML (bool)
            // cellText (TEXT)
            // horzAlign (enum) // default
            // vertAlign (enum) // default
            // bgColorType (enum) // None / matte / "Clr "
            // bgColor (Objc) // bgColorType が "Clr " なら存在
                // "Rd  " (long)
                // "Grn " (long)
                // "Bl  " (long)
                // "alpha" (long)
            // topOutset (long)
            // leftOutset (long)
            // bottomOutset (long)
            // rightOutset (long)
            pBuf = SkipDescriptorItem(pBuf, itemType);
            if (pBuf == NULL)
            {
                break;
            }
        }
        //RNoteTrace("v7 slice%d: %2d: %3d %d: %s [%-4s]: [%-4s] (%2d, %2d) - (%2d, %2d)",
        //  iSlice, GetSliceIndexFromId(id), id, groupId, origin.c_str(), sliceType.c_str(),
        //  RGetShiftJisFromUnicode(name).c_str(),
        //  bounds.left, bounds.top, bounds.right, bounds.bottom);

        //-----------------------------------------------------------------------------
        // スライス情報配列に追加します。
        const bool isImageSlice = (sliceType == "Img ");
        AppendSliceInfo(infos, id, groupId, isImageSlice, bounds, frontRects, name, cutPrefix);
        frontRects.push_back(bounds);
    }
}

//-----------------------------------------------------------------------------
//! @brief スライス情報をインデックス順にソートするための比較関数です。
//-----------------------------------------------------------------------------
//static bool SliceInfoLessIndex(const RSliceInfo*& r1, const RSliceInfo*& r2)
//{
//  return (r1->m_Index < r2->m_Index);
//}

//-----------------------------------------------------------------------------
//! @brief スライス情報を左上の座標が小さい順にソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool SliceInfoLessLeftTop(const RSliceInfo*& r1, const RSliceInfo*& r2)
{
    if (r1->m_Y == r2->m_Y)
    {
        return (r1->m_X < r2->m_X);
    }
    else
    {
        return (r1->m_Y < r2->m_Y);
    }
}

//-----------------------------------------------------------------------------
//! @brief スライス情報を取得します。
//-----------------------------------------------------------------------------
void RGetSliceInfos(
    RSliceInfoArray& infos,
    const bool isImageOnly,
    const std::string& cutPrefix
)
{
    infos.clear();

    //-----------------------------------------------------------------------------
    // 選択されているスライスを取得します。
    //const int selSliceIndex = GetSelectedSliceIndex(); // 選択されていなければ 0、選択されていれば 1, 2, 3, ...
    //const int selSliceID = GetSelectedSliceID(); // 選択されていなければ -1
    //RNoteTrace("selected slice: index = %d: ID = %d\n", selSliceIndex, selSliceID);

    //-----------------------------------------------------------------------------
    // スライスリソースを取得します。
    Handle hSlices;
    OSErr error = PIGetSlices(hSlices); // 自動処理プラグインではエラー -30901 (errPlugInPropertyUndefined)
    //RNoteTrace("get slices: %d", error);
    R_UNUSED_VARIABLE(error);

    const int slicesSize = sPSHandle->GetSize(hSlices); // スライスリソースのバイトサイズ
    //RNoteTrace("slicesSize: %d %p", slicesSize, hSlices);
    R_UNUSED_VARIABLE(slicesSize);

    Ptr pSlices = NULL;
    sPSHandle->SetLock(hSlices, true, &pSlices, NULL);
    if (pSlices != NULL)
    {
        //RNoteTrace("slice handle: %p %p %d", hSlices, pSlices, slicesSize);
        //RNoteDumpMemory(pSlices, slicesSize, "slice resource");
        //RNoteDumpMemory(pSlices, 128, "slice resource");
        //std::ofstream("D:/Download/slicesRes1.txt", ios_base::binary).write(reinterpret_cast<const char*>(pSlices), slicesSize);
        const int resVer = RGetMemBigInt(pSlices); // 6 or 7 or 8
        //RNoteTrace("slice res ver: %d", resVer);
        if (resVer >= 7)
        {
            ParseSliceResource7(infos, cutPrefix, pSlices);
        }
        else
        {
            ParseSliceResource6(infos, cutPrefix, pSlices);
        }
        sPSHandle->SetLock(hSlices, false, &pSlices, NULL); // unlock
        sPSHandle->Dispose(hSlices);
    }

    //-----------------------------------------------------------------------------
    // 左上の座標が小さい順にソートします。
    if (!infos.empty())
    {
        std::vector<const RSliceInfo*> pInfos;
        for (size_t iInfo = 0; iInfo < infos.size(); ++iInfo)
        {
            pInfos.push_back(&infos[iInfo]);
        }
        //std::sort(pInfos.begin(), pInfos.end(), SliceInfoLessIndex);
        std::sort(pInfos.begin(), pInfos.end(), SliceInfoLessLeftTop);
        RSliceInfoArray sortedInfos;
        for (size_t iInfo = 0; iInfo < pInfos.size(); ++iInfo)
        {
            sortedInfos.push_back(*pInfos[iInfo]);
        }
        infos = sortedInfos;
    }

    //-----------------------------------------------------------------------------
    // ソートした順にインデックスを再設定します。
    if (!infos.empty())
    {
        int sliceIdx = 1;
        for (size_t iInfo = 0; iInfo < infos.size(); ++iInfo)
        {
            RSliceInfo& info = infos[iInfo];
            const bool isIdxChanged = (info.m_Index != sliceIdx);
            info.m_Index = sliceIdx;
            info.SetIndexName();
            if (info.m_IsDefaultName)
            {
                info.m_Name = info.m_IndexName;
            }
            else if (isIdxChanged && info.m_IsDivided)
            {
                info.m_Name += "-" + RGetNumberString(sliceIdx, "%02d");
            }
            ++sliceIdx;
        }
    }

    //-----------------------------------------------------------------------------
    // 3 桁以上のインデックスが存在すれば名前を調整します。
    bool isDigit3 = false;
    for (size_t iInfo = 0; iInfo < infos.size(); ++iInfo)
    {
        if (infos[iInfo].m_Index >= 100)
        {
            isDigit3 = true;
            break;
        }
    }

    if (isDigit3)
    {
        for (size_t iInfo = 0; iInfo < infos.size(); ++iInfo)
        {
            RSliceInfo& info = infos[iInfo];
            if (info.m_IndexName.size() < 3)
            {
                info.m_IndexName = "0" + info.m_IndexName;
                if (info.m_IsDefaultName)
                {
                    info.m_Name = info.m_IndexName;
                }
            }
            if (info.m_ZeroBasedName.size() < 3)
            {
                info.m_ZeroBasedName = "0" + info.m_ZeroBasedName;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 画像スライスのみ抽出します。
    if (isImageOnly)
    {
        RSliceInfoArray imageInfos;
        for (size_t iInfo = 0; iInfo < infos.size(); ++iInfo)
        {
            const RSliceInfo& info = infos[iInfo];
            if (info.m_IsImage)
            {
                imageInfos.push_back(info);
            }
        }
        infos = imageInfos;
    }

    //for (size_t iInfo = 0; iInfo < infos.size(); ++iInfo) RNoteTrace("slice%d: %s", iInfo, infos[iInfo].GetString().c_str());
}

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

