﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// UpdateWinBmp PaintPreview
// RLocalProgress

//=============================================================================
// include
//=============================================================================
#include "NpsPreview.h"
#include <commctrl.h> // for slider & tooltip

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//-----------------------------------------------------------------------------
//! @brief プレビューの拡大後の表示領域とスクロールの範囲を設定します。
//-----------------------------------------------------------------------------
void RPreview::SetItemRect()
{
    const int maxW = m_MaxRect.right  - m_MaxRect.left;
    const int maxH = m_MaxRect.bottom - m_MaxRect.top;
    const int zoomW = GetZoomedW(m_OriginalW);
    const int zoomH = GetZoomedH(m_OriginalH);
    const int itemW = RMin(zoomW, maxW);
    const int itemH = RMin(zoomH, maxH);
    m_ItemRect.left   = m_MaxRect.left + (maxW - itemW) / 2;
    m_ItemRect.top    = m_MaxRect.top  + (maxH - itemH) / 2;
    m_ItemRect.right  = m_ItemRect.left + itemW;
    m_ItemRect.bottom = m_ItemRect.top  + itemH;

    // スクロールの範囲を設定します。
    m_MaxIx = RMax(zoomW - GetItemW(), 0);
    m_MaxIy = RMax(zoomH - GetItemH(), 0);
}

//-----------------------------------------------------------------------------
//! @brief プレビューのメモリを初期化します。
//-----------------------------------------------------------------------------
void RPreview::InitMemory(
    HWND hDlg,
    const RECT& totalRect,
    const int originalW,
    const int originalH
)
{
    //-----------------------------------------------------------------------------
    // メモリを解放します。
    FreeMemory();

    //-----------------------------------------------------------------------------
    // フラグなどを初期化します。
    m_DisplaysRgb = true;
    m_DisplaysAlpha = false;

    m_IsDragging = false;
    m_IsShowingOriginal = false;
    m_UpdateTimerCount = 0;

    //-----------------------------------------------------------------------------
    // 領域とサイズを設定します。
    m_MaxRect = totalRect;
    InflateRect(&m_MaxRect, -BORDER_W, -BORDER_W);

    m_OriginalW = originalW;
    m_OriginalH = originalH;

    //-----------------------------------------------------------------------------
    // 拡大率の最小最大値を設定します。
    const int maxW = m_MaxRect.right  - m_MaxRect.left;
    const int maxH = m_MaxRect.bottom - m_MaxRect.top;
    int curW = originalW;
    int curH = originalH;
    m_MinZoom = 1;
    while (curW > maxW || curH > maxH)
    {
        --m_MinZoom;
        curW >>= 1;
        curH >>= 1;
    }
    m_MaxZoom = 8;
    m_Zoom = 1;

    //-----------------------------------------------------------------------------
    // 拡大後の表示領域とスクロールの範囲を設定します。
    SetItemRect();

    m_Ix = 0;
    m_Iy = 0;

    //-----------------------------------------------------------------------------
    // ビットマップを初期化します。
    InitWinBmp(hDlg);
}

//-----------------------------------------------------------------------------
//! @brief プレビューの領域を更新リージョンに追加します。
//-----------------------------------------------------------------------------
void RPreview::InvalidateArea(HWND hDlg) const
{
    RECT rect = m_MaxRect;
    InflateRect(&rect, BORDER_W, BORDER_W);
    InvalidateRect(hDlg, &rect, FALSE);
}

//-----------------------------------------------------------------------------
//! @brief プレビューの拡大率を変更します。
//-----------------------------------------------------------------------------
void RPreview::ChangeZoom(HWND hDlg, const int zoomStep)
{
    //-----------------------------------------------------------------------------
    // get center
    const float zoomRateBefore = (m_Zoom >= 1) ? m_Zoom : 1.0f / (1 << (1 - m_Zoom));
    const float centerIx = (m_Ix + GetItemW() / 2.0f) / zoomRateBefore;
    const float centerIy = (m_Iy + GetItemH() / 2.0f) / zoomRateBefore;

    //-----------------------------------------------------------------------------
    // change zoom
    m_Zoom += zoomStep;
    m_Zoom = RClampValue(m_MinZoom, m_MaxZoom, m_Zoom);

    //-----------------------------------------------------------------------------
    // adjust scroll
    SetItemRect();
    const float zoomRate = (m_Zoom >= 1) ? m_Zoom : 1.0f / (1 << (1 - m_Zoom));
    m_Ix = RRound(centerIx * zoomRate - GetItemW() / 2.0f);
    m_Iy = RRound(centerIy * zoomRate - GetItemH() / 2.0f);
    ClampScrollValue();

    //-----------------------------------------------------------------------------
    // update control
    UpdateControl(hDlg);

    //-----------------------------------------------------------------------------
    // paint
    InvalidateArea(hDlg);
}

//-----------------------------------------------------------------------------
//! @brief プレビューのスクロール位置を変更します。
//-----------------------------------------------------------------------------
void RPreview::ChangeScroll(HWND hDlg, const int ix, const int iy)
{
    m_Ix = ix;
    m_Iy = iy;
    ClampScrollValue();

    //-----------------------------------------------------------------------------
    // paint
    InvalidateArea(hDlg);
}

//-----------------------------------------------------------------------------
//! @brief プレビューの画像の中央が表示されるようにスクロール位置を設定します。
//-----------------------------------------------------------------------------
void RPreview::CenterScroll()
{
    const int zoomW = GetZoomedW(m_OriginalW);
    const int zoomH = GetZoomedH(m_OriginalH);
    if (zoomW > GetItemW())
    {
        m_Ix = (zoomW - GetItemW()) / 2;
    }
    if (zoomH > GetItemH())
    {
        m_Iy = (zoomH - GetItemH()) / 2;
    }
}

//-----------------------------------------------------------------------------
//! @brief プレビューのコントロールの状態を更新します。
//-----------------------------------------------------------------------------
void RPreview::UpdateControl(HWND hDlg) const
{
    const bool zoomInEnable = (m_Zoom < m_MaxZoom);
    const bool zoomOtEnable = (m_Zoom > m_MinZoom);

    HWND zoomInBtn = GetDlgItem(hDlg, m_ZoomInId);
    HWND zoomOtBtn = GetDlgItem(hDlg, m_ZoomOutId);

    // フォーカスのあるボタンが無効状態になると
    // どのコントロールにもフォーカスのない状態になって
    // ホイールが効かなくなるので、フォーカスをあらかじめ移しておきます。
    HWND focus = GetFocus();
    if (focus == zoomInBtn && !zoomInEnable)
    {
        SetFocus(zoomOtBtn);
    }
    else if (focus == zoomOtBtn && !zoomOtEnable)
    {
        SetFocus(zoomInBtn);
    }

    EnableWindow(zoomOtBtn, zoomOtEnable);
    EnableWindow(zoomInBtn, zoomInEnable);

    const int zoomValue = (m_Zoom >= 1) ?
        m_Zoom * 100 : RRound(100.0f / (1 << (1 - m_Zoom)));
    SetWindowText(GetDlgItem(hDlg, m_ZoomTextId),
        RGetNumberString(zoomValue, "%d%%").c_str());
}

//-----------------------------------------------------------------------------
//! @brief プレビューの Windows ビットマップを初期化します。
//-----------------------------------------------------------------------------
void RPreview::InitWinBmp(HWND hDlg)
{
    BITMAPINFO bmi;
    memset(&bmi, 0, sizeof(BITMAPINFO));

    BITMAPINFOHEADER& header = bmi.bmiHeader;
    header.biSize        = sizeof(BITMAPINFOHEADER);
    header.biWidth       =  m_OriginalW;
    header.biHeight      = -m_OriginalH;
    header.biPlanes      = 1;
    header.biBitCount    = 32; // 24:RGBTRIPLE, 32:RGBQUAD
    header.biCompression = BI_RGB; // no compress

    HDC hDC = GetDC(hDlg);
    m_hWinBmp = CreateDIBSection(hDC, &bmi, DIB_RGB_COLORS,
        &m_pWinBmpPixels, NULL, 0);
    ReleaseDC(hDlg, hDC);
    //RNoteTrace("win bmp pixels: %p", m_pWinBmpPixels);

    memset(m_pWinBmpPixels, 0x00, m_OriginalW * m_OriginalH * R_RGBA_BYTES);
}

//-----------------------------------------------------------------------------
//! @brief プレビューの Windows ビットマップを更新します。
//-----------------------------------------------------------------------------
void RPreview::UpdateWinBmp(HWND hDlg, const void* pSrcBitmap)
{
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pSrcBitmap);
    RGBQUAD* pDst = reinterpret_cast<RGBQUAD*>(m_pWinBmpPixels);

    if (m_DisplaysRgb)
    {
        if (m_DisplaysAlpha)
        {
            //-----------------------------------------------------------------------------
            // rgb & alpha (blend)
            const float A_COL_OPACITY = 0.5f;
            const float A_COL_R = 0xff;
            const float A_COL_G = 0x00;
            const float A_COL_B = 0x00;
            for (int iy = 0; iy < m_OriginalH; ++iy)
            {
                for (int ix = 0; ix < m_OriginalW; ++ix)
                {
                    const int alpha = pSrc[3];
                    if (alpha == 0xff)
                    {
                        pDst->rgbRed   = *pSrc++;
                        pDst->rgbGreen = *pSrc++;
                        pDst->rgbBlue  = *pSrc++;
                        ++pSrc;
                    }
                    else
                    {
                        float rate0 = static_cast<float>(alpha) / 0xff;
                        rate0 = RMin(rate0 * A_COL_OPACITY + (1.0f - A_COL_OPACITY), 1.0f);
                        const float rate1 = 1.0f - rate0;
                        pDst->rgbRed   = static_cast<uint8_t>((*pSrc++) * rate0 + A_COL_R * rate1);
                        pDst->rgbGreen = static_cast<uint8_t>((*pSrc++) * rate0 + A_COL_G * rate1);
                        pDst->rgbBlue  = static_cast<uint8_t>((*pSrc++) * rate0 + A_COL_B * rate1);
                        ++pSrc;
                    }
                    ++pDst;
                }
            }
        }
        else
        {
            //-----------------------------------------------------------------------------
            // rgb only
            for (int iy = 0; iy < m_OriginalH; ++iy)
            {
                for (int ix = 0; ix < m_OriginalW; ++ix)
                {
                    pDst->rgbRed   = *pSrc++;
                    pDst->rgbGreen = *pSrc++;
                    pDst->rgbBlue  = *pSrc++;
                    ++pSrc;
                    ++pDst;
                }
            }
        }
    }
    else
    {
        if (m_DisplaysAlpha)
        {
            //-----------------------------------------------------------------------------
            // alpha only
            for (int iy = 0; iy < m_OriginalH; ++iy)
            {
                for (int ix = 0; ix < m_OriginalW; ++ix)
                {
                    pDst->rgbRed   =
                    pDst->rgbGreen =
                    pDst->rgbBlue  = pSrc[3];
                    pSrc += R_RGBA_BYTES;
                    ++pDst;
                }
            }
        }
        else
        {
            //-----------------------------------------------------------------------------
            // none
            memset(pDst, 0x00, m_OriginalW * m_OriginalH * R_RGBA_BYTES);
        }
    }

    InvalidateArea(hDlg);
}

//-----------------------------------------------------------------------------
//! @brief プレビューの表示領域のまわりの領域をダイアログの背景色でフィルします。
//-----------------------------------------------------------------------------
void RPreview::FillAroundItem(HDC& hDC) const
{
    //-----------------------------------------------------------------------------
    // 縁を含む最大領域と表示領域を取得します。
    RECT maxRect = m_MaxRect;
    InflateRect(&maxRect, BORDER_W, BORDER_W);
    RECT itemRect = m_ItemRect;
    InflateRect(&itemRect, BORDER_W, BORDER_W);

    //-----------------------------------------------------------------------------
    // left
    RECT rect;
    rect.left   = maxRect.left;
    rect.top    = maxRect.top;
    rect.right  = itemRect.left;
    rect.bottom = maxRect.bottom;
    if (rect.left < rect.right)
    {
        FillRect(hDC, &rect, reinterpret_cast<HBRUSH>(COLOR_MENU + 1));
    }

    //-----------------------------------------------------------------------------
    // right
    rect.left   = itemRect.right;
    rect.right  = maxRect.right;
    if (rect.left < rect.right)
    {
        FillRect(hDC, &rect, reinterpret_cast<HBRUSH>(COLOR_MENU + 1));
    }

    //-----------------------------------------------------------------------------
    // top
    rect.left   = itemRect.left;
    rect.top    = maxRect.top;
    rect.right  = itemRect.right;
    rect.bottom = itemRect.top;
    if (rect.top < rect.bottom)
    {
        FillRect(hDC, &rect, reinterpret_cast<HBRUSH>(COLOR_MENU + 1));
    }

    //-----------------------------------------------------------------------------
    // bottom
    rect.top    = itemRect.bottom;
    rect.bottom = maxRect.bottom;
    if (rect.top < rect.bottom)
    {
        FillRect(hDC, &rect, reinterpret_cast<HBRUSH>(COLOR_MENU + 1));
    }
}

//-----------------------------------------------------------------------------
//! @brief プレビューをペイントします。
//-----------------------------------------------------------------------------
void RPreview::Paint(HWND hDlg) const // PaintPreview
{
    //-----------------------------------------------------------------------------
    // begin paint
    PAINTSTRUCT ps;
    HDC hDC = BeginPaint(hDlg, &ps);
    // InvalidateRect の第 3 引数が TRUE の場合、この時点で背景が消去されます。

    //-----------------------------------------------------------------------------
    // no porxy
    if (!m_DisplaysRgb && !m_DisplaysAlpha)
    {
        EndPaint(hDlg, &ps);
        return;
    }

    //-----------------------------------------------------------------------------
    // fill around item
    FillAroundItem(hDC);

    //-----------------------------------------------------------------------------
    // draw border
    RECT borderRect = m_ItemRect;
    InflateRect(&borderRect, BORDER_W, BORDER_W);
    FrameRect(hDC, &borderRect, reinterpret_cast<HBRUSH>(GetStockObject(BLACK_BRUSH)));

    //-----------------------------------------------------------------------------
    // calc size & offset
    int srcIx, srcIy, ofsIx, ofsIy, srcW, srcH;
    if (m_Zoom >= 1)
    {
        srcIx = m_Ix / m_Zoom;
        srcIy = m_Iy / m_Zoom;
        ofsIx = m_Ix % m_Zoom;
        ofsIy = m_Iy % m_Zoom;
        srcW = (GetItemW() + m_Zoom - 1) / m_Zoom;
        srcH = (GetItemH() + m_Zoom - 1) / m_Zoom;
    }
    else
    {
        const int invZoom = 1 << (1 - m_Zoom);
        srcIx = m_Ix * invZoom;
        srcIy = m_Iy * invZoom;
        ofsIx = 0;
        ofsIy = 0;
        srcW = GetItemW() * invZoom;
        srcH = GetItemH() * invZoom;
    }
    if (ofsIx != 0) ++srcW;
    if (ofsIy != 0) ++srcH;
    const int dstW = GetZoomedW(srcW);
    const int dstH = GetZoomedH(srcH);

    //-----------------------------------------------------------------------------
    // draw image
    HRGN hRGN = CreateRectRgn(m_ItemRect.left, m_ItemRect.top, m_ItemRect.right, m_ItemRect.bottom);
    SelectClipRgn(hDC, hRGN);

    HDC hCDC = CreateCompatibleDC(hDC);
    SelectObject(hCDC, m_hWinBmp);

    #if 0
    BitBlt(hDC, mItem.left, mItem.top,
        m_OriginalW, m_OriginalH, hCDC, 0, 0, SRCCOPY);
    #else
    SetStretchBltMode(hDC, STRETCH_DELETESCANS);
    StretchBlt(hDC, m_ItemRect.left - ofsIx, m_ItemRect.top - ofsIy,
        dstW, dstH, hCDC, srcIx, srcIy,
        srcW, srcH, SRCCOPY);
    #endif

    DeleteDC(hCDC);

    SelectClipRgn(hDC, NULL);
    DeleteObject(hRGN);

    //-----------------------------------------------------------------------------
    // end paint
    EndPaint(hDlg, &ps);
}

//-----------------------------------------------------------------------------
//! @brief ローカルプログレスのダイアログプロシージャです。
//-----------------------------------------------------------------------------
INT_PTR CALLBACK RLocalProgress::DialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    R_UNUSED_VARIABLE(lParam);

    switch (uMsg)
    {
        case WM_INITDIALOG:
            break;
        case WM_COMMAND:
        {
            const short id = LOWORD(wParam);
            if (id == IDCANCEL)
            {
                // ダイアログを破棄する前に親ウィンドウを有効にします。
                HWND hParent = GetParent(hDlg);
                if (hParent != NULL && IsWindow(hParent))
                {
                    EnableWindow(hParent, true);
                }
                DestroyWindow(hDlg);
                return TRUE;
            }
        }
        default:
            break;
    }
    return FALSE;
}

//-----------------------------------------------------------------------------
//! @brief ローカルプログレスのコンストラクタです。
//-----------------------------------------------------------------------------
RLocalProgress::RLocalProgress(
    const int total,
    const int dialogId,
    HWND hParent,
    const int progressId,
    const int textId,
    const char* text
)
:   RProgress(total),
    m_hParent(hParent),
    m_ProgressId(progressId),
    m_IsDialogShown(false)
{
    //-----------------------------------------------------------------------------
    // ダイアログを作成します。
    m_hDialog = CreateDialog(GetDLLInstance(),
        MAKEINTRESOURCE(dialogId), m_hParent, DialogProc);
    //RNoteTrace("local progress: %p %p %d", m_hDialog, m_hParent, m_Total);

    //-----------------------------------------------------------------------------
    // テキストを設定します。
    if (textId != 0 && text != NULL)
    {
        SetDlgItemText(m_hDialog, textId, text);
    }

    //-----------------------------------------------------------------------------
    // プログレス開始時のクロックをリセットします。
    ResetStartClock();

    //-----------------------------------------------------------------------------
    // 親ウィンドウを無効にします。
    EnableParent(false);
}

//-----------------------------------------------------------------------------
//! @brief ローカルプログレスのダイアログ表示を終了します。
//-----------------------------------------------------------------------------
void RLocalProgress::End()
{
    //RNoteTrace("end local progress: %f", static_cast<float>(std::clock() - m_StartClock) / CLOCKS_PER_SEC);

    //-----------------------------------------------------------------------------
    // 親ウィンドウを有効にしてからダイアログを破棄します。
    EnableParent(true);
    if (m_hDialog != NULL && IsWindow(m_hDialog))
    {
        DestroyWindow(m_hDialog);
    }
    //SetForegroundWindow(m_hParent);

    //-----------------------------------------------------------------------------
    // ハンドルを NULL にします。
    m_hDialog = NULL;
    m_hParent = NULL;
    m_IsDialogShown = false;
}

//-----------------------------------------------------------------------------
//! @brief ローカルプログレスを更新します。
//-----------------------------------------------------------------------------
bool RLocalProgress::Update(const int step)
{
    m_Count += step;
    //RNoteTrace("progress update: %d / %d\n", m_Count, m_Total);

    if (m_hDialog != NULL   &&
        IsWindow(m_hDialog) &&
        IsWindowEnabled(m_hDialog))
    {
        //-----------------------------------------------------------------------------
        // 処理時間が掛かりそうならダイアログを表示します。
        if (!m_IsDialogShown &&
            std::clock() - m_StartClock >= CLOCKS_PER_SEC &&
            static_cast<float>(m_Count) / m_Total < 0.8f)
        {
            ShowWindow(m_hDialog, SW_SHOW);
            UpdateWindow(m_hDialog);
            m_IsDialogShown = true;
        }

        if (m_IsDialogShown)
        {
            //-----------------------------------------------------------------------------
            // ウィンドウメッセージを処理します。
            MSG msg;
            while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
            {
                // IsDialogMessage は
                // ダイアログに対するメッセージなら処理して TRUE を返します。
                if (!IsDialogMessage(m_hDialog, &msg))
                {
                    // ダイアログに対するメッセージでなければ、メインウィンドウに送ります。
                    TranslateMessage(&msg);
                    DispatchMessage(&msg);
                }
            }

            //-----------------------------------------------------------------------------
            // カーソルを設定します。
            POINT cusorPos;
            GetCursorPos(&cusorPos); // screen
            RECT wRect;
            GetWindowRect(m_hDialog, &wRect); // screen
            sPSUIHooks->SetCursor(RIsPointInRect(wRect, cusorPos) ?
                kPICursorArrow : kPICursorWatch);

            //-----------------------------------------------------------------------------
            // プログレスバーの範囲と値を設定します。
            // Windows のプログレスバーに与える値は 0xffff 以下にする必要があるので、
            // total が大きければ、done と total を割ります。
            int done = m_Count;
            int total = m_Total;
            while (total > 0xffff)
            {
                done >>= 1;
                total >>= 1;
            }
            HWND hProg = GetDlgItem(m_hDialog, m_ProgressId);
            SendMessage(hProg, PBM_SETRANGE, (WPARAM)0, MAKELPARAM(0, total));
            SendMessage(hProg, PBM_SETPOS, (WPARAM)done, 0);
        }
        else
        {
            sPSUIHooks->SetCursor(kPICursorWatch);
        }
    }

    if (m_hDialog != NULL && !IsWindow(m_hDialog))
    {
        return false;
    }

    return true;
}

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

