﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RReadFormatScriptParams RWriteFormatScriptParams

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//-----------------------------------------------------------------------------
//! @brief ファイル形式プラグイン（ライト時）のスクリプトパラメータをリードします。
//!
//! @param[in,out] globals グローバルデータです。
//!
//! @return エラーコードを返します。
//-----------------------------------------------------------------------------
OSErr RReadFormatScriptParams(GPtr globals)
{
    OSErr error = noErr;

    //-----------------------------------------------------------------------------
    // read
    DescriptorKeyIDArray keyArray =
    {
        nnFormatKeyConfigName,
        nnFormatKeyHint,
        nnFormatKeyLinearFlag,
        nnFormatKeyFormat,
        nnFormatKeyWeightedCompress,
        nnFormatKeyInitialSwizzle,
        nnFormatKeyMipLevel,
        nnFormatKeyMipGenFilter,
        nnFormatKeyDimension,
        nnFormatKeyCompSel,
        nnFormatKeyMerge,
        nnFormatKeyOverwriteInput,
        nnFormatKeyCommentText,
        nnFormatKeyCrop,
        nnFormatKeyResizeWH,
        nnFormatKeyResizeFilter,
        nnFormatKeyResizeGamma,
        nnFormatKeyIsExportTexture,
        nnFormatKeyOutputOriginalPath,
        NULLID
    };

    bool isFormatSpecified = false;
    bool isCompSelSpecified = false;
    bool isCommentTextSpecified = false;
    if (DescriptorAvailable(NULL))
    {
        //-----------------------------------------------------------------------------
        // open reader
        PIReadDescriptor token = OpenReader(keyArray);
        //RMsgBox("read token: %x", static_cast<uint32_t>(token)); // 一度も Write されてなければ 0
        if (token)
        {
            // Photoshop CC 2014 以降では PIGetText で取得した値が UTF-8 になっているので、Shift-JIS に変換します。
            const bool utf8ToSjis = RIsPhotoshopVersionGreaterEqual(PhotoshopVersionMajorCc2014, 0);

            DescriptorKeyID key = NULLID;
            DescriptorTypeID type = NULLID;
            int32 flags = 0;
            while (PIGetKey(token, &key, &type, &flags))
            {
                switch (key)
                {
                    //-----------------------------------------------------------------------------
                    // コンフィグ名
                    case nnFormatKeyConfigName:
                        RGetStringParam(globals, &g_AddInfo.m_ConfigName, token, utf8ToSjis);
                        break;

                    //-----------------------------------------------------------------------------
                    // ヒント情報
                    case nnFormatKeyHint:
                        RGetStringParam(globals, &g_AddInfo.m_Hint, token, false);
                        break;

                    //-----------------------------------------------------------------------------
                    // リニア変換フラグ
                    case nnFormatKeyLinearFlag:
                    {
                        std::string linearFlagStr;
                        if (RGetStringParam(globals, &linearFlagStr, token, false))
                        {
                            const int linearFlag = RGetLinearFlagFromOptString(linearFlagStr);
                            if (linearFlag != RLinearFlag::LINEAR_INVALID)
                            {
                                g_AddInfo.m_LinearFlag = linearFlag;
                            }
                        }
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // format
                    case nnFormatKeyFormat:
                    {
                        std::string formatStr;
                        if (RGetStringParam(globals, &formatStr, token, false))
                        {
                            const FtxFormat format = RGetTextureFormatFromString(formatStr);
                            //RNoteTrace("format: [%s] %x\n", formatStr, format);
                            if (format != FtxFormat_Invalid)
                            {
                                g_AddInfo.m_Format = format;
                                g_ImageStatus.m_IsFormatDecided = true;
                                isFormatSpecified = true;
                            }
                        }
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // weighted compress
                    case nnFormatKeyWeightedCompress:
                    {
                        Boolean weightedCompress = FALSE;
                        error = PIGetBool(token, &weightedCompress);
                        g_AddInfo.m_WeightedCompress = (weightedCompress == TRUE);
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // 初期スウィズル値
                    case nnFormatKeyInitialSwizzle:
                    {
                        int32 initialSwizzle = 1;
                        error = PIGetInt(token, &initialSwizzle);
                        if (RAddInfo::SWIZZLE_MIN <= initialSwizzle && initialSwizzle <= RAddInfo::SWIZZLE_MAX)
                        {
                            g_AddInfo.m_InitialSwizzle = initialSwizzle;
                        }
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // mipmap level
                    case nnFormatKeyMipLevel:
                    {
                        int32 mipLevel = 1;
                        error = PIGetInt(token, &mipLevel);
                        if (1 <= mipLevel && mipLevel <= RImage::MipCountMax)
                        {
                            g_AddInfo.m_MipLevel = mipLevel;
                        }
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // ミップマップ生成フィルタ
                    case nnFormatKeyMipGenFilter:
                    {
                        std::string mipGenFilterStr;
                        if (RGetStringParam(globals, &mipGenFilterStr, token, false))
                        {
                            g_AddInfo.m_MipGenFilter = RGetMipGenFilterFromString(mipGenFilterStr, true);
                        }
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // dimension
                    case nnFormatKeyDimension:
                    {
                        DescriptorEnumID dimID;
                        error = PIGetEnum(token, &dimID);
                        FtxDimension dimension = FtxDimension_2d;
                        switch (dimID)
                        {
                            case nnFormatEnum1D     : dimension = FtxDimension_1d     ; break;
                            case nnFormatEnum2D     : dimension = FtxDimension_2d     ; break;
                            case nnFormatEnum3D     : dimension = FtxDimension_3d     ; break;
                            case nnFormatEnumCube   : dimension = FtxDimension_CubeMap; break;
                            case nnFormatEnum1DArray: dimension = FtxDimension_1dArray; break;
                            case nnFormatEnum2DArray: dimension = FtxDimension_2dArray; break;
                            default: break;
                        }
                        if (dimension == FtxDimension_CubeMap &&
                            !g_ImageStatus.m_IsCubeHCEnable &&
                            !g_ImageStatus.m_IsCubeVCEnable)
                        {
                            // キューブマップ可能なサイズでなければ適用しません。
                        }
                        else
                        {
                            g_AddInfo.m_Dimension = dimension;
                        }
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // comp sel
                    case nnFormatKeyCompSel:
                    {
                        std::string compSelStr;
                        if (RGetStringParam(globals, &compSelStr, token, false))
                        {
                            g_AddInfo.m_CompSel = RGetCompSelFromOptionString(compSelStr);
                            isCompSelSpecified = true;
                        }
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // マージ
                    case nnFormatKeyMerge:
                        RGetStringParam(globals, g_AddInfo.m_MergePath, sizeof(g_AddInfo.m_MergePath), token, utf8ToSjis);
                        break;

                    //-----------------------------------------------------------------------------
                    // 入力ファイルのパスの上書き
                    case nnFormatKeyOverwriteInput:
                        RGetStringParam(globals, g_AddInfo.m_OverwriteInputPath, sizeof(g_AddInfo.m_OverwriteInputPath), token, utf8ToSjis);
                        break;

                    //-----------------------------------------------------------------------------
                    // 編集用コメント文
                    case nnFormatKeyCommentText:
                        if (RGetStringParam(globals, &g_AddInfo.m_CommentText, token, utf8ToSjis))
                        {
                            isCommentTextSpecified = true;
                        }
                        break;

                    //-----------------------------------------------------------------------------
                    // 切り抜き矩形
                    case nnFormatKeyCrop:
                        RGetStringParam(globals, &g_AddInfo.m_CropRect, token, false);
                        //RNoteTrace("crop: [%s]\n", g_AddInfo.m_CropRect.c_str());
                        break;

                    //-----------------------------------------------------------------------------
                    // サイズ変更後の幅と高さ
                    case nnFormatKeyResizeWH:
                        RGetStringParam(globals, &g_AddInfo.m_ResizeWH, token, false);
                        break;

                    //-----------------------------------------------------------------------------
                    // サイズ変更フィルタ
                    case nnFormatKeyResizeFilter:
                        RGetStringParam(globals, &g_AddInfo.m_ResizeFilter, token, false);
                        break;

                    //-----------------------------------------------------------------------------
                    // リニア空間でのサイズ変更計算フラグ
                    case nnFormatKeyResizeGamma:
                    {
                        Boolean resizesInLinear = FALSE;
                        error = PIGetBool(token, &resizesInLinear);
                        g_AddInfo.m_ResizesInLinear = (resizesInLinear == TRUE);
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // テクスチャエクスポートプラグインから呼ばれたかどうか
                    case nnFormatKeyIsExportTexture:
                    {
                        Boolean isExportTexture = TRUE;
                        error = PIGetBool(token, &isExportTexture);
                        g_AddInfo.m_IsExportTexture = (isExportTexture == TRUE);
                        break;
                    }

                    //-----------------------------------------------------------------------------
                    // 元画像ファイルの出力パス
                    case nnFormatKeyOutputOriginalPath:
                        RGetStringParam(globals, &g_AddInfo.m_OutputOriginalPath, token, false);
                        break;

                    default:
                        break;
                }
            }

            //-----------------------------------------------------------------------------
            // close reader
            error = CloseReader(&token);
            if (error == errMissingParameter)
            {
                // keyArray のキーが一部なかった場合はエラーではありません。
                error = noErr;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 成分選択が指定されていなければ
    // フォーマットに対するデフォルトの成分選択を設定します。
    if (isFormatSpecified && !isCompSelSpecified)
    {
        g_AddInfo.m_CompSel = RGetDefaultCompSel(g_AddInfo.m_Format, g_AddInfo.m_Hint);
    }

    //-----------------------------------------------------------------------------
    // テクスチャエクスポートプラグインから呼ばれた場合、
    // 編集用コメント文が指定されていなければ
    // リソースデータから取得した編集用コメント文を破棄します。
    if (g_AddInfo.m_IsExportTexture && !isCommentTextSpecified)
    {
        //RNoteTrace("clear comment: [%s]", g_AddInfo.m_CommentText.c_str());
        g_AddInfo.m_CommentText.clear();
    }

    return error;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ファイル形式プラグイン（ライト時）のスクリプトパラメータをライトします。
//!
//! @param[in] globals グローバルデータです。
//!
//! @return エラーコードを返します。
//-----------------------------------------------------------------------------
OSErr RWriteFormatScriptParams(GPtr globals)
{
    OSErr error = noErr;

    if (DescriptorAvailable(NULL))
    {
        //-----------------------------------------------------------------------------
        // open writer
        PIWriteDescriptor token = OpenWriter();
        if (token)
        {
            //-----------------------------------------------------------------------------
            // コンフィグ名
            RPutStringParam(globals, token, nnFormatKeyConfigName, g_AddInfo.m_ConfigName);

            //-----------------------------------------------------------------------------
            // ヒント情報
            RPutStringParam(globals, token, nnFormatKeyHint, g_AddInfo.m_Hint);

            //-----------------------------------------------------------------------------
            // リニア変換フラグ
            if (g_AddInfo.m_LinearFlag != RLinearFlag::LINEAR_NONE)
            {
                RPutStringParam(globals, token, nnFormatKeyLinearFlag,
                    RGetOptStringFromLinearFlag(g_AddInfo.m_LinearFlag));
            }

            //-----------------------------------------------------------------------------
            // format
            RPutStringParam(globals, token, nnFormatKeyFormat,
                RGetTextureFormatString(g_AddInfo.m_Format));

            //-----------------------------------------------------------------------------
            // weighted compress
            if (RIsBc123TextureFormat(g_AddInfo.m_Format) || RIsEtcTextureFormat(g_AddInfo.m_Format))
            {
                PIPutBool(token, nnFormatKeyWeightedCompress, g_AddInfo.m_WeightedCompress);
            }

            //-----------------------------------------------------------------------------
            // 初期スウィズル値
            PIPutInt(token, nnFormatKeyInitialSwizzle, g_AddInfo.m_InitialSwizzle);

            //-----------------------------------------------------------------------------
            // mipmap level
            PIPutInt(token, nnFormatKeyMipLevel, g_AddInfo.m_MipLevel);

            //-----------------------------------------------------------------------------
            // ミップマップ生成フィルタ
            RPutStringParam(globals, token, nnFormatKeyMipGenFilter,
                RGetMipGenFilterString(g_AddInfo.m_MipGenFilter));

            //-----------------------------------------------------------------------------
            // dimension
            DescriptorEnumID dimID = nnFormatEnum2D;
            switch (g_AddInfo.m_Dimension)
            {
                case FtxDimension_1d     : dimID = nnFormatEnum1D     ; break;
                case FtxDimension_2d     : dimID = nnFormatEnum2D     ; break;
                case FtxDimension_3d     : dimID = nnFormatEnum3D     ; break;
                case FtxDimension_CubeMap: dimID = nnFormatEnumCube   ; break;
                case FtxDimension_1dArray: dimID = nnFormatEnum1DArray; break;
                case FtxDimension_2dArray: dimID = nnFormatEnum2DArray; break;
                default: break;
            }
            PIPutEnum(token, nnFormatKeyDimension, nnFormatTypeDimension, dimID);

            //-----------------------------------------------------------------------------
            // comp sel
            RPutStringParam(globals, token, nnFormatKeyCompSel,
                RGetCompSelOptionString(g_AddInfo.m_CompSel));

            //-----------------------------------------------------------------------------
            // マージ
            if (g_AddInfo.Merges())
            {
                RPutStringParam(globals, token, nnFormatKeyMerge, g_AddInfo.m_MergePath);
            }

            //-----------------------------------------------------------------------------
            // 入力ファイルのパスの上書き
            if (g_AddInfo.OverwritesInputPath())
            {
                RPutStringParam(globals, token, nnFormatKeyOverwriteInput, g_AddInfo.m_OverwriteInputPath);
            }

            //-----------------------------------------------------------------------------
            // 編集用コメント文
            if (!g_AddInfo.m_CommentText.empty())
            {
                RPutStringParam(globals, token, nnFormatKeyCommentText, g_AddInfo.m_CommentText);
            }

            //-----------------------------------------------------------------------------
            // 切り抜き矩形
            if (!g_AddInfo.m_CropRect.empty())
            {
                RPutStringParam(globals, token, nnFormatKeyCrop, g_AddInfo.m_CropRect);
            }

            //-----------------------------------------------------------------------------
            // サイズ変更後の幅と高さ
            if (!g_AddInfo.m_ResizeWH.empty())
            {
                RPutStringParam(globals, token, nnFormatKeyResizeWH, g_AddInfo.m_ResizeWH);

                // サイズ変更フィルタ
                if (!g_AddInfo.m_ResizeFilter.empty())
                {
                    RPutStringParam(globals, token, nnFormatKeyResizeFilter, g_AddInfo.m_ResizeFilter);
                }

                // リニア空間でのサイズ変更計算フラグ
                PIPutBool(token, nnFormatKeyResizeGamma, g_AddInfo.m_ResizesInLinear);
            }

            //-----------------------------------------------------------------------------
            // close writer
            error = CloseWriter(&token);
        }
    }

    return error;
}

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

