﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <map>

#include <maya/MString.h>
#include <maya/MDagPath.h>
#include <maya/MDagPathArray.h>
#include <maya/MObjectArray.h>
#include <maya/MStringArray.h>
#include <maya/MIntArray.h>
#include <maya/MFnMesh.h>

namespace lod
{
//!
struct Bone
{
    MDagPath dag;
    bool useRigidSkin;
    bool useSmoothSkin;

    Bone() : useRigidSkin(false),useSmoothSkin(false){}

    MString GetName() const{ return dag.partialPathName(); }
};

//!
struct Shape
{
    enum SkinMode
    {
        cNoSkin,
        cRigidSkin,
        cSmoothSkin,
    };

    struct ColorSet
    {
        enum Type
        {
            cFloat,
            cInt,
            cUint,
        };

        MString name;
        int compCount; //!< 成分数です。
        bool isRa; //!< RA の 2 成分を出力するなら true です。
        Type type;
        MFnMesh::MColorRepresentation repr;

        ColorSet()
        : compCount(-1),
          isRa(false),
          type(cFloat),
          repr(MFnMesh::kRGB)
        {
        }
    };

    MDagPath dag;
    MDagPath m_MeshPath; //!< mesh ノードの DAG パスです。
    MObject faceComp;
    MObject material;
    SkinMode skinMode;
    std::vector<Bone*> bones;
    MStringArray uvSets;
    std::vector<ColorSet> colorSets;

    std::map<const Bone*,MIntArray> boneToVtxIds;

    MIntArray smoothVtxIds;
    int m_VertexSkinningCount; //!< スキニングのウェイト値を分配した 1 頂点当たりの最大ノード数です。

    Shape() : skinMode(cNoSkin){}

    bool IsUsedBone(const Bone* bone) const;

    MString GetMeshName() const;
    MString GetMaterialName() const;
};


//!
class Model
{

private:
    std::vector<Bone*> m_bones;
    std::vector<Shape*> m_shapes;

    MDagPath m_dagRoot;
    MDagPathArray m_exportableNodes;

public:
    Model();
    ~Model();

    void SetRootNode(const MDagPath& dagRoot);

    MString GetName() const;

    int GetNodeCount() const{ return m_exportableNodes.length(); }
    const MDagPath& GetNodeAt(int index) const{ return m_exportableNodes[index]; }

    int GetShapeCount() const{ return static_cast<int>(m_shapes.size()); }
    Shape* GetShapeAt(int index) const{ return m_shapes[index]; }
    Shape* FindCorrespondingShape(const Shape* other) const;

    Shape* GetShapeByDagAndMaterial(const MDagPath& dag,const MObject& material);

    int GetMaterials(MObjectArray& materials) const;

    int GetBoneCount() const{ return static_cast<int>(m_bones.size()); }
    Bone* GetBoneAt(int index) const{ return (index < m_bones.size()) ? m_bones[index] : NULL; }
    Bone* GetBoneByDag(const MDagPath& dag) const;

    Bone* FindCorrespondingBone(const Bone *other) const;

private:
    void RegisterNode(const MDagPath& dag);
    void RegisterShape(const MDagPath& dag,const MObject& faceComp,const MObject& sg);

    void SetShapeUVSetNames(Shape* pShape) const;
    void SetShapeColorSetNames(Shape* pShape) const;


    bool IsExportableNode(const MDagPath& dag) const;
};

} // namespace lod

