﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// YFindValueInArray
// CreateTransformMtx
// GetCurrentColorSetName

// FindTextureNode
// GetImageInfoForMaya ReadImageFileForMaya

//=============================================================================
// include
//=============================================================================
#include "Export.h"
#include "Deform.h"
#include "DccShape.h"
#include <shlobj.h> // file dialog

#include <maya/MImage.h>

#define MNoVersionString
#define MNoPluginEntry
#include <maya/MFnPlugin.h>

using namespace std;
using namespace nn::gfx::tool::dcc;

//-----------------------------------------------------------------------------
//! @brief エラーを表示します。
//-----------------------------------------------------------------------------
void YShowError(
    RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const RStringArray& args
)
{
    RScene& rscene = *pScene;
    if (!rscene.IsEvaluating())
    {
        //-----------------------------------------------------------------------------
        // フォーマット文字列に従ってメッセージを作成します。
        const std::string& format = (rscene.IsJapaneseUi() && !formatJp.empty()) ?
            formatJp : formatEn;
        const std::string msg = RReplaceArgumentsInString(format, args);

        //-----------------------------------------------------------------------------
        // メッセージを表示します。
        if (rscene.ErrPtr() != NULL)
        {
            rscene.Err() << "Error: " << msg.c_str() << endl;
        }
        if (MGlobal::mayaState() == MGlobal::kInteractive)
        {
            MGlobal::displayError(msg.c_str());
        }
    }

    rscene.AddErrorCount(1); // エラー数をインクリメントします。
}

//-----------------------------------------------------------------------------
//! @brief 警告を表示します。
//-----------------------------------------------------------------------------
void YShowWarning(
    RScene* pScene,
    const std::string& formatJp,
    const std::string& formatEn,
    const RStringArray& args
)
{
    RScene& rscene = *pScene;
    if (!rscene.IsEvaluating())
    {
        //-----------------------------------------------------------------------------
        // フォーマット文字列に従ってメッセージを作成します。
        const std::string& format = (rscene.IsJapaneseUi() && !formatJp.empty()) ?
            formatJp : formatEn;
        const std::string msg = RReplaceArgumentsInString(format, args);

        //-----------------------------------------------------------------------------
        // メッセージを表示します。
        if (rscene.ErrPtr() != NULL)
        {
            rscene.Err() << "Warning: " << msg.c_str() << endl;
        }
        if (MGlobal::mayaState() == MGlobal::kInteractive)
        {
            MGlobal::displayWarning(msg.c_str());
        }
    }

    rscene.AddWarningCount(1); // 警告数をインクリメントします。
}

//-----------------------------------------------------------------------------
//! @brief ヘルプラインにメッセージを表示します。
//-----------------------------------------------------------------------------
void YSetHelpLine(const char* format, ...)
{
    const int STR_SIZE = 1024;
    char* pStr = new char[STR_SIZE];
    va_list args;
    va_start(args, format);
    const int size = _vsnprintf_s(pStr, STR_SIZE, _TRUNCATE, format, args);
    va_end(args);
    MFeedbackLine::setFormat(pStr);
    delete[] pStr;
    R_UNUSED_VARIABLE(size);

    RefleshIfMaya2011();
}

//-----------------------------------------------------------------------------
//! @brief 2 つの MColor の誤差が許容値未満なら true を返します。
//-----------------------------------------------------------------------------
bool IsEquivalentColor(const MColor& c0, const MColor& c1, const float tolerance)
{
    return (
        RIsSame(c0.r, c1.r, tolerance) &&
        RIsSame(c0.g, c1.g, tolerance) &&
        RIsSame(c0.b, c1.b, tolerance) &&
        RIsSame(c0.a, c1.a, tolerance));
}

//-----------------------------------------------------------------------------
//! @brief MIntArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int YFindValueInArray(const MIntArray& array, const int value, const int startIdx)
{
    const int count = array.length();
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief MFloatArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int YFindValueInArray(const MFloatArray& array, const float value, const int startIdx)
{
    const int count = array.length();
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief MDoubleArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int YFindValueInArray(const MDoubleArray& array, const double value, const int startIdx)
{
    const int count = array.length();
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief MStringArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int YFindValueInArray(const MStringArray& array, const MString& value, const int startIdx)
{
    const int count = array.length();
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief MColorArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int YFindValueInArray(const MColorArray& array, const MColor& value, const int startIdx)
{
    const int count = array.length();
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief MObjectArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int FindObjectInArray(const MObjectArray& array, const MObject& obj, const int startIdx)
{
    const int count = array.length();
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == obj)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief MDagPathArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int FindDagPathInArray(const MDagPathArray& array, const MDagPath& path, const int startIdx)
{
    const int count = array.length();
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == path)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief CDagPathArray から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int FindDagPathInArray(const CDagPathArray& array, const MDagPath& path, const int startIdx)
{
    const int count = static_cast<int>(array.size());
    for (int index = startIdx; index < count; ++index)
    {
        if (array[index] == path)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief MIntArray から MIntArray を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int YFindArrayInArray(const MIntArray& base, const MIntArray& comp)
{
    const int compCount = comp.length();
    const int ibaseMax = base.length() - compCount;
    for (int ibase = 0; ibase <= ibaseMax; ++ibase)
    {
        bool sameFlag = true;
        for (int index = 0; index < compCount; ++index)
        {
            if (base[ibase + index] != comp[index])
            {
                sameFlag = false;
                break;
            }
        }
        if (sameFlag)
        {
            return ibase;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief MFloatArray から MFloatArray を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int YFindArrayInArray(const MFloatArray& base, const MFloatArray& comp)
{
    const int compCount = comp.length();
    const int ibaseMax = base.length() - compCount;
    for (int ibase = 0; ibase <= ibaseMax; ++ibase)
    {
        bool sameFlag = true;
        for (int index = 0; index < compCount; ++index)
        {
            if (!RIsSame(base[ibase + index], comp[index]))
            {
                sameFlag = false;
                break;
            }
        }
        if (sameFlag)
        {
            return ibase;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief MIntArray に別の MIntArray の内容をすべて追加します。
//-----------------------------------------------------------------------------
void YAppendArrayToArray(MIntArray& dst, const MIntArray& src)
{
    const int count = src.length();
    for (int iSrc = 0; iSrc < count; ++iSrc)
    {
        dst.append(src[iSrc]);
    }
}

//-----------------------------------------------------------------------------
//! @brief MFloatArray に別の MFloatArray の内容をすべて追加します。
//-----------------------------------------------------------------------------
void YAppendArrayToArray(MFloatArray& dst, const MFloatArray& src)
{
    const int count = src.length();
    for (int iSrc = 0; iSrc < count; ++iSrc)
    {
        dst.append(src[iSrc]);
    }
}

//-----------------------------------------------------------------------------
//! @brief MDoubleArray に別の MDoubleArray の内容をすべて追加します。
//-----------------------------------------------------------------------------
void YAppendArrayToArray(MDoubleArray& dst, const MDoubleArray& src)
{
    const int count = src.length();
    for (int iSrc = 0; iSrc < count; ++iSrc)
    {
        dst.append(src[iSrc]);
    }
}

//-----------------------------------------------------------------------------
//! @brief MIntArray から値を検索し、存在しなければ値を追加します。
//!
//! @return 値が存在すれば最初にマッチした位置のインデックス、
//!         値が存在しなければ追加した位置のインデックスを返します。
//-----------------------------------------------------------------------------
int YFindAppendValue(MIntArray& array, const int value)
{
    int index = YFindValueInArray(array, value);
    if (index == -1)
    {
        index = array.length();
        array.append(value);
    }
    return index;
}

//-----------------------------------------------------------------------------
//! @brief MIntArray から別の MIntArray を検索し、存在しなければ内容をすべて
//!        追加します。
//!
//! @return 値が存在すれば最初にマッチした位置のインデックス、
//!         値が存在しなければ追加した位置のインデックスを返します。
//-----------------------------------------------------------------------------
int YFindAppendArray(MIntArray& dst, const MIntArray& src)
{
    int index = YFindArrayInArray(dst, src);
    if (index == -1)
    {
        index = dst.length();
        YAppendArrayToArray(dst, src);
    }
    return index;
}

//-----------------------------------------------------------------------------
//! @brief MVector を RVec3 に変換して返します。
//-----------------------------------------------------------------------------
RVec3 GetRVec3(const MVector& v, const bool snapToZero)
{
    RVec3 vec = RVec3(static_cast<float>(v.x), static_cast<float>(v.y), static_cast<float>(v.z));
    if (snapToZero)
    {
        vec.SnapToZero();
    }
    return vec;
}

//-----------------------------------------------------------------------------
//! @brief MFloatVector を RVec3 に変換して返します。
//-----------------------------------------------------------------------------
RVec3 GetRVec3(const MFloatVector& v, const bool snapToZero)
{
    RVec3 vec = RVec3(v.x, v.y, v.z);
    if (snapToZero)
    {
        vec.SnapToZero();
    }
    return vec;
}

//-----------------------------------------------------------------------------
//! @brief MPoint を RVec3 に変換して返します。
//-----------------------------------------------------------------------------
RVec3 GetRVec3(const MPoint& v, const bool snapToZero)
{
    RVec3 vec = RVec3(static_cast<float>(v.x), static_cast<float>(v.y), static_cast<float>(v.z));
    if (snapToZero)
    {
        vec.SnapToZero();
    }
    return vec;
}

//-----------------------------------------------------------------------------
//! @brief RVec3 を MVector に変換して返します。
//-----------------------------------------------------------------------------
MVector GetMVector(const RVec3& v)
{
    return MVector(v.x, v.y, v.z);
}

//-----------------------------------------------------------------------------
//! @brief RVec3 を MFloatVector に変換して返します。
//-----------------------------------------------------------------------------
MFloatVector GetMFloatVector(const RVec3& v)
{
    return MFloatVector(v.x, v.y, v.z);
}

//-----------------------------------------------------------------------------
//! @brief RVec3 を MPoint に変換して返します。
//-----------------------------------------------------------------------------
MPoint GetMPoint(const RVec3& v)
{
    return MPoint(v.x, v.y, v.z);
}

//-----------------------------------------------------------------------------
//! @brief MColor を RVec3 に変換して返します。
//-----------------------------------------------------------------------------
RVec3 GetRVec3Color(const MColor& mc, const bool snapToZero, const bool clamps)
{
    if (mc == Y_MCOLOR_NULL)
    {
        return RVec3::kZero;
    }
    else
    {
        RVec3 col(mc.r, mc.g, mc.b);
        if (clamps)
        {
            col.x = RClampValue(0.0f, 1.0f, col.x);
            col.y = RClampValue(0.0f, 1.0f, col.y);
            col.z = RClampValue(0.0f, 1.0f, col.z);
        }
        if (snapToZero)
        {
            col.SnapToZero();
        }
        return col;
    }
}

//-----------------------------------------------------------------------------
//! @brief MColor を RVec4 に変換して返します。
//-----------------------------------------------------------------------------
RVec4 GetRVec4Color(const MColor& mc, const bool snapToZero, const bool clamps)
{
    if (mc == Y_MCOLOR_NULL)
    {
        return RVec4::kBlack;
    }
    else
    {
        RVec4 col(mc.r, mc.g, mc.b, mc.a);
        if (clamps)
        {
            col.x = RClampValue(0.0f, 1.0f, col.x);
            col.y = RClampValue(0.0f, 1.0f, col.y);
            col.z = RClampValue(0.0f, 1.0f, col.z);
            col.w = RClampValue(0.0f, 1.0f, col.w);
        }
        if (snapToZero)
        {
            col.SnapToZero();
        }
        return col;
    }
}

//-----------------------------------------------------------------------------
//! @brief MColor を RRgbaColor に変換して返します。
//-----------------------------------------------------------------------------
RRgbaColor GetRRgbaColor(const MColor& mc)
{
    int r, g, b, a;

    if (mc == Y_MCOLOR_NULL)
    {
        r = g = b = 0;
        a = RRgbaColor::COL_MAX;
    }
    else
    {
        r = RRound(mc.r * static_cast<float>(RRgbaColor::COL_MAX));
        g = RRound(mc.g * static_cast<float>(RRgbaColor::COL_MAX));
        b = RRound(mc.b * static_cast<float>(RRgbaColor::COL_MAX));
        a = RRound(mc.a * static_cast<float>(RRgbaColor::COL_MAX));
        if (r < 0) r = 0;
        if (g < 0) g = 0;
        if (b < 0) b = 0;
        if (a < 0) a = 0;
        if (r > RRgbaColor::COL_MAX) r = RRgbaColor::COL_MAX;
        if (g > RRgbaColor::COL_MAX) g = RRgbaColor::COL_MAX;
        if (b > RRgbaColor::COL_MAX) b = RRgbaColor::COL_MAX;
        if (a > RRgbaColor::COL_MAX) a = RRgbaColor::COL_MAX;
    }

    return RRgbaColor(r, g, b, a);
}

//-----------------------------------------------------------------------------
//! @brief MMatrix を RMtx44 に変換して返します。
//-----------------------------------------------------------------------------
RMtx44 GetRMtx44(const MMatrix& mm)
{
    RMtx44 rm;
    for (int row = 0; row < 4; ++row)
    {
        for (int col = 0; col < 4; ++col)
        {
            rm[row][col] = static_cast<float>(mm[row][col]);
        }
    }
    return rm;
}

//-----------------------------------------------------------------------------
//! @brief RStatus を MStatus に変換して返します。
//-----------------------------------------------------------------------------
MStatus GetMStatus(const RStatus& rstatus)
{
    if (rstatus)
    {
        return MS::kSuccess;
    }
    else
    {
        return MS::kFailure;
    }
}

//-----------------------------------------------------------------------------
//! @brief RStatus を MStatus に変換して返します。
//!        RStatus が失敗ならエラーメッセージを表示します。
//-----------------------------------------------------------------------------
MStatus GetMStatusDisplayError(RScene* pScene, const RStatus& rstatus)
{
    if (rstatus)
    {
        return MS::kSuccess;
    }
    else
    {
        const std::string msgJp = rstatus.GetJapaneseMessage();
        const std::string msgEn = rstatus.GetMessage();
        if (!msgJp.empty() || !msgEn.empty())
        {
            YShowError(pScene, msgJp, msgEn);
        }
        return MS::kFailure;
    }
}

//-----------------------------------------------------------------------------
//! @brief 親ノードが joint ノードなら true を返します。
//-----------------------------------------------------------------------------
bool HasMayaJointParent(const MDagPath& dagPath)
{
    if (dagPath.length() > 1) // "|joint1" の length は 1
    {
        MDagPath parentPath = dagPath;
        parentPath.pop();
        return (parentPath.apiType() == MFn::kJoint);
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 親ノードのスケールが inverseScale アトリビュートに接続されていれば true を返します。
//-----------------------------------------------------------------------------
bool IsParentScaleConnectedToMayaJoint(const MDagPath& dagPath)
{
    if (dagPath.node().apiType() == MFn::kJoint &&
        dagPath.length() > 1)
    {
        MDagPath parentPath = dagPath;
        parentPath.pop();
        if (parentPath.apiType() == MFn::kJoint)
        {
            MPlugArray plugArray;
            MFnDagNode(dagPath).findPlug("inverseScale").connectedTo(plugArray, true, false);
            if (plugArray.length() != 0 &&
                plugArray[0] == MFnDagNode(parentPath).findPlug("scale"))
            {
                return true;
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief nodeState アトリビュートの値が有効（通常）なら true を返します。
//-----------------------------------------------------------------------------
bool IsNodeStateEffective(const MObject& obj)
{
    int nodeState;
    MFnDependencyNode(obj).findPlug("nodeState").getValue(nodeState);
    if (nodeState == 1 ||   // HasNoEffect
        nodeState == 2)     // Blocking
    {
        return false;
    }
    else
    {
        return true;
    }
}


//-----------------------------------------------------------------------------
//! @brief ノードが圧縮不可なら true を返します。
//-----------------------------------------------------------------------------
bool IsNoCuttingNode(const MObject& obj)
{
    MStatus status;

    MFnDependencyNode depFn(obj);
    MPlug noCompressPlug = FindPlugQuiet(depFn, "nw4fNoCompressNode", &status);
    if (!status)
    {
        return false;
    }

    bool noCompressFlag;
    noCompressPlug.getValue(noCompressFlag);

    return noCompressFlag;
}

//-----------------------------------------------------------------------------
//! @brief ノードが有効（出力対象）なら true を返します。
//-----------------------------------------------------------------------------
bool IsEffectiveNode(const MObject& obj)
{
    MFnDependencyNode depFn(obj);

    bool visibleFlag, templateFlag, intermediateFlag;

#if 1
    // ノードから primaryVisibility の値を取得します。
    MStatus status;
    MPlug visPlug = FindPlugQuiet(depFn, "primaryVisibility", &status);
    //MPlug visPlug = depFn.findPlug("primaryVisibility", &status);
    if (status)
    {
        visPlug.getValue(visibleFlag);
    }
    else
    {   // アトリビュートが取得できなかった場合は常に primaryVisibility=ON とみなす
        visibleFlag = true;
    }
    //cerr << depFn.name() << " " << visibleFlag << R_ENDL;
#else
    depFn.findPlug("visibility").getValue(visibleFlag);
#endif

    // ノードがテンプレートオブジェクトか中間オブジェクトであるか調べる
    depFn.findPlug("template").getValue(templateFlag);
    depFn.findPlug("intermediateObject").getValue(intermediateFlag);
    // primaryVisibility = ON, template = OFF, intermediateObject = OFF であることを確認します。
    if (!visibleFlag || templateFlag || intermediateFlag)
    {
        return false;
    }

    // overrideEnabled アトリビュートの値を確認し OFF なら無効と判断します。
    bool overrideFlag;
    depFn.findPlug("overrideEnabled").getValue(overrideFlag);
    if (!overrideFlag)
    {
        return true;
    }

    // overrideVisibility アトリビュートの値を確認し OFF なら無効と判断します。
    depFn.findPlug("overrideVisibility").getValue(visibleFlag);
    if (!visibleFlag)
    {
        return false;
    }

    // overrideShading アトリビュートの値を確認し OFF なら無効と判断します。
    short shadingFlag;
    depFn.findPlug("overrideShading").getValue(shadingFlag);
    if (!shadingFlag)
    {
        return false;
    }

    // overrideDisplayType アトリビュートの値を確認し、
    // template ではないことを確認します。
    short displayType;  // 0=Normal, 1=Template, 2=Reference
    depFn.findPlug("overrideDisplayType").getValue(displayType);

    return (displayType != 1);  // effective if not template
}

//-----------------------------------------------------------------------------
//! @brief ノードが可視なら true を返します。
//-----------------------------------------------------------------------------
bool IsVisibleNode(const MObject& obj)
{
    MFnDependencyNode depFn(obj);

    bool visibleFlag;
    depFn.findPlug("visibility").getValue(visibleFlag);
    if (!visibleFlag)
    {
        return false;
    }

    bool overrideFlag;
    depFn.findPlug("overrideEnabled").getValue(overrideFlag);
    if (!overrideFlag)
    {
        return true;
    }

    depFn.findPlug("overrideVisibility").getValue(visibleFlag);
    if (!visibleFlag)
    {
        return false;
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief 親ノードも含めて DAG ノードが可視なら true を返します。
//-----------------------------------------------------------------------------
bool IsVisibleNodeIncludeParent(const MDagPath& dagPath)
{
    MDagPath curPath = dagPath;
    do
    {
        if (!IsVisibleNode(curPath.node()))
        {
            return false;
        }
        curPath.pop();
    } while (curPath.length() > 0);
    return true;
}

//-----------------------------------------------------------------------------
//! @brief ノードがテンプレート状態なら true を返します。
//-----------------------------------------------------------------------------
bool IsTemplateNode(const MObject& obj)
{
    MFnDependencyNode depFn(obj);

    bool templateFlag;
    depFn.findPlug("template").getValue(templateFlag);
    if (templateFlag)
    {
        return true;
    }

    bool overrideFlag;
    depFn.findPlug("overrideEnabled").getValue(overrideFlag);
    if (!overrideFlag)
    {
        return false;
    }

    short displayType;  // 0=Normal, 1=Template, 2=Reference
    depFn.findPlug("overrideDisplayType").getValue(displayType);

    return (displayType == 1);
}

//-----------------------------------------------------------------------------
//! @brief ノードが指定されたタイプの子ノードを持つなら true を返します。
//-----------------------------------------------------------------------------
bool NodeHasSpecificChild(const MFnDagNode& dagFn, MFn::Type type)
{
    int childCount = dagFn.childCount();
    for (int ichild = 0; ichild < childCount; ++ichild)
    {
        if (dagFn.child(ichild).apiType() == type)
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ノードが指定されたファンクションタイプの子ノードを持つなら true を返します。
//-----------------------------------------------------------------------------
bool NodeHasSpecificFunctionChild(const MFnDagNode& dagFn, MFn::Type type)
{
    int childCount = dagFn.childCount();
    for (int ichild = 0; ichild < childCount; ++ichild)
    {
        if (dagFn.child(ichild).hasFn(type))
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ノードが指定されたタイプの親ノードを持つなら true を返します。
//-----------------------------------------------------------------------------
bool NodeHasSpecificParent(MDagPath dagPath, MFn::Type type)
{
    if (dagPath.pop())
    {
        if (dagPath.node().apiType() == type)
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ノードが有効なトランスフォームの子ノードを持つなら true を返します。
//-----------------------------------------------------------------------------
bool NodeHasEffectiveTransformChild(const MFnDagNode& dagFn)
{
    int childCount = dagFn.childCount();
    for (int ichild = 0; ichild < childCount; ++ichild)
    {
        // DAGノードがトランスフォームを子供に持つかどうかを検索して調べる
        MObject childObj = dagFn.child(ichild);
        if (childObj.hasFn(MFn::kTransform))
        {
            // 見つかったシェイプに対しアトリビュートが有効な設定になているか確認します。
            // アトリビュートが有効になっていなくても「Set No Compress Node」
            // プラグインにより削除されない設定になっている場合は有効とみなす。
            if (IsEffectiveNode(childObj) || IsNoCuttingNode(childObj))
            {
                return true;
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ノードが有効な mesh ノードの子ノードを持つなら true を返します。
//-----------------------------------------------------------------------------
bool NodeHasEffectiveMeshChild(const MFnDagNode& dagFn)
{
    int childCount = dagFn.childCount();
    for (int ichild = 0; ichild < childCount; ++ichild)
    {
        MObject childObj = dagFn.child(ichild);
        if (childObj.apiType() == MFn::kMesh &&
            IsEffectiveNode(childObj))
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
// joint ラベルタイプを表す中間ファイルの文字列配列です。
//-----------------------------------------------------------------------------
static const char* s_JointLabelTypeStrings[Y_JOINT_LABEL_TYPE_COUNT] =
{
    "", // none
    "nw_root",
    "nw_hip",
    "nw_knee",
    "nw_foot",
    "nw_toe",
    "nw_spine",
    "nw_neck",
    "nw_head",
    "nw_collar",
    "nw_shoulder",
    "nw_elbow",
    "nw_hand",
    "nw_finger",
    "nw_thumb",
    "nw_prop_a",
    "nw_prop_b",
    "nw_prop_c",
    "", // other（その他の場合は「その他のタイプ」の文字列を出力するので使用されません）。
    "nw_index_finger",
    "nw_middle_finger",
    "nw_ring_finger",
    "nw_pinky_finger",
    "nw_extra_finger",
    "nw_big_toe",
    "nw_index_toe",
    "nw_middle_toe",
    "nw_ring_toe",
    "nw_pinky_toe",
    "nw_foot_thumb",
};

//-----------------------------------------------------------------------------
//! @brief joint ラベルタイプを表す中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string YGetJointLabelTypeString(const YJointLabelType jointLabelType)
{
    return (jointLabelType < Y_JOINT_LABEL_TYPE_COUNT) ? s_JointLabelTypeStrings[jointLabelType] : "";
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列から joint ラベルタイプを取得します。
//-----------------------------------------------------------------------------
YJointLabelType YGetJointLabelTypeFromString(const std::string& str)
{
    if (!str.empty())
    {
        for (int jointLabelType = 0; jointLabelType < Y_JOINT_LABEL_TYPE_COUNT; ++jointLabelType)
        {
            if (str == s_JointLabelTypeStrings[jointLabelType])
            {
                return static_cast<YJointLabelType>(jointLabelType);
            }
        }
        return Y_JOINT_LABEL_TYPE_OTHER;
    }
    return Y_JOINT_LABEL_TYPE_NONE;
}

//-----------------------------------------------------------------------------
//【現在は使用されていません】
//! @brief 選択リストにスキンに影響しているノードの DAG パスを追加します。
//!
//! @param[in,out] paths 選択リストの DAG パス配列です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
//static MStatus AddSkinInfluencePathsToSelectionList(CDagPathArray& paths)
//{
//  int pathCount = paths.size();
//  for (int ipath = 0; ipath < pathCount; ++ipath)
//  {
//      //-----------------------------------------------------------------------------
//      // get mesh path
//      MDagPath xformPath = paths[ipath];
//      MFnDagNode dagFn(xformPath);
//      if (!NodeHasEffectiveMeshChild(dagFn))
//          continue;
//      MDagPath shapePath = xformPath;
//      shapePath.extendToShape();
//      //cerr << "shape: " << shapePath.partialPathName() << R_ENDL;
//
//      //-----------------------------------------------------------------------------
//      // get influence path
//      CDagPathArray infs;
//      int infCount = GetSkinInfluencePathsForShape(infs, shapePath);
//      for (int iinf = 0; iinf < infCount; ++iinf)
//      {
//          MDagPath infPath = infs[iinf];
//          // スキンに影響するノードは最上位ノードまで含めて出力
//          while (infPath.length() > 0)
//          {
//              if (FindDagPathInArray(paths, infPath) == -1)
//              {
//                  paths.push_back(infPath);
//#ifdef DEBUG_PRINT_SW
//                  cerr << "add inf node: " << infPath.partialPathName() << R_ENDL;
//#endif
//                  //cerr << "add inf: " << infPath.fullPathName() << "," << infPath.length() << R_ENDL;
//              }
//              infPath.pop();
//          }
//      }
//
//      //-----------------------------------------------------------------------------
//      // スキンされたノードの親も最上位まで出力
//      if (infCount > 0)
//      {
//          MDagPath parentPath = xformPath;
//          while (parentPath.length() > 1)
//          {
//              parentPath.pop();
//              if (FindDagPathInArray(paths, parentPath) == -1)
//              {
//                  paths.push_back(parentPath);
//#ifdef DEBUG_PRINT_SW
//                  cerr << "add skin parent: " << parentPath.partialPathName() << R_ENDL;
//#endif
//              }
//          }
//      }
//  }
//
//  return MS::kSuccess;
//}

//-----------------------------------------------------------------------------
//! @brief DAG パスの階層の深さを比較します。
//!     二つの DAG パスの階層の深さを比較し、結果を返します。
//!
//! @param[in] r1 比較する DAG パスです。
//! @param[in] r2 比較する DAG パスです。
//!
//! @return r2 より r1 の階層が深ければ true を返します。
//!         r1 より r2 の階層が深いか r1 と r2 が同じ階層なら false を返します。
//-----------------------------------------------------------------------------
static bool DagPathDepthLess(const MDagPath& r1, const MDagPath& r2)
{
    return (r1.length() < r2.length());
}

//-----------------------------------------------------------------------------
//! @brief インスタンスを考慮して指定した親ノードの直接の子の DAG パスを取得します。
//-----------------------------------------------------------------------------
MDagPath GetDirectChildDagPath(
    const MDagPath& parentPath,
    const MObject& childObj
)
{
    // オブジェクトのDagノードを作成
    MFnDagNode dagFn(childObj);
    // Dagノードの全てのパスを取得します。
    // インスタンス化されている場合、allPaths には複数の DAG パスが設定されます。
    MDagPathArray allPaths;
    dagFn.getAllPaths(allPaths);

    // 取得した DAG パスの親が parentPath に一致するものを検索します。
    const int pathCount = allPaths.length();
    for (int ipath = 0; ipath < pathCount; ++ipath)
    {
        MDagPath testPath = allPaths[ipath];
        // パスを一つ戻して親のパスに変換します。
        testPath.pop();
        // parentPathと一致すればそのパスを返します。
        if (testPath == parentPath)
        {
            return allPaths[ipath];
        }
    }
    // 親が間違っていない限りここには来ない
    MDagPath childPath;
    dagFn.getPath(childPath);
    return childPath;
}

//-----------------------------------------------------------------------------
//! @brief 指定した transform ノード以下の階層の transform ノード群の DAG パスを取得します。
//!        再帰的に呼ばれます。
//!
//! @param[in,out] pXformPaths transform ノードの DAG パス配列へのポインターです。
//!                            配列に存在しない transform ノードがあれば追加します。
//! @param[in] xformPath 追加する階層構造の最上位 transform ノードの DAG パスです。
//-----------------------------------------------------------------------------
static void GetTransformNodesBelow(CDagPathArray* pXformPaths, const MDagPath& xformPath)
{
    //-----------------------------------------------------------------------------
    // 指定された transform ノードが配列に存在しなければ追加します。
    if (FindDagPathInArray(*pXformPaths, xformPath) == -1)
    {
        pXformPaths->push_back(xformPath);
    }

    //-----------------------------------------------------------------------------
    // 指定された transform ノードの下位階層に存在する transform ノードを取得します。
    const int childCount = xformPath.childCount();
    for (int childIdx = 0; childIdx < childCount; ++childIdx)
    {
        MObject childObj = xformPath.child(childIdx);
        if (childObj.hasFn(MFn::kTransform))
        {
            // 階層インスタンスに対応できるように getAllPath を使用し、
            // 親が xformPath に一致するものを子ノードのパスとして採用します。
            const MDagPath childPath = GetDirectChildDagPath(xformPath, childObj);
            GetTransformNodesBelow(pXformPaths, childPath);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 選択された transform ノード群以下に含まれる transform ノード群の
//!        DAG パス配列を取得します。
//-----------------------------------------------------------------------------
void GetSelectedTransformNodesBelow(CDagPathArray* pXformPaths, const MSelectionList& slist)
{
    pXformPaths->clear();
    MItSelectionList sIter(slist, MFn::kTransform);
    for ( ; !sIter.isDone(); sIter.next())
    {
        MDagPath xformPath;
        if (sIter.getDagPath(xformPath))
        {
            GetTransformNodesBelow(pXformPaths, xformPath);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 選択された transform ノード以下の階層も含む選択リストを取得します。
//-----------------------------------------------------------------------------
void GetTreeSelectionList(MSelectionList* pSlist)
{
    //-----------------------------------------------------------------------------
    // 選択された transform ノード群以下に含まれる transform ノード群を取得します。
    CDagPathArray xformPaths;
    GetSelectedTransformNodesBelow(&xformPaths, *pSlist);

    //-----------------------------------------------------------------------------
    // add skin influence paths to selection list
    //AddSkinInfluencePathsToSelectionList(xformPaths);

    //-----------------------------------------------------------------------------
    // DAG パスの階層レベルが浅い順にソートします。
    std::sort(xformPaths.begin(), xformPaths.end(), DagPathDepthLess);

    //-----------------------------------------------------------------------------
    // 選択リストを再作成します。
    pSlist->clear();
    const int xformCount = static_cast<int>(xformPaths.size());
    for (int xformIdx = 0; xformIdx < xformCount; ++xformIdx)
    {
        pSlist->add(xformPaths[xformIdx]);
    }
}

//-----------------------------------------------------------------------------
//! @brief エッジ群のソフト／ハードを設定します。
//!        MFnMesh::setEdgeSmoothing は重いので、MEL コマンド polySoftEdge を使用します。
//-----------------------------------------------------------------------------
void YSetEdgesSoftHard(const MDagPath& meshPath, MIntArray& iEdges, const bool isSoft)
{
    if (iEdges.length() != 0)
    {
        MSelectionList selBak;
        MGlobal::getActiveSelectionList(selBak);

        MFnSingleIndexedComponent edgeCompFn;
        MObject edgeComp = edgeCompFn.create(MFn::kMeshEdgeComponent);
        edgeCompFn.addElements(iEdges);
        MGlobal::select(meshPath, edgeComp, MGlobal::kReplaceList);
        MGlobal::executeCommand(MString("polySoftEdge -ch 0 -a ") + ((isSoft) ? "360" : "0"));

        MGlobal::setActiveSelectionList(selBak);
    }
}

//-----------------------------------------------------------------------------
//! @brief カラーセットのファミリー名を取得します。
//-----------------------------------------------------------------------------
MString GetColorSetFamilyName(const MDagPath& meshPath, const MString& colSet)
{
    const MFnMesh meshFn(meshPath);
    if (meshFn.isColorSetPerInstance(colSet))
    {
        MStringArray familyNames;
        meshFn.getColorSetFamilyNames(familyNames);
        for (int iFamily = 0; iFamily < static_cast<int>(familyNames.length()); ++iFamily)
        {
            const MString& familyName = familyNames[iFamily];
            MStringArray colSetsInFamily;
            meshFn.getColorSetsInFamily(familyName, colSetsInFamily);
            if (YFindValueInArray(colSetsInFamily, colSet) != -1)
            {
                return familyName;
            }
        }
    }
    return colSet;
}

//-----------------------------------------------------------------------------
//! @brief 指定した mesh ノード用のカラーセットなら true を返します。
//!
//! @param[in] meshPath mesh ノードの DAG パスです。
//! @param[in] colSet カラーセット名です。
//!
//! @return 指定した mesh ノード用のカラーセットなら true を返します。
//-----------------------------------------------------------------------------
static bool IsColorSetForMesh(const MDagPath& meshPath, const MString& colSet)
{
    const MFnMesh meshFn(meshPath);
    if (meshFn.isColorSetPerInstance(colSet))
    {
        // インスタンスごとのカラーセットの場合は、このカラーセットが対象とする
        // インスタンスのインデックス配列を取得し、パスのインスタンス番号が
        // そのインデックス配列に含まれているかどうかを調べます。
        // インデックス配列にインスタンス番号が含まれていればカラーセットは
        // このメッシュのインスタンスに対するものと判断できます。
        MIntArray instances;
        meshFn.getAssociatedColorSetInstances(colSet, instances);
        if (YFindValueInArray(instances, meshPath.instanceNumber()) == -1)
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief mesh ノードのカレントのカラーセット名を取得します。
//-----------------------------------------------------------------------------
MString GetCurrentColorSetName(const MDagPath& meshPath)
{
    const MFnMesh meshFn(meshPath);
    if (meshFn.numColorSets() == 0)
    {
        return "";
    }

    MString curColSet = meshFn.currentColorSetName(meshPath.instanceNumber());

    //↑2009 (無印）でこれらを実行すると Output Window に getting colorset for ... と表示されてしまいます。
    //curColSet = meshFn.findPlug("currentColorSet").asString();
    //cerr << "gccsn: " << meshPath.partialPathName() << " [" << curColSet<< "]"<< endl;
    //if (meshFn.isColorSetPerInstance(curColSet))
    //{
    //  MStringArray names;
    //  meshFn.getColorSetsInFamily(curColSet, names);
    //  int nameCount = static_cast<int>(names.length());
    //  if (nameCount >= 2)
    //  {
    //      for (int iname = 0; iname < nameCount; ++iname)
    //      {
    //          MIntArray instances;
    //          meshFn.getAssociatedColorSetInstances(names[iname], instances);
    //          if (YFindValueInArray(instances, meshPath.instanceNumber()) != -1)
    //          {
    //              curColSet = names[iname];
    //              break;
    //          }
    //      }
    //  }
    //}

    //cerr << "color set: " << meshPath.partialPathName() << ": " << meshPath.instanceNumber() << ": [" << curColSet << "]" << R_ENDL;
    return curColSet;
}

//-----------------------------------------------------------------------------
//! @brief カラーセットの成分数を取得します。
//-----------------------------------------------------------------------------
int GetColorSetComponentCount(bool* pIsRa, const MDagPath& meshPath, const MString& colSet)
{
    //-----------------------------------------------------------------------------
    // カラーセット名から成分数を取得します。
    if (pIsRa != nullptr)
    {
        *pIsRa = false;
    }
    const MString familyName = GetColorSetFamilyName(meshPath, colSet);
    const int length = static_cast<int>(familyName.length());
    const int iUnder = familyName.rindex('_'); // 最後の _ を検索します。
    if (iUnder != -1 && iUnder + 3 == length) // 3 = "_NT" (N = 1,2,3,4, T = i,u,f) の形式
    {
        int compCount = familyName.asChar()[iUnder + 1] - '0';
        if (compCount == ColorSetComponentCountRa)
        {
            compCount = 2;
            if (pIsRa != nullptr)
            {
                *pIsRa = true;
            }
        }
        if (1 <= compCount && compCount <= 4) // 成分数が正しいことを確認します。
        {
            const char t = familyName.asChar()[length - 1];
            if (t == 'i' || t == 'u' || t == 'f') // 値の型が正しいことを確認します。
            {
                return compCount;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // カラーセットのコンポーネントタイプから成分数を取得します。
    const MFnMesh::MColorRepresentation repr =
        MFnMesh(meshPath).getColorRepresentation(colSet);
    return (repr == MFnMesh::kRGB) ? 3 : 4;
}

//-----------------------------------------------------------------------------
//! @brief カラーセット名から頂点カラー属性のインデックスを取得します。
//-----------------------------------------------------------------------------
int GetVertexColorAttrIndex(const MDagPath& meshPath, const MString& colSet)
{
    const MString familyName = GetColorSetFamilyName(meshPath, colSet);
    for (int iCol = 0; iCol < RPrimVtx::VTX_COL_MAX; ++iCol)
    {
        const MString attrNameTop = MString("nw4f_color") + RGetNumberString(iCol).c_str();
        if (familyName == attrNameTop) // データ型指定なし
        {
            return iCol;
        }
        else if (strncmp(familyName.asChar(), attrNameTop.asChar(), attrNameTop.length()) == 0)
        {
            const MString typeStr = familyName.substring(attrNameTop.length(), familyName.length() - 1);
            const int typeLen = static_cast<int>(typeStr.length());
            if ((typeLen == 2 || typeLen == 3) && typeStr.asChar()[0] == '_')
            {
                int compCount = (typeLen == 3) ?
                    typeStr.asChar()[1] - '0' : 1;
                compCount = (compCount == ColorSetComponentCountRa) ? 2 : compCount;
                if (1 <= compCount && compCount <= 4) // 成分数が正しいことを確認します。
                {
                    const char t = typeStr.asChar()[typeLen - 1];
                    if (t == 'i' || t == 'u' || t == 'f') // 値の型が正しいことを確認します。
                    {
                        return iCol;
                    }
                }
            }
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief 無視する（出力しない）カラーセットなら true を返します。
//!
//! @param[in] colSetName カラーセット名です。
//!
//! @return 無視する（出力しない）カラーセットなら true を返します。
//-----------------------------------------------------------------------------
static bool IgnoresColorSet(const MString& colSetName)
{
    return (colSetName.indexW("nw_ignore") == 0);
}

//-----------------------------------------------------------------------------
//! @brief mesh ノードから出力する頂点カラー属性のカラーセット名配列を取得します。
//-----------------------------------------------------------------------------
MStatus GetMeshVertexColorAttrs(
    MString* meshColSets,
    RScene& rscene,
    const MDagPath& meshPath
)
{
    //-----------------------------------------------------------------------------
    // 各カラーセット名を空文字で初期化します。
    for (int colIdx = 0; colIdx < RPrimVtx::VTX_COL_MAX; ++colIdx)
    {
        meshColSets[colIdx].clear();
    }

    //-----------------------------------------------------------------------------
    // mesh ノードの全カラーセットを調べてマルチ頂点カラー用の名前なら
    // カラーセット名配列に格納します。
    const MFnMesh meshFn(meshPath);
    MStringArray meshColSetNames;
    meshFn.getColorSetNames(meshColSetNames);
    //cerr << "all color sets: " << meshPath.partialPathName() << ": " << meshColSetNames << endl;
    //MStringArray familyNames;
    //meshFn.getColorSetFamilyNames(familyNames);
    //cerr << "family names: " << meshPath.partialPathName() << ": " << familyNames << endl;
    SortArray(meshColSetNames); // アルファベット順にソートします。
    MString firstColSetName; // マルチ頂点カラー用の名前でないアルファベット順で最初のカラーセット名です。
    bool isColor0Found = false;

    for (int meshColSetIdx = 0; meshColSetIdx < static_cast<int>(meshColSetNames.length()); ++meshColSetIdx)
    {
        const MString& colSetName = meshColSetNames[meshColSetIdx];
        if (IgnoresColorSet(colSetName))
        {
            continue;
        }
        const int colIdx = GetVertexColorAttrIndex(meshPath, colSetName);
        if (colIdx == 0)
        {
            isColor0Found = true;
        }
        // カラーセットにカラーが設定されていて、
        // カラーセットがこの mesh ノードのインスタンスに対するのもであることを確認します。
        if (meshFn.numColors(colSetName) > 0 &&
            IsColorSetForMesh(meshPath, colSetName))
        {
            if (colIdx != -1)
            {
                if (meshColSets[colIdx].length() != 0)
                {
                    YShowError(&rscene, // Color set for same vertex color exists: %s: %s
                        "同じ番号の頂点カラーに対応するカラーセットが複数存在します: {0}: {1} \n"
                        "たとえば 1 つのオブジェクトに「nw4f_color1」と「nw4f_color1_i」という名前のカラーセットが存在するような状態です。\n"
                        "不要なカラーセットを削除するか名前を変更してください。",
                        "More than one color set corresponds to the same vertex color number: {0}: {1} \n"
                        "This error may occur when, for example, color sets named nw4f_color1 and nw4f_color1_i both exist for the same object. \n"
                        "Either delete unnecessary color sets, or change their names.",
                        meshPath.partialPathName().asChar(), colSetName.asChar());
                    return MS::kFailure;
                }
                meshColSets[colIdx] = colSetName;
            }
            else if (firstColSetName.length() == 0)
            {
                firstColSetName = colSetName;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // color0 用の名前のカラーセットが存在しない場合、
    // カレントのカラーセットまたは firstColSetName を格納します。
    if (!isColor0Found && meshFn.numColorSets() > 0)
    {
        const MString curColSetName = GetCurrentColorSetName(meshPath);
        if (!IgnoresColorSet(curColSetName))
        {
            meshColSets[0] = (GetVertexColorAttrIndex(meshPath, curColSetName) == -1) ?
                curColSetName : firstColSetName;
        }
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief mesh ノードのフェース群に頂点カラーが設定されていれば true を返します。
//-----------------------------------------------------------------------------
bool ComponentVertexColorExists(
    const MDagPath& meshPath,
    MObject& comp,
    const MString& colSet
)
{
    MItMeshPolygon pIter(meshPath, comp);
    for ( ; !pIter.isDone(); pIter.next())
    {
        int colCount;
        pIter.numColors(colCount, &colSet);
        if (colCount > 0) // フェースに頂点カラーが設定されている
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief mesh ノードのフェース群に設定されている頂点カラーがすべて不透明なら true を返します。
//-----------------------------------------------------------------------------
bool IsComponentVertexColorOpaque(
    const MDagPath& meshPath,
    MObject& comp,
    const MString& colSet
)
{
    if (colSet.length() == 0) // no vertex color
    {
        return true;
    }

    MItMeshPolygon pIter(meshPath, comp);
    for ( ; !pIter.isDone(); pIter.next())
    {
        int colCount;
        pIter.numColors(colCount, &colSet);
        if (colCount > 0)
        {
            MColorArray vtxColors;
            pIter.getColors(vtxColors, &colSet); // 頂点カラーがない頂点に対しては（-1, -1, -1, -1) が入ります。
            const int vtxCount = pIter.polygonVertexCount();
            for (int ivtx = 0; ivtx < vtxCount; ++ivtx)
            {
                if (vtxColors[ivtx] != Y_MCOLOR_NULL &&
                    vtxColors[ivtx].a < 1.0f)
                {
                    return false;
                }
            }
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief シェーダオブジェクトからカラーのプラグを取得します。
//-----------------------------------------------------------------------------
MPlug GetShaderColorPlug(const MObject& shaderObj)
{
    MString plugName;
    MFn::Type type = shaderObj.apiType();
    if (type == MFn::kSurfaceShader)
    {
        plugName = "outColor";
    }
    else if (type == MFn::kShadingMap)
    {
        plugName = "shadingMapColor";
    }
    else
    {
        plugName = "color";
    }
    MFnDependencyNode depFn(shaderObj);
    return depFn.findPlug(plugName);
}

//-----------------------------------------------------------------------------
//! @brief シェーディンググループからシェーダオブジェクトを取得します。
//-----------------------------------------------------------------------------
MObject FindShaderObject(const MObject& sgObj)
{
    MFnDependencyNode sgFn(sgObj);
    MPlugArray plugArray;
    sgFn.findPlug("surfaceShader").connectedTo(plugArray, true, false);
    if (plugArray.length() == 0)
    {
        return MObject::kNullObj;
    }
    else
    {
        return plugArray[0].node();
    }
}

//-----------------------------------------------------------------------------
//! @brief デフォルトの UV セット名を取得します。
//!        UV セットが複数ある場合、インデックスが最小のものを取得します。
//!
//! @param[in] shapePath シェイプノードの DAG パスです。
//!
//! @return デフォルトの UV セット名を返します。
//-----------------------------------------------------------------------------
MString GetDefaultUvSetName(const MDagPath& shapePath)
{
    MStatus status;
    MString uvSetName = "map1";
    MPlug uvSetsPlug = MFnDependencyNode(shapePath.node()).findPlug("uvSet"); // controlPoint のアトリビュート
    for (int iUvSet = 0; iUvSet < static_cast<int>(uvSetsPlug.numElements()); ++iUvSet)
    {
        MPlug uvSetPlug = uvSetsPlug.elementByLogicalIndex(iUvSet, &status);
        if (status)
        {
            //const int uvCount = uvSetPlug.child(1).numElements(); // uvSetPoints (float2)
            //MString name; uvSetPlug.child(0).getValue(name); cerr << "uv set: " << name << ": " << uvCount << endl;
            uvSetPlug.child(0).getValue(uvSetName); // uvSetName (string)
            break;
        }
    }
    return uvSetName;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーパターンアニメーション用のファイルパスを取得します。
//!
//! @param[in] filePre フレーム拡張子より前のファイルパスです（フォルダのパスも含みます）。
//! @param[in] fileExt フレーム拡張子より後のファイルパスです。
//! @param[in] fe フレーム拡張子です。
//!
//! @return フレーム拡張子に対応したファイルが存在すればファイルパスを返します。
//!         存在しなければ空文字を返します。
//-----------------------------------------------------------------------------
std::string GetFilePathForTexPatAnim(
    const std::string& filePre,
    const std::string& fileExt,
    const int fe
)
{
    const int FMT_COUNT = 4;
    static const char* const testFmt[FMT_COUNT] = { "%d", "%02d", "%03d", "%04d" };
    for (int iFmt = 0; iFmt < FMT_COUNT; ++iFmt)
    {
        std::string filePath = filePre +
            RGetNumberString(fe, testFmt[iFmt]) + fileExt;
        if (RFileExists(filePath))
        {
            return filePath;
        }
    }
    return std::string();
}

//-----------------------------------------------------------------------------
//! @brief file ノードのファイルテクスチャー名（イメージの名前）を絶対パスで取得します。
//!        文字列 <f> は現在のフレーム拡張子に置き換えます。
//!
//! @param[in] fileObj file ノードオブジェクトです。
//! @param[in] rootFolder ファイルテクスチャー名が相対パスの場合のルートフォルダーのパスです。
//!
//! @return ファイルテクスチャー名（イメージの名前）の絶対パスを返します。
//-----------------------------------------------------------------------------
std::string GetFileTextureName(const MObject& fileObj, const std::string& rootFolder)
{
    static const std::string FrameTag = "<f>";

    MFnDependencyNode texFn(fileObj);
    MString filePathTmp;
    texFn.findPlug("fileTextureName").getValue(filePathTmp);
    std::string filePath = filePathTmp.asChar();

    if (!filePath.empty())
    {
        // 絶対パスに変換します。
        filePath = YGetFullFilePath(filePath, rootFolder);

        // 文字列 <f> が含まれていれば現在のフレーム拡張子に置き換えます。
        int curFe = 0;
        bool isCurFeGot = false;
        size_t frameTagIdx = filePath.find(FrameTag);
        while (frameTagIdx != std::string::npos)
        {
            if (!isCurFeGot)
            {
                texFn.findPlug("frameExtension").getValue(curFe);
                isCurFeGot = true;
            }
            const std::string filePre = filePath.substr(0, frameTagIdx);
            const std::string fileExt = filePath.substr(frameTagIdx + FrameTag.size());
            filePath = GetFilePathForTexPatAnim(filePre, fileExt, curFe);
            if (filePath.empty())
            {
                filePath = filePre + RGetNumberString(curFe) + fileExt;
            }
            frameTagIdx = filePath.find(FrameTag);
        }
    }
    return filePath;
}

//-----------------------------------------------------------------------------
//! @brief レイヤテクスチャに接続されたテクスチャノード群
//!        （file または envCube ノード）を取得します。
//!
//! @param[out] texObjs テクスチャノード配列を格納します。
//! @param[in] leyeredObj レイヤテクスチャオブジェクトです。
//! @param[in] getsEnvCube envCube ノードも取得するなら true です。
//!
//! @return 取得したテクスチャノード群の数を返します。
//-----------------------------------------------------------------------------
int GetLayeredTextureInputs(
    MObjectArray& texObjs,
    const MObject& leyeredObj,
    const bool getsEnvCube
)
{
    texObjs.clear();
    // 各テクスチャはレイヤテクスチャの inputs[n].color プラグに接続されています。
    MPlug inputsPlug = MFnDependencyNode(leyeredObj).findPlug("inputs");
    const int layerCount = inputsPlug.numElements();
    for (int iLayer = 0; iLayer < layerCount; ++iLayer)
    {
        MPlug colorPlug = inputsPlug.elementByPhysicalIndex(iLayer).child(0);
        MPlugArray plugArray;
        colorPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() != 0)
        {
            MObject texObj = plugArray[0].node();
            const MFn::Type type = texObj.apiType();
            if (type == MFn::kFileTexture ||
                (getsEnvCube && type == MFn::kEnvCube))
            {
                texObjs.append(texObj);
            }
        }
    }
    return texObjs.length();
}

//-----------------------------------------------------------------------------
//! @brief レイヤテクスチャに接続された最初の指定タイプノードを取得します。
//!
//! @param[in] leyeredObj layeredTexture ノードです。
//! @param[in] inputType 取得するノードのタイプです。
//!
//! @return 指定タイプノードを返します。見つからなければ kNullObj を返します。
//-----------------------------------------------------------------------------
static MObject FindLayeredTextureInput(
    const MObject& layeredObj,
    const MFn::Type inputType
)
{
    // 各テクスチャはレイヤテクスチャの inputs[n].color プラグに接続されています。
    MPlug inputsPlug = MFnDependencyNode(layeredObj).findPlug("inputs");
    const int layerCount = inputsPlug.numElements();
    for (int iLayer = 0; iLayer < layerCount; ++iLayer)
    {
        MStatus status;
        MPlug inPlug = inputsPlug.elementByPhysicalIndex(iLayer, &status);
        if (!status)
        {
            break;
        }
        MPlug colorPlug = inPlug.child(0);
        MPlugArray plugArray;
        colorPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() != 0)
        {
            MObject texObj = plugArray[0].node();
            const MFn::Type type = texObj.apiType();
            if (type == inputType)
            {
                return texObj;
            }
        }
    }
    return MObject::kNullObj;
}

//-----------------------------------------------------------------------------
//! @brief レイヤテクスチャに接続された最初の file ノードを取得します。
//!
//! @param[in] leyeredObj layeredTexture ノードです。
//!
//! @return file ノードを返します。見つからなければ kNullObj を返します。
//-----------------------------------------------------------------------------
MObject GetFirstLayeredFileTexture(const MObject& layeredObj)
{
    return FindLayeredTextureInput(layeredObj, MFn::kFileTexture);
}

//-----------------------------------------------------------------------------
//! @brief 環境ボールに接続された file ノードを取得します。
//!
//! @param[in] envBallObj envBall ノードです。
//!
//! @return file ノードを返します。
//!         file ノードが接続されていなければ MObject::kNullObj を返します。
//-----------------------------------------------------------------------------
MObject GetEnvBallFileTexture(const MObject& envBallObj)
{
    MPlug plug = MFnDependencyNode(envBallObj).findPlug("image");
    MPlugArray plugArray;
    plug.connectedTo(plugArray, true, false);
    if (plugArray.length() != 0)
    {
        MObject obj = plugArray[0].node();
        if (obj.apiType() == MFn::kFileTexture)
        {
            return obj;
        }
    }
    return MObject::kNullObj;
}

//-----------------------------------------------------------------------------
//! @brief 環境立方体のフェースのアトリビュート名を返します。
//!
//! @param[in] faceIdx フェースのインデックスです。
//!
//! @return アトリビュート名を返します。
//-----------------------------------------------------------------------------
std::string GetEnvCubeFaceAttrName(const int faceIdx)
{
    static const char* const envCubeAttrNames[RImage::CUBE_FACE_COUNT] =
    {
        "right", "left", "top", "bottom", "front", "back"
    };
    assert(0 <= faceIdx && faceIdx < RImage::CUBE_FACE_COUNT);
    return std::string(envCubeAttrNames[faceIdx]);
}

//-----------------------------------------------------------------------------
//! @brief 環境立方体に接続された file ノードを取得します。
//!
//! @param[in] envCubeObj envCube ノードです。
//! @param[in] faceIdx フェースのインデックスです。
//!
//! @return file ノードを返します。
//!         指定したフェースに file ノードが接続されていなければ MObject::kNullObj を返します。
//-----------------------------------------------------------------------------
MObject GetEnvCubeFileTexture(const MObject& envCubeObj, const int faceIdx)
{
    if (0 <= faceIdx && faceIdx < RImage::CUBE_FACE_COUNT)
    {
        MPlug plug = MFnDependencyNode(envCubeObj).findPlug(GetEnvCubeFaceAttrName(faceIdx).c_str());
        MPlugArray plugArray;
        plug.connectedTo(plugArray, true, false);
        if (plugArray.length() != 0)
        {
            MObject obj = plugArray[0].node();
            if (obj.apiType() == MFn::kFileTexture)
            {
                return obj;
            }
        }
    }
    return MObject::kNullObj;
}

//-----------------------------------------------------------------------------
//! @brief 環境立方体の +X 面にキューブマップの DDS ファイルが接続されていれば
//!        true を返します。
//-----------------------------------------------------------------------------
bool RIsSingleEnvCube(const MObject& envCubeObj, const std::string& projectPath)
{
    const MObject fileObj = GetEnvCubeFileTexture(envCubeObj, RImage::CUBE_FACE_PX);
    if (!fileObj.isNull())
    {
        const std::string filePath = GetFileTextureName(fileObj, projectPath);
        if (RIsCubeMapImage(filePath))
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 環境立方体が有効なら true を返します。
//!        すべてのフェースに file ノードが接続されているか、
//!        +X 面にキューブマップの DDS ファイルが接続されていれば有効と判定します。
//!
//! @param[out] iInvalidFace 無効な場合に file ノードが接続されていないフェース
//!                          のインデックスを格納します。
//! @param[in] envCubeObj envCube ノードです。
//! @param[in] projectPath Maya のプロジェクトフォルダのパスです。
//!
//! @return 環境立方体が有効なら true を返します。
//-----------------------------------------------------------------------------
static bool IsValidEnvCubeNode(
    int& iInvalidFace,
    const MObject& envCubeObj,
    const std::string& projectPath
)
{
    iInvalidFace = 0;

    if (RIsSingleEnvCube(envCubeObj, projectPath))
    {
        return true;
    }

    for (int iFace = 0; iFace < RImage::CUBE_FACE_COUNT; ++iFace)
    {
        if (GetEnvCubeFileTexture(envCubeObj, iFace).isNull())
        {
            iInvalidFace = iFace;
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャオブジェクトのメインの file ノードを取得します。
//!
//! @param[in] texObj テクスチャオブジェクト（file、layeredTexture、envBall、envCube）です。
//!
//! @return file ノードを返します。
//!         texObj が file ノードでなく、
//!         texObj に file ノードが接続されていなければ MObject::kNullObj を返します。
//-----------------------------------------------------------------------------
MObject GetMainFileTexture(const MObject& texObj)
{
    const MFn::Type type = texObj.apiType();
    if (type == MFn::kFileTexture)
    {
        return texObj;
    }
    else if (type == MFn::kLayeredTexture)
    {
        return GetFirstLayeredFileTexture(texObj); // 最上位レイヤー
    }
    else if (type == MFn::kEnvBall)
    {
        return GetEnvBallFileTexture(texObj);
    }
    else if (type == MFn::kEnvCube)
    {
        MObject fileObj = GetEnvCubeFileTexture(texObj, RImage::CUBE_FACE_PZ); // 前面
        if (fileObj.isNull())
        {
            fileObj = GetEnvCubeFileTexture(texObj, RImage::CUBE_FACE_PX); // 右面
        }
        return fileObj;
    }
    else
    {
        return MObject::kNullObj;
    }
}

//-----------------------------------------------------------------------------
//! @brief プラグに接続された有効なテクスチャオブジェクトを取得します。
//!        file ノードまたは、
//!        file ノードが接続された layeredTexture ノードのみ有効とみなします。
//-----------------------------------------------------------------------------
MObject FindTextureNode(
    RScene& rscene,
    const MPlug& dstPlug,
    const bool checkElementFlag
)
{
    MObject texObj = MObject::kNullObj;

    MPlugArray plugArray;
    dstPlug.connectedTo(plugArray, true, false);
    if (plugArray.length() > 0)
    {
        const MObject srcObj = plugArray[0].node();
        const MFn::Type type = srcObj.apiType();
        const MString srcObjName = MFnDependencyNode(srcObj).name();

        if (type == MFn::kBulge          ||
            type == MFn::kChecker        ||
            type == MFn::kCloth          ||
            type == MFn::kFluidTexture2D ||
            type == MFn::kFractal        ||
            type == MFn::kGrid           ||
            type == MFn::kMountain       ||
            type == MFn::kNoise          ||
            type == MFn::kOcean          ||
            type == MFn::kRamp           ||
            type == MFn::kWater)
        {
            if (!checkElementFlag)
            {
                YShowWarning(&rscene, // Procedural texture is ignored: %s
                    "プロシージャテクスチャー（チェッカ、ランプなど）はそのままでは出力できません（無視されます）: {0} \n"
                    "「ファイルテクスチャーに変換」でファイルに変換してから出力してください。",
                    "Procedural textures such as checker and ramp cannot be exported as is, so they are ignored: {0} \n"
                    "First, convert to a file by using Convert to File Texture, and then export.",
                    srcObjName.asChar());
            }
        }
        else if (type == MFn::kPsdFileTexture)
        {
            if (!checkElementFlag)
            {
                YShowWarning(&rscene, // PSD file texture is ignored: %s
                    "PSD ファイルテクスチャー（psdFileTex ノード）は出力されません（無視されます）: {0}",
                    "PSD file textures (psdFileTex nodes) are not exported, so they are ignored: {0}",
                    srcObjName.asChar());
            }
        }
        else if (type == MFn::kProjection)
        {
            if (!checkElementFlag)
            {
                YShowWarning(&rscene, // Projection mapping is not supported: %s
                    "投影ユーティリティによる投影マッピングが使用されています。テクスチャーは出力されません: {0}",
                    "Projection mapping with a projection utility is used. The texture is not exported: {0}",
                    srcObjName.asChar());
            }
        }
        else if (type == MFn::kStencil)
        {
            if (!checkElementFlag)
            {
                YShowWarning(&rscene, // Stencil mapping is not supported: %s
                    "ステンシルマッピングが使用されています。テクスチャーは出力されません: {0}",
                    "Stencil mapping is used. The texture is not exported: {0}",
                    srcObjName.asChar());
            }
        }
        else if (type == MFn::kLayeredTexture)
        {
            //-----------------------------------------------------------------------------
            // レイヤテクスチャの場合、1 つ以上の file ノードが接続されていれば有効とみなします。
            if (GetFirstLayeredFileTexture(srcObj).isNull())
            {
                if (!checkElementFlag)
                {
                    YShowWarning(&rscene, // File texture is not connected: %s
                        "layeredTexture ノードに file ノードが接続されていません: {0} \n"
                        "テクスチャーが貼られていないとして出力します。",
                        "A file node is not connected to a layeredTexture node: {0} \n"
                        "Exports as if no texture has been bound.",
                        srcObjName.asChar());
                }
            }
            else
            {
                texObj =  srcObj;
            }
        }
        else if (type == MFn::kFileTexture)
        {
            //-----------------------------------------------------------------------------
            // file ノードならそのまま返します。
            texObj = srcObj;
        }
    }

    return texObj;
}

//-----------------------------------------------------------------------------
//! @brief 法線マップのテクスチャオブジェクトを取得します。
//!        file ノードまたは、
//!        file ノードが接続された layeredTexture ノードのみ有効とみなします。
//!
//! @param[in] shaderObj シェーダオブジェクトです。
//!
//! @return テクスチャオブジェクトを返します。
//!         有効なテクスチャが接続されていなければ MObject::kNullObj を返します。
//-----------------------------------------------------------------------------
MObject FindNormalMapTextureNode(
    RScene& rscene,
    const MObject& shaderObj,
    const bool checkElementFlag
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // get normalCamera input
    MFnDependencyNode shaderFn(shaderObj);
    MPlug normalPlug = FindPlugQuiet(shaderFn, "normalCamera", &status);
    if (!status)
    {
        return MObject::kNullObj;
    }
    MPlugArray plugArray;
    normalPlug.connectedTo(plugArray, true, false);
    if (plugArray.length() == 0)
    {
        return MObject::kNullObj;
    }

    MObject texObj = plugArray[0].node();
    if (texObj.apiType() == MFn::kBump)
    {
        // bump2d ノードが接続されていれば bumpValue プラグの入力を取得します。
        return FindTextureNode(rscene, MFnDependencyNode(texObj).findPlug("bumpValue"), checkElementFlag);
    }
    else
    {
        // bump2d ノードが接続されてなければ normalCamera プラグの入力を取得します。
        return FindTextureNode(rscene, normalPlug, checkElementFlag);
    }
}

//-----------------------------------------------------------------------------
//! @brief 環境マップのテクスチャオブジェクトを取得します。
//!        file、layeredTexture、envBall、envCube ノードのみ有効とみなします。
//!        layeredTexture ノードには file または envCube ノードが接続されている必要があります。
//-----------------------------------------------------------------------------
MObject FindEnvMapTextureNode(
    RScene& rscene,
    const MPlug& dstPlug,
    const std::string& projectPath,
    const bool checkElementFlag
)
{
    MObject texObj = MObject::kNullObj;
    MPlugArray plugArray;
    dstPlug.connectedTo(plugArray, true, false);
    if (plugArray.length() > 0)
    {
        const MObject srcObj = plugArray[0].node();
        const MFn::Type type = srcObj.apiType();
        const MString srcObjName = MFnDependencyNode(srcObj).name();

        if (type == MFn::kFileTexture)
        {
            texObj = srcObj;
        }
        else if (type == MFn::kLayeredTexture)
        {
            //-----------------------------------------------------------------------------
            // レイヤテクスチャの場合、1 つ以上の file または envCube ノードが接続されていれば有効とみなします。
            MObjectArray inputObjs;
            const int inputCount = GetLayeredTextureInputs(inputObjs, srcObj, true);
            if (inputCount == 0)
            {
                if (!checkElementFlag)
                {
                    YShowWarning(&rscene, // File or EnvCube texture is not connected: %s
                        "反射カラーアトリビュートに接続された layeredTexture ノードに file ノードまたは envCube ノードが接続されていません: {0} \n"
                        "テクスチャーが貼られていないとして出力します。",
                        "A file node or envCube node is not connected to the layeredTexture node connected to the Reflected Color attribute: {0} \n"
                        "Exports as if no texture has been bound.",
                        srcObjName.asChar());
                }
            }
            else
            {
                texObj = srcObj;
                for (int iInput = 0; iInput < inputCount; ++iInput)
                {
                    const MObject& inputObj = inputObjs[iInput];
                    int iInvalidFace;
                    if (inputObj.apiType() == MFn::kEnvCube &&
                        !IsValidEnvCubeNode(iInvalidFace, inputObj, projectPath))
                    {
                        if (!checkElementFlag)
                        {
                            YShowWarning(&rscene, // File texture is not connected (%s): %s
                                "envCube ノードに file ノードが接続されていません: {0}: {1} \n"
                                "テクスチャーが貼られていないとして出力します。",
                                "A file node is not connected to a envCube node: {0}: {1} \n"
                                "Exports as if no texture has been bound.",
                                GetEnvCubeFaceAttrName(iInvalidFace),
                                MFnDependencyNode(inputObj).name().asChar());
                        }
                        texObj = MObject::kNullObj;
                        break;
                    }
                }
            }
        }
        else if (type == MFn::kEnvBall)
        {
            //-----------------------------------------------------------------------------
            // 環境ボールの場合、image アトリビュートに file ノードが接続されていれば有効とみなします。
            if (GetEnvBallFileTexture(srcObj).isNull())
            {
                if (!checkElementFlag)
                {
                    YShowWarning(&rscene, // File texture is not connected: %s
                        "envBall ノードに file ノードが接続されていません: {0} \n"
                        "テクスチャーが貼られていないとして出力します。",
                        "A file node is not connected to a envBall node: {0} \n"
                        "Exports as if no texture has been bound.",
                        srcObjName.asChar());
                }
            }
            else
            {
                texObj = srcObj;
            }
        }
        else if (type == MFn::kEnvCube)
        {
            //-----------------------------------------------------------------------------
            // 環境立方体の場合、6 面すべてに file ノードが接続されているか、
            // +X 面にキューブマップの DDS ファイルが接続されていれば有効とみなします。
            int iInvalidFace;
            if (!IsValidEnvCubeNode(iInvalidFace, srcObj, projectPath))
            {
                if (!checkElementFlag)
                {
                    YShowWarning(&rscene, // File texture is not connected (%s): %s
                        "envCube ノードに file ノードが接続されていません: {0}: {1} \n"
                        "テクスチャーが貼られていないとして出力します。",
                        "A file node is not connected to a envCube node: {0}: {1} \n"
                        "Exports as if no texture has been bound.",
                        GetEnvCubeFaceAttrName(iInvalidFace), srcObjName.asChar());
                }
            }
            else
            {
                texObj = srcObj;
            }
        }
    }
    return texObj;
}

//-----------------------------------------------------------------------------
//! @brief シェイプに適用されたシェーディンググループ群と
//!        それに対応するコンポーネント群を取得します。
//!        Maya API で取得したシェーディンググループ群に重複があれば、
//!        合成したコンポーネントオブジェクトを作成して返します。
//-----------------------------------------------------------------------------
bool GetShadingGroupsAndComponents(
    CObjectArray& outSgs,
    CObjectArray& outComps,
    const MDagPath& shapePath
)
{
    //-----------------------------------------------------------------------------
    // clear
    outSgs.clear();
    outComps.clear();

    //-----------------------------------------------------------------------------
    // シェーディンググループを取得します。

    // シェーディンググループグループを取得するインスタンス番号を取得します。
    const int instanceNum = (shapePath.isInstanced()) ?
        shapePath.instanceNumber() : 0;
    const MFnMesh meshFn(shapePath);
    // 指定したメッシュインスタンスに適用されているシェーディンググループオブジェクト
    // とコンポーネント情報を取得します。
    MObjectArray sgs;
    MObjectArray comps;
    meshFn.getConnectedSetsAndMembers(instanceNum, sgs, comps, true); // 第４引数は renderableSetsOnly
    const int sgCount = sgs.length();
    // MeshPolygonComponent 型のコンポーネント情報の数を取得します。
    int compCount = 0;
    for (int isg = 0; isg < sgCount; ++isg)
    {
        if (comps[isg].apiType() == MFn::kMeshPolygonComponent)
        {
            ++compCount;
        }
    }

    // シェーディンググループとコンポーネントを保存します。
    for (int isg = 0; isg < sgCount; ++isg)
    {
        MObject sgObj = sgs[isg];
        MObject comp = comps[isg];
        // コンポーネントがないか、ポリゴンが含まれていなければその
        // シェーディンググループを登録しない
        if (compCount > 0 && comp.isNull())
        {
            continue; // no polygon in component list
        }
        outSgs.push_back(sgObj);
        outComps.push_back(comp);
    }

    //-----------------------------------------------------------------------------
    // adjust
    if (outSgs.size() >= 2)
    {
        CObjectArray adjSgs;
        CObjectArray adjComps;
        if (RIsConstantArray(outSgs))
        {
            //cerr << "all same SG: " << shapePath.partialPathName() << endl;
            adjSgs.push_back(outSgs[0]);
            adjComps.push_back(MObject::kNullObj);
        }
        else
        {
            for (int isg = 0; isg < static_cast<int>(outSgs.size()); ++isg)
            {
                const MObject sgObj = outSgs[isg];
                const MObject comp  = outComps[isg];
                const int idst = RFindValueInArray(adjSgs, sgObj);
                if (idst == -1)
                {
                    adjSgs.push_back(sgObj);
                    adjComps.push_back(comp);
                }
                else
                {
                    MObject adjComp = adjComps[idst];
                    if (!comp.isNull() && !adjComp.isNull())
                    {
                        MIntArray elements;
                        MFnSingleIndexedComponent(comp).getElements(elements);
                        MFnSingleIndexedComponent(adjComp).addElements(elements);
                    }
                }
            }
            if (adjSgs.size() != outSgs.size())
            {
                //cerr << "partial same SG: " << shapePath.partialPathName() << endl;
            }
            else
            {
                adjSgs.clear();
            }
        }

        if (adjSgs.size() != 0)
        {
            outSgs   = adjSgs;
            outComps = adjComps;
            return true;
        }
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief オブジェクトからプラグを取得します。
//!        MFnDependencyNode::findPlug と違って、
//!        プラグが存在しなかった場合にエラーログにエラーを出力しません。
//-----------------------------------------------------------------------------
MPlug FindPlugQuiet(
    const MFnDependencyNode& depFn,
    const MString& name,
    MStatus* pStatus
)
{
    // オブジェクトのアトリビュートを走査し指定のアトリビュートを探します。
    int attrCount = depFn.attributeCount();
    for (int iattr = 0; iattr < attrCount; ++iattr)
    {
        // アトリビュートオブジェクトを取得します。
        MObject attrObj = depFn.attribute(iattr);
        MFnAttribute attrFn(attrObj);
        // 名前を比較し、該当のアトリビュートかどうかを判断します。
        if (attrFn.name() == name)
        {
            // アトリビュートが見つかったら MS::kSuccess を結果に設定します。
            if (pStatus != NULL)
            {
                *pStatus = MS::kSuccess;
            }
            // プラグを返します。
            return MPlug(depFn.object(), attrObj);
        }
    }
    // アトリビュートが見つからなければ結果に MS::kFailure を設定し、
    // 空のプラグを返します。
    if (pStatus != NULL)
    {
        *pStatus = MS::kFailure;
    }
    return MPlug(); // empty plug
}

//-----------------------------------------------------------------------------
//! @brief 指定したアトリビュート名の子プラグのインデックスを取得します。
//-----------------------------------------------------------------------------
int FindChildPlugIndexByName(const MPlug& plug, const MString& name)
{
    const int childCount = static_cast<int>(plug.numChildren());
    for (int childIdx = 0; childIdx < childCount; ++childIdx)
    {
        const MPlug childPlug = plug.child(childIdx);
        if (MFnAttribute(childPlug.attribute()).name() == name)
        {
            return childIdx;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief 指定したアトリビュート名の子プラグを取得します。
//-----------------------------------------------------------------------------
MPlug FindChildPlugByName(const MPlug& plug, const MString& name)
{
    const int childCount = static_cast<int>(plug.numChildren());
    for (int childIdx = 0; childIdx < childCount; ++childIdx)
    {
        const MPlug childPlug = plug.child(childIdx);
        if (MFnAttribute(childPlug.attribute()).name() == name)
        {
            return childPlug;
        }
    }
    return MPlug();
}

//-----------------------------------------------------------------------------
//! @brief ノードのローカル変換行列が単位行列で、
//!        スケール・回転・移動に何も接続されていなければ true を返します。
//-----------------------------------------------------------------------------
bool IsTransformIdentity(const MDagPath& dagPath)
{
    const int attrCount = 9;
    static const char* const attrStrs[attrCount] =
    {
        "sx", "sy", "sz",
        "rx", "ry", "rz",
        "tx", "ty", "tz",
    };

    const MFnTransform xformFn(dagPath);
    if (xformFn.transformationMatrix() != MMatrix::identity)
    {
        return false;
    }

    for (int iattr = 0; iattr < attrCount; ++iattr)
    {
        MPlug plug = xformFn.findPlug(attrStrs[iattr]);
        MPlugArray plugArray;
        plug.connectedTo(plugArray, true, false);
        if (plugArray.length() > 0)
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 指定したノードの親ノードのローカル変換行列が単位行列で、
//!        親ノードのスケール・回転・移動に何も接続されていなければ true を返します。
//!        親ノードがない場合は常に true を返します。
//-----------------------------------------------------------------------------
bool IsParentTransformIdentity(MDagPath dagPath)
{
    while (dagPath.length() >= 2)
    {
        dagPath.pop();
        if (!IsTransformIdentity(dagPath))
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 親ノードの変換を継承するなら true を返します。
//-----------------------------------------------------------------------------
bool IsInheritsTransform(const MDagPath& dagPath)
{
    bool inheritFlag;
    MFnDagNode(dagPath).findPlug("inheritsTransform").getValue(inheritFlag);
    return inheritFlag;
}

//-----------------------------------------------------------------------------
//! @brief 取得した回転を出力のために調整する必要があるなら true を返します。
//-----------------------------------------------------------------------------
bool IsRotateNeededToAdjust(
    const MDagPath& dagPath,
    const MTransformationMatrix::RotationOrder targetRotationOrder
)
{
    const MFnTransform xformFn(dagPath);
    MEulerRotation jo = MEulerRotation::identity;
    if (dagPath.node().apiType() == MFn::kJoint)
    {
        MFnIkJoint(dagPath).getOrientation(jo);
    }

    return (
        xformFn.rotationOrder() != targetRotationOrder ||
        xformFn.rotateOrientation(MSpace::kTransform) != MQuaternion::identity ||
        jo != MEulerRotation::identity);
}

//-----------------------------------------------------------------------------
//! @brief 取得した移動を出力のために調整する必要があるなら true を返します。
//-----------------------------------------------------------------------------
bool IsTranslateNeededToAdjust(const MDagPath& dagPath)
{
    const MFnTransform xformFn(dagPath);
    const MVector sp  = xformFn.scalePivot(MSpace::kTransform);
    const MVector rp  = xformFn.rotatePivot(MSpace::kTransform);
    const MVector spt = xformFn.scalePivotTranslation(MSpace::kTransform);
    const MVector rpt = xformFn.rotatePivotTranslation(MSpace::kTransform);
    const MVector pivotMiddleOfs = (sp + spt - rp);
    if (!pivotMiddleOfs.isEquivalent(MVector::zero))
    {
        return true;
    }
    MVector translateOfs = rpt + rp;
    if (dagPath.length() >= 2)
    {
        MDagPath parentPath = dagPath;
        parentPath.pop();
        translateOfs -= MFnTransform(parentPath).scalePivot(MSpace::kTransform);
    }
    return !translateOfs.isEquivalent(MVector::zero);
}

//-----------------------------------------------------------------------------
//! @brief transform ノードからスケール・回転・移動を取得します。
//-----------------------------------------------------------------------------
void GetTransformFromNode(
    RVec3& scale,
    RVec3& rotate,
    RVec3& translate,
    const MDagPath& dagPath,
    const float magnify,
    const bool useWorldTransform,
    const bool snapToZero
)
{
    if (useWorldTransform)
    {
        GetRMtx44(dagPath.inclusiveMatrix()).GetTransform(
            scale, rotate, translate);
    }
    else
    {
        const MFnTransform xformFn(dagPath);

        double scaleValues[3];
        xformFn.getScale(scaleValues);
        scale = RVec3(static_cast<float>(scaleValues[0]), static_cast<float>(scaleValues[1]), static_cast<float>(scaleValues[2]));

        MEulerRotation eulerRotate;
        xformFn.getRotation(eulerRotate);
        rotate = RVec3(static_cast<float>(eulerRotate.x), static_cast<float>(eulerRotate.y), static_cast<float>(eulerRotate.z)) *
            static_cast<float>(R_M_RAD_TO_DEG);

        translate = GetRVec3(xformFn.getTranslation(MSpace::kTransform));
    }
    translate *= magnify;

    if (snapToZero)
    {
        scale.SnapToZero();
        rotate.SnapToZero();
        translate.SnapToZero();
    }
}

//-----------------------------------------------------------------------------
//! @brief ワールド回転行列を取得します。
//!        [Ro R Jo] [Ro R Jo] ...
//-----------------------------------------------------------------------------
MMatrix GetWorldRotateMtx(MDagPath dagPath)
{
    MMatrix mtx;

    while (dagPath.node().hasFn(MFn::kTransform))
    {
        //cerr << "gwrm: " << dagPath.fullPathName() << R_ENDL;
        MFnTransform xformFn(dagPath);

        MQuaternion ro = xformFn.rotateOrientation(MSpace::kTransform);
        mtx *= ro.asMatrix();

        MEulerRotation rot;
        xformFn.getRotation(rot);
        mtx *= rot.asMatrix();

        if (dagPath.node().apiType() == MFn::kJoint)
        {
            MQuaternion jo;
            MFnIkJoint(dagPath).getOrientation(jo);
            mtx *= jo.asMatrix();
        }

        if (!IsInheritsTransform(dagPath))
        {
            break;
        }

        if (!dagPath.pop())
        {
            break;
        }
    }

    return mtx;
}

//-----------------------------------------------------------------------------
//! @brief 親ノードのワールド回転行列を取得します。
//!        親ノードがない場合は常に単位行列を返します。
//-----------------------------------------------------------------------------
MMatrix GetParentWorldRotateMtx(MDagPath dagPath)
{
    MMatrix mtx;

    if (dagPath.length() > 1)   // |pCube1|pCube2 = 2
    {
        if (IsInheritsTransform(dagPath))
        {
            dagPath.pop();
            mtx *= GetWorldRotateMtx(dagPath);
        }
    }

    return mtx;
}

//-----------------------------------------------------------------------------
//! @brief ファイルパスをフルパスに変換します。
//-----------------------------------------------------------------------------
std::string YGetFullFilePath(const std::string& path, const std::string& rootFolder)
{
    //-----------------------------------------------------------------------------
    // 環境変数を展開します。
    std::string dst = path;
    if (dst.find('%') != std::string::npos ||
        dst.find('$') != std::string::npos)
    {
        dst = MString(dst.c_str()).expandEnvironmentVariablesAndTilde().asChar();
        //cerr << "env: " << path << " -> " << dst << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // 相対パスならルートフォルダのパスを付加します。
    if (!RIsFullFilePath(dst) && !rootFolder.empty())
    {
        dst = rootFolder + "/" + dst;
    }

    //-----------------------------------------------------------------------------
    // ".." などを解消したフルパスを返します。
    return RGetFullFilePath(dst, true);
}

//-----------------------------------------------------------------------------
//! @brief プラグからファイルパスを取得してフルパスで返します。
//-----------------------------------------------------------------------------
MString GetFilePathAttribute(const MPlug& plug, const std::string& rootFolder)
{
    MString path;
    plug.getValue(path);
    path = RTrimString(path.asChar()).c_str();
    if (path.length() != 0)
    {
        path = YGetFullFilePath(path.asChar(), rootFolder).c_str();
    }
    return path;
}

//-----------------------------------------------------------------------------
//! @brief プラグインの UI を日本語で表示するなら true を返します。
//-----------------------------------------------------------------------------
bool IsJapaneseUi()
{
    if (RGetEnvVar("NINTENDO_MAYA_FORCE_JAPANESE_UI") == "1")
    {
        return true;
    }

    MString lang;
    MGlobal::executeCommand("about -uil", lang);
    return (lang == "ja_JP");
}

//-----------------------------------------------------------------------------
//! @brief 特定のウィンドウをアクティブにするためのウィンドウ列挙コールバックです。
//!
//! @param[in] hWnd ウィンドウハンドルです。
//! @param[in] lparam アクティブにするウィンドウのタイトルです。
//!
//! @return 列挙を継続するなら TRUE を返します。
//-----------------------------------------------------------------------------
static BOOL CALLBACK ProcSetWindowForeground(HWND hWnd, LPARAM lparam)
{
    if (!IsWindowEnabled(hWnd))
    {
        return TRUE;
    }

    // ウィンドウのタイトルを取得します。
    const int TEXT_SIZE = 256;
    char text[TEXT_SIZE] = { 0 };
    GetWindowTextA(hWnd, text, TEXT_SIZE);
    //if (text[0]) cerr << "text: [" << text << "] " << strlen(text) << R_ENDL;

    //const int CLASS_NAME_SIZE = 256;
    //char className[CLASS_NAME_SIZE];
    //GetClassName(hWnd, className, CLASS_NAME_SIZE);
    //cerr << "class: [" << className << "] " << strlen(className) << R_ENDL;

    // ウィンドウのタイトルが指定されたウィンドウ名と一致するか確認します。
    const char* name = reinterpret_cast<const char*>(lparam);
    if (strcmp(text, name) == 0) // changed (2007/03/02)
    {
        // 一致したウィンドウを表示させ、アクティブ化します。
        ShowWindow(hWnd, SW_SHOWNORMAL);    // min & max -> normal size
        SetForegroundWindow(hWnd);
        //cerr << "- find -" << R_ENDL;
        // 該当のウィンドウを処理したら、列挙を中止させるため FALSE を返します。
        return FALSE;
    }

    // 次のウィンドウを処理します。
    return TRUE;
}

//-----------------------------------------------------------------------------
//! @brief Maya の Output Window をアクティブにして前面に表示します。
//-----------------------------------------------------------------------------
void SetMayaOutputWindowForeground()
{
    // Maya の言語設定によって有効にする Output Window のウィンドウ名を決定します。
    #if (MAYA_API_VERSION >= 201100)
    const char* winName = "Output Window";
    #else
    const char* winName = (!IsJapaneseUi()) ? "Output Window" : "出力ウィンドウ";
    #endif
    //cerr << "output: " << winName << R_ENDL;
    // ウィンドウを列挙し、該当の名前を持つウィンドウを前面に表示させます。
    EnumWindows(ProcSetWindowForeground, reinterpret_cast<LPARAM>(winName));
}

//-----------------------------------------------------------------------------
//! @brief MIntArray の値の範囲を取得します。
//-----------------------------------------------------------------------------
void GetArrayRange(const MIntArray& array, int& valueMin, int& valueMax)
{
    int valueCount = array.length();
    if (valueCount == 0)
    {
        valueMin = 0;
        valueMax = 0;
        return;
    }

    valueMin = LONG_MAX;
    valueMax = LONG_MIN;
    for (int ival = 0; ival < valueCount; ++ival)
    {
        int val = array[ival];
        if (val < valueMin)
        {
            valueMin = val;
        }
        if (val > valueMax)
        {
            valueMax = val;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief MFloatArray の値の範囲を取得します。
//-----------------------------------------------------------------------------
void GetArrayRange(const MFloatArray& array, float& valueMin, float& valueMax)
{
    int valueCount = array.length();
    if (valueCount == 0)
    {
        valueMin = 0;
        valueMax = 0;
        return;
    }

    valueMin = static_cast<float>(LONG_MAX);
    valueMax = static_cast<float>(LONG_MIN);
    for (int ival = 0; ival < valueCount; ++ival)
    {
        float val = array[ival];
        if (val < valueMin)
        {
            valueMin = val;
        }
        if (val > valueMax)
        {
            valueMax = val;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief MPointArray の値の範囲を取得します。
//-----------------------------------------------------------------------------
void GetArrayRange(const MPointArray& array, MPoint& valueMin, MPoint& valueMax)
{
    int pntCount = array.length();
    if (pntCount == 0)
    {
        valueMin.x = valueMin.y = valueMin.z = 0.0;
        valueMax.x = valueMax.y = valueMax.z = 0.0;
        return;
    }

    valueMin.x = valueMin.y = valueMin.z = static_cast<double>(LONG_MAX);
    valueMax.x = valueMax.y = valueMax.z = static_cast<double>(LONG_MIN);
    for (int ipnt = 0; ipnt < pntCount; ++ipnt)
    {
        const MPoint& pnt = array[ipnt];
        if (pnt.x < valueMin.x) valueMin.x = pnt.x;
        if (pnt.y < valueMin.y) valueMin.y = pnt.y;
        if (pnt.z < valueMin.z) valueMin.z = pnt.z;
        if (pnt.x > valueMax.x) valueMax.x = pnt.x;
        if (pnt.y > valueMax.y) valueMax.y = pnt.y;
        if (pnt.z > valueMax.z) valueMax.z = pnt.z;
    }
}

//-----------------------------------------------------------------------------
//! @brief MPointArray の原点からの距離の最大値を返します。
//-----------------------------------------------------------------------------
double GetPointArrayMaxLength(const MPointArray& array)
{
    double maxLen = 0.0;
    int pntCount = array.length();
    for (int ipnt = 0; ipnt < pntCount; ++ipnt)
    {
        double len = array[ipnt].distanceTo(MPoint::origin);
        if (len > maxLen)
        {
            maxLen = len;
        }
    }
    return maxLen;
}

//-----------------------------------------------------------------------------
//! @brief MPointArray をスケールします。
//-----------------------------------------------------------------------------
void ScaleArray(MPointArray& array, double scale)
{
    int pntCount = array.length();
    for (int ipnt = 0; ipnt < pntCount; ++ipnt)
    {
        array[ipnt] = array[ipnt] * scale;
    }
}

//-----------------------------------------------------------------------------
//! @brief MFloatArray を固定小数点数配列に変換します。
//-----------------------------------------------------------------------------
void GetFixedPointArray(
    MIntArray& dst,
    const MFloatArray& src,
    int mul,
    int valueMin,
    int valueMax
)
{
    int valueCount = src.length();
    for (int ival = 0; ival < valueCount; ++ival)
    {
        dst.append(RClampValue(valueMin, valueMax, RRound(src[ival] * mul)));
    }
}

//-----------------------------------------------------------------------------
//! @brief MIntArray を値の順にソートします。
//-----------------------------------------------------------------------------
void SortArray(MIntArray& array, const bool isDescending)
{
    const int size = array.length();
    if (!isDescending)
    {
        for (int i0 = 0; i0 < size - 1; ++i0)
        {
            for (int i1 = i0 + 1; i1 < size; ++i1)
            {
                if (array[i0] > array[i1])
                {
                    const int tmpVal = array[i0];
                    array[i0] = array[i1];
                    array[i1] = tmpVal;
                }
            }
        }
    }
    else
    {
        for (int i0 = 0; i0 < size - 1; ++i0)
        {
            for (int i1 = i0 + 1; i1 < size; ++i1)
            {
                if (array[i0] < array[i1])
                {
                    const int tmpVal = array[i0];
                    array[i0] = array[i1];
                    array[i1] = tmpVal;
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief MStringArray をアルファベット順にソートします。
//-----------------------------------------------------------------------------
void SortArray(MStringArray& array, const bool isDescending)
{
    const int size = array.length();
    for (int i0 = 0; i0 < size - 1; ++i0)
    {
        for (int i1 = i0 + 1; i1 < size; ++i1)
        {
            const int cmp = strcmp(array[i0].asChar(), array[i1].asChar());
            if ((!isDescending && cmp > 0) ||
                ( isDescending && cmp < 0))
            {
                const MString tmpStr = array[i0];
                array[i0] = array[i1];
                array[i1] = tmpStr;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief MFloatVectorArray に長さ 0 のベクトルが含まれるなら true を返します。
//-----------------------------------------------------------------------------
bool IsZeroVectorExist(const MFloatVectorArray& array)
{
    int size = array.length();
    for (int index = 0; index < size; ++index)
    {
        if (array[index] == MFloatVector::zero)
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 変換行列を作成します。
//-----------------------------------------------------------------------------
void CreateTransformMtx(
    MMatrix& m,
    const MVector& s,
    const MVector& r,
    const MVector& t
)
{
    double sinx = sin(r.x * R_M_DEG_TO_RAD);
    double siny = sin(r.y * R_M_DEG_TO_RAD);
    double sinz = sin(r.z * R_M_DEG_TO_RAD);
    double cosx = cos(r.x * R_M_DEG_TO_RAD);
    double cosy = cos(r.y * R_M_DEG_TO_RAD);
    double cosz = cos(r.z * R_M_DEG_TO_RAD);

    /* SRT */
    m[0][0] = (cosy * cosz) * s.x;
    m[0][1] = (cosy * sinz) * s.x;
    m[0][2] = (-siny) * s.x;

    m[1][0] = (sinx * siny * cosz - cosx * sinz) * s.y;
    m[1][1] = (sinx * siny * sinz + cosx * cosz) * s.y;
    m[1][2] = (sinx * cosy) * s.y;

    m[2][0] = (cosx * siny * cosz + sinx * sinz) * s.z;
    m[2][1] = (cosx * siny * sinz - sinx * cosz) * s.z;
    m[2][2] = (cosx * cosy) * s.z;

    m[3][0] = t.x;
    m[3][1] = t.y;
    m[3][2] = t.z;

    m[0][3] = m[1][3] = m[2][3] = 0.0;
    m[3][3] = 1.0;
}

//-----------------------------------------------------------------------------
//! @brief セグメントスケール補正を考慮した変換行列を作成します。
//!        親のスケールを打ち消す変換行列を作成します。
//-----------------------------------------------------------------------------
void CreateTransformMtxScaleCompensate(
    MMatrix& m,
    const MVector& s,
    const MVector& r,
    const MVector& t,
    const MVector& ps
)
{
    double sinx = sin(r.x * R_M_DEG_TO_RAD);
    double siny = sin(r.y * R_M_DEG_TO_RAD);
    double sinz = sin(r.z * R_M_DEG_TO_RAD);
    double cosx = cos(r.x * R_M_DEG_TO_RAD);
    double cosy = cos(r.y * R_M_DEG_TO_RAD);
    double cosz = cos(r.z * R_M_DEG_TO_RAD);

    double ix = (ps.x != 0.0) ? (1.0 / ps.x) : 1.0;
    double iy = (ps.y != 0.0) ? (1.0 / ps.y) : 1.0;
    double iz = (ps.z != 0.0) ? (1.0 / ps.z) : 1.0;

    /* SRT */
    m[0][0] = (cosy * cosz) * s.x * ix;
    m[0][1] = (cosy * sinz) * s.x * iy;
    m[0][2] = (-siny)       * s.x * iz;

    m[1][0] = (sinx * siny * cosz - cosx * sinz) * s.y * ix;
    m[1][1] = (sinx * siny * sinz + cosx * cosz) * s.y * iy;
    m[1][2] = (sinx * cosy)                      * s.y * iz;

    m[2][0] = (cosx * siny * cosz + sinx * sinz) * s.z * ix;
    m[2][1] = (cosx * siny * sinz - sinx * cosz) * s.z * iy;
    m[2][2] = (cosx * cosy)                      * s.z * iz;

    m[3][0] = t.x;
    m[3][1] = t.y;
    m[3][2] = t.z;

    m[0][3] = m[1][3] = m[2][3] = 0.0;
    m[3][3] = 1.0;
}

//-----------------------------------------------------------------------------
//! @brief 変換行列からスケール・回転・移動を取得します。
//!        変換行列は座標を左から掛けるタイプになっている必要があります。
//-----------------------------------------------------------------------------
void GetTransformFromMtx(
    MVector& scale, MVector& rotate, MVector& translate,
    const MMatrix& mtxLocal)
{
    RVec3 s, r, t;
    GetRMtx44(mtxLocal).GetTransform(s, r, t);
    scale     = GetMVector(s);
    rotate    = GetMVector(r);
    translate = GetMVector(t);
}

//-----------------------------------------------------------------------------
//! @brief 変換行列を正規化します。
//-----------------------------------------------------------------------------
void NormalizeMtx(MFloatMatrix& mf)
{
    for (int iaxis = 0; iaxis < 3; ++iaxis)
    {
        float f0 = mf[iaxis][0];
        float f1 = mf[iaxis][1];
        float f2 = mf[iaxis][2];
        float len = sqrtf(f0 * f0 + f1 * f1 + f2 * f2);
        if (len  > 0.0f)
        {
            float mag = 1.0f / len;
            mf[iaxis][0] *= mag;
            mf[iaxis][1] *= mag;
            mf[iaxis][2] *= mag;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 法線に行列を掛けた後に正規化します。
//-----------------------------------------------------------------------------
MFloatVector MulNormalAndMtxPostNormalize(
    const MFloatVector& v,
    const MMatrix& m
)
{
    MFloatVector dst;
    dst.x = v.x * static_cast<float>(m[0][0]) + v.y * static_cast<float>(m[1][0]) + v.z * static_cast<float>(m[2][0]);
    dst.y = v.x * static_cast<float>(m[0][1]) + v.y * static_cast<float>(m[1][1]) + v.z * static_cast<float>(m[2][1]);
    dst.z = v.x * static_cast<float>(m[0][2]) + v.y * static_cast<float>(m[1][2]) + v.z * static_cast<float>(m[2][2]);
    dst.normalize();
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 指定した頂点を含む頂点フェースの法線の平均を返します。
//-----------------------------------------------------------------------------
MVector GetAverageVertexNormal(const MDagPath& meshPath, const int ivtxNode)
{
    MItMeshVertex vIter(meshPath);
    int ivtxPrev;
    vIter.setIndex(ivtxNode, ivtxPrev);
    MVectorArray nrms;
    vIter.getNormals(nrms, MSpace::kObject);

    MVector vtxNrm(0.0, 0.0, 0.0);
    int nrmCount = nrms.length();
    for (int inrm = 0; inrm < nrmCount; ++inrm)
    {
        vtxNrm += nrms[inrm];
    }
    if (nrmCount >= 2)
    {
        vtxNrm *= 1.0 / nrmCount;
        vtxNrm.normalize();
    }

    //MFnMesh(meshPath).getVertexNormal(ivtxNode, vtxNrm, MSpace::kObject); // 法線を編集した場合に反映されない！

    return vtxNrm;
}

//-----------------------------------------------------------------------------
//! @brief 整数値の文字列から MIntArray を取得します。
//!        同じ値はまとめられます。
//-----------------------------------------------------------------------------
MStatus GetArrayFromStringCutSame(MIntArray& array, const char* str)
{
    const int MINUS = -10;
    const int NUM_SIZE = 15; // 数値の最大桁数

    //-----------------------------------------------------------------------------
    // extract numbers
    MIntArray numbers;
    const char* p = str;
    char c, buf[NUM_SIZE + 1];
    int ibuf = 0;
    bool minusOk = true;

    while ((c = *p) != '\0')
    {
        if (c == ' ' || c == ',' || c == '-')
        {
            if (ibuf != 0)
            {
                int num = atol(buf);
                numbers.append(num);
                ibuf = 0;
                minusOk = true;
            }
            if (c == '-')
            {
                if (!minusOk)
                {
                    return MS::kFailure;
                }
                numbers.append(MINUS);
            }
            if (c == ',')
            {
                minusOk = false;
            }
        }
        else if ('0' <= c && c <= '9')
        {
            if (ibuf >= NUM_SIZE)
            {
                return MS::kFailure;
            }
            buf[ibuf] = c;
            buf[++ibuf] = '\0';
        }
        else
        {
            return MS::kFailure;
        }
        ++p;
    }

    if (ibuf != 0)
    {
        int num = atol(buf);
        numbers.append(num);
        ibuf = 0;
    }

    //-----------------------------------------------------------------------------
    // append
    MIntArray result;
    int size = numbers.length();
    for (int i = 0; i < size; ++i)
    {
        int num = numbers[i];
        if (num == MINUS)
        {
            return MS::kFailure;
        }
        else if (i < size - 1 && numbers[i + 1] == MINUS)
        {
            if (i == size - 2)  // like "1,2,5-"
            {
                return MS::kFailure;
            }
            int next = numbers[i + 2];
            if (next == MINUS || next < num)
            {
                return MS::kFailure;
            }
            for (int j = num; j <= next; ++j)
            {
                if (YFindValueInArray(result, j) == -1)
                {
                    result.append(j);
                }
            }
            i += 2;
        }
        else
        {
            if (YFindValueInArray(result, num) == -1)
            {
                result.append(num);
            }
        }
    }

    //cerr << "gifscs: " << result << R_ENDL;

    array = result;

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief MImage でリードする画像ファイル群なら true を返します。
//!
//! @param[in] filePaths 画像ファイルのパス配列です。
//!
//! @return MImage でリードする画像ファイル群なら true を返します。
//-----------------------------------------------------------------------------
static bool UsesMImageForRead(const RStringArray& filePaths)
{
    for (size_t fileIdx = 0; fileIdx < filePaths.size(); ++fileIdx)
    {
        const std::string& filePath = filePaths[fileIdx];
        const std::string ext = RGetExtensionFromFilePath(filePath);
        if (ext == "png")
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 画像ファイルの情報を取得します。
//-----------------------------------------------------------------------------
MStatus GetImageInfoForMaya(
    int& imageW,
    int& imageH,
    RScene& rscene,
    const std::string& filePath,
    const bool checkElementFlag
)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // init
    imageW = imageH = 1;

    //-----------------------------------------------------------------------------
    // get info
    const std::string ext = RGetExtensionFromFilePath(filePath);
    if (ext == "tga" || ext == "png" || ext == "dds")
    {
        //-----------------------------------------------------------------------------
        // TGA、PNG、DDS
        RStatus rstatus = RGetImageInfo(&imageW, &imageH, filePath);
        if (!checkElementFlag)
        {
            return GetMStatusDisplayError(&rscene, rstatus);
        }
        else
        {
            return GetMStatus(rstatus);
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // other

        //-----------------------------------------------------------------------------
        // read file (use MImage)
        MImage mimg;
        status = mimg.readFromFile(filePath.c_str());
        if (!status)
        {
            if (!checkElementFlag)
            {
                YShowError(&rscene, "ファイルをリードできません: {0}", "Cannot read file: {0}", filePath);
            }
            return status;
        }

        //-----------------------------------------------------------------------------
        // get size
        unsigned int fullW, fullH;
        status = mimg.getSize(fullW, fullH);
        if (!status)
        {
            if (!checkElementFlag)
            {
                YShowError(&rscene, // Cannot get texture size: %s // 通常は発生しない
                    "テクスチャーファイルの内容が不正なため、画像の幅と高さを取得できません: {0}",
                    "Cannot get the height and width of an image due to invalid data in the texture file: {0}",
                    filePath);
            }
            return status;
        }
        imageW = static_cast<int>(fullW);
        imageH = static_cast<int>(fullH);
        return MS::kSuccess;
    }
}

//-----------------------------------------------------------------------------
//! @brief MImage からピクセルデータを取得します。
//!
//! @param[out] pDstPixels 出力ピクセルデータへのポインタです。
//! @param[in,out] pHasAlpha アルファ成分が存在すれば true を格納します。
//! @param[in] faceW フェースの幅です。
//! @param[in] faceH フェースの高さです。
//! @param[in] isFloat 出力ピクセルデータのデータ型が浮動小数点数なら true、整数なら false です。
//! @param[in] mimg MImage オブジェクトです。
//-----------------------------------------------------------------------------
static void GetMImagePixels(
    uint8_t* pDstPixels,
    bool* pHasAlpha,
    const int faceW,
    const int faceH,
    const bool isFloat,
    const MImage& mimg
)
{
    const bool srcIsFloat = (mimg.pixelType() == MImage::kFloat);
    const size_t srcValueBytes = (srcIsFloat) ? sizeof(float) : sizeof(uint8_t);
    const size_t srcPixelBytes = mimg.depth();
    const int srcChanCount = static_cast<int>(srcPixelBytes / srcValueBytes);
    if (srcChanCount == 2 || srcChanCount == 4)
    {
        *pHasAlpha = true;
    }

    uint8_t* pDstU8 = pDstPixels;
    float* pDstF32 = reinterpret_cast<float*>(pDstPixels);
    if (!srcIsFloat) // 入力が整数型
    {
        const uint8_t* pSrcU8Top = reinterpret_cast<const uint8_t*>(mimg.pixels());
        const size_t srcRowBytes = srcPixelBytes * faceW;
        for (int iy = 0; iy < faceH; ++iy)
        {
            const uint8_t* pSrcU8 = pSrcU8Top + srcRowBytes * (faceH - 1 - iy); // 垂直に反転
            for (int ix = 0; ix < faceW; ++ix)
            {
                uint8_t cr = 0;
                uint8_t cg = 0;
                uint8_t cb = 0;
                uint8_t ca = 0xff;
                if (srcChanCount >= 3)
                {
                    cr = *pSrcU8++;
                    cg = *pSrcU8++;
                    cb = *pSrcU8++;
                    if (srcChanCount == 4)
                    {
                        ca = *pSrcU8++;
                    }
                }
                else // 1 or 2
                {
                    cr = cg = cb = *pSrcU8++;
                    if (srcChanCount == 2)
                    {
                        ca = *pSrcU8++;
                    }
                }
                if (!isFloat)
                {
                    pDstU8[0] = cr;
                    pDstU8[1] = cg;
                    pDstU8[2] = cb;
                    pDstU8[3] = ca;
                    pDstU8 += R_RGBA_COUNT;
                }
                else
                {
                    pDstF32[0] = static_cast<float>(cr) / 0xff;
                    pDstF32[1] = static_cast<float>(cg) / 0xff;
                    pDstF32[2] = static_cast<float>(cb) / 0xff;
                    pDstF32[3] = static_cast<float>(ca) / 0xff;
                    pDstF32 += R_RGBA_COUNT;
                }
            }
        }
    }
    else // 入力が浮動小数点数型
    {
        const float* pSrcF32Top = mimg.floatPixels();
        const size_t srcRowValueCount = srcChanCount * faceW;
        for (int iy = 0; iy < faceH; ++iy)
        {
            const float* pSrcF32 = pSrcF32Top + srcRowValueCount * (faceH - 1 - iy); // 垂直に反転
            for (int ix = 0; ix < faceW; ++ix)
            {
                float fr = 0.0f;
                float fg = 0.0f;
                float fb = 0.0f;
                float fa = 1.0f;
                if (srcChanCount >= 3)
                {
                    fr = *pSrcF32++;
                    fg = *pSrcF32++;
                    fb = *pSrcF32++;
                    if (srcChanCount == 4)
                    {
                        fa = *pSrcF32++;
                    }
                }
                else // 1 or 2
                {
                    fr = fg = fb = *pSrcF32++;
                    if (srcChanCount == 2)
                    {
                        fa = *pSrcF32++;
                    }
                }
                if (!isFloat)
                {
                    pDstU8[0] = static_cast<uint8_t>(RClampValue(0, 0xff, RRound(fr * 0xff)));
                    pDstU8[1] = static_cast<uint8_t>(RClampValue(0, 0xff, RRound(fg * 0xff)));
                    pDstU8[2] = static_cast<uint8_t>(RClampValue(0, 0xff, RRound(fb * 0xff)));
                    pDstU8[3] = static_cast<uint8_t>(RClampValue(0, 0xff, RRound(fa * 0xff)));
                    pDstU8 += R_RGBA_COUNT;
                }
                else
                {
                    pDstF32[0] = fr;
                    pDstF32[1] = fg;
                    pDstF32[2] = fb;
                    pDstF32[3] = fa;
                    pDstF32 += R_RGBA_COUNT;
                }
            }
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief MImage で画像ファイルをリードして、テクスチャデータを設定します。
//!
//! @param[out] rimg テクスチャイメージです。
//! @param[in] filePaths 画像ファイルのパス配列です。
//! @param[in] mergePath マージする ftx ファイルのパスです。空文字ならマージしません。
//! @param[in] dccPreset DCC プリセット名です。
//! @param[in] hint ヒント情報です。
//! @param[in] linearFlag リニア変換フラグです。
//! @param[in] usesSrgbFetch sRGB フォーマットを使用するなら true です。
//! @param[in] dimension 次元です。
//! @param[in] initialSwizzle 初期スウィズル値です。
//! @param[in] comment 編集用コメント文です。
//! @param[in] checkElementFlag 要素チェック中（エラー抑止）なら true を指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus ReadImageFileByMImage(
    RImage& rimg,
    RScene& rscene,
    const RStringArray& filePaths,
    const std::string& mergePath,
    const std::string& dccPreset,
    const std::string& hint,
    const int linearFlag,
    const bool usesSrgbFetch,
    const RImage::Dimension dimension,
    const int initialSwizzle,
    const std::string& comment,
    const bool checkElementFlag
)
{
    MStatus status;

    uint8_t* pBitmap = nullptr;
    int imageW = 1;
    int imageH = 1;
    bool hasAlpha = false;
    bool isFloat = false;
    size_t faceBytes = 0;

    for (size_t fileIdx = 0; fileIdx < filePaths.size(); ++fileIdx)
    {
        //-----------------------------------------------------------------------------
        // リードします。
        const std::string& filePath = filePaths[fileIdx];
        //cerr << "read: " << fileIdx << ": " << filePath << endl;
        MImage mimg;
        status = mimg.readFromFile(filePath.c_str());
        if (!status)
        {
            if (!checkElementFlag)
            {
                YShowError(&rscene, "ファイルをリードできません: {0}", "Cannot read file: {0}", filePath);
            }
            break;
        }

        //-----------------------------------------------------------------------------
        // 情報を取得します。
        unsigned int unsignedW;
        unsigned int unsignedH;
        status = mimg.getSize(unsignedW, unsignedH);
        if (!status)
        {
            if (!checkElementFlag)
            {
                YShowError(&rscene, // Cannot get texture size: %s
                    "テクスチャーファイルの内容が不正なため、画像の幅と高さを取得できません: {0}",
                    "Cannot get the height and width of an image due to invalid data in the texture file: {0}",
                    filePath);
            }
            break;
        }
        const int faceW = unsignedW;
        const int faceH = unsignedH;

        //-----------------------------------------------------------------------------
        // ピクセルデータを取得します。
        if (fileIdx == 0)
        {
            imageW = faceW;
            imageH = faceH;
            isFloat = (mimg.pixelType() == MImage::kFloat ||
                RGetPngBitCount(filePath) == 16);
            const size_t pixelBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
            faceBytes = pixelBytes * faceW * faceH;
            pBitmap = new uint8_t[faceBytes * filePaths.size()];
            GetMImagePixels(pBitmap, &hasAlpha, faceW, faceH, isFloat, mimg);
        }
        else
        {
            if (faceW != imageW ||
                faceH != imageH)
            {
                if (!checkElementFlag)
                {
                    YShowError(&rscene, // Cube map width or height is not identical: %s
                        "キューブマップの各面の幅と高さがテクスチャーファイルによって異なります。"
                        "幅と高さが同じテクスチャーファイルを使用してください: {0}",
                        "The width or height of the cube map differs depending on the texture file. "
                        "Make sure that you use texture files that have the same width and height: {0}",
                        filePath);
                }
                status = MS::kFailure;
                break;
            }
            GetMImagePixels(pBitmap + faceBytes * fileIdx, &hasAlpha, faceW, faceH, isFloat, mimg);
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャイメージにビットマップをリードします。
    if (status && pBitmap != nullptr)
    {
        rimg.ReadBitmap(filePaths, mergePath,
            dccPreset, hint, linearFlag, usesSrgbFetch, dimension, initialSwizzle, comment,
            pBitmap, imageW, imageH, hasAlpha, isFloat); // pBitmap は rimg 内にコピーされます。
    }
    if (pBitmap != nullptr)
    {
        delete[] pBitmap;
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 画像ファイルをリードしてデータを取得します。
//-----------------------------------------------------------------------------
MStatus ReadImageFileForMaya(
    RImage& rimg,
    RScene& rscene,
    const RStringArray& filePaths,
    const std::string& mergePath,
    const std::string& dccPreset,
    const std::string& hint,
    const int linearFlag,
    const bool usesSrgbFetch,
    const RImage::Dimension dimension,
    const int initialSwizzle,
    const std::string& comment,
    const bool checkElementFlag
)
{
    //-----------------------------------------------------------------------------
    // 拡張子をチェックします。
    for (size_t fileIdx = 0; fileIdx < filePaths.size(); ++fileIdx)
    {
        const std::string& filePath = filePaths[fileIdx];
        const std::string ext = RGetExtensionFromFilePath(filePath);
        if (ext != "tga" && ext != "png" && ext != "dds")
        {
            if (!checkElementFlag)
            {
                YShowError(&rscene, // Image type is not supported: %s
                    "サポートしていない画像ファイルが使用されています: {0} \n"
                    "現在 TGA / PNG / DDS ファイルのみテクスチャーファイルとして使用できます。",
                    "An unsupported image file is being used: {0} \n"
                    "Currently, only TGA / PNG / DDS files can be used as texture files.",
                    filePath);
            }
            return MS::kFailure;
        }
    }

    //-----------------------------------------------------------------------------
    // リードします。
    if (UsesMImageForRead(filePaths))
    {
        return ReadImageFileByMImage(rimg, rscene, filePaths, mergePath,
            dccPreset, hint, linearFlag, usesSrgbFetch, dimension, initialSwizzle, comment,
            checkElementFlag);
    }
    else
    {
        RStatus rstatus = rimg.ReadFile(filePaths, mergePath,
            dccPreset, hint, linearFlag, usesSrgbFetch, dimension, initialSwizzle, comment);
        if (!checkElementFlag)
        {
            return GetMStatusDisplayError(&rscene, rstatus);
        }
        else
        {
            return GetMStatus(rstatus);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief シーン内の全 IK ハンドルのデータを取得します。
//-----------------------------------------------------------------------------
MStatus GetIkHandleData(YIkHandleArray& ikHandleArray)
{
    //-----------------------------------------------------------------------------
    // 階層構造から IK ハンドルを幅優先探索で走査します。
    MItDag dagIter(MItDag::kBreadthFirst, MFn::kIkHandle);
    for ( ; !dagIter.isDone(); dagIter.next())
    {
        // IK ハンドルの中間データ
        YIkHandle data;
        // IK ハンドルの DAG パスを取得し、IK ハンドルオブジェクトを作成します。
        dagIter.getPath(data.m_HandlePath);
        MFnIkHandle handleFn(data.m_HandlePath);

        //-----------------------------------------------------------------------------
        // IK ハンドルからジョイントの先頭とエフェクターを取得し、
        // 中間データに記録します。
        handleFn.getStartJoint(data.m_StartPath);
        handleFn.getEffector(data.m_EffPath);

        //-----------------------------------------------------------------------------
        // エフェクターの translateX パラメータに接続されている Dagノード を取得します。
        // そのノードがジョイントの終端となる
        MPlug txPlug = MFnDagNode(data.m_EffPath).findPlug("translateX");
        MPlugArray plugArray;
        txPlug.connectedTo(plugArray, true, false);
        if (plugArray.length() > 0)
        {
            MFnDagNode(plugArray[0].node()).getPath(data.m_EndPath);
        }

        //-----------------------------------------------------------------------------
        // IK ハンドルに登録されているジョイントの先頭から終端までを paths 配列に設定します。
        CDagPathArray paths;
        MDagPath curPath = data.m_EffPath;
        // ジョイントの終端から一つづつ辿っていき、先頭まで配列に登録
        while (curPath.pop())
        {
            if (curPath == data.m_StartPath ||
                curPath.length() == 0) // root
            {
                break;
            }
            paths.push_back(curPath);
        }
        // YIkHandle::m_ChainPaths に IK ハンドルから取得したジョイントの先頭から終端
        // までを登録します。
        data.m_ChainPaths.push_back(data.m_StartPath);
        int chainCount = static_cast<int>(paths.size());
        for (int ipath = 0; ipath < chainCount; ++ipath)
        {
            data.m_ChainPaths.push_back(paths[chainCount - 1 - ipath]);
        }

        //-----------------------------------------------------------------------------
        // ikHandleArray に YIkHandle を登録します。
        ikHandleArray.push_back(data);
    }

    //-----------------------------------------------------------------------------
    // debug
    #if 0
    int handleCount = ikHandleArray.size();
    for (int ihandle = 0; ihandle < handleCount; ++ihandle)
    {
        const YIkHandle& data = ikHandleArray[ihandle];
        cerr << "ik: " << data.m_HandlePath.partialPathName() << ": "
             << data.m_StartPath.partialPathName() << " -> "
             << data.m_EndPath.partialPathName() << " ("
             << data.m_EffPath.partialPathName() << ") : ";
        int chainCount = data.m_ChainPaths.size();
        for (int ipath = 0; ipath < chainCount; ++ipath)
        {
            cerr << data.m_ChainPaths[ipath].partialPathName() << " ";
        }
        cerr << R_ENDL;
    }
    #endif

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief ノードをコントロールしている IK ハンドルの配列内のインデックスを取得します。
//-----------------------------------------------------------------------------
int FindIkHandleDataFromChainJoint(
    const YIkHandleArray& ikHandleArray,
    const MDagPath& path
)
{
    // 配列内のIKハンドルデータで path で指定されたノードを管理していないか
    // 順番に確認します。
    int handleCount = static_cast<int>(ikHandleArray.size());
    for (int ihandle = 0; ihandle < handleCount; ++ihandle)
    {
        // IKハンドルが管理しているChainパス配列に path で指定された DAG パスが
        // 含まれているか確認します。
        const YIkHandle& data = ikHandleArray[ihandle];
        int chainCount = static_cast<int>(data.m_ChainPaths.size());
        for (int ipath = 0; ipath < chainCount; ++ipath)
        {
            // 引数で指定された DAG パスが見つかったらそのIKハンドルの添字を返します。
            if (data.m_ChainPaths[ipath] == path)
                return ihandle;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータの値を解析します。
//!
//! @param[in,out] data ユーザーデータです。
//! @param[in] valStr 値の文字列です。
//!
//! @return 処理成功なら true を返します。
//!         値の文字列が不正なら false を返します。
//-----------------------------------------------------------------------------
static bool ParseUserDataValue(RUserData& data, const MString& valStr)
{
    if (data.m_Type == RUserData::INT   ||
        data.m_Type == RUserData::FLOAT)
    {
        MStringArray words;
        valStr.split(' ', words);
        const int wordCount = words.length();
        if (data.m_Type == RUserData::INT)
        {
            for (int iWord = 0; iWord < wordCount; ++iWord)
            {
                const MString& word = words[iWord];
                if (word.length() > 0)
                {
                    data.m_IntValues.push_back(atol(word.asChar()));
                }
            }
            return true;
        }
        else
        {
            for (int iWord = 0; iWord < wordCount; ++iWord)
            {
                const MString& word = words[iWord];
                if (word.length() > 0)
                {
                    data.m_FloatValues.push_back(static_cast<float>(atof(word.asChar())));
                }
            }
            return true;
        }
    }
    else // STRING, WSTRING
    {
        bool isValid = true;
        MStringArray words;
        valStr.split('\n', words);
        const int wordCount = words.length();
        for (int iWord = 0; iWord < wordCount; ++iWord)
        {
            const MString& word = words[iWord];
            if (word.length() > 0 &&
                RUserData::IsValidString(word.asChar()))
            {
                data.m_StringValues.push_back(word.asChar());
            }
            else
            {
                isValid = false;
            }
        }
        return isValid;
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya のノードに設定されているユーザーデータを取得します。
//-----------------------------------------------------------------------------
MStatus GetUserData(RUserDataArray& datas, RScene& rscene, const MObject& nodeObj)
{
    MStatus status;

    datas.clear();

    //-----------------------------------------------------------------------------
    // ユーザーデータ数を取得します。
    MFnDependencyNode depFn(nodeObj);
    MPlug dataCountPlug = FindPlugQuiet(depFn, "nw4fUserDataCount", &status);
    if (!status)
    {
        return MS::kSuccess;
    }
    int dataCount = 0;
    dataCountPlug.getValue(dataCount);
    if (dataCount == 0)
    {
        return MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // ユーザーデータ値を取得します。
    MPlug dataPlug = FindPlugQuiet(depFn, "nw4fUserData", &status);
    if (!status)
    {
        return MS::kSuccess;
    }

    for (int iData = 0; iData < dataCount; ++iData)
    {
        //-----------------------------------------------------------------------------
        // エンコードされた文字列を取得します。
        MPlug valuePlug = dataPlug.elementByLogicalIndex(iData, &status);
        if (!status)
        {
            break;
        }
        MString valStr;
        valuePlug.getValue(valStr);
        //cerr << "ud: " << iData << ": " << valStr << R_ENDL;

        //-----------------------------------------------------------------------------
        // ユーザーデータ名を取得します。
        const int iDq = valStr.index('\"');
        if (iDq < 1)
        {
            continue;
        }
        RUserData data;
        data.m_Name = valStr.substring(0, iDq - 1).asChar();

        //-----------------------------------------------------------------------------
        // 値の型を取得します。
        MString typeStr = valStr.substring(iDq + 1, iDq + 1);
        if (typeStr == "i")
        {
            data.m_Type = RUserData::INT;
        }
        else if (typeStr == "f")
        {
            data.m_Type = RUserData::FLOAT;
        }
        else if (typeStr == "s")
        {
            data.m_Type = RUserData::STRING;
        }
        else
        {
            data.m_Type = RUserData::WSTRING;
        }

        //-----------------------------------------------------------------------------
        // 配列の要素数を取得します。
        int valueCount = 0;
        valStr = valStr.substring(iDq + 3, valStr.length() - 1);
        const int iSpace = valStr.index(' ');
        if (iSpace > 0)
        {
            valueCount = atol(valStr.substring(0, iSpace - 1).asChar());
        }

        //-----------------------------------------------------------------------------
        // 値を取得します。
        if (iSpace > 0 && iSpace + 1 < static_cast<int>(valStr.length()))
        {
            valStr = valStr.substring(iSpace + 1, valStr.length() - 1);
        }
        else
        {
            valStr = "";
        }
        //cerr << "ud: " << iData << ": " << data.m_Name.c_str() << " (" << typeStr << ") [" << valueCount << "] \"" << valStr << "\"" << R_ENDL;
        if (!ParseUserDataValue(data, valStr))
        {
            YShowWarning(&rscene, // User data is wrong: %s: %s // 通常は発生しない
                "ユーザーデータが不正です: {0}: {1}",
                "User data is wrong: {0}: {1}",
                depFn.name().asChar(), data.m_Name);
            continue;
        }

        //-----------------------------------------------------------------------------
        // append
        datas.push_back(data);
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 出力用の要素名を返します。
//-----------------------------------------------------------------------------
std::string GetOutElementName(const std::string& src, const bool removeNamespace)
{
    std::string name = src;
    if (removeNamespace)
    {
        int icolon = static_cast<int>(name.find_last_of(':'));
        if (icolon != static_cast<int>(std::string::npos) &&
            icolon + 1 < static_cast<int>(name.size()))
        {
            name = std::string(name.c_str() + icolon + 1);
        }
    }
    return RAdjustElementNameString(name);
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Maya プラグインルートパスを返します。
//!        環境変数が定義されていなければ "./" を返します。
//-----------------------------------------------------------------------------
std::string GetNintendoMayaRootPath()
{
    std::string path = RGetEnvVar("NINTENDO_MAYA_ROOT");
    if (path.empty())
    {
        path = RGetEnvVar("NW4F_MAYA_ROOT");
        if (path.empty())
        {
            path = RGetEnvVar("NW4F_G3D_ROOT");
            if (path.empty())
            {
                path = RGetEnvVar("NW4F_ROOT");
            }

            if (!path.empty())
            {
                path += "\\Tool\\DccPlugin\\Maya";
            }
            else
            {
                path = ".";
            }
        }
    }
    path = RGetFilePathWithEndingSlash(RGetUnixFilePath(path));
    return path;
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Maya プラグインが使用するグラフィックスツールフォルダのパスを返します。
//-----------------------------------------------------------------------------
std::string GetNintendoMayaGraphicsToolsPath()
{
    const std::string rootPath = GetNintendoMayaRootPath();

    std::string gfxToolsPath = RGetFullFilePath(rootPath + "../GraphicsTools/", true);
    if (RFolderExists(gfxToolsPath))
    {
        return gfxToolsPath;
    }

    gfxToolsPath = RGetFullFilePath(rootPath + "../../G3dTool/", true);
    #ifdef _WIN64
    gfxToolsPath += "win64/";
    #else
    gfxToolsPath += "win32/";
    #endif
    return gfxToolsPath;
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Maya プラグインが使用する 3D ツールフォルダのパスを返します。
//-----------------------------------------------------------------------------
std::string GetNintendoMaya3dToolsPath()
{
    const std::string rootPath = GetNintendoMayaRootPath();

    std::string g3dToolPath = RGetFullFilePath(rootPath + "../3dTools/", true);
    if (RFolderExists(g3dToolPath))
    {
        return g3dToolPath;
    }

    g3dToolPath = RGetFullFilePath(rootPath + "../../G3dTool/", true);
    #ifdef _WIN64
    g3dToolPath += "win64/";
    #else
    g3dToolPath += "win32/";
    #endif
    return g3dToolPath;
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Maya プラグインが使用する 3D テクスチャーコンバーターのパスを返します。
//-----------------------------------------------------------------------------
std::string GetNintendoMaya3dTextureConverterPath()
{
    const std::string g3dToolsPath = GetNintendoMaya3dToolsPath();
    std::string converterPath = g3dToolsPath + "3dTextureConverter.exe";
    if (!RFileExists(converterPath))
    {
        const std::string nw4fConverterPath = g3dToolsPath + "NW4F_g3dtexcvtr.exe";
        if (RFileExists(nw4fConverterPath))
        {
            converterPath = nw4fConverterPath;
        }
    }
    return converterPath;
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Maya プラグインのコンフィグフォルダのパスを返します。
//!
//! @param[in] getsCofigEnv 専用の環境変数による指定を取得するなら true です。
//-----------------------------------------------------------------------------
std::string GetNintendoMayaConfigFolderPath(const bool getsCofigEnv)
{
    std::string path = RGetEnvVar("NINTENDO_MAYA_CONFIG");
    if (path.empty())
    {
        path = RGetEnvVar("NW4F_MAYA_CONFIG");
    }

    if (!path.empty() && getsCofigEnv)
    {
        path = RGetFilePathWithEndingSlash(RGetUnixFilePath(path));
    }
    else
    {
        path = GetNintendoMayaRootPath() + "Config/";
    }
    return path;
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Maya プラグインのプリセットフォルダのパスを返します。
//-----------------------------------------------------------------------------
std::string GetNintendoMayaPresetsFolderPath()
{
    std::string path = RGetEnvVar("NINTENDO_MAYA_PRESETS");
    if (path.empty())
    {
        path = RGetEnvVar("NW4F_MAYA_PRESETS");
    }

    if (!path.empty())
    {
        path = RGetFilePathWithEndingSlash(RGetUnixFilePath(path));
    }
    else
    {
        path = GetNintendoMayaConfigFolderPath(true) + "Presets/";
    }
    return path;
}

