﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityCommon.h"
#include "DccUtilityMaterial.h"

#include "DccShape.h"
#include <list>

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

/**
 */
class FOutShapeInfo : public nn::gfx::tool::dcc::ROutShapeInfo
{
public:
    /**
    コンストラクタ
    */
    FOutShapeInfo()
    {
        m_Index = 0;
        m_MatName = "";
        m_Name = "";
        m_pUvAttribNames = &m_uvAttribNames;
        m_pUvHintIdxs = &m_uvHintIdxs;
    }

    /**
    頂点配列の複製を作成
    */
    FOutShapeInfo* duplicate(void) const;

    /**
    頂点属性毎の、データ配列の要素数を取得
    \param attr 要素数を知りたい頂点属性
    \return データ配列の要素数
    \par 解説：
    　mVtxInfos[attr].mArrayPtr 配列の要素数を返す。
    この変数は void* 型として宣言されており、
    頂点属性毎に nn::gfx::tool::dcc::RVec2Array　、 nn::gfx::tool::dcc::RVec3Array もしくは nn::gfx::tool::dcc::RVec4Array へのキャストが必要になる。
    本関数はこのキャストを隠蔽する。
    */
    size_t getNumOfVertexInfo(nn::gfx::tool::dcc::RPrimVtx::VtxAttr attr) const;

    //! @brief UV セットの頂点属性名配列へのポインタです。出力する UV セット数と同じ長さです。
    //!        NULL ならすべての UV セットの頂点属性名をヒント情報から自動決定します。
    //!        部分的に空文字であれば、空文字の UV セットの頂点属性名のみ自動決定します。
    nn::gfx::tool::dcc::RStringArray m_uvAttribNames;

    //! @brief UV セットのヒント情報インデックス配列へのポインタです。出力する UV セット数と同じ長さです。
    //!        NULL なら uv0 から順番に割り当てます。
    nn::gfx::tool::dcc::RIntArray m_uvHintIdxs;
};

/**
 *	FDC の RShape だけでは保持できないデータを処理するために作成しました。
 *	Maya では ShapeData というクラスを作成していましたが Max/Softimage で共通のコードを
 *	使用するために utility レベルで対応します。
*/
class FShape : public nn::gfx::tool::dcc::RShape
{
public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] skinningMode スキニングモードです。
    //! @param[in] mtxPalCount 行列パレットサイズです。
    //! @param[in] orgName シェイプ名（デバッグ表示用）です。設定しなくても構いません。
    //! @param[in] progShapePtr シェイプ処理の進行状況を伝えるオブジェクトです。設定しなくても構いません。
    //!
    explicit FShape(
        const nn::gfx::tool::dcc::RShape::SkinningMode skinningMode,
        const int mtxPalCount,
        const char* orgName = nullptr,
        nn::gfx::tool::dcc::RProgShape* progShapePtr = nullptr) :
            nn::gfx::tool::dcc::RShape( skinningMode, mtxPalCount, orgName, progShapePtr ),
            m_MaterialId(-1),
            m_Index(0),
            m_Priority(0),
            m_ShapeInfo(nullptr)
        {
        }

    virtual ~FShape()
    {
        if( m_ShapeInfo != nullptr )
        {
            delete m_ShapeInfo;
            m_ShapeInfo = nullptr;
        }
    }

    int GetMaterialId( void ) const { return m_MaterialId; }
    void SetMaterialId( int nId ){ m_MaterialId = nId; }
    int m_MaterialId;
    int m_Index; //!< 出力用のインデックスです。
    int m_Priority; //!< 描画優先度です。

    /**
    頂点属性データを関連付け
    \param shapeInfo このシェイプが参照する頂点属性データ
    \par 解説：
    　関連付けられた頂点属性データは mergeSameVertexData() 、 deleteUnusedVertexData()
    および out() で使用される。
    この関数内で頂点属性データの複製が作成されるため
    mergeSameVertexData() や deleteUnusedVertexData()
    での変更は他のシェイプには影響しない。
    <br />
    　この関数は一度だけ呼び出すことができる。
    2回目の呼び出しではアサートされる。
    */
    void setVertexData(const FOutShapeInfo &shapeInfo);

    /**
    同じ値を持つ頂点属性データをマージし、プリミティブのインデックス配列を修正
    \par 解説：
    　同じ値を持つ頂点属性データをマージし、
    mPrimitives に登録されているプリミティブのインデックス配列を修正する。
    shape に対してトライアングルの登録をした後で、
    SetAndOptimize() を呼び出す前に本処理をする。
    <br />
    　setVertexData() で頂点属性データを関連付けた後に本関数を呼び出す。
    setVertexData() が呼び出されていないとアサートされる。
    */
    bool mergeSameVertexData(void);

    /**
    モデルの頂点情報を拡大縮小する
    \par 解説：
    　モデルの頂点情報を拡大縮小する。
    　setVertexData() で頂点属性データを関連付けた後に本関数を呼び出す。
    setVertexData() が呼び出されていないとアサートされる。
    */
    bool magnifyVertexData(float magnify);

    /**
    参照されていない頂点属性データを削除
    \par 解説：
    　参照されていない頂点属性データを削除する。
    setVertexData() で頂点属性データを関連付けた後に本関数を呼び出す。
    setVertexData() が呼び出されていないとアサートされる。
    */
    void deleteUnusedVertexData(void);

    /**
    シェイプと頂点情報から法線マップ用のタンジェントを計算する。
    \param texCh 計算に使用するUV座標チャンネル
    \param isUVBackward UV座標が逆向きに貼られている場合(3dsmax)
    \return
    \par 解説：
    　シェイプと頂点情報(ShapeInfo)から法線マップに必要なタンジェントを計算して
    設定する。
    */
    nn::gfx::tool::dcc::RStatus calcTangent(nn::gfx::tool::dcc::RPrimVtx::VtxAttr texCh, bool isUVBackward = false);

    /**
    頂点座標、法線ベクトルを変換する
    \param mtx 変換行列
    \par 解説：
    　 mShapeInfo 内の頂点座標、法線ベクトルを指定された変換行列で変換する。
    中間フォーマットではスキニングが有効な場合、
    頂点をワールド座標系で出力する必要がある。
    その変換のために本関数を利用する。
    本関数は setVertexData() の後に呼び出す必要がある。
    */
    void transformVertexData(const nn::gfx::tool::dcc::RMtx44& mtx);

    /**
    頂点毎に頂点座標、法線ベクトルを参照するボーンの座標系へ変換する
    \param mtxToBone ボーン毎の変換行列
    \par 解説：
    mShapeInfo 内の頂点座標、法線ベクトルを、参照するボーンの座標系へ変換する。
    ボーン毎の変換行列は mtxs 引数に渡され、ボーンの getNodeArrayIndex() で参照する。
    中間フォーマットではスキニングが有効で各頂点がそれぞれ一つのボーンのみを参照する場合、
    リジッドスキニングのために頂点をボーン座標系で出力する必要がある。
    その変換のために本関数を利用する。
    本関数は setVertexData() の後に呼び出す必要がある。
    */
    void transformVertexDataToBone(const nn::gfx::tool::dcc::RMtx44* mtxToBone, const nn::gfx::tool::dcc::RVtxMtxArray& vtxMtxs);

    /**
    このシェイプ内で参照しているRVtxMtxのインデックスリストを取得します。
    \param[out] list	追加するリスト
    \par 解説：
    */
    void GetRefVmtIndexList( std::list<int>&	list );

    FOutShapeInfo* getShapeInfo( void ) const
    {
        return m_ShapeInfo;
    }
    bool FShape::deleteSameKeyShapes(void);

    // マテリアルから参照されなかったUV座標を削除する
    void deleteUnusedUV( FMaterial* fmat );

    /**
    頂点属性データ
    \par 解説：
    setVertexData() 内で複製された頂点属性データ。
    複製元は RModel::mShapeInfo 。
    */
    FOutShapeInfo	*m_ShapeInfo;
};



/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
