﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DccUtilityCommon.h"
#include "DccUtilityOptimizer.h"
#include "DccUtilityLogger.h"
#include <ostream>

namespace Dcc = nn::gfx::tool::dcc;

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

//----------------------------------------------------------------------------
ROptimizer::ROptimizer()
{
}

//----------------------------------------------------------------------------
ROptimizer::~ROptimizer()
{
}

#define DO_OPTIMIZER_PARALLEL // 中間ファイルオプティマイザを並列実行するなら定義します。

//----------------------------------------------------------------------------
bool ROptimizer::Optimize(
    const std::string& outFolderPath,
    const Dcc::RExpOpt&	opt )
{
    //-----------------------------------------------------------------------------
    // check option
    if (!opt.Uses3dOptimizer())
    {
        return true;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザの exe ファイルが存在するかチェックします。
    std::string cvtrPath = opt.m_G3dToolPath + "3dIntermediateFileOptimizer.exe";
    if (!Dcc::RFileExists(cvtrPath))
    {
        cvtrPath = opt.m_G3dToolPath + "NW4F_g3doptcvtr.exe";
    }
    if (!Dcc::RFileExists(cvtrPath))
    {
        RLogger::LogMessagebyID(RLogger::kLogERR_FindOptimizer, cvtrPath.c_str());
        return false;
    }

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザの引数を設定します。
    std::string cmdTop = "\"" + RGetWindowsFilePath(cvtrPath) + "\"";
    if (!opt.m_DisplaysProfile)
    {
        cmdTop += " -s"; // silent
    }
    RStringArray cmds;

    if (opt.OptimizesFmd() || opt.MergesFmd())
    {
        cmds.push_back(cmdTop + Get3dOptimizerFmdArgs(opt, outFolderPath));
    }

    if (opt.OptimizesFsk() ||
        (opt.MergesFsk() && RFileExists(opt.GetMergeAnimFilePath(RExpOpt::FSK))))
    {
        cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(opt, RExpOpt::FSK, outFolderPath));
    }

    if (opt.OptimizesFvb() ||
        (opt.MergesFvb() && RFileExists(opt.GetMergeAnimFilePath(RExpOpt::FVB))))
    {
        cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(opt, RExpOpt::FVB, outFolderPath));
    }

    if (opt.UsesSingleFma())
    {
        if (opt.OptimizesFma() ||
            (opt.MergesFma() && RFileExists(opt.GetMergeAnimFilePath(RExpOpt::FCL))))
        {
            cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(opt, RExpOpt::FCL, outFolderPath));
        }
    }
    else
    {
        if (opt.OptimizesFcl() ||
            (opt.MergesFcl() && RFileExists(opt.GetMergeAnimFilePath(RExpOpt::FCL))))
        {
            cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(opt, RExpOpt::FCL, outFolderPath));
        }

        if (opt.OptimizesFts() ||
            (opt.MergesFts() && RFileExists(opt.GetMergeAnimFilePath(RExpOpt::FTS))))
        {
            cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(opt, RExpOpt::FTS, outFolderPath));
        }

        if (opt.OptimizesFtp() ||
            (opt.MergesFtp() && RFileExists(opt.GetMergeAnimFilePath(RExpOpt::FTP))))
        {
            cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(opt, RExpOpt::FTP, outFolderPath));
        }
    }

    if (opt.OptimizesFsh() ||
        (opt.MergesFsh() && RFileExists(opt.GetMergeAnimFilePath(RExpOpt::FSH))))
    {
        cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(opt, RExpOpt::FSH, outFolderPath));
    }

    if (opt.OptimizesFsn() ||
        (opt.MergesFsn() && RFileExists(opt.GetMergeAnimFilePath(RExpOpt::FSN))))
    {
        cmds.push_back(cmdTop + Get3dOptimizerAnimArgs(opt, RExpOpt::FSN, outFolderPath));
    }

#if 0
    if (opt.OptimizesFmd())
    {
        //-----------------------------------------------------------------------------
        // fmd ファイルの最適化指定です。
        std::string cmd = cmdTop + " \"" + fmdPath + "\"";
        if (opt.MergesFmd())
        {
            cmd += " --merge --merge-options=\"--merge-file='" + opt.m_MergeFmdPath + "'";
            if (opt.m_SamplerMergePriority)
            {
                cmd += " --sampler-merge-priority";
            }
            cmd += "\"";
        }

        Add3dOptimizerCompressBoneArg(&cmd, opt);

        if (opt.m_CompressMaterial)
        {
            cmd += " --compress-material";
        }
        if (opt.m_CompressShape)
        {
            cmd += " --compress-shape";
        }

        Add3dOptimizerPrimitiveArg(&cmd, opt);
        Add3dOptimizerQuantizationAnalysisArg(&cmd, opt);
        cmd += " -o \"" + fmdPath + "\"";
        cmds.push_back(cmd);
    }

    if (opt.OptimizesFsk() ||
        (mergeByDCC && opt.MergesFsk() && Dcc::RFileExists(mergeAnimPathTop + opt.GetExtension(Dcc::RExpOpt::FSK))))
    {
        //-----------------------------------------------------------------------------
        // fsk ファイルの最適化指定です。
        std::string cmd = cmdTop + " \"" + fskPath + "\"";
        Add3dOptimizerMergeAnimArg(&cmd, opt, Dcc::RExpOpt::FSK);
        Add3dOptimizerCompressBoneArg(&cmd, opt);
        Add3dOptimizerQuantizationAnalysisArg(&cmd, opt);
        cmd += " -o \"" + fskPath + "\"";
        cmds.push_back(cmd);
    }

    if (opt.OptimizesFvb() ||
        (mergeByDCC && opt.MergesFvb() && Dcc::RFileExists(mergeAnimPathTop + opt.GetExtension(Dcc::RExpOpt::FVB))))
    {
        //-----------------------------------------------------------------------------
        // fvb ファイルの最適化指定です。
        std::string cmd = cmdTop + " \"" + fvbPath + "\"";
        Add3dOptimizerMergeAnimArg(&cmd, opt, Dcc::RExpOpt::FVB);
        Add3dOptimizerCompressBoneArg(&cmd, opt);
        Add3dOptimizerQuantizationAnalysisArg(&cmd, opt);
        cmd += " -o \"" + fvbPath + "\"";
    }

    if (opt.OptimizesFcl() ||
        (mergeByDCC && opt.MergesFcl() && Dcc::RFileExists(mergeAnimPathTop + opt.GetExtension(Dcc::RExpOpt::FCL))))
    {
        //-----------------------------------------------------------------------------
        // fcl ファイルの最適化指定です。
        std::string cmd = cmdTop + " \"" + fclPath + "\"";
        Add3dOptimizerMergeAnimArg(&cmd, opt, Dcc::RExpOpt::FCL);
        Add3dOptimizerQuantizationAnalysisArg(&cmd, opt);
        cmd += " -o \"" + fclPath + "\"";
        cmds.push_back(cmd);
    }

    if (opt.OptimizesFts() ||
        (mergeByDCC && opt.MergesFts() && Dcc::RFileExists(mergeAnimPathTop + opt.GetExtension(Dcc::RExpOpt::FTS))))
    {
        //-----------------------------------------------------------------------------
        // fts ファイルの最適化指定です。
        std::string cmd = cmdTop + " \"" + ftsPath + "\"";
        Add3dOptimizerMergeAnimArg(&cmd, opt, Dcc::RExpOpt::FTS);
        Add3dOptimizerQuantizationAnalysisArg(&cmd, opt);
        cmd += " -o \"" + ftsPath + "\"";
        cmds.push_back(cmd);
    }

    if (opt.OptimizesFtp() ||
        (mergeByDCC && opt.MergesFtp() && Dcc::RFileExists(mergeAnimPathTop + opt.GetExtension(Dcc::RExpOpt::FTP))))
    {
        //-----------------------------------------------------------------------------
        // ftp ファイルの最適化指定です。
        std::string cmd = cmdTop + " \"" + ftpPath + "\"";
        Add3dOptimizerMergeAnimArg(&cmd, opt, Dcc::RExpOpt::FTP);
        Add3dOptimizerQuantizationAnalysisArg(&cmd, opt);
        cmd += " -o \"" + ftpPath + "\"";
        cmds.push_back(cmd);
    }

    if (opt.OptimizesFsh() ||
        (mergeByDCC && opt.MergesFsh() && Dcc::RFileExists(mergeAnimPathTop + opt.GetExtension(Dcc::RExpOpt::FSH))))
    {
        //-----------------------------------------------------------------------------
        // fsh ファイルの最適化指定です。
        std::string cmd = cmdTop + " \"" + fshPath + "\"";
        Add3dOptimizerMergeAnimArg(&cmd, opt, Dcc::RExpOpt::FSH);
        Add3dOptimizerQuantizationAnalysisArg(&cmd, opt);
        cmd += " -o \"" + fshPath + "\"";
        cmds.push_back(cmd);
    }

    if (opt.OptimizesFsn() ||
        (mergeByDCC && opt.MergesFsn() && Dcc::RFileExists(mergeAnimPathTop + opt.GetExtension(Dcc::RExpOpt::FSN))))
    {
        //-----------------------------------------------------------------------------
        // fsn ファイルの最適化指定です。
        std::string cmd = cmdTop + " \"" + fsnPath + "\"";
        Add3dOptimizerMergeAnimArg(&cmd, opt, Dcc::RExpOpt::FSN);
        Add3dOptimizerQuantizationAnalysisArg(&cmd, opt);
        cmd += " -o \"" + fsnPath + "\"";
        cmds.push_back(cmd);
    }
#endif

    //-----------------------------------------------------------------------------
    // 中間ファイルオプティマイザを実行します。
    //const clock_t startClock = clock();
    if (opt.OptimizesSome() && (opt.MergesFmd() || opt.MergesAnim()))
    {
        //	TODO:
        //YSetHelpLine("Optimizing and Merging fmd file ...");
    }
    else if (opt.OptimizesSome())
    {
        //	TODO:
        //YSetHelpLine("Optimizing %s ...", targetStr.c_str());
    }
    else
    {
        //	TODO:
        //YSetHelpLine("Merging fmd file ...");
    }

    int exitCode = 0;
    std::string outMsg;
    std::string errMsg;
    std::string* pOutMsg = (opt.m_DisplaysProfile) ? &outMsg : nullptr;
#ifdef DO_OPTIMIZER_PARALLEL
    if (opt.m_DisplaysProfile)
    {
        for (size_t cmdIdx = 0; cmdIdx < cmds.size(); ++cmdIdx)
        {
            RLogger::LogMessage(cmds[cmdIdx], RLogger::kDump);
        }
    }
    exitCode = RExecProcessParallel(nullptr, pOutMsg, &errMsg, cmds, SW_HIDE);
    if (!outMsg.empty())
    {
        RLogger::LogMessage(outMsg, RLogger::kDump);
    }
#else
    for (size_t cmdIdx = 0; cmdIdx < cmds.size(); ++cmdIdx)
    {
        const std::string& cmd = cmds[cmdIdx];
        if (opt.m_DisplaysProfile)
        {
            RLogger::LogMessage(cmd, RLogger::kDump);
        }
        exitCode = RExecProcessWithPipe(nullptr, pOutMsg, &errMsg, cmd.c_str(), SW_HIDE);
        if (!outMsg.empty())
        {
            RLogger::LogMessage(outMsg, RLogger::kDump);
        }
        if (exitCode != 0)
        {
            break;
        }
    }
#endif

    if (!errMsg.empty())
    {
        if (exitCode == 0) // オプティマイザ警告
        {
            RLogger::LogMessagebyID(RLogger::kLogWRN_Optimization);
            RLogger::LogMessage(errMsg, RLogger::kDump);
        }
        else // オプティマイザエラー
        {
            RLogger::LogMessagebyID(RLogger::kLogERR_Optimization);
            RLogger::LogMessage(errMsg, RLogger::kDump);
            return false;
        }
    }


    //-----------------------------------------------------------------------------
    // profile
    //	TODO:
    //yscene.m_OptimizeClocks += clock() - startClock;

    return true;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザのアニメーション中間ファイルマージ引数を追加します。
//!
//! @param[in,out] pArgStr 中間ファイルオプティマイザの引数の文字列へのポインタです。
//! @param[in] opt Export オプションです。
//! @param[in] fileType ファイルタイプです。
//-----------------------------------------------------------------------------
void ROptimizer::Add3dOptimizerMergeAnimArg(std::string* pArgStr, const Dcc::RExpOpt& opt, const Dcc::RExpOpt::FileType fileType)
{
    if (opt.m_MergeAnimFlag)
    {
        const std::string mergeAnimPath = opt.m_MergeAnimFolder +
            opt.m_MergeAnimName + "." + opt.GetExtension(fileType);
        if (Dcc::RFileExists(mergeAnimPath))
        {
            *pArgStr += " --merge --merge-options=\"--merge-file='" + mergeAnimPath + "'\"";
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザのボーン圧縮引数を追加します。
//!
//! @param[in,out] pArgStr 中間ファイルオプティマイザの引数の文字列へのポインタです。
//! @param[in] opt Export オプションです。
//-----------------------------------------------------------------------------
void ROptimizer::Add3dOptimizerCompressBoneArg(std::string* pArgStr, const Dcc::RExpOpt& opt)
{
    if (opt.m_CompressBone == Dcc::RExpOpt::COMPRESS_BONE_CULL)
    {
        *pArgStr += " --compress-bone-cull";
    }
    else if (opt.m_CompressBone == Dcc::RExpOpt::COMPRESS_BONE_MERGE)
    {
        *pArgStr += " --compress-bone-merge";
    }
    else if (opt.m_CompressBone == Dcc::RExpOpt::COMPRESS_BONE_UNITE)
    {
        *pArgStr += " --compress-bone-unite";
    }
    else if (opt.m_CompressBone == Dcc::RExpOpt::COMPRESS_BONE_UNITE_ALL)
    {
        *pArgStr += " --compress-bone-unite-all";
    }

    if (opt.UnitesChildBone())
    {
        *pArgStr += " --compress-bone-unite-child";
    }
}
//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザのプリミティブ最適化引数を追加します。
//!
//! @param[in,out] pArgStr 中間ファイルオプティマイザの引数の文字列へのポインタです。
//! @param[in] opt エクスポートオプションです。
//-----------------------------------------------------------------------------
void ROptimizer::Add3dOptimizerPrimitiveArg(std::string* pArgStr, const Dcc::RExpOpt& opt)
{
    if (opt.m_OptimizePrimitive != RExpOpt::OPTIMIZE_PRIMITIVE_NONE)
    {
        *pArgStr += " --optimize-primitive --optimize-primitive-options=\"--mode=";
        if (opt.m_OptimizePrimitive == RExpOpt::OPTIMIZE_PRIMITIVE_DEFAULT)
        {
            *pArgStr += "Normal";
        }
        else if (opt.m_OptimizePrimitive == RExpOpt::OPTIMIZE_PRIMITIVE_FORCE)
        {
            *pArgStr += "Force";
        }
        else if (opt.m_OptimizePrimitive == RExpOpt::OPTIMIZE_PRIMITIVE_BRUTE_FORCE)
        {
            *pArgStr += "BruteForce";
        }
        else if (opt.m_OptimizePrimitive == RExpOpt::OPTIMIZE_PRIMITIVE_FORSYTH)
        {
            *pArgStr += "Forsyth";
        }
        *pArgStr += "\"";
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザの量子化分析引数を追加します。
//!
//! @param[in,out] pArgStr 中間ファイルオプティマイザの引数の文字列へのポインタです。
//! @param[in] opt エクスポートオプションです。
//-----------------------------------------------------------------------------
void ROptimizer::Add3dOptimizerQuantizationAnalysisArg(std::string* pArgStr, const Dcc::RExpOpt& opt)
{
    if (opt.m_QuantizationAnalysis)
    {
        *pArgStr += " --quantization-analysis";
    }
}


//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザの fmd 最適化引数群を取得します。
//!
//! @param[in] opt エクスポートオプションです。
//-----------------------------------------------------------------------------
std::string ROptimizer::Get3dOptimizerFmdArgs(const RExpOpt& rOpt, const std::string tempPath)
{
    const std::string fmdPath = tempPath + rOpt.GetFileName(RExpOpt::FMD);
    std::string argStr = " \"" + fmdPath + "\"";
    if (rOpt.MergesFmd())
    {
        argStr += " -m --merge-options=\"--merge-file='" + rOpt.m_MergeFmdPath + "'";
        if (rOpt.m_SamplerMergePriority)
        {
            argStr += " --sampler-merge-priority";
        }
        if (rOpt.m_BoneVisMergePriority)
        {
            argStr += " --bone-visibility-merge-priority";
        }
        argStr += "\"";
    }
    Add3dOptimizerCompressBoneArg(&argStr, rOpt);
    if (rOpt.m_CompressMaterial)
    {
        argStr += " --compress-material";
    }
    if (rOpt.m_CompressShape)
    {
        argStr += " --compress-shape";
        if (rOpt.m_CompressesIgnoringVertexSkinningCount)
        {
            argStr += " --compress-shape-options=\"";
            argStr += "--ignore-skinning-count";
            argStr += "\"";
        }
    }
    if (rOpt.m_DivideSubmesh && !rOpt.m_DivideSubmeshMode.empty())
    {
        argStr += " --divide-submesh --divide-submesh-options=\"" + rOpt.m_DivideSubmeshMode + "\"";
    }
    if (rOpt.m_PolyReduction && !rOpt.m_PolyReductionMode.empty())
    {
        argStr += " --polygon-reduction --polygon-reduction-options=\"" + rOpt.m_PolyReductionMode + "\"";
    }

    if (!rOpt.m_ExportsLod)
    {
        Add3dOptimizerPrimitiveArg(&argStr, rOpt);
        Add3dOptimizerQuantizationAnalysisArg(&argStr, rOpt);
    }

    if (!rOpt.m_DeleteNearVertex.empty())
    {
        argStr += " --delete-near-vertex --delete-near-vertex-options=\"" + rOpt.m_DeleteNearVertex + "\"";
    }
    if (!rOpt.m_OptimizerExtraOptionFmd.empty())
    {
        argStr += " " + rOpt.m_OptimizerExtraOptionFmd;
    }
    argStr += " -o \"" + fmdPath + "\"";
    return argStr;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルオプティマイザのアニメーション最適化引数群を取得します。
//!
//! @param[in] opt エクスポートオプションです。
//! @param[in] fileType ファイルタイプです。
//-----------------------------------------------------------------------------
std::string ROptimizer::Get3dOptimizerAnimArgs(const Dcc::RExpOpt& rOpt, const Dcc::RExpOpt::FileType fileType, const std::string tempPath)
{
    const std::string filePath = tempPath + rOpt.GetFileName(fileType);
    std::string argStr = " \"" + filePath + "\"";
    Add3dOptimizerMergeAnimArg(&argStr, rOpt, fileType);
    if (fileType == RExpOpt::FSK ||
        fileType == RExpOpt::FVB)
    {
        Add3dOptimizerCompressBoneArg(&argStr, rOpt);
    }
    Add3dOptimizerQuantizationAnalysisArg(&argStr, rOpt);
    argStr += " -o \"" + filePath + "\"";
    return argStr;
}



/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
