﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RGetImageInfo
// ReadFile
// DecodeTgaHeader WriteTga
// DecodeDdsHeader DecodeDdsBitmap
// ReadBitmap

// SetImageData SetFormatFromBitmap
// GetInfoFromMergeFile

// DecodeAddInfo ParseAITexelFormat

// GetConverterOption

//=============================================================================
// 処理選択用マクロです。
//=============================================================================

// fixed
//#define USE_TEX_ADD_INFO_SW // TGA ファイルの付加情報を使用する場合に定義します。

// debug

//=============================================================================
// include
//=============================================================================
#include "DccImage.h"

using namespace std;

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
// ヒント情報の定数です。
const std::string RImage::HINT_ALBEDO     = "albedo";     //!< アルベド用のヒント情報です。
const std::string RImage::HINT_OPACITY    = "opacity";    //!< 不透明度用のヒント情報です。
const std::string RImage::HINT_EMISSION   = "emission";   //!< エミッション用のヒント情報です。
const std::string RImage::HINT_NORMAL     = "normal";     //!< 法線用のヒント情報です。
const std::string RImage::HINT_TANGENT    = "tangent";    //!< 接線用のヒント情報です。
const std::string RImage::HINT_SPECULAR   = "specular";   //!< スペキュラ用のヒント情報です。
const std::string RImage::HINT_REFLECTION = "reflection"; //!< 反射用のヒント情報です。

//-----------------------------------------------------------------------------
// TGA ファイル関連の定数です。
const std::string TgaExtension = "tga"; //!< TGA ファイルの拡張子です。
const int TGA_HEADER_SIZE = 0x12; //!< TGA ファイルのヘッダサイズです。

const uint32_t NINTENDO_TGA_ID_SIZE = 0x14;
const uint32_t NINTENDO_TGA_ID_AI_OFS = 0x10;
const char NINTENDO_TGA_ID_TOP[] = "NintendoTga Ver1.0";

//-----------------------------------------------------------------------------
// PNG ファイル関連の定数です。
const std::string PngExtension = "png"; //!< PNG ファイルの拡張子です。
const int PngFileSignatureBytes = 0x08; //!< PNG ファイルのファイルシグネチャのバイト数です。
const int PngImageHeaderBytes = 0x19; //!< PNG ファイルのイメージヘッダのバイト数です。
const uint8_t PngFileSignature[PngFileSignatureBytes] = //!< PNG ファイルのファイルシグネチャの値です。
{
    0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a
};

//-----------------------------------------------------------------------------
// DDS ファイル関連の定数です。
const std::string DdsExtension = "dds"; //!< DDS ファイルの拡張子です。
const int DDS_HEADER_SIZE = 0x80; //!< DDS ファイルのヘッダサイズです。
const int DDS_DX10_HEADER_SIZE = 20; //!< DDS ファイルの DX10 ヘッダサイズです。
const std::string DDS_SIGNATURE = "DDS "; //!< DDS ファイルの先頭 4 バイトの識別子です。

const uint32_t DDPF_FOURCC       = 0x00000004; // dwFourCC が有効
const uint32_t DDPF_RGB          = 0x00000040; // dwRGBBitCount/dwRBitMask/dwGBitMask/dwBBitMask/dwRGBAlphaBitMask によってフォーマットが定義されている
const uint32_t DDPF_LUMINANCE    = 0x00020000; // 1ch のデータが R G B すべてに展開される

const uint32_t DDSCAPS2_CUBEMAP  = 0x00000200; // Cubemap が存在する場合
const uint32_t DDSCAPS2_VOLUME   = 0x00200000; // VolumeTexture の場合

const uint32_t DDS_DXT1          = 0x31545844;
const uint32_t DDS_DXT2          = 0x32545844;
const uint32_t DDS_DXT3          = 0x33545844;
const uint32_t DDS_DXT4          = 0x34545844;
const uint32_t DDS_DXT5          = 0x35545844;
const uint32_t DDS_ATI2          = 0x32495441;
const uint32_t DDS_R16F          = 0x6f; // 111
const uint32_t DDS_G16R16F       = 0x70; // 112
const uint32_t DDS_A16B16G16R16F = 0x71; // 113
const uint32_t DDS_R32F          = 0x72; // 114
const uint32_t DDS_G32R32F       = 0x73; // 115
const uint32_t DDS_A32B32G32R32F = 0x74; // 116
const uint32_t DDS_DX10          = 0x30315844; // DX10

const uint32_t DXGI_R32G32B32A32_FLOAT = 0x02; //  2
const uint32_t DXGI_R16G16B16A16_FLOAT = 0x0a; // 10
const uint32_t DXGI_R8G8B8A8_UNORM     = 0x1c; // 28

const uint32_t DDS_DIM_1D = 0x02;
const uint32_t DDS_DIM_2D = 0x03;
const uint32_t DDS_DIM_3D = 0x04;

const uint32_t DDS_MISC_TEXTURECUBE = 0x04;

//-----------------------------------------------------------------------------
// add info
const uint32_t AI_TAG_SIZE = 8;
const uint32_t AI_SIZE_SIZE = 4;
const uint32_t AI_HEADER_SIZE = AI_TAG_SIZE + AI_SIZE_SIZE;

const char AI_TAG_TEXEL_FORMAT[] = "nw4f_tfm";
const char AI_TAG_CUBE_MAP[]     = "nw4f_cub";
const char AI_TAG_MIP_SIZE[]     = "nw4f_mps";
const char AI_TAG_TEXEL_DATA[]   = "nw4f_txd";
const char AI_TAG_USER_DATA[]    = "nw4f_udt";

const char AI_TAG_END_BLOCK[]    = "nw4f_end";

//-----------------------------------------------------------------------------
// リニア変換
static const int s_LinearRgbaFlags[R_RGBA_COUNT] =
{
    RImage::LINEAR_R,
    RImage::LINEAR_G,
    RImage::LINEAR_B,
    RImage::LINEAR_A,
};

//-----------------------------------------------------------------------------
// cube map
static const int s_CubeMapFacePossHC[RImage::CUBE_FACE_COUNT][2] =
{
    //      [+Y]
    // [-X] [-Z] [+X] [+Z]
    //      [-Y]

    { 2, 1 }, // +X
    { 0, 1 }, // -X
    { 1, 0 }, // +Y
    { 1, 2 }, // -Y
    { 3, 1 }, // +Z
    { 1, 1 }, // -Z
};

static const int s_CubeMapFacePossVC[RImage::CUBE_FACE_COUNT][2] =
{
    //      [+Y]
    // [-X] [-Z] [+X]
    //      [-Y]
    //      [+Z]

    { 2, 1 }, // +X
    { 0, 1 }, // -X
    { 1, 0 }, // +Y
    { 1, 2 }, // -Y
    { 1, 3 }, // +Z
    { 1, 1 }, // -Z
};

//-----------------------------------------------------------------------------
// dimension
static const char* const s_DimensionNames[RImage::DIM_COUNT] =
{
    "1d",
    "2d",
    "3d",
    "cube", // separate
    "cube", // HC
    "cube", // VC
    "cube", // input
    "1d_array",
    "2d_array",
};

//-----------------------------------------------------------------------------
// format
static const char* const s_FormatNames[RImage::Format_Count] =
{
    "unorm_8",
    "snorm_8",
    "uint_8",
    "sint_8",

    "unorm_4_4",
    "unorm_8_8",
    "snorm_8_8",
    "uint_8_8",
    "sint_8_8",

    "unorm_5_6_5",

    "unorm_5_5_5_1",
    "unorm_4_4_4_4",

    "unorm_8_8_8_8",
    "snorm_8_8_8_8",
    "srgb_8_8_8_8",
    "uint_8_8_8_8",
    "sint_8_8_8_8",

    "float_16",
    "float_32",
    "float_16_16",
    "float_32_32",
    "float_11_11_10",
    "float_16_16_16_16",
    "float_32_32_32_32",

    "unorm_bc1",
    "srgb_bc1",
    "unorm_bc2",
    "srgb_bc2",
    "unorm_bc3",
    "srgb_bc3",
    "unorm_bc4",
    "snorm_bc4",
    "unorm_bc5",
    "snorm_bc5",

    "ufloat_bc6",
    "float_bc6",
    "unorm_bc7",
    "srgb_bc7",

    "unorm_etc1",
    "unorm_etc2",
    "srgb_etc2",
    "unorm_etc2_mask",
    "srgb_etc2_mask",
    "unorm_etc2_alpha",
    "srgb_etc2_alpha",

    "unorm_eac_11",
    "snorm_eac_11",
    "unorm_eac_11_11",
    "snorm_eac_11_11",

    "unorm_pvrtc1_2bpp",
    "srgb_pvrtc1_2bpp",
    "unorm_pvrtc1_4bpp",
    "srgb_pvrtc1_4bpp",
    "unorm_pvrtc1_alpha_2bpp",
    "srgb_pvrtc1_alpha_2bpp",
    "unorm_pvrtc1_alpha_4bpp",
    "srgb_pvrtc1_alpha_4bpp",
    "unorm_pvrtc2_alpha_2bpp",
    "srgb_pvrtc2_alpha_2bpp",
    "unorm_pvrtc2_alpha_4bpp",
    "srgb_pvrtc2_alpha_4bpp",

    "unorm_astc_4x4",
    "srgb_astc_4x4",
    "unorm_astc_5x4",
    "srgb_astc_5x4",
    "unorm_astc_5x5",
    "srgb_astc_5x5",
    "unorm_astc_6x5",
    "srgb_astc_6x5",
    "unorm_astc_6x6",
    "srgb_astc_6x6",
    "unorm_astc_8x5",
    "srgb_astc_8x5",
    "unorm_astc_8x6",
    "srgb_astc_8x6",
    "unorm_astc_8x8",
    "srgb_astc_8x8",
    "unorm_astc_10x5",
    "srgb_astc_10x5",
    "unorm_astc_10x6",
    "srgb_astc_10x6",
    "unorm_astc_10x8",
    "srgb_astc_10x8",
    "unorm_astc_10x10",
    "srgb_astc_10x10",
    "unorm_astc_12x10",
    "srgb_astc_12x10",
    "unorm_astc_12x12",
    "srgb_astc_12x12",
};

//-----------------------------------------------------------------------------
//! @brief テクスチャーファイルを開けなかった場合の処理結果ステータスを取得します。
//-----------------------------------------------------------------------------
static RStatus GetTexFileOpenFailureStatus(const std::string& filePath)
{
    if (!RFileExists(filePath))
    {
        return RStatus(RStatus::FAILURE,
            "テクスチャーファイルが存在しません: {0}",
            "The texture file cannot be found: {0}", filePath);
    }
    else
    {
        return RStatus(RStatus::FAILURE,
            "テクスチャーファイルを開けません: {0}",
            "The texture file cannot be opened: {0}", filePath);
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からリニア変換フラグを取得します。
//!
//! @param[in] str リニア変換フラグを表す文字列です。
//!
//! @return リニア変換フラグを返します。
//-----------------------------------------------------------------------------
static int GetLinearFlagFromString(const std::string& str)
{
    RStringArray tokens;
    if (RTokenizeString(tokens, str) == R_RGBA_COUNT)
    {
        int linearFlag = RImage::LINEAR_NONE;
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            if (tokens[iRgba] == "true")
            {
                linearFlag |= s_LinearRgbaFlags[iRgba];
            }
        }
        return linearFlag;
    }
    return RImage::LINEAR_NONE;
}

//-----------------------------------------------------------------------------
//! @brief オプション文字列からリニア変換フラグを取得します。
//-----------------------------------------------------------------------------
int RImage::GetLinearFlagFromOptString(const std::string& str)
{
    if (str.size() == R_RGBA_COUNT)
    {
        int linearFlag = LINEAR_NONE;
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            const char c = str[iRgba];
            if (c == '0')
            {

            }
            else if (c == '1')
            {
                linearFlag |= s_LinearRgbaFlags[iRgba];
            }
            else
            {
                return LINEAR_INVALID;
            }
        }
        return linearFlag;
    }
    return LINEAR_INVALID;
}

//-----------------------------------------------------------------------------
//! @brief リニア変換フラグからオプション文字列を取得します。
//!
//! @param[in] linearFlag リニア変換フラグです。
//!
//! @return オプション文字列を返します。
//-----------------------------------------------------------------------------
std::string RImage::GetOptStringFromLinearFlag(const int linearFlag)
{
    if (linearFlag < 0)
    {
        return "0000";
    }
    else
    {
        char strBuf[R_RGBA_COUNT];
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            strBuf[iRgba] = ((linearFlag & s_LinearRgbaFlags[iRgba]) != 0) ? '1' : '0';
        }
        return std::string(strBuf, sizeof(strBuf));
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーイメージの次元名を返します。
//-----------------------------------------------------------------------------
std::string RImage::GetDimensionName(const Dimension dimension)
{
    if (dimension < 0 || dimension >= DIM_COUNT)
    {
        return s_DimensionNames[DIM_2D];
    }
    else
    {
        return s_DimensionNames[dimension];
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーイメージの次元名から次元を返します。
//-----------------------------------------------------------------------------
RImage::Dimension RImage::GetDimensionFromName(const std::string& name)
{
    if (name == "cube")
    {
        return DIM_CUBE_INPUT;
    }
    if (name == "cube_array")
    {
        return DIM_CUBE_INPUT;
    }

    for (int iDimension = 0; iDimension < DIM_COUNT; ++iDimension)
    {
        if (name == s_DimensionNames[iDimension])
        {
            return static_cast<Dimension>(iDimension);
        }
    }
    return DIM_2D;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーイメージのフォーマット名を返します。
//-----------------------------------------------------------------------------
std::string RImage::GetFormatName(const Format format)
{
    if (format < 0 || format >= Format_Count)
    {
        return "invalid";
    }
    else
    {
        return s_FormatNames[format];
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーイメージのフォーマット名からフォーマットを返します。
//-----------------------------------------------------------------------------
RImage::Format RImage::GetFormatFromName(const std::string& name)
{
    for (int iFormat = 0; iFormat < Format_Count; ++iFormat)
    {
        if (name == s_FormatNames[iFormat])
        {
            return static_cast<Format>(iFormat);
        }
    }
    return Format_Invalid;
}

//-----------------------------------------------------------------------------
//! @brief フォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsBc123Format(const Format format)
{
    switch (format)
    {
    case Format_Unorm_Bc1:
    case Format_Srgb_Bc1:
    case Format_Unorm_Bc2:
    case Format_Srgb_Bc2:
    case Format_Unorm_Bc3:
    case Format_Srgb_Bc3:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief フォーマットが BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsBcFormat(const Format format)
{
    if (IsBc123Format(format))
    {
        return true;
    }
    else
    {
        switch (format)
        {
        case Format_Unorm_Bc4:
        case Format_Snorm_Bc4:
        case Format_Unorm_Bc5:
        case Format_Snorm_Bc5:
        case Format_Ufloat_Bc6:
        case Format_Float_Bc6:
        case Format_Unorm_Bc7:
        case Format_Srgb_Bc7:
            return true;
        default:
            return false;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief フォーマットが浮動小数点数型なら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsFloatFormat(const Format format)
{
    switch (format)
    {
    case Format_Float_16:
    case Format_Float_32:
    case Format_Float_16_16:
    case Format_Float_32_32:
    case Format_Float_11_11_10:
    case Format_Float_16_16_16_16:
    case Format_Float_32_32_32_32:
    case Format_Ufloat_Bc6:
    case Format_Float_Bc6:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief フォーマットから成分数を取得します。
//-----------------------------------------------------------------------------
int RImage::GetCompSize(const Format format)
{
    switch (format)
    {
    case Format_Unorm_8:
    case Format_Snorm_8:
    case Format_Uint_8:
    case Format_Sint_8:
    case Format_Float_16:
    case Format_Float_32:
    case Format_Unorm_Bc4:
    case Format_Snorm_Bc4:
    case Format_Unorm_Eac_11:
    case Format_Snorm_Eac_11:
        return 1;

    case Format_Unorm_4_4:
    case Format_Unorm_8_8:
    case Format_Snorm_8_8:
    case Format_Uint_8_8:
    case Format_Sint_8_8:
    case Format_Float_16_16:
    case Format_Float_32_32:
    case Format_Unorm_Bc5:
    case Format_Snorm_Bc5:
    case Format_Unorm_Eac_11_11:
    case Format_Snorm_Eac_11_11:
        return 2;

    case Format_Unorm_5_6_5:
    case Format_Float_11_11_10:
    case Format_Ufloat_Bc6:
    case Format_Float_Bc6:
    case Format_Unorm_Etc1:
    case Format_Unorm_Etc2:
    case Format_Srgb_Etc2:
        return 3;

    default:
        return 4;
    }
}

//-----------------------------------------------------------------------------
//! @brief 有効な成分選択なら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsValidCompSel(const std::string& compSel)
{
    const int COMP_CHAR_COUNT = 6;
    const char compSelChars[COMP_CHAR_COUNT] =
    {
        'r', 'g', 'b', 'a', '0', '1'
    };

    int validCount = 0;
    if (compSel.size() == R_RGBA_COUNT)
    {
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            const char c = compSel[iRgba];
            for (int iComp = 0; iComp < COMP_CHAR_COUNT; ++iComp)
            {
                if (c == compSelChars[iComp])
                {
                    ++validCount;
                    break;
                }
            }
        }
    }
    return (validCount == R_RGBA_COUNT);
}

//-----------------------------------------------------------------------------
//! @brief 水平十字キューブマップの幅と高さなら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsCubeHCWH(const int imageW, const int imageH)
{
    if (imageW % CUBE_HC_COUNT_H == 0 &&
        imageH % CUBE_HC_COUNT_V == 0)
    {
        const int faceW = imageW / CUBE_HC_COUNT_H;
        const int faceH = imageH / CUBE_HC_COUNT_V;
        return (faceW == faceH && faceW >= WidthHeightMin);
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 垂直十字キューブマップの幅と高さなら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsCubeVCWH(const int imageW, const int imageH)
{
    if (imageW % CUBE_VC_COUNT_H == 0 &&
        imageH % CUBE_VC_COUNT_V == 0)
    {
        const int faceW = imageW / CUBE_VC_COUNT_H;
        const int faceH = imageH / CUBE_VC_COUNT_V;
        return (faceW == faceH && faceW >= WidthHeightMin);
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief read add info to memory
//!
//! @return NULL if not exist
//-----------------------------------------------------------------------------
//static const uint8_t* ReadAddInfoToMemory(uint32_t& aiSize, const std::string& filePath)
//{
//  aiSize = 0;
//
//  #ifdef USE_TEX_ADD_INFO_SW
//  //-----------------------------------------------------------------------------
//  // check file type
//  const std::string ext = RGetExtensionFromFilePath(filePath);
//  if (ext !=  TgaExtension)
//  {
//      return NULL;
//  }
//
//  //-----------------------------------------------------------------------------
//  // open file
//  ifstream ifs(filePath.c_str(), ios_base::binary);
//  if (!ifs)
//  {
//      return NULL;
//  }
//
//  //-----------------------------------------------------------------------------
//  // parse header
//  int aiOfs = 0;
//
//  uint8_t header[TGA_HEADER_SIZE];
//  ifs.read(reinterpret_cast<char*>(header), TGA_HEADER_SIZE);
//  if (ifs)
//  {
//      const int idSize = static_cast<int>(header[0x00]);
//      if (idSize >= NINTENDO_TGA_ID_SIZE)
//      {
//          uint8_t* idBuf = new uint8_t[idSize];
//          ifs.read(reinterpret_cast<char*>(idBuf), idSize);
//          if (strncmp(reinterpret_cast<char*>(idBuf), NINTENDO_TGA_ID_TOP, strlen(NINTENDO_TGA_ID_TOP)) == 0 &&
//              idBuf[strlen(NINTENDO_TGA_ID_TOP)] == 0x00)
//          {
//              RGetMemLittle(aiOfs, idBuf + NINTENDO_TGA_ID_AI_OFS);
//          }
//          delete[] idBuf;
//      }
//  }
//
//  //-----------------------------------------------------------------------------
//  // read add info
//  const uint8_t* aiBuf = NULL;
//  if (aiOfs > 0)
//  {
//      ifs.seekg(0L, ios_base::end);
//      const int fileSize = static_cast<int>(ifs.tellg());
//      if (aiOfs < fileSize)
//      {
//          aiSize = fileSize - aiOfs;
//          aiBuf = new uint8_t[aiSize];
//          ifs.seekg(aiOfs, ios_base::beg);
//          ifs.read(reinterpret_cast<char*>(aiBuf), aiSize);
//      }
//  }
//
//  ifs.close();
//  return aiBuf;
//  #else
//  R_UNUSED_VARIABLE(filePath);
//  return NULL;
//  #endif
//}

//-----------------------------------------------------------------------------
//! @brief get texel format from AI data
//-----------------------------------------------------------------------------
static int GetTexelFormatFromAIData(const uint8_t* dataBuf, const uint32_t dataSize)
{
    for (int iFormat = 0; iFormat < RImage::Format_Count; ++iFormat)
    {
        const std::string formatName =
            RImage::GetFormatName(static_cast<RImage::Format>(iFormat));
        const uint32_t formatNameSize = (uint32_t)formatName.size();
        if (formatNameSize == dataSize &&
            strncmp(formatName.c_str(), (const char*)dataBuf, formatNameSize) == 0)
        {
            return iFormat;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief get mipmap level from AI data
//-----------------------------------------------------------------------------
static int GetMipLevelFromAIData(const uint8_t* dataBuf, const uint32_t dataSize)
{
    int mipLevel = dataBuf[0] - '0';
    if (dataSize == 2)
    {
        mipLevel *= 10;
        mipLevel += dataBuf[1] - '0';
    }
    return mipLevel;
}

//-----------------------------------------------------------------------------
//! @brief 付加情報を解析して画像ファイルの情報を取得します。
//!
//! @param[out] dimension 次元を格納します。
//! @param[out] addInfoFormat 付加情報のフォーマットを格納します。
//!                           付加情報がない場合は -1 を格納します。
//! @param[out] mipLevel ミップマップのレベル数を格納します。
//!                      ミップマップでない場合と付加情報がない場合は 1 を格納します。
//! @param[in] filePath 画像ファイルのフルパスを指定します。
//!
//-----------------------------------------------------------------------------
//static void ParseAddInfoForGetInfo(
//  RImage::Dimension& dimension,
//  int& addInfoFormat,
//  int& mipLevel,
//  const std::string& filePath
//)
//{
//  //-----------------------------------------------------------------------------
//  // init
//  dimension = RImage::DIM_2D;
//  addInfoFormat = -1;
//  mipLevel = 1;
//
//  //-----------------------------------------------------------------------------
//  // read add info
//  uint32_t aiSize;
//  const uint8_t* aiBuf = ReadAddInfoToMemory(aiSize, filePath);
//  if (aiBuf == NULL)
//  {
//      return;
//  }
//
//  //-----------------------------------------------------------------------------
//  // parse add info
//  const uint8_t* dataPtr = aiBuf;
//  const uint8_t* const dataPtrEnd = dataPtr + aiSize;
//  while (dataPtr <= dataPtrEnd - AI_HEADER_SIZE)
//  {
//      uint32_t dataBlockSize;
//      RGetMemLittle(dataBlockSize, dataPtr + AI_TAG_SIZE);
//      if (dataBlockSize == 0 ||
//          dataBlockSize > aiSize ||
//          strncmp(reinterpret_cast<const char*>(dataPtr), AI_TAG_END_BLOCK, AI_TAG_SIZE) == 0)
//      {
//          break;
//      }
//      const char* tagName = reinterpret_cast<const char*>(dataPtr);
//      const uint8_t* valueTop = dataPtr + AI_HEADER_SIZE;
//      const uint32_t valueSize = dataBlockSize - AI_HEADER_SIZE;
//      //cerr << "paifgi: [" << std::string(tagName, AI_TAG_SIZE) << "] (" << hex << static_cast<int>(dataBlockSize) << dec << ")" << R_ENDL;
//
//      if (strncmp(tagName, AI_TAG_TEXEL_FORMAT, AI_TAG_SIZE) == 0)
//      {
//          addInfoFormat = GetTexelFormatFromAIData(valueTop, valueSize);
//      }
//      else if (strncmp(tagName, AI_TAG_MIP_SIZE, AI_TAG_SIZE) == 0)
//      {
//          mipLevel = GetMipLevelFromAIData(valueTop, valueSize);
//      }
//      else if (strncmp(tagName, AI_TAG_CUBE_MAP, AI_TAG_SIZE) == 0)
//      {
//          dimension = RImage::DIM_CUBE_HC;
//      }
//      dataPtr += dataBlockSize;
//  }
//  delete[] aiBuf;
//}

//-----------------------------------------------------------------------------
//! @brief 画像ファイルの情報を取得します。
//!        現在 TGA、PNG、DDS ファイルに対応しています。
//-----------------------------------------------------------------------------
RStatus RGetImageInfo(
    int* pImageW,
    int* pImageH,
    const std::string& filePath
)
{
    //-----------------------------------------------------------------------------
    // デフォルトの情報を格納します。
    if (pImageW != nullptr)
    {
        *pImageW = 1;
    }
    if (pImageH != nullptr)
    {
        *pImageH = 1;
    }

    //-----------------------------------------------------------------------------
    // 拡張子をチェックします。
    const std::string ext = RGetExtensionFromFilePath(filePath);
    if (ext != TgaExtension &&
        ext != PngExtension &&
        ext != DdsExtension)
    {
        return RStatus(RStatus::FAILURE, // RShowError: Image type is not supported
            "サポートしていない画像ファイルが使用されています: {0}",
            "An unsupported image file is being used: {0}",
            filePath);
    }

    //-----------------------------------------------------------------------------
    // ファイルをオープンします。
    std::ifstream ifs(filePath.c_str(), std::ios::binary);
    if (!ifs)
    {
        return GetTexFileOpenFailureStatus(filePath); // RShowError: Cannot open file
    }

    //-----------------------------------------------------------------------------
    // ファイルを解析します。
    int imageW = 1;
    int imageH = 1;
    bool successFlag = false;
    if (ext == TgaExtension)
    {
        //-----------------------------------------------------------------------------
        // TGA ファイルのヘッダから情報を取得します。
        uint8_t header[TGA_HEADER_SIZE];
        ifs.read(reinterpret_cast<char*>(header), sizeof(header));
        if (ifs)
        {
            imageW = header[0x0c] + (header[0x0d] << 8);
            imageH = header[0x0e] + (header[0x0f] << 8);
            successFlag = true;
        }
    }
    else if (ext == PngExtension)
    {
        //-----------------------------------------------------------------------------
        // PNG ファイルのヘッダから情報を取得します。
        uint8_t header[PngFileSignatureBytes + PngImageHeaderBytes];
        ifs.read(reinterpret_cast<char*>(header), sizeof(header));
        if (ifs)
        {
            if (memcmp(header, PngFileSignature, PngFileSignatureBytes) == 0)
            {
                imageW = RGetMemBigInt(header + 0x10);
                imageH = RGetMemBigInt(header + 0x14);
                successFlag = true;
            }
        }
    }
    else if (ext == DdsExtension)
    {
        //-----------------------------------------------------------------------------
        // DDS ファイルのヘッダから情報を取得します。
        uint8_t header[DDS_HEADER_SIZE];
        ifs.read(reinterpret_cast<char*>(header), sizeof(header));
        if (ifs)
        {
            if (std::string(reinterpret_cast<const char*>(header), 4) == DDS_SIGNATURE)
            {
                imageH = RGetMemLittleInt(header + 0x0C);
                imageW = RGetMemLittleInt(header + 0x10);
                successFlag = true;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // ファイルをクローズします。
    ifs.close();

    //-----------------------------------------------------------------------------
    // 情報を格納します。
    if (pImageW != nullptr)
    {
        *pImageW = imageW;
    }
    if (pImageH != nullptr)
    {
        *pImageH = imageH;
    }

    //-----------------------------------------------------------------------------
    // 処理失敗ならエラーを返します。
    if (!successFlag)
    {
        return RStatus(RStatus::FAILURE, // RShowError: Cannot get image information // 通常は発生しない
            "画像ファイルの情報を取得できません: {0}",
            "Cannot get image information: {0}", filePath);
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 画像ファイルがキューブマップなら true を返します。
//!        現在キューブマップの DDS ファイルのみ true を返します。
//-----------------------------------------------------------------------------
bool RIsCubeMapImage(const std::string& filePath)
{
    bool isCube = false;
    const std::string ext = RGetExtensionFromFilePath(filePath);
    if (ext == DdsExtension)
    {
        ifstream ifs(filePath.c_str(), ios_base::binary);
        if (ifs)
        {
            uint8_t header[DDS_HEADER_SIZE];
            ifs.read(reinterpret_cast<char*>(header), sizeof(header));
            if (ifs)
            {
                if (std::string(reinterpret_cast<const char*>(header), 4) == DDS_SIGNATURE)
                {
                    const uint32_t pixelFormat = RGetMemLittleUInt(header + 0x54);
                    if (pixelFormat != DDS_DX10)
                    {
                        const uint32_t caps2 = RGetMemLittleUInt(header + 0x70);
                        isCube = ((caps2 & DDSCAPS2_CUBEMAP) != 0);
                    }
                    else
                    {
                        uint8_t dx10Header[DDS_DX10_HEADER_SIZE];
                        ifs.read(reinterpret_cast<char*>(dx10Header), sizeof(dx10Header));
                        const uint32_t miscFlag = RGetMemLittleUInt(dx10Header + 0x08);
                        isCube = ((miscFlag & DDS_MISC_TEXTURECUBE) != 0);
                    }
                }
            }
            ifs.close();
        }
    }
    return isCube;
}

//-----------------------------------------------------------------------------
//! @brief PNG ファイルの 1 成分あたりビット数を返します。
//-----------------------------------------------------------------------------
int RGetPngBitCount(const std::string& filePath)
{
    int bitCount = 0;
    const std::string ext = RGetExtensionFromFilePath(filePath);
    if (ext == PngExtension)
    {
        ifstream ifs(filePath.c_str(), std::ios::binary);
        if (ifs)
        {
            uint8_t header[PngFileSignatureBytes + PngImageHeaderBytes];
            ifs.read(reinterpret_cast<char*>(header), sizeof(header));
            if (ifs)
            {
                if (memcmp(header, PngFileSignature, PngFileSignatureBytes) == 0)
                {
                    bitCount = header[0x18];
                }
            }
            ifs.close();
        }
    }
    return bitCount;
}

//-----------------------------------------------------------------------------
//! @brief イメージの幅、高さ、奥行きを設定します。
//!        パス配列、次元、ビットマップの幅と高さを設定してから呼びます。
//-----------------------------------------------------------------------------
void RImage::SetImageWHD()
{
    //-----------------------------------------------------------------------------
    // 幅と高さを設定します。
    m_ImageW = m_FullW;
    m_ImageH = m_FullH;
    if (m_Dimension == DIM_CUBE_HC)
    {
        m_ImageW /= CUBE_HC_COUNT_H;
        m_ImageH /= CUBE_HC_COUNT_V;
    }
    else if (m_Dimension == DIM_CUBE_VC)
    {
        m_ImageW /= CUBE_VC_COUNT_H;
        m_ImageH /= CUBE_VC_COUNT_V;
    }

    //-----------------------------------------------------------------------------
    // 奥行きを設定します。
    m_ImageD = 1;
    if (m_Dimension == DIM_CUBE_SEPARATE)
    {
        m_ImageD = static_cast<int>(m_FilePaths.size());
    }
    else if (m_Dimension == DIM_CUBE_HC || m_Dimension == DIM_CUBE_VC)
    {
        m_ImageD = CUBE_FACE_COUNT;
    }
}

//-----------------------------------------------------------------------------
//! @brief ビットマップのピクセルがすべてグレースケール（R=G=B）なら true を返します。
//!        現在はミップマップ非対応です。
//-----------------------------------------------------------------------------
static bool RIsAllGrayBitmap(
    const uint8_t* pFullBitmap,
    const int fullW,
    const int imageW,
    const int imageH,
    const int imageD,
    const RImage::Dimension dimension,
    const bool isFloat
)
{
    const int (*cubeMapFacePoss)[2] = (dimension == RImage::DIM_CUBE_HC) ?
        s_CubeMapFacePossHC : s_CubeMapFacePossVC;

    const int curW = imageW;
    const int curH = imageH;
    for (int iz = 0; iz < imageD; ++iz)
    {
        int srcIx = 0;
        int srcIy = iz * curH;
        if (dimension == RImage::DIM_CUBE_HC ||
            dimension == RImage::DIM_CUBE_VC)
        {
            const int* facePoss = cubeMapFacePoss[iz];
            srcIx = facePoss[0] * curW;
            srcIy = facePoss[1] * curH;
        }

        if (!isFloat)
        {
            const uint8_t* pSrc = pFullBitmap + (srcIx + srcIy * fullW) * R_RGBA_COUNT;
            for (int iy = 0; iy < curH; ++iy)
            {
                for (int ix = 0; ix < curW; ++ix)
                {
                    if (pSrc[0] != pSrc[1] || pSrc[0] != pSrc[2])
                    {
                        return false;
                    }
                    pSrc += R_RGBA_COUNT;
                }
                pSrc += (fullW - curW) * R_RGBA_COUNT;
            }
        }
        else
        {
            const float* pSrcF32 = reinterpret_cast<const float*>(pFullBitmap) +
                (srcIx + srcIy * fullW) * R_RGBA_COUNT;
            for (int iy = 0; iy < curH; ++iy)
            {
                for (int ix = 0; ix < curW; ++ix)
                {
                    if (pSrcF32[0] != pSrcF32[1] || pSrcF32[0] != pSrcF32[2])
                    {
                        return false;
                    }
                    pSrcF32 += R_RGBA_COUNT;
                }
                pSrcF32 += (fullW - curW) * R_RGBA_COUNT;
            }
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief ビットマップのアルファ成分から透明モードを取得します。
//!        現在はミップマップ非対応です。
//-----------------------------------------------------------------------------
static RImage::TransparencyMode GetBitmapTransparencyMode(
    const uint8_t* pFullBitmap,
    const int fullW,
    const int imageW,
    const int imageH,
    const int imageD,
    const RImage::Dimension dimension,
    const bool hasAlpha,
    const bool isFloat
)
{
    if (!hasAlpha)
    {
        return RImage::OPA;
    }

    RImage::TransparencyMode xpaMode = RImage::OPA;
    const int (*cubeMapFacePoss)[2] = (dimension == RImage::DIM_CUBE_HC) ?
        s_CubeMapFacePossHC : s_CubeMapFacePossVC;

    const int curW = imageW;
    const int curH = imageH;
    for (int iz = 0; iz < imageD; ++iz)
    {
        int srcIx = 0;
        int srcIy = iz * curH;
        if (dimension == RImage::DIM_CUBE_HC ||
            dimension == RImage::DIM_CUBE_VC)
        {
            const int* facePoss = cubeMapFacePoss[iz];
            srcIx = facePoss[0] * curW;
            srcIy = facePoss[1] * curH;
        }
        //cerr << "gbtm: " << iz << ": " << srcIx << " " << srcIy << endl;

        if (!isFloat)
        {
            const uint8_t* pSrc = pFullBitmap + (srcIx + srcIy * fullW) * R_RGBA_COUNT;
            for (int iy = 0; iy < curH; ++iy)
            {
                for (int ix = 0; ix < curW; ++ix)
                {
                    const uint8_t alpha = pSrc[3];
                    if (alpha == 0x00)
                    {
                        xpaMode = RImage::MASK;
                        // 残りの部分に中間的なアルファが存在する可能性があるので
                        // 判定を継続します。
                    }
                    else if (alpha < 0xff)
                    {
                        return RImage::XLU;
                    }
                    pSrc += R_RGBA_COUNT;
                }
                pSrc += (fullW - curW) * R_RGBA_COUNT;
            }
        }
        else
        {
            const float* pSrcF32 = reinterpret_cast<const float*>(pFullBitmap) +
                (srcIx + srcIy * fullW) * R_RGBA_COUNT;
            for (int iy = 0; iy < curH; ++iy)
            {
                for (int ix = 0; ix < curW; ++ix)
                {
                    const float alpha = pSrcF32[3];
                    if (alpha == 0.0f)
                    {
                        xpaMode = RImage::MASK;
                        // 残りの部分に中間的なアルファが存在する可能性があるので
                        // 判定を継続します。
                    }
                    else if (alpha < 1.0f)
                    {
                        return RImage::XLU;
                    }
                    pSrcF32 += R_RGBA_COUNT;
                }
                pSrcF32 += (fullW - curW) * R_RGBA_COUNT;
            }
        }
    }
    return xpaMode;
}

//-----------------------------------------------------------------------------
//! @brief ビットマップからフォーマットを設定します。
//-----------------------------------------------------------------------------
void RImage::SetFormatFromBitmap()
{
    //-----------------------------------------------------------------------------
    // check gray
    const bool grayFlag = RIsAllGrayBitmap(m_pFullBitmap, m_FullW,
        m_ImageW, m_ImageH, m_ImageD, m_Dimension, m_IsFloat);

    //-----------------------------------------------------------------------------
    // set format
    if (m_IsFloat)
    {
        m_Format = Format_Float_16_16_16_16;
    }
    else if (m_Dimension == DIM_1D       ||
             m_Dimension == DIM_1D_ARRAY)
    {
        if ((m_LinearFlag == LINEAR_RGB || m_LinearFlag == LINEAR_RGBA) && m_UsesSrgbFetch)
        {
            m_Format = Format_Srgb_8_8_8_8;
        }
        else
        {
            m_Format = Format_Unorm_8_8_8_8;
        }
    }
    else if (grayFlag)
    {
        m_Format = (m_TransparencyMode == OPA) ? Format_Unorm_Bc4 : Format_Unorm_Bc5;
    }
    else // color
    {
        if ((m_LinearFlag == LINEAR_RGB || m_LinearFlag == LINEAR_RGBA) && m_UsesSrgbFetch)
        {
            m_Format = (m_TransparencyMode == XLU) ? Format_Srgb_Bc3 : Format_Srgb_Bc1;
        }
        else
        {
            m_Format = (m_TransparencyMode == XLU) ? Format_Unorm_Bc3 : Format_Unorm_Bc1;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief TGA ファイルのヘッダをデコードします。
//!
//! @param[out] imageW 画像の幅を格納します。
//! @param[out] imageH 画像の高さを格納します。
//! @param[out] hasAlpha アルファ成分の有無を格納します。
//! @param[out] addInfoOfs 付加情報の画像ファイルの先頭からのオフセットを格納します。
//! @param[in] fileBuf ファイルバッファです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static RStatus DecodeTgaHeader(
    int& imageW,
    int& imageH,
    bool& hasAlpha,
    int& addInfoOfs,
    const RFileBuf& fileBuf
)
{
    //  header size = 18 (0x12) bytes, binary data is little endian
    //  00      ID length after header
    //  01      color palette flag (0=no palette, 1=use palette)
    //  02      image type
    //           0=no image data
    //           1=uncompressed, color-mapped
    //           2=uncompressed, RGB
    //           3=uncompressed, black and white
    //           9=runlength encoded, color-mapped
    //          10=runlength encoded, RGB
    //          11=compressed, black and white
    //          32=compressed color-mapped, Huffman, Delta, RLE
    //          33=compressed color-mapped, Huffman, Delta, RLE. 4-pass quadtree-type process
    //
    //  03-04   color index min
    //  05-06   palette color size
    //  07      palette color bit size
    //  08-09   x origin of image
    //  0a-0b   y origin of image
    //  0c-0d   width
    //  0e-0f   height
    //  10      image pixel bit size (8, 16, 24, 32)
    //  11      image descriptor byte
    //          bit 3-0 number of attribute bits (0=CI or GRAY or 16/24bit RGB, 1=16bit RGBA, 8=32bit RGBA)
    //          bit 4   x direction (0=left to right, 1=right to left)
    //          bit 5   y direction (0=lower to upper, 1=upper to lower)
    //          bit 7-6 data storage interleaving flag
    //                  00=non-interleaved
    //                  01=two-way (even/odd) interleaving
    //                  10=four way interleaving
    //                  11=reserved

    //-----------------------------------------------------------------------------
    // check header
    const int imageType = fileBuf[0x02];
    if (imageType !=  1 &&  // CI
        imageType !=  2 &&  // RGB
        imageType !=  3 &&  // GRAY
        imageType !=  9 &&  // CI (RLE)
        imageType != 10 &&  // RGB (RLE)
        imageType != 11)    // GRAY (RLE)
    {
        return RStatus(RStatus::FAILURE, // RShowError: TGA file is wrong: (invalid image type)
            "不正な TGA ファイルです:（不正なイメージタイプ）: {0}",
            "TGA file is wrong: (invalid image type): {0}", fileBuf.GetPath());
    }

    const int bitPerPixel = fileBuf[0x10];
    if (bitPerPixel !=  8 &&
        bitPerPixel != 15 &&
        bitPerPixel != 16 &&
        bitPerPixel != 24 &&
        bitPerPixel != 32)
    {
        return RStatus(RStatus::FAILURE, // RShowError: TGA file is wrong: (invalid bit per pixel)
            "不正な TGA ファイルです:（不正なピクセル当たりのビット数）: {0}",
            "TGA file is wrong: (invalid bit per pixel): {0}", fileBuf.GetPath());
    }

    //-----------------------------------------------------------------------------
    // get size
    imageW = fileBuf[0x0c] + (fileBuf[0x0d] << 8);
    imageH = fileBuf[0x0e] + (fileBuf[0x0f] << 8);

    //-----------------------------------------------------------------------------
    // get alpha flag
    if (fileBuf[0x01] != 0) // color palette flag
    {
        // CI
        hasAlpha = (fileBuf[0x07] == 32); // palette color bit size
    }
    else
    {
        // RGB or GRAY
        const int attrBitSize = fileBuf[0x11] & 0x0f;
        hasAlpha = (attrBitSize != 0);
        if ((imageType == 3 || imageType == 11) && // GRAY
            bitPerPixel == 8)
        {
            hasAlpha = false;
        }
    }

    //-----------------------------------------------------------------------------
    // check ID field for add info
    addInfoOfs = 0;
    #ifdef USE_TEX_ADD_INFO_SW
    const int idSize = static_cast<int>(fileBuf[0x00]);
    const uint8_t* idField = fileBuf.GetBuf() + TGA_HEADER_SIZE;
    if (idSize >= NINTENDO_TGA_ID_SIZE &&
        strncmp(reinterpret_cast<char*>(idField), NINTENDO_TGA_ID_TOP, strlen(NINTENDO_TGA_ID_TOP)) == 0 &&
        idField[strlen(NINTENDO_TGA_ID_TOP)] == 0x00)
    {
        RGetMemLittle(addInfoOfs, idField + NINTENDO_TGA_ID_AI_OFS);
        if (addInfoOfs < TGA_HEADER_SIZE + NINTENDO_TGA_ID_SIZE ||
            addInfoOfs >= fileBuf.GetSize())
        {
            addInfoOfs = 0;
        }
    }
    #endif

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief TGA ファイルのビットマップをデコードします。
//!
//! @param[out] pDst ビットマップバッファのポインタです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] fileBuf ファイルバッファです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static void DecodeTgaBitmap(
    uint8_t* pDst,
    const int imageW,
    const int imageH,
    const RFileBuf& fileBuf
)
{
    //-----------------------------------------------------------------------------
    // ヘッダを解析します。
    const uint8_t* pSrc = fileBuf.GetBuf();
    const int idSize = static_cast<int>(pSrc[0x00]);
    int iBuf = TGA_HEADER_SIZE + idSize;

    const bool hasPal = (pSrc[0x01] != 0);
    const int imageType = pSrc[0x02];
    const int bitPerPixel = pSrc[0x10];

    // decode image descriptor
    //const int attrBitSize = pSrc[0x11] & 0x0f;
    const bool rightToLeft  = ((pSrc[0x11] & 0x10) != 0);
    const bool upperToLower = ((pSrc[0x11] & 0x20) != 0);
    const int interLeave    =  (pSrc[0x11] & 0xc0) >> 6; // 0=none, 1=two, 2=four

    //-----------------------------------------------------------------------------
    // パレットをデコードします。
    uint8_t* pPalBuf = NULL;
    if (hasPal)
    {
        const int ciMin   = pSrc[0x03] | (pSrc[0x04] << 8);
        const int colSize = pSrc[0x05] | (pSrc[0x06] << 8);
        const int ciMax = ciMin + colSize;
        const int colBitSize = pSrc[0x07];
        pPalBuf = new uint8_t[ciMax * R_RGBA_BYTES];
        int iCol;
        for (iCol = 0; iCol < ciMin; ++iCol)
        {
            // ciMin までを黒で埋めます。
            int ofs = iCol * R_RGBA_BYTES;
            pPalBuf[ofs    ] = pPalBuf[ofs + 1] = pPalBuf[ofs + 2] = 0x00;
            pPalBuf[ofs + 3] = 0xff;
        }
        for ( ; iCol < ciMax; ++iCol)
        {
            uint8_t cr, cg, cb, ca;
            if (colBitSize == 8) // GRAY 8 bit
            {
                cr = cg = cb = ca = pSrc[iBuf++];
            }
            else if (colBitSize == 15 ||    // RGB(A) 15/16 bit
                     colBitSize == 16)
            {
                int val = pSrc[iBuf] | (pSrc[iBuf + 1] << 8);
                iBuf += 2;
                cr = static_cast<uint8_t>((val >> 10) & 0x1f);
                cg = static_cast<uint8_t>((val >>  5) & 0x1f);
                cb = static_cast<uint8_t>((val      ) & 0x1f);
                cr = (cr << 3) | (cr >> 2);
                cg = (cg << 3) | (cg >> 2);
                cb = (cb << 3) | (cb >> 2);
                ca = 0xff;
                //if (attrBitSize == 0)
                //{
                //  ca = 0xff;
                //}
                //else
                //{
                //  ca = (val & 0x8000) ? 0xff : 0x00;
                //}
            }
            else if (colBitSize == 24) // RGB 24 bit
            {
                cb = pSrc[iBuf    ];
                cg = pSrc[iBuf + 1];
                cr = pSrc[iBuf + 2];
                ca = 0xff;
                iBuf += 3;
            }
            else if (colBitSize == 32) // RGBA 32 bit
            {
                cb = pSrc[iBuf    ];
                cg = pSrc[iBuf + 1];
                cr = pSrc[iBuf + 2];
                ca = pSrc[iBuf + 3];
                iBuf += 4;
            }
            else
            {
                cr = cg = cb = ca = 0;
            }
            const int palOfs = iCol * R_RGBA_BYTES;
            pPalBuf[palOfs    ] = cr;
            pPalBuf[palOfs + 1] = cg;
            pPalBuf[palOfs + 2] = cb;
            pPalBuf[palOfs + 3] = ca;
        }
    }

    //-----------------------------------------------------------------------------
    // ビットマップをデコードします。
    const bool rleFlag = (imageType == 9 || imageType == 10 || imageType == 11);
    int rleCount = 0;
    bool sameFlag = false;
    uint8_t cr = 0, cg = 0, cb = 0, ca = 0;
    int iyOfs = 0;
    int iyCur = iyOfs;
    for (int iy = 0; iy < imageH; ++iy)
    {
        const int iyDst = (upperToLower) ? iyCur : imageH - 1 - iyCur;
        int dstOfs = iyDst * imageW * R_RGBA_BYTES;
        if (rightToLeft)
        {
            dstOfs += (imageW - 1) * R_RGBA_BYTES;
        }
        for (int ix = 0; ix < imageW; ++ix)
        {
            bool getColorFlag = true;
            if (rleFlag)
            {
                if (rleCount == 0)
                {
                    int count = pSrc[iBuf++];
                    if (count & 0x80)
                    {
                        rleCount = count - 0x80;
                    }
                    else
                    {
                        rleCount = count;
                    }
                    sameFlag = ((count & 0x80) != 0);
                }
                else
                {
                    --rleCount;
                    getColorFlag = !sameFlag;
                }
            }

            if (getColorFlag)
            {
                if (hasPal)                 // CI 8/16 bit
                {
                    int palOfs;
                    if (bitPerPixel == 8)
                    {
                        palOfs = pSrc[iBuf++] * R_RGBA_BYTES;
                    }
                    else
                    {
                        palOfs = (pSrc[iBuf] | (pSrc[iBuf + 1] << 8)) * R_RGBA_BYTES;
                        iBuf += 2;
                    }
                    cr = pPalBuf[palOfs    ];
                    cg = pPalBuf[palOfs + 1];
                    cb = pPalBuf[palOfs + 2];
                    ca = pPalBuf[palOfs + 3];
                }
                else if (bitPerPixel == 8)  // GRAY 8 bit
                {
                    cr = cg = cb = pSrc[iBuf++];
                    ca = 0xff;
                }
                else if (bitPerPixel == 15 ||
                         bitPerPixel == 16) // RGB(A) 16 bit
                {
                    int val = pSrc[iBuf] | (pSrc[iBuf + 1] << 8);
                    iBuf += 2;
                    cr = static_cast<uint8_t>((val >> 10) & 0x1f);
                    cg = static_cast<uint8_t>((val >>  5) & 0x1f);
                    cb = static_cast<uint8_t>((val      ) & 0x1f);
                    cr = (cr << 3) | (cr >> 2);
                    cg = (cg << 3) | (cg >> 2);
                    cb = (cb << 3) | (cb >> 2);
                    ca = 0xff;
                    //if (attrBitSize == 0)
                    //{
                    //  ca = 0xff;
                    //}
                    //else
                    //{
                    //  ca = (val & 0x8000) ? 0xff : 0x00;
                    //}
                }
                else if (bitPerPixel == 24) // RGB 24 bit
                {
                    cb = pSrc[iBuf    ];
                    cg = pSrc[iBuf + 1];
                    cr = pSrc[iBuf + 2];
                    ca = 0xff;
                    iBuf += 3;
                }
                else if (bitPerPixel == 32) // RGBA 32 bit
                {
                    cb = pSrc[iBuf    ];
                    cg = pSrc[iBuf + 1];
                    cr = pSrc[iBuf + 2];
                    ca = pSrc[iBuf + 3];
                    iBuf += 4;
                }
            }
            pDst[dstOfs    ] = cr;
            pDst[dstOfs + 1] = cg;
            pDst[dstOfs + 2] = cb;
            pDst[dstOfs + 3] = ca;
            if (rightToLeft)
            {
                dstOfs -= 4;
            }
            else
            {
                dstOfs += 4;
            }
        }
        if (interLeave == 2)        // 4
        {
            iyCur += 4;
        }
        else if (interLeave == 1)   // 2
        {
            iyCur += 2;
        }
        else                        // 0
        {
            ++iyCur;
        }
        if (iyCur >= imageH)
        {
            iyCur = ++iyOfs;
        }
    }

    //-----------------------------------------------------------------------------
    // パレットを解放します。
    if (pPalBuf != NULL)
    {
        delete[] pPalBuf;
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief TGA ファイルをライトします（動作確認用）。
//-----------------------------------------------------------------------------
RStatus RImage::WriteTga(const std::string& path) const
{
    //-----------------------------------------------------------------------------
    // ファイルをオープンします。
    ofstream ofs(path.c_str(), ios_base::out | ios_base::binary);
    if (!ofs)
    {
        return RStatus(RStatus::FAILURE, // RShowError: Cannot open file
            "ファイルを開けません。上書き禁止になっていないか確認してください: {0}",
            "The file cannot be opened. Confirm whether the file can be overwritten: {0}", path);
    }
    //cerr << "write: " << path << endl;

    //-----------------------------------------------------------------------------
    // TGA のビットマップを作成します。
    const int dstD = (m_Dimension == DIM_CUBE_HC || m_Dimension == DIM_CUBE_VC) ?
        1 : m_ImageD;
    const int dstW = m_FullW;
    const int dstH = m_FullH * dstD;

    const int bitmapSize = dstW * dstH * R_RGBA_BYTES;
    uint8_t* bitmap = new uint8_t[bitmapSize];
    memset(bitmap, 0xff, bitmapSize);

    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(m_pFullBitmap);
    int ofsX = 0;
    for (int level = 0; level < 1; ++level)
    {
        const int curW = RMax(m_FullW >> level, 1);
        const int curH = RMax(m_FullH >> level, 1);
        int ofsY = 0;
        for (int iz = 0; iz < dstD; ++iz)
        {
            uint8_t* pDst = bitmap + (ofsX + ofsY * dstW) * R_RGBA_BYTES;
            for (int iy = 0; iy < curH; ++iy)
            {
                for (int ix = 0; ix < curW; ++ix)
                {
                    // TGA のビットマップは BGRA の順に格納します。
                    *pDst++ = pSrc[2]; // B
                    *pDst++ = pSrc[1]; // G
                    *pDst++ = pSrc[0]; // R
                    *pDst++ = pSrc[3]; // A
                    pSrc += R_RGBA_BYTES;
                }
                pDst += (dstW - curW) * R_RGBA_BYTES;
            }
            ofsY += m_FullH;
        }
        ofsX += curW;
    }

    //-----------------------------------------------------------------------------
    // ライトします。
    const uint8_t header[TGA_HEADER_SIZE] =
    {
        0x00,       // id size
        0x00,       // palette flag
        0x02,       // image type (0x02 = RGB)
        0x00,       // color index min L
        0x00,       // color index min H
        0x00,       // palette color size L
        0x00,       // palette color size H
        0x00,       // palette color bit size
        0x00,       // x origin of image L
        0x00,       // x origin of image H
        0x00,       // y origin of image L
        0x00,       // y origin of image H
        static_cast<uint8_t>(dstW & 0xff), // width L
        static_cast<uint8_t>(dstW >> 8  ), // width H
        static_cast<uint8_t>(dstH & 0xff), // height L
        static_cast<uint8_t>(dstH >> 8  ), // height H
        0x20,       // pixel bit size
        0x28,       // descriptor (0x20 = upper to lower, 0x08 = 32bit RGBA)
    };
    ofs.write(reinterpret_cast<const char*>(header), sizeof(header));
    ofs.write(reinterpret_cast<const char*>(bitmap), bitmapSize);
    delete[] bitmap;

    return RStatus::SUCCESS;
}

//=============================================================================
//! @brief DDS ファイルマスク情報のクラスです。
//=============================================================================
class RDdsMaskInfo
{
public:
    uint32_t m_Mask; //!< マスク値です。
    int m_Size; //!< 値のビット数です。
    int m_Shift; //!< シフト数です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] mask マスク値です。
    //!
    explicit RDdsMaskInfo(uint32_t mask = 0)
    {
        m_Mask = mask;
        m_Size = m_Shift = 0;
        if (mask != 0)
        {
            while ((mask & 1) == 0 && m_Shift < 32)
            {
                m_Shift += 1;
                mask >>= 1;
            }
            while ((mask & 1) != 0 && m_Shift + m_Size < 32)
            {
                m_Size += 1;
                mask >>= 1;
            }
        }
    }
};

//=============================================================================
//! @brief DDS ファイル情報のクラスです。
//=============================================================================
class RDdsInfo
{
public:
    uint32_t m_HeaderSize; //!< ヘッダのサイズです。
    uint32_t m_PfFlags; //!< ピクセルフォーマットフラグです。
    uint32_t m_PixelFormat; //!< ピクセルフォーマットです。
    uint32_t m_BitCount; //!< 1 ピクセルのビット数です。
    RDdsMaskInfo m_MaskInfoR; //!< R 成分のマスクです。
    RDdsMaskInfo m_MaskInfoG; //!< G 成分のマスク情報です。
    RDdsMaskInfo m_MaskInfoB; //!< B 成分のマスク情報です。
    RDdsMaskInfo m_MaskInfoA; //!< A 成分のマスク情報です。
    uint32_t m_Caps; //!< ミップマップなどのフラグです。
    uint32_t m_Caps2; //!< キューブマップ／ボリュームなどのフラグです。

    bool m_IsDX10; //!< DX10 形式なら true です。
    uint32_t m_DxgiFormat; //!< DXGI フォーマットです。
    uint32_t m_ResourceDimension; //!< リソースの次元です。
    uint32_t m_MiscFlag; //!< 各種フラグです。
    uint32_t m_ArraySize; //!< 配列サイズです。

    bool m_Compressed; //!< 圧縮フォーマットなら true です。

public:
    //! フォーマットを取得します。
    RImage::Format GetFormat() const;

    //! 1 ピクセルのビット数を返します。
    int GetBitCount() const;
};

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのフォーマットを取得します。
//-----------------------------------------------------------------------------
RImage::Format RDdsInfo::GetFormat() const
{
    if ((m_PfFlags & DDPF_FOURCC) != 0)
    {
        switch (m_PixelFormat)
        {
        case DDS_DXT1:          return RImage::Format_Unorm_Bc1;
        case DDS_DXT2:          return RImage::Format_Unorm_Bc2;
        case DDS_DXT3:          return RImage::Format_Unorm_Bc2;
        case DDS_DXT4:          return RImage::Format_Unorm_Bc3;
        case DDS_DXT5:          return RImage::Format_Unorm_Bc3;
        case DDS_ATI2:          return RImage::Format_Unorm_Bc5;
        case DDS_R16F:          return RImage::Format_Float_16;
        case DDS_G16R16F:       return RImage::Format_Float_16_16;
        case DDS_A16B16G16R16F: return RImage::Format_Float_16_16_16_16;
        case DDS_R32F:          return RImage::Format_Float_32;
        case DDS_G32R32F:       return RImage::Format_Float_32_32;
        case DDS_A32B32G32R32F: return RImage::Format_Float_32_32_32_32;
        default:                break;
        }
    }
    else if (m_BitCount == 8)
    {
        if (m_MaskInfoR.m_Size == 4) return RImage::Format_Unorm_4_4;
        if (m_MaskInfoR.m_Size == 8) return RImage::Format_Unorm_8;
    }
    else if (m_BitCount == 16)
    {
        if (m_MaskInfoR.m_Size ==  4 && m_MaskInfoA.m_Size == 4) return RImage::Format_Unorm_4_4_4_4;
        if (m_MaskInfoR.m_Size ==  5 && m_MaskInfoG.m_Size == 6) return RImage::Format_Unorm_5_6_5;
        if (m_MaskInfoR.m_Size ==  5 && m_MaskInfoA.m_Size == 1) return RImage::Format_Unorm_5_5_5_1;
        if (m_MaskInfoR.m_Size ==  8) return RImage::Format_Unorm_8_8;
        //if (m_MaskInfoR.m_Size == 16) return RImage::Format_Unorm_16;
    }
    else if (m_BitCount == 24)
    {
        if (m_MaskInfoR.m_Size == 8) return RImage::Format_Unorm_8_8_8_8;
    }
    else if (m_BitCount == 32)
    {
        if (m_MaskInfoR.m_Size ==  8) return RImage::Format_Unorm_8_8_8_8;
        //if (m_MaskInfoR.m_Size == 10)
        //{
        //  return (m_MaskInfoR.m_Shift == 0) ?
        //      RImage::Format_Unorm_2_10_10_10 :
        //      RImage::Format_Unorm_10_10_10_2;
        //}
        //if (m_MaskInfoR.m_Size == 16) return RImage::Format_Unorm_16_16;
    }
    else if (m_BitCount == 64)
    {
        //if (m_MaskInfoR.m_Size == 16) return RImage::Format_Unorm_16_16_16_16;
    }
    return RImage::Format_Invalid;
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの 1 ピクセルのビット数を返します。
//-----------------------------------------------------------------------------
int RDdsInfo::GetBitCount() const
{
    if ((m_PfFlags & DDPF_FOURCC) != 0)
    {
        switch (m_PixelFormat)
        {
        default:
        case DDS_DXT1:          return 4;
        case DDS_DXT2:          return 8;
        case DDS_DXT3:          return 8;
        case DDS_DXT4:          return 8;
        case DDS_DXT5:          return 8;
        case DDS_ATI2:          return 8;
        case DDS_R16F:          return 16;
        case DDS_G16R16F:       return 32;
        case DDS_A16B16G16R16F: return 64;
        case DDS_R32F:          return 32;
        case DDS_G32R32F:       return 64;
        case DDS_A32B32G32R32F: return 128;
        }
    }
    else
    {
        return m_BitCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのヘッダをデコードします。
//!
//! @param[out] dimension 次元を格納します。
//! @param[out] imageW 画像の幅を格納します。
//! @param[out] imageH 画像の高さを格納します。
//! @param[out] imageD 画像の奥行きを格納します。
//! @param[out] mipLevel ミップマップのレベル数を格納します。
//! @param[out] hasAlpha アルファ成分の有無を格納します。
//! @param[out] isFloat 浮動小数点数型かどうかを格納します。
//! @param[out] ddsInfo DDS ファイル情報を格納します。
//! @param[in] fileBuf ファイルバッファです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static RStatus DecodeDdsHeader(
    RImage::Dimension& dimension,
    int& imageW,
    int& imageH,
    int& imageD,
    int& mipLevel,
    bool& hasAlpha,
    bool& isFloat,
    RDdsInfo& ddsInfo,
    const RFileBuf& fileBuf
)
{
    //-----------------------------------------------------------------------------
    // ヘッダを解析します。
    const uint8_t* pBuf = fileBuf.GetBuf();
    if (fileBuf.GetSize() < DDS_HEADER_SIZE)
    {
        return RStatus(RStatus::FAILURE, // RShowError: DDS file is wrong: (invalid size)
            "不正な DDS ファイルです:（不正なサイズ）: {0}",
            "DDS file is wrong: (invalid size): {0}", fileBuf.GetPath());
    }
    if (std::string(reinterpret_cast<const char*>(pBuf), 4) != "DDS ")
    {
        return RStatus(RStatus::FAILURE, // RShowError: DDS file is wrong: (invalid signature)
            "不正な DDS ファイルです:（不正なシグネチャ）: {0}",
            "DDS file is wrong: (invalid signature): {0}", fileBuf.GetPath());
    }
    ddsInfo.m_HeaderSize = RGetMemLittleUInt(pBuf + 0x04); // 124
    imageH = RGetMemLittleInt(pBuf + 0x0c);
    imageW = RGetMemLittleInt(pBuf + 0x10);
    imageD = RMax(RGetMemLittleInt(pBuf + 0x18), 1); // ボリューム以外は 0
    mipLevel = RMax(RGetMemLittleInt(pBuf + 0x1c), 1);

    ddsInfo.m_PfFlags     = RGetMemLittleUInt(pBuf + 0x50);
    ddsInfo.m_PixelFormat = RGetMemLittleUInt(pBuf + 0x54);
    ddsInfo.m_BitCount    = RGetMemLittleUInt(pBuf + 0x58);
    uint32_t maskR = RGetMemLittleUInt(pBuf + 0x5c);
    uint32_t maskG = RGetMemLittleUInt(pBuf + 0x60);
    uint32_t maskB = RGetMemLittleUInt(pBuf + 0x64);
    uint32_t maskA = RGetMemLittleUInt(pBuf + 0x68);
    ddsInfo.m_Caps  = RGetMemLittleUInt(pBuf + 0x6c);
    ddsInfo.m_Caps2 = RGetMemLittleUInt(pBuf + 0x70);

    //-----------------------------------------------------------------------------
    // DX10 ヘッダを解析します。
    // 可能なら DX9 形式でデコードできる状態のパラメータを設定します。
    ddsInfo.m_DxgiFormat = 0;
    ddsInfo.m_ResourceDimension = DDS_DIM_2D;
    ddsInfo.m_MiscFlag = 0;
    ddsInfo.m_ArraySize = 1;
    ddsInfo.m_IsDX10 = (ddsInfo.m_PfFlags & DDPF_FOURCC) != 0 && ddsInfo.m_PixelFormat == DDS_DX10;
    if (ddsInfo.m_IsDX10)
    {
        ddsInfo.m_DxgiFormat        = RGetMemLittleUInt(pBuf + 0x80);
        ddsInfo.m_ResourceDimension = RGetMemLittleUInt(pBuf + 0x84);
        ddsInfo.m_MiscFlag          = RGetMemLittleUInt(pBuf + 0x88);
        ddsInfo.m_ArraySize         = RGetMemLittleUInt(pBuf + 0x8C);
        if (ddsInfo.m_DxgiFormat == DXGI_R8G8B8A8_UNORM)
        {
            ddsInfo.m_PfFlags &= ~DDPF_FOURCC;
            ddsInfo.m_BitCount = 32;
            maskR = 0x000000ff;
            maskG = 0x0000ff00;
            maskB = 0x00ff0000;
            maskA = 0xff000000;
        }
        else if (ddsInfo.m_DxgiFormat == DXGI_R16G16B16A16_FLOAT)
        {
            ddsInfo.m_PixelFormat = DDS_A16B16G16R16F;
        }
        else if (ddsInfo.m_DxgiFormat == DXGI_R32G32B32A32_FLOAT)
        {
            ddsInfo.m_PixelFormat = DDS_A32B32G32R32F;
        }
    }

    //-----------------------------------------------------------------------------
    // 圧縮フォーマットフラグを設定します。
    ddsInfo.m_Compressed = (ddsInfo.m_PfFlags & DDPF_FOURCC) != 0 &&
        (ddsInfo.m_PixelFormat == DDS_DXT1 ||
         ddsInfo.m_PixelFormat == DDS_DXT2 ||
         ddsInfo.m_PixelFormat == DDS_DXT3 ||
         ddsInfo.m_PixelFormat == DDS_DXT4 ||
         ddsInfo.m_PixelFormat == DDS_DXT5 ||
         ddsInfo.m_PixelFormat == DDS_ATI2);

    //-----------------------------------------------------------------------------
    // フォーマットを取得します。
    ddsInfo.m_MaskInfoR = RDdsMaskInfo(maskR);
    ddsInfo.m_MaskInfoG = RDdsMaskInfo(maskG);
    ddsInfo.m_MaskInfoB = RDdsMaskInfo(maskB);
    ddsInfo.m_MaskInfoA = RDdsMaskInfo(maskA);
    const RImage::Format format = ddsInfo.GetFormat();
    isFloat = RImage::IsFloatFormat(format);
    const bool isBcFormat = RImage::IsBcFormat(format);
    if (format != RImage::Format_Unorm_8_8_8_8 &&
        !(isFloat && !isBcFormat))
    {
        return RStatus(RStatus::FAILURE, // RShowError: DDS file is wrong: (unsupported format)
            "サポート外のフォーマットの DDS ファイルです: {0} \n"
            "R8G8B8、A8R8G8B8、A16B16G16R16F、A32B32G32R32F フォーマットのみ対応しています。",
            "The DDS file uses an unsupported format: {0} \n"
            "Supported formats are only R8G8B8, A8R8G8B8, A16B16G16R16F, A32B32G32R32F.",
            fileBuf.GetPath());
    }

    //-----------------------------------------------------------------------------
    // 次元を取得します。
    if (!ddsInfo.m_IsDX10)
    {
        if ((ddsInfo.m_Caps2 & DDSCAPS2_CUBEMAP) != 0)
        {
            dimension = RImage::DIM_CUBE_INPUT;
            imageD = RImage::CUBE_FACE_COUNT;
        }
        else if ((ddsInfo.m_Caps2 & DDSCAPS2_VOLUME) != 0)
        {
            dimension = RImage::DIM_3D;
        }
        else if (imageD >= 2)
        {
            dimension = RImage::DIM_2D_ARRAY;
        }
        else
        {
            dimension = RImage::DIM_2D;
        }
    }
    else
    {
        if ((ddsInfo.m_MiscFlag & DDS_MISC_TEXTURECUBE) != 0)
        {
            dimension = RImage::DIM_CUBE_INPUT;
            imageD = RImage::CUBE_FACE_COUNT * ddsInfo.m_ArraySize;
        }
        else
        {
            imageD *= ddsInfo.m_ArraySize;
            if (ddsInfo.m_ResourceDimension == DDS_DIM_1D)
            {
                dimension = (imageD == 1) ? RImage::DIM_1D : RImage::DIM_1D_ARRAY;
            }
            else if (ddsInfo.m_ResourceDimension == DDS_DIM_2D)
            {
                dimension = (imageD == 1) ? RImage::DIM_2D : RImage::DIM_2D_ARRAY;
            }
            else // 3D
            {
                dimension = RImage::DIM_3D;
            }
        }
    }
    //cerr << "dds dim: " << RImage::GetDimensionName(dimension) << endl;

    //-----------------------------------------------------------------------------
    // アルファ成分の有無を取得します。
    if ((ddsInfo.m_PfFlags & DDPF_FOURCC) != 0)
    {
        hasAlpha = (
            (ddsInfo.m_Compressed && ddsInfo.m_PixelFormat != DDS_ATI2) ||
            ddsInfo.m_PixelFormat == DDS_A16B16G16R16F                  ||
            ddsInfo.m_PixelFormat == DDS_A32B32G32R32F);
    }
    else
    {
        hasAlpha = (ddsInfo.m_MaskInfoA.m_Size != 0);
    }

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのカラー成分を 8bit に変換した値を返します。
//-----------------------------------------------------------------------------
static uint8_t GetDds8Bit(const uint32_t val, const int bitSize)
{
    if      (bitSize ==  8) return static_cast<uint8_t>(val);
    else if (bitSize ==  6) return static_cast<uint8_t>((val << 2) | (val >> 4));
    else if (bitSize ==  5) return static_cast<uint8_t>((val << 3) | (val >> 2));
    else if (bitSize ==  4) return static_cast<uint8_t>((val << 4) | val);
    else if (bitSize ==  1) return static_cast<uint8_t>(val * 0xff);
    else if (bitSize ==  0) return 0x00;
    else if (bitSize == 16) return static_cast<uint8_t>(val >> 8);
    else if (bitSize == 10) return static_cast<uint8_t>(val >> 2);
    else                    return static_cast<uint8_t>(val);
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの float16 を float32 に変換して返します。
//-----------------------------------------------------------------------------
static float GetDdsF16(const uint8_t* pMemory)
{
    const uint32_t iv = (pMemory[1] << 8) | pMemory[0];
    if (iv == 0)
    {
        return 0.0f;
    }
    const uint32_t s =   iv & 0x8000;
    const uint32_t e = ((iv & 0x7c00) >> 10) - 15 + 127;
    const uint32_t f =   iv & 0x03ff;
    const uint32_t fv = (s << 16) | ((e << 23) & 0x7f800000) | (f << 13);
    return *reinterpret_cast<const float*>(&fv);
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのビットマップの 1 面をデコードします。
//!
//! @param[out] pDst デコードしたデータを格納するバッファのポインタです。
//!                  curW * curH * (4 or 16) のサイズが必要です。
//! @param[in] pSrc 入力データのバッファのポインタです。
//! @param[in] curW 1 面の幅です。
//! @param[in] curH 1 面の高さです。
//! @param[in] ddsInfo DDS ファイル情報です。
//!
//! @return インクリメントした入力データのバッファのポインタを返します。
//-----------------------------------------------------------------------------
static const uint8_t* DecodeDdsOneFace(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int curW,
    const int curH,
    const RDdsInfo& ddsInfo
)
{
    if (ddsInfo.m_Compressed)
    {
        //-----------------------------------------------------------------------------
        // compressed
        // 現在は非対応です。
    }
    else if ((ddsInfo.m_PfFlags & DDPF_FOURCC) != 0)
    {
        //-----------------------------------------------------------------------------
        // float
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (int iy = 0; iy < curH; ++iy)
        {
            for (int ix = 0; ix < curW; ++ix)
            {
                float fr = 0.0f;
                float fg = 0.0f;
                float fb = 0.0f;
                float fa = 1.0f;
                if (ddsInfo.m_PixelFormat == DDS_R16F)
                {
                    fr = GetDdsF16(pSrc + 0);
                    pSrc += sizeof(uint16_t);
                }
                else if (ddsInfo.m_PixelFormat == DDS_G16R16F)
                {
                    fr = GetDdsF16(pSrc + 0);
                    fg = GetDdsF16(pSrc + 2);
                    pSrc += 2 * sizeof(uint16_t);
                }
                else if (ddsInfo.m_PixelFormat == DDS_A16B16G16R16F)
                {
                    fr = GetDdsF16(pSrc + 0);
                    fg = GetDdsF16(pSrc + 2);
                    fb = GetDdsF16(pSrc + 4);
                    fa = GetDdsF16(pSrc + 6);
                    pSrc += R_RGBA_COUNT * sizeof(uint16_t);
                }
                else if (ddsInfo.m_PixelFormat == DDS_R32F)
                {
                    RGetMemLittle(fr, pSrc +  0);
                    pSrc += sizeof(float);
                }
                else if (ddsInfo.m_PixelFormat == DDS_G32R32F)
                {
                    RGetMemLittle(fr, pSrc +  0);
                    RGetMemLittle(fg, pSrc +  4);
                    pSrc += 2 * sizeof(float);
                }
                else if (ddsInfo.m_PixelFormat == DDS_A32B32G32R32F)
                {
                    RGetMemLittle(fr, pSrc +  0);
                    RGetMemLittle(fg, pSrc +  4);
                    RGetMemLittle(fb, pSrc +  8);
                    RGetMemLittle(fa, pSrc + 12);
                    pSrc += R_RGBA_COUNT * sizeof(float);
                }
                *pDstF32++ = fr;
                *pDstF32++ = fg;
                *pDstF32++ = fb;
                *pDstF32++ = fa;
            }
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // RGB, ALPHA, LUMINANCE
        // 64 bps 以上は未対応
        const bool isLuminance = ((ddsInfo.m_PfFlags & DDPF_LUMINANCE) != 0);
        for (int iy = 0; iy < curH; ++iy)
        {
            for (int ix = 0; ix < curW; ++ix)
            {
                uint32_t val;
                if (ddsInfo.m_BitCount == 8)
                {
                    val = *pSrc++;
                }
                else if (ddsInfo.m_BitCount == 16)
                {
                    val = RGetMemLittleUShort(pSrc);
                    pSrc += sizeof(uint16_t);
                }
                else if (ddsInfo.m_BitCount == 24)
                {
                    val = RGetMemLittleUShort(pSrc);
                    val |= pSrc[2] << 16;
                    pSrc += 3;
                }
                else // 32
                {
                    val = RGetMemLittleUInt(pSrc);
                    pSrc += sizeof(uint32_t);
                }
                uint8_t cr = GetDds8Bit((val & ddsInfo.m_MaskInfoR.m_Mask) >> ddsInfo.m_MaskInfoR.m_Shift, ddsInfo.m_MaskInfoR.m_Size);
                uint8_t cg = GetDds8Bit((val & ddsInfo.m_MaskInfoG.m_Mask) >> ddsInfo.m_MaskInfoG.m_Shift, ddsInfo.m_MaskInfoG.m_Size);
                uint8_t cb = GetDds8Bit((val & ddsInfo.m_MaskInfoB.m_Mask) >> ddsInfo.m_MaskInfoB.m_Shift, ddsInfo.m_MaskInfoB.m_Size);
                uint8_t ca = 0xff;
                if (ddsInfo.m_MaskInfoA.m_Mask != 0)
                {
                    ca = GetDds8Bit((val & ddsInfo.m_MaskInfoA.m_Mask) >> ddsInfo.m_MaskInfoA.m_Shift, ddsInfo.m_MaskInfoA.m_Size);
                }
                if (isLuminance)
                {
                    cg = cb = cr;
                }
                *pDst++ = cr;
                *pDst++ = cg;
                *pDst++ = cb;
                *pDst++ = ca;
            }
        }
    }
    return pSrc;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのビットマップをデコードします。
//!
//! @param[out] pDstBuf ビットマップバッファのポインタです。
//! @param[in] dimension 次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] mipLevel ミップマップのレベル数です。
//! @param[in] isFloat 浮動小数点数型かどうかです。
//! @param[in] ddsInfo DDS ファイル情報です。
//! @param[in] fileBuf ファイルバッファです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static RStatus DecodeDdsBitmap(
    uint8_t* pDstBuf,
    const RImage::Dimension dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipLevel,
    const bool isFloat,
    const RDdsInfo& ddsInfo,
    const RFileBuf& fileBuf
)
{
    const int colBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const uint8_t* pSrc = fileBuf.GetBuf() + 4 + ddsInfo.m_HeaderSize;
    if (ddsInfo.m_IsDX10)
    {
        pSrc += DDS_DX10_HEADER_SIZE;
    }

    if (dimension == RImage::DIM_3D)
    {
        uint8_t* pDst = reinterpret_cast<uint8_t*>(pDstBuf);
        for (int level = 0; level < 1; ++level) // 最上位レベルのみ取得します。
        {
            const int curW = RMax(imageW >> level, 1);
            const int curH = RMax(imageH >> level, 1);
            const int curD = (dimension == RImage::DIM_3D) ?
                RMax(imageD >> level, 1) : imageD;
            for (int iz = 0; iz < curD; ++iz)
            {
                pSrc = DecodeDdsOneFace(pDst, pSrc, curW, curH, ddsInfo);
                pDst += curW * curH * colBytes;
            }
        }
    }
    else
    {
        for (int iz = 0; iz < imageD; ++iz)
        {
            for (int level = 0; level < mipLevel; ++level)
            {
                const int curW = RMax(imageW >> level, 1);
                const int curH = RMax(imageH >> level, 1);
                const int faceBytes = curW * curH * colBytes;

                int iDstFace = iz;
                if (dimension == RImage::DIM_CUBE_INPUT)
                {
                    // DDS は左手座標系で +X、-X、+Y、-Y、+Z、-Z 面の順なので、
                    // 右手座標系で +X、-X、+Y、-Y、+Z、-Z 面の順になるように格納します。
                    int iCubeFace = iz % RImage::CUBE_FACE_COUNT;
                    if      (iCubeFace == 4) iCubeFace = 5;
                    else if (iCubeFace == 5) iCubeFace = 4;
                    iDstFace = (iz / RImage::CUBE_FACE_COUNT) * RImage::CUBE_FACE_COUNT + iCubeFace;
                }

                if (level == 0) // 最上位レベルのみ取得します。
                {
                    uint8_t* pDst = reinterpret_cast<uint8_t*>(pDstBuf) + iDstFace * faceBytes;
                    pSrc = DecodeDdsOneFace(pDst, pSrc, curW, curH, ddsInfo);
                }
                else
                {
                    pSrc += curW * curH * ddsInfo.GetBitCount() / 8;
                }
            }
        }
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief マージする ftx ファイルから情報（ヒント情報、リニア変換フラグ、
//!        次元、フォーマット、ミップマップ、初期スウィズル値）を取得します。
//-----------------------------------------------------------------------------
RStatus RImage::GetInfoFromMergeFile()
{
    //-----------------------------------------------------------------------------
    // ファイルの先頭部分をリードします。
    std::ifstream ifs(m_MergePath.c_str(), ios_base::binary);
    if (!ifs)
    {
        return RStatus(RStatus::FAILURE, // RShowError: Cannot open file
            "ファイルを開けません: {0}",
            "The file cannot be opened: {0}", m_MergePath);
    }

    const int READ_BUF_SIZE = 1024;
    char* readBuf = new char[READ_BUF_SIZE];
    ifs.read(readBuf, READ_BUF_SIZE);
    std::string header(readBuf, static_cast<size_t>(ifs.gcount()));
    ifs.close();
    delete[] readBuf;

    //-----------------------------------------------------------------------------
    // <texture_info> 要素を取得します。
    bool isValid = false;
    const size_t infoOfs = header.find("<texture_info");
    if (infoOfs != std::string::npos)
    {
        header = header.substr(infoOfs);
        const size_t infoEndOfs = header.find(">");
        if (infoEndOfs != std::string::npos)
        {
            header = header.substr(0, infoEndOfs + 1);
            //cerr << "ftx header: [" << header << "]" << endl;
            isValid = true;
        }
    }

    //-----------------------------------------------------------------------------
    // <texture_info> 要素を解析します。
    if (isValid)
    {
        //-----------------------------------------------------------------------------
        // 次元
        const Dimension dimension = GetDimensionFromName(RGetXmlAttr(header, "dimension"));
        if (!IsCubeDimension(m_Dimension) &&
            !IsCubeDimension(dimension))
        {
            m_Dimension = dimension;
        }

        //-----------------------------------------------------------------------------
        // フォーマット
        const Format format = GetFormatFromName(RGetXmlAttr(header, "quantize_type"));
        if (format != Format_Invalid)
        {
            m_Format = format;
        }
        m_WeightedCompress = (RGetXmlAttr(header, "weighted_compress") == "true");

        //-----------------------------------------------------------------------------
        // ミップマップ
        const int mergeW = atol(RGetXmlAttr(header, "width" ).c_str());
        const int mergeH = atol(RGetXmlAttr(header, "height").c_str());
        const int mergeMipLevel = atol(RGetXmlAttr(header, "mip_level").c_str());

        m_MipLevel = 1;
        if (mergeMipLevel >= 2)
        {
            if (m_ImageW != 0 && m_ImageH != 0)
            {
                // マージする ftx ファイルの mip_level が 2 以上の場合、
                // 最小レベルの幅と高さが、マージするファイルの最小レベルの幅と高さ以下になる
                // レベルまでミップマップを作成します。
                int curW = m_ImageW;
                int curH = m_ImageH;
                const int minShift = (mergeMipLevel - 1);
                const int minW = RMax(mergeW >> minShift, 1);
                const int minH = RMax(mergeH >> minShift, 1);
                while (curW > minW || curH > minH)
                {
                    ++m_MipLevel;
                    curW >>= 1;
                    curH >>= 1;
                }
            }
            else
            {
                // 幅と高さを未取得の場合（サポート外の画像ファイルタイプの場合）は
                // ミップマップ数を最大とします。
                m_MipLevel = MipCountMax;
            }
        }
        //cerr << "merge mip: " << m_MipLevel << endl;

        //-----------------------------------------------------------------------------
        // ヒント情報
        const std::string defaultHint = (m_Format == Format_Snorm_Bc5) ?
            HINT_NORMAL : "";
        m_Hint = RGetXmlAttr(header, "hint", defaultHint.c_str()); // 2.2.0 で追加

        //-----------------------------------------------------------------------------
        // リニア変換フラグ
        const std::string linearStr = RGetXmlAttr(header, "linear"); // 2.2.0 で追加
        m_LinearFlag = (!linearStr.empty()) ? GetLinearFlagFromString(linearStr) : LINEAR_NONE;

        //-----------------------------------------------------------------------------
        // 初期スウィズル値
        const std::string initialSwizzle = RGetXmlAttr(header, "initial_swizzle"); // 2.2.0 で追加
        m_InitialSwizzle = (!initialSwizzle.empty()) ? atoi(initialSwizzle.c_str()) : 0;
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ビットマップの内容からテクスチャーイメージデータを設定します。
//!        次元、用途、ビットマップを設定してから呼びます。
//-----------------------------------------------------------------------------
RStatus RImage::SetImageData()
{
    //-----------------------------------------------------------------------------
    // 幅と高さをチェックします。
    if (m_ImageW < WidthHeightMin ||
        m_ImageH < WidthHeightMin ||
        m_ImageW > WidthHeightMax ||
        m_ImageH > WidthHeightMax)
    {
        const std::string sizeErrInfo = GetMainFilePath() + " (" +
            RGetNumberString(m_ImageW) + " x " + RGetNumberString(m_ImageH) + ")";
        return RStatus(RStatus::FAILURE, // RShowError: Texture size is wrong
            "テクスチャー画像の幅または高さがサポート外のサイズです: {0}",
            "The texture width or height is unsupported size: {0}", sizeErrInfo);
    }

    //-----------------------------------------------------------------------------
    // 透明モードを取得します。
    if (m_Hint == HINT_NORMAL)
    {
        m_TransparencyMode = OPA;
    }
    else
    {
        m_TransparencyMode = GetBitmapTransparencyMode(m_pFullBitmap, m_FullW,
            m_ImageW, m_ImageH, m_ImageD, m_Dimension, m_HasAlpha, m_IsFloat);
    }

    //-----------------------------------------------------------------------------
    // マージする ftx ファイルから情報を取得します。
    if (!m_MergePath.empty())
    {
        return GetInfoFromMergeFile();
    }

    //-----------------------------------------------------------------------------
    // フォーマットを設定します。
    if (m_Hint == HINT_NORMAL)
    {
        if (m_Dimension == DIM_1D       ||
            m_Dimension == DIM_1D_ARRAY)
        {
            m_Format = Format_Snorm_8_8;
        }
        else
        {
            m_Format = (m_IsFloat) ? Format_Float_16_16_16_16 : Format_Snorm_Bc5;
        }
    }
    else
    {
        SetFormatFromBitmap();
    }

    //-----------------------------------------------------------------------------
    // ミップマップは 1 x 1 のレベルまで作成します。
    m_MipLevel = RImage::GetMaxMipLevel(m_ImageW, m_ImageH);

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief image :: parse AI texel format
//-----------------------------------------------------------------------------
RStatus RImage::ParseAITexelFormat(const uint8_t* dataBuf, const uint32_t dataSize)
{
    const int format = GetTexelFormatFromAIData(dataBuf, dataSize);
    if (format != -1)
    {
        m_Format = (Format)format;
        return RStatus::SUCCESS;
    }
    else
    {
        return RStatus(RStatus::FAILURE, // RShowError: Texture format is wrong // 現在は発生しない
            "Texture format is wrong: " + std::string(reinterpret_cast<const char*>(dataBuf), dataSize));
    }
}

//-----------------------------------------------------------------------------
//! @brief image :: parse AI cube map
//-----------------------------------------------------------------------------
RStatus RImage::ParseAICubeMap(const uint8_t* dataBuf, const uint32_t dataSize)
{
    R_UNUSED_VARIABLE(dataBuf);
    R_UNUSED_VARIABLE(dataSize);

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief image :: parse AI mipmap level
//-----------------------------------------------------------------------------
RStatus RImage::ParseAIMipLevel(const uint8_t* dataBuf, const uint32_t dataSize)
{
    m_MipLevel = GetMipLevelFromAIData(dataBuf, dataSize);
    if (m_MipLevel <= 0 || m_MipLevel > MipCountMax)
    {
        return RStatus(RStatus::FAILURE, // RShowError: Mipmap size is wrong // 現在は発生しない
            "Mipmap size is wrong: " + std::string(reinterpret_cast<const char*>(dataBuf), dataSize));
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief image :: parse AI texel data
//-----------------------------------------------------------------------------
RStatus RImage::ParseAITexelData(const uint8_t* dataBuf, const uint32_t dataSize)
{
    m_TexelDataSize = dataSize;
    m_pTexelData = new uint8_t[m_TexelDataSize];
    memcpy(m_pTexelData, dataBuf, dataSize);
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief image :: decode add info
//-----------------------------------------------------------------------------
RStatus RImage::DecodeAddInfo(const uint8_t* fileBuf, const int fileSize)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // set default
    m_ImageW = m_FullW;
    m_ImageH = m_FullH;

    //-----------------------------------------------------------------------------
    // parse data block
    const uint8_t* dataPtr = fileBuf + m_AddInfoOfs;
    const uint32_t totalDataSize = fileSize - m_AddInfoOfs;
    const uint8_t* const dataPtrEnd = dataPtr + totalDataSize;
    while (dataPtr <= dataPtrEnd - AI_HEADER_SIZE)
    {
        uint32_t dataBlockSize;
        RGetMemLittle(dataBlockSize, dataPtr + AI_TAG_SIZE);
        if (dataBlockSize == 0 || dataBlockSize > totalDataSize)
        {
            break;
        }

        // check end block
        if (strncmp(reinterpret_cast<const char*>(dataPtr), AI_TAG_END_BLOCK, AI_TAG_SIZE) == 0)
        {
            break;
        }
        //cerr << "ai parse: [" << reinterpret_cast<char*>(dataPtr) << "] (" << hex << static_cast<int>(dataBlockSize) << dec << ")" << R_ENDL;

        const char* tagName = reinterpret_cast<const char*>(dataPtr);
        const uint8_t* valueTop = dataPtr + AI_HEADER_SIZE;
        const uint32_t valueSize = dataBlockSize - AI_HEADER_SIZE;
        if (strncmp(tagName, AI_TAG_TEXEL_FORMAT, AI_TAG_SIZE) == 0)
        {
            status = ParseAITexelFormat(valueTop, valueSize);
        }
        else if (strncmp(tagName, AI_TAG_CUBE_MAP, AI_TAG_SIZE) == 0)
        {
            status = ParseAICubeMap(valueTop, valueSize);
        }
        else if (strncmp(tagName, AI_TAG_MIP_SIZE, AI_TAG_SIZE) == 0)
        {
            status = ParseAIMipLevel(valueTop, valueSize);
        }
        else if (strncmp(tagName, AI_TAG_TEXEL_DATA, AI_TAG_SIZE) == 0)
        {
            status = ParseAITexelData(valueTop, valueSize);
        }
        RCheckStatus(status);
        dataPtr += dataBlockSize;
    }

    //-----------------------------------------------------------------------------
    // check data pointer
    if (m_pTexelData == NULL)
    {
        return RStatus(RStatus::FAILURE, // RShowError: Additional information is wrong // 現在は発生しない
            "Additional information is wrong: " + GetMainFilePath());
    }

    //-----------------------------------------------------------------------------
    // set size
    m_ImageW = m_FullW;
    m_ImageH = m_FullH;

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 画像ファイルのパス配列を設定します。
//!        パスからテクスチャー名も設定されます。
//!        次元を設定してから呼びます。
//-----------------------------------------------------------------------------
void RImage::SetFilePaths(const RStringArray& filePaths)
{
    m_FilePaths = filePaths;
    m_OrgName = RGetNoExtensionFilePath(RGetFileNameFromFilePath(GetMainFilePath()));
    m_Name = RAdjustElementNameString(m_OrgName);
}

//-----------------------------------------------------------------------------
//! @brief 画像ファイルタイプを返します。
//-----------------------------------------------------------------------------
static RImage::FileType GetImageFileType(const std::string& filePath)
{
    const std::string ext = RGetExtensionFromFilePath(filePath);
    if (ext == TgaExtension)
    {
        return RImage::TGA;
    }
    else if (ext == DdsExtension)
    {
        return RImage::DDS;
    }
    else
    {
        return RImage::FILE_OTHER;
    }
}

//-----------------------------------------------------------------------------
//! @brief 画像ファイルをリードしてデータを取得します。
//-----------------------------------------------------------------------------
RStatus RImage::ReadFile(
    const RStringArray& filePaths,
    const std::string& mergePath,
    const std::string& dccPreset,
    const std::string& hint,
    const int linearFlag,
    const bool usesSrgbFetch,
    const Dimension dimension,
    const int initialSwizzle,
    const std::string& comment
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // ファイルパスの数をチェックします。
    if (filePaths.size() == 0)
    {
        return RStatus(RStatus::FAILURE, // RShowError: File path array is empty // 通常は発生しない
            "ファイルパス配列が空です。",
            "File path array is empty.");
    }

    //-----------------------------------------------------------------------------
    // 初期化します。
    Init(dccPreset, hint, linearFlag, usesSrgbFetch, dimension, initialSwizzle, comment.c_str());
    SetFilePaths(filePaths);
    m_MergePath = mergePath;

    const bool isSpecialCube = (
        dimension == DIM_CUBE_SEPARATE ||
        dimension == DIM_CUBE_HC       ||
        dimension == DIM_CUBE_VC);

    //-----------------------------------------------------------------------------
    // 最初のファイルをリードします。
    const std::string& firstFilePath = m_FilePaths[0];
    m_FileType = GetImageFileType(firstFilePath);
    if (m_FileType == FILE_OTHER)
    {
        // サポート外の画像ファイルタイプの場合、ミップマップありとみなし、
        // マージする ftx ファイルが存在すれば情報を取得します。
        m_MipLevel = MipCountMax;
        if (!m_MergePath.empty())
        {
            GetInfoFromMergeFile();
        }
        return RStatus(RStatus::FAILURE, // RShowError: Image type is not supported
            "サポートしていない画像ファイルが使用されています: {0}",
            "An unsupported image file is being used: {0}",
            firstFilePath);
    }

    RFileBuf fileBuf(firstFilePath);
    if (!fileBuf)
    {
        return GetTexFileOpenFailureStatus(firstFilePath); // RShowError: Cannot open file
    }

    //-----------------------------------------------------------------------------
    // 最初のファイルのヘッダをデコードします。
    RDdsInfo ddsInfo;
    if (m_FileType == TGA)
    {
        status = DecodeTgaHeader(m_FullW, m_FullH, m_HasAlpha, m_AddInfoOfs, fileBuf);
    }
    else
    {
        Dimension ddsDim;
        status = DecodeDdsHeader(ddsDim, m_FullW, m_FullH, m_ImageD,
            m_MipLevel, m_HasAlpha, m_IsFloat, ddsInfo, fileBuf);
        if (!isSpecialCube)
        {
            m_Dimension = ddsDim;
        }
    }
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // イメージの幅、高さ、奥行きを設定します。
    SetImageWHD();

    //-----------------------------------------------------------------------------
    // ビットマップをデコードします。
    if (m_AddInfoOfs == 0)
    {
        //-----------------------------------------------------------------------------
        // 最初のファイルのビットマップをデコードします。
        const size_t colBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
        const size_t fullFaceBytes = colBytes * m_FullW * m_FullH;
        const int fileCount = static_cast<int>(m_FilePaths.size());
        const size_t fullBitmapBytes = (isSpecialCube) ?
             fullFaceBytes * fileCount :
             fullFaceBytes * m_ImageD;
        m_pFullBitmap = new uint8_t[fullBitmapBytes];
        if (m_FileType == TGA)
        {
            DecodeTgaBitmap(m_pFullBitmap, m_FullW, m_FullH, fileBuf);
        }
        else
        {
            const int fullD = (isSpecialCube) ? 1 : m_ImageD;
            DecodeDdsBitmap(m_pFullBitmap, m_Dimension, m_FullW, m_FullH, fullD,
                m_MipLevel, m_IsFloat, ddsInfo, fileBuf);
        }

        //-----------------------------------------------------------------------------
        // 2 番目以降のファイルをデコードします。
        for (int iFile = 1; iFile < fileCount; ++iFile)
        {
            //-----------------------------------------------------------------------------
            // 2 番目以降のファイルをリードします。
            const std::string& otherPath = m_FilePaths[iFile];
            const RImage::FileType otherType = GetImageFileType(otherPath);
            if (otherType == FILE_OTHER)
            {
                return RStatus(RStatus::FAILURE, // RShowError: Image type is not supported
                    "サポートしていない画像ファイルが使用されています: {0}",
                    "An unsupported image file is being used: {0}",
                    otherPath);
            }
            RFileBuf otherBuf(otherPath);
            if (!otherBuf)
            {
                return GetTexFileOpenFailureStatus(otherPath); // RShowError: Cannot open file
            }

            //-----------------------------------------------------------------------------
            // 2 番目以降のファイルのヘッダをデコードします。
            Dimension otherDim = RImage::DIM_2D;
            int otherW;
            int otherH;
            int otherD = 1;
            int otherMipLevel = 1;
            bool otherHasAlpha;
            bool otherIsFloat = false;
            RDdsInfo otherDdsInfo;
            int otherAddInfoOfs = 0;
            if (otherType == TGA)
            {
                status = DecodeTgaHeader(otherW, otherH, otherHasAlpha, otherAddInfoOfs, otherBuf);
            }
            else
            {
                status = DecodeDdsHeader(otherDim, otherW, otherH, otherD,
                    otherMipLevel, otherHasAlpha, otherIsFloat, otherDdsInfo, otherBuf);
            }
            RCheckStatus(status);
            if (otherW != m_FullW || otherH != m_FullH)
            {
                return RStatus(RStatus::FAILURE, // RShowError: Cube map width or height is not identical
                    "キューブマップの各面の幅と高さがテクスチャーファイルによって異なります。"
                    "幅と高さが同じテクスチャーファイルを使用してください: {0}",
                    "The width or height of the cube map differs depending on the texture file. "
                    "Make sure that you use texture files that have the same width and height: {0}",
                    otherPath);
            }
            if (otherIsFloat != m_IsFloat)
            {
                return RStatus(RStatus::FAILURE, // RShowError: Cube map format is different from other face
                    "キューブマップの各面のフォーマット（整数型か実数型か）がテクスチャーファイルによって異なります。"
                    "フォーマットが同じテクスチャーファイルを使用してください: {0}",
                    "The format (integer or real number type) used for each face of a cube map differs by texture file. "
                    "Use texture files that all have the same format: {0}",
                    otherPath);
            }
            if (otherHasAlpha) // 1 つでもアルファを使用するファイルがあればアルファ使用フラグを ON にします。
            {
                m_HasAlpha = true;
            }

            //-----------------------------------------------------------------------------
            // 2 番目以降のビットマップをデコードします。
            uint8_t* pOtherBitmap = m_pFullBitmap + iFile * fullFaceBytes;
            if (otherType == TGA)
            {
                DecodeTgaBitmap(pOtherBitmap, m_FullW, m_FullH, otherBuf);
            }
            else
            {
                DecodeDdsBitmap(pOtherBitmap, otherDim, m_FullW, m_FullH, 1,
                    otherMipLevel, otherIsFloat, otherDdsInfo, otherBuf);
            }
        }

        //-----------------------------------------------------------------------------
        // write TGA test
        //const std::string writePath = "D:/Download/" + RGetNoExtensionFilePath(RGetFileNameFromFilePath(GetMainFilePath())) + ".tga";
        //WriteTga(writePath);
    }

    //-----------------------------------------------------------------------------
    // decode add info
    #ifdef USE_TEX_ADD_INFO_SW
    if (m_AddInfoOfs != 0)
    {
        return DecodeAddInfo(fileBuf.GetBuf(), fileBuf.GetSize());
    }
    #endif

    //-----------------------------------------------------------------------------
    // set image data
    return SetImageData();
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief メモリ上のビットマップをリードしてデータを取得します。
//-----------------------------------------------------------------------------
void RImage::ReadBitmap(
    const RStringArray& filePaths,
    const std::string& mergePath,
    const std::string& dccPreset,
    const std::string& hint,
    const int linearFlag,
    const bool usesSrgbFetch,
    const Dimension dimension,
    const int initialSwizzle,
    const std::string& comment,
    const uint8_t* pBitmap,
    const int fullW,
    const int fullH,
    const bool hasAlpha,
    const bool isFloat
)
{
    Init(dccPreset, hint, linearFlag, usesSrgbFetch, dimension, initialSwizzle, comment.c_str());
    SetFilePaths(filePaths);
    m_MergePath = mergePath;

    m_FullW = fullW;
    m_FullH = fullH;
    m_HasAlpha = hasAlpha;
    m_IsFloat = isFloat;

    SetImageWHD();

    const size_t colBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t fullFaceBytes = colBytes * m_FullW * m_FullH;
    const int fullD = (dimension == DIM_CUBE_HC || dimension == DIM_CUBE_VC) ?
        1 : m_ImageD;
    const size_t fullBitmapBytes = fullFaceBytes * fullD;
    m_pFullBitmap = new uint8_t[fullBitmapBytes];
    memcpy(m_pFullBitmap, pBitmap, fullBitmapBytes);

    SetImageData();
}

//-----------------------------------------------------------------------------
//! @brief 3D テクスチャーコンバーターのオプション文字列を返します。
//-----------------------------------------------------------------------------
std::string RImage::GetConverterOption(const std::string& ftxPath) const
{
    std::string opt;

    //-----------------------------------------------------------------------------
    // 入力ファイルのパス指定です。
    if (m_Dimension == DIM_CUBE_SEPARATE)
    {
        opt = "\""                   + m_FilePaths[CUBE_FACE_PZ] +
            "\" --cubemap-back=\""   + m_FilePaths[CUBE_FACE_NZ] +
            "\" --cubemap-left=\""   + m_FilePaths[CUBE_FACE_NX] +
            "\" --cubemap-right=\""  + m_FilePaths[CUBE_FACE_PX] +
            "\" --cubemap-top=\""    + m_FilePaths[CUBE_FACE_PY] +
            "\" --cubemap-bottom=\"" + m_FilePaths[CUBE_FACE_NY] +
            "\"";
    }
    else
    {
        opt = "\"" + m_FilePaths[0] + "\"";
        if (m_Dimension == DIM_CUBE_HC ||
            m_Dimension == DIM_CUBE_VC)
        {
            opt += " --dimension=\"cube\"";
        }
    }

    //-----------------------------------------------------------------------------
    // 出力ファイルのパス指定です。
    opt += " --output=\"" + ftxPath + "\"";

    //-----------------------------------------------------------------------------
    // フォーマットなどの指定です。
    if (!m_ProjectRootPath.empty())
    {
        opt += " --project-root=\"" + m_ProjectRootPath + "\"";
    }
    if (!m_DccPreset.empty())
    {
        opt += " --dcc-preset=\"" + m_DccPreset + "\"";
    }

    if (m_MergePath.empty())
    {
        opt += " --hint=\"" + m_Hint + "\"";
        if (m_LinearFlag != LINEAR_NONE)
        {
            opt += " --linear=\"" + GetOptStringFromLinearFlag(m_LinearFlag) + "\"";
        }

        opt += " --format=\"" + GetFormatName() +
            "\" --mip-level=\"" + RGetNumberString(m_MipLevel) + "\"";

        if (!m_CompSel.empty())
        {
            opt += " --comp-sel=\"" + m_CompSel + "\"";
        }
        else if (m_Hint == HINT_OPACITY && GetCompSize(m_Format) == 1)
        {
            opt += " --comp-sel=\"111r\"";
        }

        opt += " --weighted-compress=\"" + std::string(RBoolStr(m_WeightedCompress)) + "\"";
        opt += " --swizzle=\"" + RGetNumberString(m_InitialSwizzle & INITIAL_SWIZZLE_MASK) + "\"";
    }
    else
    {
        opt += " --merge=\"" + m_MergePath + "\"";
    }

    if (!m_CommentText.empty())
    {
        opt += " --comment=\"" + m_CommentText + "\"";
    }

    return opt;
}

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

