﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NintendoExportMax.h"
#include "NodeTree.h"

#include "DccShape.h"

namespace Dcc = nn::gfx::tool::dcc;
using namespace nn::gfx::tool::dcc::utility;

typedef vector<pair<TriObject*, bool>> TriBoolPairVector;

//	TODO: 置き場所を考えたい。
Dcc::RVtxMtxArray	g_VtxMtxs;

static int AppendVtxMtx( const Dcc::RVtxMtx& vtxMtx )
{
    //-----------------------------------------------------------------------------
    // vtxMtx に渡されたデータが既に追加されていないか確認します。
    const int vmtCount = static_cast<int>(g_VtxMtxs.size());
    for (int iVmt = 0; iVmt < vmtCount; ++iVmt)
    {
        // 既に追加済みのデータであればそのデータの配列内のインデックスを返します。
        if (g_VtxMtxs[iVmt] == vtxMtx)
        {
            return iVmt;
        }
    }

    //-----------------------------------------------------------------------------
    // まだ追加されていないデータであれば、ymodel.m_VtxMtxs に追加します。
    g_VtxMtxs.push_back(vtxMtx);

    // 追加されたデータのインデックスを返します。
    return static_cast<int>(g_VtxMtxs.size()) - 1;
}


void killNaN(Point3& p)
{
    for(int i = 0; i < 3; i++)
    {
        if(_isnan(p[i]))
        {
            p = Point3(.0f,.0f,.0f);
            return;
        }
    }
}

void killNaN(Quat& q)
{
    for(int i = 0; i < 4; i++)
    {
        if(_isnan(q[i]))
        {
            q = Quat();
            return;
        }
    }
}

void killNaN(float& f)
{
    if(_isnan(f))
    {
        f = 0.0f;
    }
}

void killNaN(Matrix3& tm)
{
    Point3 s;
    Quat q;
    SpectralDecomp(tm, s, q);
    AffineParts ap;
    decomp_affine(tm, &ap);
    killNaN(ap.t);
    killNaN(ap.q);
    killNaN(ap.u);
    killNaN(ap.k);
    killNaN(ap.f);

    Matrix3 srtm, rtm, ptm, stm, ftm;
    ptm.IdentityMatrix();
    ptm.SetTrans(ap.t);
    ap.q.MakeMatrix(rtm);
    ap.u.MakeMatrix(srtm);
    stm = ScaleMatrix(ap.k);
    ftm = ScaleMatrix(Point3(ap.f,ap.f,ap.f));
    tm = Inverse(srtm) * stm * srtm * rtm * ftm * ptm;
}

// 0に近いスケール値を適当な小さな値を入れる。
Matrix3 kill0scale(Matrix3 tm)
{
    //Point3 s;
    //Quat q;
    //SpectralDecomp(tm, s, q);
    AffineParts ap;
    decomp_affine(tm, &ap);
    const float smallval = 1.0e-5F;

    // 0に近いスケール値は適当な小さな値を入れる。
    if(fabsf(ap.k.x) < smallval) ap.k.x = (ap.k.x >= 0.0f)? smallval: -smallval;
    if(fabsf(ap.k.y) < smallval) ap.k.y = (ap.k.y >= 0.0f)? smallval: -smallval;
    if(fabsf(ap.k.z) < smallval) ap.k.z = (ap.k.z >= 0.0f)? smallval: -smallval;
    if(fabsf(ap.f) < smallval) ap.f = (ap.f >= 0.0f)? smallval: -smallval;

    Matrix3 srtm, rtm, ptm, stm, ftm, rettm;
    ptm.IdentityMatrix();
    ptm.SetTrans(ap.t);
    ap.q.MakeMatrix(rtm);
    ap.u.MakeMatrix(srtm);
    stm = ScaleMatrix(ap.k);
    ftm = ScaleMatrix(Point3(ap.f,ap.f,ap.f));
    rettm = Inverse(srtm) * stm * srtm * rtm * ftm * ptm;

    return rettm;
}



// 0に近いスケール値を考慮したInverse
Matrix3 inverse_ns0(Matrix3 tm)
{
    //Point3 s;
    //Quat q;
    //SpectralDecomp(tm, s, q);
    AffineParts ap;
    decomp_affine(tm, &ap);
    const float smallval = Dcc::R_SAME_TOLERANCE_F;//1.0e-5F;

    // 0に近いスケール値は適当な小さな値を入れる。
    if(fabsf(ap.k.x) < smallval) ap.k.x = (ap.k.x > 0.0f)? smallval: -smallval;
    if(fabsf(ap.k.y) < smallval) ap.k.y = (ap.k.y > 0.0f)? smallval: -smallval;
    if(fabsf(ap.k.z) < smallval) ap.k.z = (ap.k.z > 0.0f)? smallval: -smallval;
    if(fabsf(ap.f) < smallval) ap.f = (ap.f > 0.0f)? smallval: -smallval;

    Matrix3 srtm, rtm, ptm, stm, ftm, rettm;
    ptm.IdentityMatrix();
    ptm.SetTrans(ap.t);
    ap.q.MakeMatrix(rtm);
    ap.u.MakeMatrix(srtm);
    stm = ScaleMatrix(ap.k);
    ftm = ScaleMatrix(Point3(ap.f,ap.f,ap.f));
    rettm = Inverse(srtm) * stm * srtm * rtm * ftm * ptm;

    return Inverse(rettm);
}

// 3dsmaxの行列からRMtx44に変換する
// また、座標系も変換する。
Dcc::RMtx44 maxMtxToRMtx(const Matrix3 tm)
{
    Point3 trans, scale;
    Quat quat;
    AffineParts ap;

    // 行列をTRS要素に分解
    //DecomposeMatrix(tm, trans, quat, scale);
    decomp_affine(tm, &ap);
    killNaN(ap.t);
    killNaN(ap.q);
    killNaN(ap.u);
    killNaN(ap.k);
    killNaN(ap.f);

    trans = ap.t;
    quat = ap.q;
    //scale = ap.k * ap.f;
    scale = ScaleValue(ap.k, ap.u).s;
    // translate
    // 座標系変更
#if 0
    RVec3 rScale;
    RVec3 rRot;
    RVec3 rTrans;
#endif
#if 0
    rTrans.x = Dcc::RSnapToZero(trans.x);
    rTrans.y = Dcc::RSnapToZero(trans.z);
    rTrans.z = Dcc::RSnapToZero(-trans.y);
#endif
    // rotate
    float euler[3];
    Matrix3 rTm;
    quat.MakeMatrix(rTm);
    rTm = ScaleMatrix(Point3(ap.f, ap.f, ap.f)) * rTm;
    //decomp_affine(rTm, &ap);
    //MatrixToEuler(rTm, euler, EULERTYPE_XZY);
    //QuatToEuler(ap.q, euler, EULERTYPE_XZY);

#if 0
    QuatToEuler(quat, euler, EULERTYPE_XZY);

    // 座標系変更
    rRot.x = Dcc::RSnapToZero(euler[0]);
    rRot.y = Dcc::RSnapToZero(euler[1]);
    rRot.z = Dcc::RSnapToZero(-euler[2]);
    //rRot.y *= ap.f;

    rRot *= RAD_TO_DEG;
#endif

    // scale
    scale *= ap.f;

#if 0
    rScale.x = Dcc::RSnapToZero(scale.x);
    rScale.y = Dcc::RSnapToZero(scale.z);
    rScale.z = Dcc::RSnapToZero(scale.y);
#endif

    // 座標系変更
    Dcc::RVec3 rScale;
    Dcc::RVec3 rRot;
    Dcc::RVec3 rTrans;
    if(GetNintendoOptions()->IsEnableZYAxisConversion())
    {
        rTrans.x = Dcc::RSnapToZero(trans.x);
        rTrans.y = Dcc::RSnapToZero(trans.z);
        rTrans.z = Dcc::RSnapToZero(-trans.y);

        QuatToEuler(quat, euler, EULERTYPE_XZY);
        rRot.x = Dcc::RSnapToZero(euler[0]);
        rRot.y = Dcc::RSnapToZero(euler[1]);
        rRot.z = Dcc::RSnapToZero(-euler[2]);

        rScale.x = Dcc::RSnapToZero(scale.x);
        rScale.y = Dcc::RSnapToZero(scale.z);
        rScale.z = Dcc::RSnapToZero(scale.y);
    }
    else
    {
        rTrans.x = Dcc::RSnapToZero(trans.x);
        rTrans.y = Dcc::RSnapToZero(trans.y);
        rTrans.z = Dcc::RSnapToZero(trans.z);

        QuatToEuler(quat, euler, EULERTYPE_XYZ);
        rRot.x = Dcc::RSnapToZero(euler[0]);
        rRot.y = Dcc::RSnapToZero(euler[1]);
        rRot.z = Dcc::RSnapToZero(euler[2]);

        rScale.x = Dcc::RSnapToZero(scale.x);
        rScale.y = Dcc::RSnapToZero(scale.y);
        rScale.z = Dcc::RSnapToZero(scale.z);
    }

    rRot *= RAD_TO_DEG;

    // ノードのマトリクスを設定
    Dcc::RMtx44 mtx;
    mtx.SetTransform(rScale, rRot, rTrans);

    return mtx;
}


///////////////////////////////////////////////////////////////////////////
// CNodeTree
/*
Dcc::RExpOpt::Quantize CNodeTree::m_QuantPos_ = Dcc::RExpOpt::QuantizeFloat;
Dcc::RExpOpt::Quantize CNodeTree::m_QuantNml_ = Dcc::RExpOpt::QuantizeFloat;
Dcc::RExpOpt::Quantize CNodeTree::m_QuantTex_ = Dcc::RExpOpt::QuantizeFloat;
*/
bool CNodeTree::m_UseFigureMode = false;
bool CNodeTree::m_ConvertToModel_ = false;
bool CNodeTree::m_NonUniformScale_ = false;
int CNodeTree::m_ReservedUniformRegister_ = 0;


CNodeTree::CNodeTree(void) : m_pMaxNode(NULL), m_pMaxParent(NULL) ,m_pRNode(NULL)
    ,m_pRParent(NULL),m_DoesExportModel(false), m_Name(""),m_IsBone(false)
    ,m_pSkinMod(NULL),m_pMorphMod(NULL)
    ,m_IsNeedDelTriObj(false), m_pTriObj(NULL)
    ,m_BillboardType(0),m_CombineGroup(-1)
    ,m_DoesExportScaleKey(false), m_DoesExportRotateKey(false), m_DoesExportTransKey(false), m_DoesExportVisibility(false)
{
    Init();
    m_pParent = NULL;
}


CNodeTree::CNodeTree(CNodeTree* node) :
        m_pMaxNode(NULL), m_pMaxParent(NULL) ,m_pRNode(NULL)	,m_pRParent(NULL),
        m_DoesExportModel(false), m_Name(""), m_pSkinMod(NULL), m_pMorphMod(NULL),
        m_IsBone(false), m_IsNeedDelTriObj(false), m_pTriObj(NULL), m_BillboardType(0),
        m_CombineGroup(-1)
{
    Init();
    m_pParent = node;
}

CNodeTree::~CNodeTree(void)
{
    Init();
}

void CNodeTree::Init(void)
{
    m_pParent = NULL;
    m_pMaxNode = m_pMaxParent = NULL;
    m_pRParent = NULL;
    m_pSkinMod = m_pMorphMod = NULL;
    m_Name = "";
    m_DoesExportModel = m_IsBone = false;
    m_IsVisible = false;
    m_IsNoCompress = false;
    m_BillboardType = 0;
    m_CombineGroup = -1;
    m_UserOptions = "";

    //m_QuantPos_ = m_QuantNml_ = m_QuantTex_ = R_QUANTIZE_FLOAT;
    m_DoesExportScaleKey = m_DoesExportRotateKey = m_DoesExportTransKey = m_DoesExportVisibility =false;

    // morph
    m_ExportShapePosition = m_ExportShapeNormal = true;
    m_ExportShapeColor = false;


    m_MaxBones.Init();
    m_RBones.clear();
    m_BonesInitMtx.clear();
    m_SkinVtx.clear();

    if(m_IsNeedDelTriObj)
    {
        m_pTriObj->DeleteMe();
        m_pTriObj = NULL;
        m_IsNeedDelTriObj = false;
    }

    for(nodeTreeList::iterator it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->Init();
        //delete (*it);
    }

    m_Children.clear();
}

// ノードを設定する。
void CNodeTree::SetNode(INode* node, string forceName)
{
    //if(!IsValidINode(node)) return ;
    if(!node) return;

    m_pMaxNode = node;
    m_pMaxParent = node->GetParentNode();
    if(forceName == "")
    {
        string orgName = M_2_A(node->GetName());
        m_Name = Dcc::RAdjustElementNameString(orgName);
        if(m_Name != orgName && GetNintendoOptions()->m_Opt.m_WarnsNodeNameChanged)
        {
            RLogger::LogMessagebyID(RLogger::kLogWRN_NodeNameIsChanged, orgName);
        }
    }
    else
    {
        m_Name = forceName;
    }


    // ノードのNoCompressプラグインで設定されたフラグを取得
    m_IsNoCompress = false;

    // ノードのSet Shape Animation プラグインで設定されたフラグを取得
    m_ExportShapePosition = true;
    m_ExportShapeNormal = true;
    m_ExportShapeColor = false;

    ICustAttribContainer* cc = node->GetCustAttribContainer();
    if(cc)
    {
        CustAttrib* attrib = NULL;
        // すでにNw4fNodeCustAttribが存在すればそれを返す
        for(int i = 0; i < cc->GetNumCustAttribs(); i++)
        {
            if(cc->GetCustAttrib(i)->ClassID() == NINTENDOEXPORT_NODE_ATTRIB_CLASS_ID)
            {
                attrib = cc->GetCustAttrib(i);
                break;
            }
        }
        // Nw4fNodeCustAttribが有る場合はフラグをチェック
        if(attrib)
        {
            IParamBlock2* pblock = attrib->GetParamBlock(0);
            if(pblock)
            {
                m_IsNoCompress = (pblock->GetInt(nw4f_node_noCompress) == TRUE);
                m_ExportShapePosition = (pblock->GetInt(nw4f_node_shapeExportPosition) == TRUE);
                m_ExportShapeNormal	=	(pblock->GetInt(nw4f_node_shapeExportNormal) == TRUE);
                m_ExportShapeColor =	(pblock->GetInt(nw4f_node_shapeExportColor) == TRUE);
                const MCHAR* opt = pblock->GetStr(nw4f_node_userSettingString);
                if(opt)
                {
                    m_UserOptions = std::string(M_2_A(opt));
                }
            }
        }
    }
}
static int gNumAnim = 0;

static void debugOutMtx(const char* name, Matrix3& m)
{
    AffineParts ap;
    decomp_affine(m, &ap);
    DebugPrint(_M_TEXT("node: %s\n"), name);
    DebugPrint(_M_TEXT("\tscale : %f, %f, %f\n"), ap.k.x, ap.k.y, ap.k.z);
    DebugPrint(_M_TEXT("\trotate: %f, %f, %f, %f\n"), ap.q.x, ap.q.y, ap.q.z, ap.q.w);
    DebugPrint(_M_TEXT("\ttrans : %f, %f, %f\n"), ap.t.x, ap.t.y, ap.t.z);
}


// 指定された期間のSRTアニメーションを設定する
bool CNodeTree::SetAnimation(int start, int end, int frameTicks)
{
    if(!m_pMaxNode || !m_pRNode) return false;
    Interface* ip =  GetCOREInterface();
    Interval ivalid;
    Matrix3 localTm, nodeTm, parentTm, parentTmInv;
    //Control* c = NULL;
    //c = m_pMaxNode->GetTMController();
    //DebugPrint("SetAnimation: %s(%d)\n", m_Name.c_str(), gNumAnim++);

    Dcc::RVec3 scale, rotate, trans;
    //DebugPrint("\nnode:%s\n", GetName().c_str());

    Point3 mtrans, mscale;
    Quat quat;
    //float euler[3];
    AffineParts ap;

    for(int t = start; t <= end; t++)
    {
        Control* tmc = m_pMaxNode->GetTMController();
        tmc = NULL;
        if(tmc && tmc->SuperClassID() == CTRL_MATRIX3_CLASS_ID)
        {
            localTm = m_pMaxNode->GetParentTM(t * frameTicks);
            tmc->GetValue(t, (void*)&localTm, ivalid, CTRL_RELATIVE);
        }
        else
        {
            nodeTm = m_pMaxNode->GetNodeTM(t * frameTicks);
            parentTm = m_pMaxNode->GetParentTM(t * frameTicks);
            Point3 s;
            Quat q;
            AffineParts ap;
#if 0
            SpectralDecomp(parentTm, s, q);
#else
            decomp_affine(parentTm, &ap);
            s = ap.k;
#endif
            // 親のスケールが0の場合はとりあえず何もしない
            if( (fabs(s.x) > Dcc::R_SAME_TOLERANCE_F) &&
                (fabs(s.y) > Dcc::R_SAME_TOLERANCE_F) &&
                (fabs(s.z) > Dcc::R_SAME_TOLERANCE_F))
            {
                nodeTm = kill0scale(nodeTm);
                parentTm = kill0scale(parentTm);
                killNaN(parentTm);
                //parentTmInv = Inverse(parentTm);
                parentTmInv = inverse_ns0(parentTm);
                killNaN(parentTmInv);
                localTm = nodeTm * parentTmInv;
                killNaN(localTm);
            }
            else
            {
                //localTm = parentTm;
                //localTm.IdentityMatrix();
                localTm.Zero();
                //DebugPrint("Parent Scale is Zero. Node %s, Frame %d\n", m_Name.c_str(), t);
                //debugOutMtx("parent", parentTm);
                //debugOutMtx("this  ", nodeTm);
            }
        }
        Dcc::RMtx44 mtx = maxMtxToRMtx(localTm);
        m_pRNode->setLocalMatrix(t, mtx);
/*
        mtx.GetTransform(scale, rotate, trans);
        RMtx44 mtx2;
        mtx2.SetTransform(scale, rotate,trans);
        DebugPrint("CafeScaleAfter (%3d) x:%f, y:%f, z:%f\n", t, scale.x, scale.y, scale.z);
        DebugPrint("CafeRotAfter   (%3d) x:%f, y:%f, z:%f\n", t, rotate.x, rotate.y, rotate.z);
        DebugPrint("CafeTransAfter (%3d) x:%f, y:%f, z:%f\n", t, trans.x, trans.y, trans.z);
*/
    }

    m_pRNode->convertMatrixAnimToSRTAnim();
    return true;
}

// 再帰的に指定された期間のSRTアニメーションを設定する
bool CNodeTree::SetAnimationRecursive(int start, int end, int frameTicks)
{
    SetAnimation(start, end, frameTicks);

    for(nodeTreeList::iterator it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->SetAnimationRecursive(start, end, frameTicks);
    }
    return true;
}

// 再帰的に指定された期間のビジビリティアニメーションを設定する
bool CNodeTree::SetVisibilityAnimationRecursive(int start, int end, int frameTicks)
{
    if(!m_pMaxNode || !m_pRNode) return false;

    // 非表示ノードのビジビリティアニメーションは出力しない
    if(!m_pMaxNode->IsNodeHidden())
    {
        Control* vc = m_pMaxNode->GetVisController();
        if(vc)
        {
            if((vc->NumKeys() > 0) || vc->IsAnimated())
            {
                for(int t = start; t <= end; t++)
                {
                    float vis;
                    vis = m_pMaxNode->GetVisibility(t * frameTicks);
                    vis = (Dcc::RSnapToZero(vis) <= 0.0f)? 0.0f : 1.0f;
                    m_pRNode->setVisibilityAnim(vis);
                }
            }
        }
    }

    for(nodeTreeList::iterator it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->SetVisibilityAnimationRecursive(start, end, frameTicks);
    }
    return true;
}

// 再帰的に指定された期間のシェイプアニメーションを設定する。
bool CNodeTree::SetShapeAnimationRecursive( int start, int end, int frameTicks )
{
    if(!m_pMaxNode || !m_pRNode) return false;

    if(m_pMorphMod && m_MorphCh.Count() > 0)
    {
        for(int ishape = 0; ishape < m_MorphCh.Count(); ishape++)
        {
            int ch = m_MorphCh[ishape];
            morphChannel const & chInfo = m_pMorphMod->chanBank[ch];
            IParamBlock* pb = chInfo.cblock;
            if(pb)
            {
                Control* mc = pb->GetController(0);
                Interval valid = FOREVER;
                float weight;
                if(mc)
                {
                    if(mc->NumKeys() > 0 || mc->IsAnimated())
                    {
                        for(int t = start; t <= end; t++)
                        {
                            mc->GetValue(t * frameTicks, &weight, valid);
                            weight = Dcc::RSnapToZero(weight) * 0.01f; //パラメータには％で入っている?
                            m_pRNode->setShapeAnim(ishape, weight);
                        }
                    }
                    else // アニメーションがない場合もキーを一つ打っておく
                    {
                        mc->GetValue(start * frameTicks, &weight, valid);
                        m_pRNode->setShapeAnim(ishape, weight);
                    }
                }
            }
        }

    }

    for(nodeTreeList::iterator it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->SetShapeAnimationRecursive(start, end, frameTicks);
    }
    return true;
}


// ノードをINodeをキーに検索する。
CNodeTree* CNodeTree::FindChild(INode* node)
{
    if(!IsValidINode(node)) return NULL;

    nodeTreeList::iterator it;

    // まず子を全て探す
    for(it = m_Children.begin(); it != m_Children.end(); it++)
    {
        if(node == (*it)->m_pMaxNode)
        {
            return (*it);
        }
    }
    return NULL;
}

// 指定のノード以下をINodeをキーに検索する。
CNodeTree* CNodeTree::FindChildRecursive(INode* node)
{
    if(!IsValidINode(node)) return NULL;

    // まず子を全て探す
    CNodeTree* cNode = FindChild(node);

    if(cNode)
    {
        return cNode;
    }

    // 子のさらに子を再帰的に探す
    nodeTreeList::iterator it;
    for(it = m_Children.begin(); it != m_Children.end(); it++)
    {
        cNode = (*it)->FindChildRecursive(node);
        if(cNode)
        {
            return cNode;
        }
    }

    // 見つからなかった場合はNULL
    return NULL;
}

// 子ノードを追加する。
CNodeTree* CNodeTree::AddChild(INode* node)
{
    if(!IsValidINode(node)) return NULL;

    CNodeTree* cNode = FindChild(node);
    if(cNode)
    {
        return cNode;
    }

    //m_Children
    cNode = new CNodeTree(this);
    cNode->SetNode(node);
    m_Children.push_back(cNode);

    return cNode;
}

// 指定のノード以下を再帰的に登録する。
void CNodeTree::AddChildrenRecursive()
{
    if(!m_pMaxNode) return;

    INode* child;
    for (int idx=0; idx < m_pMaxNode->NumberOfChildren(); idx++)
    {
        child = m_pMaxNode->GetChildNode(idx);
        if(child)
        {
            // カメラとライトは追加しない
            ObjectState os;
            os = child->EvalWorldState(0);
            if (!os.obj) continue;
            /*
            // ノードがターゲットの場合は、ルックアットノードを調べる
            if(child->IsTarget())
            {
                INode* lookat = child->GetLookatNode();
                if(lookat)
                {
                    os = lookat->EvalWorldState(0);
                }
            }
            else
            {
                os = child->EvalWorldState(0);
            }
            if (!os.obj) continue;
            if (os.obj->SuperClassID() == LIGHT_CLASS_ID
                || os.obj->SuperClassID() == CAMERA_CLASS_ID
                )
            {
                continue;
            }
            */

            CNodeTree* cChild = AddChild(child);
            if(cChild)
            {
                cChild->InquiryAsModel();
                cChild->AddChildrenRecursive();
            }
        }
    }
}

// RNodeを再帰的に作成する。
void CNodeTree::CreateRNodeRecursive(CNodeTree* parent, INodeToRNodeMap& nodeMap, int f)
{
    assert(m_pMaxNode);

    // ライトやカメラなどはRNodeを作成しない(その子も)
    {
        ObjectState os;
        //INode* node = NULL;
        //os = m_pMaxNode->EvalWorldState(0);
        //if (!os.obj) continue;
        // ノードがターゲットの場合は、ルックアットノードを調べる
        if(m_pMaxNode->IsTarget())
        {
            //node = m_pMaxNode->GetLookatNode();
            INode* lookat = m_pMaxNode->GetLookatNode();
            if(lookat)
            {
                os = lookat->EvalWorldState(0);
            }
        }
        else
        {
            //node = m_pMaxNode;
            os = m_pMaxNode->EvalWorldState(0);
        }

        if (os.obj)
        {
            if(os.obj->SuperClassID() == LIGHT_CLASS_ID
                || os.obj->SuperClassID() == CAMERA_CLASS_ID
              )
            {
                return;
            }
        }
    }
    // RNodeを作成し設定
    m_pRNode = new RNode;
    m_pRNode->setName(m_Name);
    //DebugPrint("Create Node %s\n", name.c_str());

    if(parent)
    {
        m_pRNode->setParent(parent->m_pRNode);
    }

    // ノードのリンク情報をチェック(ROOTはチェックしない)
    Control* c = m_pMaxNode->GetTMController();
    if(c)
    {
        DWORD flags = c->GetInheritanceFlags();
        bool err = false;

        //変換コントローラの種類で「継承」をチェックするか決める
        Class_ID cid = c->ClassID();
        if(cid == Class_ID(PRS_CONTROL_CLASS_ID,0)
            //|| cid == LINKCTRL_CLASSID // リンクコンストレイントは継承がセットできるが取得できない
          )
        {
            if(flags != 0)
            {
                err = true;
            }
        }
        else if(cid == Class_ID(LOOKAT_CONTROL_CLASS_ID,0))
        {
            // LookAtの場合は(ライトカメラなど)移動のフラグのみチェック
            if(flags & (INHERIT_POS_X +INHERIT_POS_Y + INHERIT_POS_Z))
            {
                err = true;
            }
        }

        if(err)
        {
            string mes = "Inherits of node link info is off(";
            mes = mes + std::string(M_2_A(m_pMaxNode->GetName())) + ")";
            RLogger::LogMessage(mes, RLogger::kWarning);
        }
    }

    nodeMap.insert(INodeToRNodeMap::value_type(m_pMaxNode, m_pRNode));

    // 行列を設定する。
    TimeValue t = f * GetTicksPerFrame();
    //Matrix3 localTm = m_pMaxNode->GetNodeTM(t) * Inverse(m_pMaxNode->GetParentTM(t));
    Matrix3 localTm, nodeTm, parentTm;
    nodeTm = kill0scale(m_pMaxNode->GetNodeTM(t));
    parentTm = kill0scale(m_pMaxNode->GetParentTM(t));
    localTm = nodeTm * Inverse(parentTm);

#if 0
    {
        Control* tmc = maxNode->GetTMController();
        if(tmc && tmc->SuperClassID() == CTRL_MATRIX3_CLASS_ID)
        {
            //Matrix3Indirect pmat(maxNode->GetParentTM(t));
            Matrix3Indirect pmat(Matrix3(1));
            Point3 pos;
            ScaleValue scale;
            float rot[4];
            Interval rIval, sIval, pIval;
            TMComponentsArg tmarg(&pos, &pIval, rot, &rIval, &scale, &sIval);
            bool ret = tmc->GetLocalTMComponents(t, tmarg, pmat);
            if(ret)
            {
                DebugPrint("%s: time:%d pos:(%f,%f,%f)\n", maxNode->GetName(), t, pos.x, pos.y, pos.z);
            }
            else
            {
                DebugPrint("%s: time:%d Can't get LocalTMComponents\n");
            }
        }
    }
#endif
    Dcc::RMtx44 mtx = maxMtxToRMtx(localTm);
    m_pRNode->setLocalMatrix(f, mtx);
    if(m_IsBone)
    {
        //m_pRNode->setBindMatrix(initLocalMtx);
        m_pRNode->setBindMatrix(m_InitMtx);
    }
    // ノードアトリビュート情報を設定
    m_pRNode->setCompressEnableFlag(!m_IsNoCompress);
    //m_pRNode->setBillboardMode((RBillboard)m_BillboardType);
    //m_pRNode->setCombineGroup(m_CombineGroup);
    //m_pRNode->setForceExportKeyFlag(m_DoesExportScaleKey, m_DoesExportRotateKey, m_DoesExportTransKey, m_DoesExportVisibility);
    Dcc::RUserDataArray dataArray;
    SetUserDataFromString(m_UserOptions, NULL, m_pRNode);


    // 子ノードを再帰的に作成する。
    nodeTreeList::iterator it;
    for (it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->CreateRNodeRecursive(this, nodeMap, f);
    }
}


static bool isMapChExist(Mesh* mesh, int ch)
{
    if(!mesh) return false;

    if(mesh->mapSupport(ch) == FALSE) return false;
    MeshMap& mmap = mesh->Map(ch);
    // 頂点か面の数が0なら出力しない
    if(mmap.GetFlag(MESHMAP_USED) == FALSE) return false;
    if(mmap.getNumVerts() == 0) return false;
    if(mmap.getNumFaces() == 0) return false;

    return true;
}

// マップチャンネルから「nw4f_color0-7」を探す
static void getCafeVColToMaxVColMap(INode* node, Mesh* mesh, intintmap &cafeColToMaxCol)
{
    assert(node);
    assert(mesh);

    for(int i = 0; i < Dcc::RPrimVtx::VTX_COL_MAX; i++)
    {
        cafeColToMaxCol[i] = MAP_ALPHA - 1;
    }

    for (int maxUVCh = MAP_ALPHA; maxUVCh < MAX_MESHMAPS; maxUVCh++)
    {
        if(!isMapChExist(mesh, maxUVCh)) continue;
        MeshMap& mmap = mesh->Map(maxUVCh);
        //if(!mmap.GetFlag(MESHMAP_VERTCOLOR)) continue;

        // マップチャンネル名を取得する
        MSTR key, name;
        key.printf(_M_TEXT("MapChannel:%d"), maxUVCh);
        name = _M_TEXT("");
        if (node->UserPropExists(key))
        {
            node->GetUserPropString(key, name);
            // マップチャンネル名がnw4f_colorから始まっていればvcolChに追加
            std::string sname = M_2_A(name.data());
            if(strstr(sname.c_str(), "nw4f_color") == sname.c_str())
            {
                // "nw4f_colorN" のNが０～７の時に出力する
                char c = sname[10];
                if('0' <= c && c <= '7')
                {
                    int num = c - '0';
                    if(cafeColToMaxCol[num] < MAP_ALPHA)
                    {
                        cafeColToMaxCol[num] = maxUVCh;
                    }
                    else
                    {
                        // colorN batting
                    }
                }
            }
        }
    }
}

bool compare_maxUVInfo(const maxUVInfo& left, const maxUVInfo& right)
{
    return left.maxUVName < right.maxUVName;
}


// RModelを再帰的に作成する。
bool CNodeTree::CreateRModelRecursive( CNodeTree* parent, RSceneMaterials& sceneMaterials, INodeToRNodeMap& nodeMap, std::vector<RNode*>& bones, Dcc::RExpOpt& rOpt )
{
    //DebugPrint("CreateRModelRecursive node:%s, m_pMaxNode:%X, m_pRNode:%X\n", name.c_str(), m_pMaxNode, m_pRNode);
    bool ret = true;
    assert(m_pMaxNode);
    // RNodeがない場合は（ライト、カメラなど）何もしない
    if(!m_pRNode)
    {
        return true;
    }

    // スキンモディファイアがある場合
    bool hasSkin = false;
    if(m_pTriObj && m_SkinVtx.size() > 0 && m_MaxBones.Count() > 0)
    {
        hasSkin = true;
    }

    // メッシュを準備する。
    if(m_pTriObj)
    {
        Mesh* mesh = &m_pTriObj->GetMesh();
        // 法線を計算
        //mesh->buildNormals();
        mesh->checkNormals(TRUE);

        const int mtxPalCount = Dcc::RShape::GetStandardMtxPalCount(false, 0);
        RModel* model = new RModel(
            mtxPalCount,
            GetName().c_str());
            //RGetUtf8FromShiftJis(string(m_pMaxNode->GetName())).c_str());
        //DebugPrint("Create Model %s\n", GetName().c_str());

        // スキン情報
        std::vector<int>	VtxMtxIdxs;
        if(hasSkin)
        {
            // スキン情報が有る場合、RVtxBoneRefArrayを作成して
            // ボーンとウェイトの情報を設定する。
            skinVtxInfoList::iterator	v_it; // vertex
            skinVtxInfo::iterator		w_it; // weight in vertex
            for(v_it = m_SkinVtx.begin(); v_it != m_SkinVtx.end(); v_it++)
            {
                float weightSum = 0.0f; // ウェイト値の合計
                Dcc::RVtxMtx	vmt;
                for(w_it = v_it->begin(); w_it != v_it->end(); w_it++)
                {
                    weightSum += w_it->second;
                    INodeToRNodeMap::iterator n_it;
                    n_it = nodeMap.find(m_MaxBones[w_it->first]);
                    assert(n_it != nodeMap.end());
                    RNode* rBone = n_it->second;
                    int weight = int(w_it->second * Dcc::RVtxWeight::WEIGHT_MAX + 0.5f);
                    weight = Dcc::RClampValue( 0, static_cast<int>(Dcc::RVtxWeight::WEIGHT_MAX), weight );
                    //	使用するボーンのインデックスとウエイトを設定します。
                    if(weight > 0)
                    {
                        vmt.Append(rBone->getNodeArrayIndex(), weight);
                    }
                }
                // ウェイトの合計が 1.0 から 0.005 以上離れて入ればエラーとします。
                if (!Dcc::RIsSame(weightSum, 1.0f, 0.005f))
                {
                    RLogger::LogMessagebyID(RLogger::kLogERR_WrongWeightedVertices, this->GetName());
                    return false;
                }

                vmt.AdjustWeight();

                // コンフィグで指定した数を超えるノードにウェイト値が分配されていればエラーとします。
                if (vmt.GetBoneCount() > rOpt.m_MaxVertexSkinningCount)
                {
                    //YShowError(yscene, "The number of weighted influence is over %d: %s.vtx[%d]",
                    //	yopt.m_MaxVertexSkinningCount, ynode.m_ShapePath.partialPathName().asChar(), vIter.index());
                    string mes = "The number of weighted influence is over ";
                    mes +=  Dcc::RGetNumberString(rOpt.m_MaxVertexSkinningCount);
                    mes += ": ";
                    mes += GetName();
                    RLogger::LogMessage(mes);
                    return false;
                }

                //	頂点行列をリストへ追加して、追加されたインデックスを頂点に対応するデータとして保持します。
                const int iVmt = AppendVtxMtx( vmt );
                VtxMtxIdxs.push_back( iVmt );
            }

#if 1
            // shapeInfoに登録(フォーマットのみ)
            FOutShapeInfo& shapeInfo = model->getShapeInfo();
            shapeInfo.m_VtxInfos[Dcc::RPrimVtx::IDX0] = Dcc::ROutShapeVtxInfo(4, Dcc::RPrimVtx::VALUE_UINT, NULL);
            shapeInfo.m_VtxInfos[Dcc::RPrimVtx::WGT0] = Dcc::ROutShapeVtxInfo(4, Dcc::RPrimVtx::VALUE_FLOAT, NULL);
#else // スキンウェイトを無効化
            #pragma message(TODO("共通モジュールでスキニングが未実装のため、スキンウェイトは無効化されている。"))
#endif
        }
        else
        {
            // スキニングしていない場合でも階層構造を表現するのにスキニング機能を使用するため、
            // 全てのメッシュに対して RVtxMtx を設定する必要があります。
            const int iVmt = AppendVtxMtx(Dcc::RVtxMtx(m_pRNode->getNodeArrayIndex()));
            for (int iVtx = 0; iVtx < mesh->getNumVerts(); ++iVtx)
            {
                VtxMtxIdxs.push_back( iVmt );
            }
            m_pRNode->m_RigidBody = true;
        }



        // マテリアル情報の取得
        if(m_pMaxNode->GetMtl() == NULL)
        {
            //mprintf("WARNING: node(%s) has no material\n", node->GetName());
            string mes = "Node has no material(";
            mes += M_2_A(m_pMaxNode->GetName());
            mes += ")";
            RLogger::LogMessage(mes, RLogger::kWarning);
        }
        MultiMtlInfo multiMtl;
        bool ret = multiMtl.Init(m_pMaxNode->GetMtl(), sceneMaterials.GetDoesExportTexture());
        assert(ret);

        // NW4FのUV座標チャンネルとMaxのUV座標チャンネルとの対応
        //		intintmap cafeUVToMaxUV;
        /*struct maxUVInfo
        {
            int maxUV;
            string maxUVName;
        };*/



        cafeToMaxUVInfoVec fixedMaxUV; // ch固定のUV
        cafeToMaxUVInfoVec namedMaxUV; // 名前指定のUV
        cafeToMaxUVInfoVec forcedMaxUV; // 明示的に指定されたUV
        cafeToMaxUVInfoVec usedMaxUV; // テクスチャで使用されているUV

        const std::string EXTRA_UV_SET_PREFIX = "nw4f_uv";
        const std::string FIX_UV_SET_PREFIX = "nw4f_fix";

        // 明示的にエクスポート指定、またはチャンネル指定されたUVチャンネルを探す
        for(int ch = 1; ch < MAX_MESHMAPS; ch++)
        {
            if(!isMapChExist(mesh, ch)) continue;
            MSTR key, name;
            key.printf(_M_TEXT("MapChannel:%d"), ch);
            name = _M_TEXT("");
            if (m_pMaxNode->UserPropExists(key))
            {
                m_pMaxNode->GetUserPropString(key, name);
                std::string sname = M_2_A(name.data());

                bool isNeedAdd = false;
                int cafeCh = -1;
                std::string cafeChName = "";

                // マップチャンネル名がnw4f_uvから始まっていればuserUVmapに追加
                if(sname.find(EXTRA_UV_SET_PREFIX) == 0)
                {
                    //maxUVInfo info = {-1, ch, sname};
                    //forcedMaxUV.push_back(info);
                    isNeedAdd = true;
                }
                // マップチャンネル名がnw4f_fixから始まっている場合
                else if(sname.find(FIX_UV_SET_PREFIX) == 0)
                {
                    std::string idxStr = sname.substr(FIX_UV_SET_PREFIX.length(), sname.length() - 1);
                    if (idxStr.length() > 0)
                    {
                        const size_t iUnder = idxStr.find("_");
                        if (iUnder == 0) // fix_SSSS 形式
                        {
                            if(idxStr.length() >= 2) // fix_ の場合は警告
                            {
                                cafeChName = idxStr.substr(1, idxStr.length() - 1);
                                isNeedAdd = true;
                            }
                            else
                            {
                                string mes = GetName();
                                mes += ": ";
                                mes += sname;
                                RLogger::LogMessagebyID(RLogger::kLogWRN_UvSetNameIsWrong, mes);
                            }
                        }
                        else
                        {
                            if (iUnder != -1) // fixNN_SSS 形式
                            {
                                if(iUnder < idxStr.length() - 1) // fixNN_ の場合は無視
                                {
                                    //cafeChName = idxStr.substr(iUnder+1, idxStr.length() - 1);
                                    cafeChName = idxStr.substr(iUnder+1);
                                }
                                idxStr = idxStr.substr(0, iUnder);
                            }
                            int iHint;
                            BOOL parseRet = false;
                            // nw4f_fixN で N が数値に変換できなければスキップ
                            parseRet = StrToIntExA(idxStr.c_str(), STIF_DEFAULT, &iHint);
                            if(parseRet && iHint >= 0 && iHint <= Dcc::RVtxAttrib::HINT_INDEX_MAX)
                            {
                                cafeCh = iHint;
                                isNeedAdd = true;
                            }
                            else
                            {
                                string mes = GetName();
                                mes += ": ";
                                mes += sname;
                                RLogger::LogMessagebyID(RLogger::kLogWRN_UvSetNameIsWrong, mes);
                            }
                        }
                    }
                    else
                    {
                        string mes = GetName();
                        mes += ": ";
                        mes += sname;
                        RLogger::LogMessagebyID(RLogger::kLogWRN_UvSetNameIsWrong, mes);
                    }
                }

                if(!isNeedAdd) continue;

                if(cafeCh >= 0)
                {
                    bool sameHintExist = false;
                    // すでに固定用にCafeのUVチャンネルが登録されていないか調べる
                    for(cafeToMaxUVInfoVec::iterator it = fixedMaxUV.begin(); it != fixedMaxUV.end(); it++)
                    {
                        if(it->cafeUV == cafeCh)
                        {
                            sameHintExist = true;
                            break;
                        }
                    }

                    if(sameHintExist) //登録済みならエラーとする
                    {
                        RLogger::LogMessagebyID(RLogger::kLogERR_UvSetSameHintExists, sname);
                        return false;
                    }
                    else
                    {
                        maxUVInfo info = {cafeCh, ch, cafeChName, true};
                        fixedMaxUV.push_back(info);
                    }
                }
                else // ch固定でない場合
                {
                    if(cafeChName.size() > 0) //名前指定の場合
                    {
                        maxUVInfo info = {-1, ch, cafeChName, true};
                        namedMaxUV.push_back(info);
                    }
                    else
                    {
                        maxUVInfo info = {-1, ch, sname, true};
                        forcedMaxUV.push_back(info);
                    }
                }
            }
        }


        // マテリアルで使用する全てのテクスチャ座標を収集する
        for(int i = 0; i < multiMtl.GetNumSubMtl(); i++)
        {
            MtlInfo* mtlinfo = multiMtl.GetSubMtl(i);
            if(mtlinfo == NULL) continue;

            // すべてのテクスチャを調べる
            for(UINT j = 0; j < mtlinfo->GetNumBitmap(); j++)
            {
                int maxUVch = mtlinfo->GetBitmap(j)->mapch;
                if(!isMapChExist(mesh, maxUVch)) continue;

                // すでにMaxのUVチャンネルが登録されていないか調べる
                for(cafeToMaxUVInfoVec::iterator it = usedMaxUV.begin(); it != usedMaxUV.end(); it++)
                {
                    if(it->maxUV == maxUVch)
                    {
                        maxUVch = -1;
                        break;
                    }
                }
                if(maxUVch < 0) continue; //登録済みなら追加しない

                // すでに固定用にMaxのUVチャンネルが登録されていないか調べる
                for(cafeToMaxUVInfoVec::iterator it = fixedMaxUV.begin(); it != fixedMaxUV.end(); it++)
                {
                    if(it->maxUV == maxUVch)
                    {
                        maxUVch = -1;
                        break;
                    }
                }
                if(maxUVch < 0) continue; //登録済みなら追加しない

                maxUVInfo info = {-1, maxUVch, "", false};
                usedMaxUV.push_back(info);
            }
        }

        // cafe uv map (255)
        cafeToMaxUVInfoVec cafeUVToMaxUV;
        for(int i = 0; i <= Dcc::RVtxAttrib::HINT_INDEX_MAX; i++)
        {
            maxUVInfo info = {-1, -1, std::string(""), false};
            cafeUVToMaxUV.push_back(info);
        }

        // 全ての登録されたUVチャンネルをcafeUVに割り振る

        // 固定CH
        for(cafeToMaxUVInfoVec::iterator it = fixedMaxUV.begin(); it != fixedMaxUV.end(); it++)
        {
            cafeUVToMaxUV[it->cafeUV] = *it;
            //cafeUVToMaxUV[it->cafeUV].maxUV = it->maxUV;
            //cafeUVToMaxUV[it->cafeUV].maxUVName = it->maxUVName;
            //cafeUVToMaxUV[it->cafeUV].useForce = true;
        }

        // 使用UV
        for(cafeToMaxUVInfoVec::iterator it = usedMaxUV.begin(); it != usedMaxUV.end(); it++)
        {
            //int cafeCh = -1;
            // 登録済みMaxUV CHを探す
            int cafeCh = findValidCh(cafeUVToMaxUV, it->maxUV);

            if(cafeCh < 0)
            {
                // error
                //#pragma message(TODO("warning"))
            }
            else
            {
                cafeUVToMaxUV[cafeCh].maxUV = it->maxUV;
            }
        }

        // 名前指定UV
        std::sort(namedMaxUV.begin(), namedMaxUV.end(), compare_maxUVInfo); //名前でソート
        for(cafeToMaxUVInfoVec::iterator it = namedMaxUV.begin(); it != namedMaxUV.end(); it++)
        {
            //int cafeCh = -1;
            // 登録済みMaxUV CHを探す
            int cafeCh = findValidCh(cafeUVToMaxUV, it->maxUV);

            if(cafeCh < 0)
            {
                // error
                //#pragma message(TODO("warning"))

            }
            else
            {
                cafeUVToMaxUV[cafeCh].maxUV = it->maxUV;
                cafeUVToMaxUV[cafeCh].maxUVName = it->maxUVName;
                cafeUVToMaxUV[cafeCh].useForce = true;
            }
        }

        // 明示的UV
        std::sort(forcedMaxUV.begin(), forcedMaxUV.end(), compare_maxUVInfo); //名前でソート
        for(cafeToMaxUVInfoVec::iterator it = forcedMaxUV.begin(); it != forcedMaxUV.end(); it++)
        {
            //int cafeCh = -1;
            // 登録済みMaxUV CHを探す
            int cafeCh = findValidCh(cafeUVToMaxUV, it->maxUV);

            if(cafeCh < 0)
            {
                // error
                //#pragma message(TODO("warning"))
            }
            else
            {
                cafeUVToMaxUV[cafeCh].maxUV = it->maxUV;
                cafeUVToMaxUV[cafeCh].useForce = true;
            }
        }

        // 最終的に使用するUVを決定
        cafeToMaxUVInfoVec finalCafeUVToMaxUV;
        for(int i = 0; i <= Dcc::RVtxAttrib::HINT_INDEX_MAX; i++)
        {
            maxUVInfo& info = cafeUVToMaxUV[i];
            if(info.maxUV < 0) continue;

            //info.cafeUV = i;
            finalCafeUVToMaxUV.push_back(info);
        }

        // UV数が8を超えたら警告してリサイズ
        if(finalCafeUVToMaxUV.size() > Dcc::RPrimVtx::VTX_TEX_MAX)
        {
            //#pragma message(TODO("warning"))
            finalCafeUVToMaxUV.resize(Dcc::RPrimVtx::VTX_TEX_MAX);
        }

        // UVの使用フラグを立てる
        for(int i = 0; i < finalCafeUVToMaxUV.size(); i++)
        {
            multiMtl.m_ForceUseCafeUV[i] = finalCafeUVToMaxUV[i].useForce;
        }


        // NW4Fの頂点カラーチャンネルとMaxの頂点カラーチャンネルとの対応
        // Cafeの頂点カラーチャンネルをキーとしたMap
        intintmap cafeColToMaxCol;

        getCafeVColToMaxVColMap(m_pMaxNode, mesh, cafeColToMaxCol);


        // 頂点座標などを取得しエクスポート
        ret = GetGeomVertex(mesh ,*model, finalCafeUVToMaxUV, cafeColToMaxCol);
        if(ret)
        {
            ret = ExportGeomObject(mesh, *model, VtxMtxIdxs
                , sceneMaterials, multiMtl, finalCafeUVToMaxUV, cafeColToMaxCol);
            //	全てのシェイプの登録を完了したので
            //	シェイプごとのスキニングモードを設定する。
            model->getShapeSkinningMode( m_pRNode->getNodeArrayIndex(), g_VtxMtxs, bones, rOpt.m_AdjustsSmoothSkinning );
        }

        if(m_IsNeedDelTriObj)
        {
            m_pTriObj->DeleteMe();
            m_pTriObj = NULL;
            m_IsNeedDelTriObj = false;
        }
        m_pTriObj = NULL;

        // RNodeにモデルを設定する。
        if(ret)
        {
            m_pRNode->setModel(model);
            m_pRNode->setVisible(m_IsVisible);
        }
        else
        {
            return false;
        }
    }

    // 子ノードを再帰的にエクスポートする。
    nodeTreeList::iterator it;
    for (it = m_Children.begin(); it != m_Children.end(); it++)
    {
        ret = (*it)->CreateRModelRecursive(this, sceneMaterials, nodeMap, bones, rOpt);
        if(!ret) return false;
    }
    return true;
}

// モデルとして調査する
bool CNodeTree::InquiryAsModel(void)
{
    if(!m_pMaxNode || m_pMaxNode->IsNodeHidden()) return false;
    //if(!m_pMaxNode) return false;
    //MCHAR* nodename = m_pMaxNode->GetName();

    ObjectState os = m_pMaxNode->EvalWorldState(0);
    if (!os.obj) return false;

    // 表示フラグを取得
    m_IsVisible = (m_pMaxNode->GetVisibility(0) > 0.0001f);

    // スキンまたはPhysiqueを取得
    if(GetSkinModFromNode())
    {
        GetSkinInfoFromModifier(m_pSkinMod);
    }

    PerpareObject();

    // モーフを取得
    if(GetMorphFromNode())
    {

    }


    return true;
}

// ノードからオブジェクトを取り出す
bool CNodeTree::PerpareObject(void)
{
    if(!m_pMaxNode) return false;

    // ノードのスーパークラスがジオメトリでない場合はメッシュを出力しない。
    ObjectState tos = m_pMaxNode->EvalWorldState(0);
    if(tos.obj->SuperClassID() != GEOMOBJECT_CLASS_ID)
    {
        return false;
    }

    // bone,bipedのメッシュを出力しない。
#if 1
    Class_ID cid = tos.obj->ClassID();
    if(		cid == BONE_OBJ_CLASSID
        ||	cid == FOOTPRINT_CLASS_ID
        ||	cid == SKELOBJ_CLASS_ID	)
    {
        return false;
    }
    Control* c = m_pMaxNode->GetTMController();
    if( (c->ClassID() == BIPSLAVE_CONTROL_CLASS_ID) ||
        (c->ClassID() == BIPBODY_CONTROL_CLASS_ID) ||
        (c->ClassID() == FOOTPRINT_CLASS_ID) )
    {
        return false;
    }
#endif

    Object* obj = GetObjBeforeModifier();
    if(!obj) return false;
    // メッシュに変換できない物は出力しない
    if(		 (obj->SuperClassID() != GEOMOBJECT_CLASS_ID)
        ||	 (obj->ClassID() == Class_ID(TARGET_CLASS_ID, 0))
        ||	!(obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0))) )
    {
        return false;
    }

    m_pTriObj = (TriObject *) obj->ConvertToType(0, Class_ID(TRIOBJ_CLASS_ID, 0));
    if (!m_pTriObj) return false;

    // os.objがTriobjectではない場合は明示的に削除が必要
    if (obj != m_pTriObj) m_IsNeedDelTriObj = true;

    return true;
}

// モディファイア適用前のObjectを取得
Object* CNodeTree::GetObjBeforeModifier()
{
    if(!m_pMaxNode) return NULL;

    // モディファイアがない場合
    // あるいはフィギュアモードを使う場合
    if(!(m_pSkinMod || m_pMorphMod) || m_UseFigureMode)
    {
        ObjectState os = m_pMaxNode->EvalWorldState(0);
        return os.obj;
    }


    // skinとmorphが見つかったかどうかのフラグ
    // モディファイアがない場合は最初からtrueにしておく
    bool skinModFound = (m_pSkinMod == NULL);
    bool morphModFound = (m_pMorphMod == NULL);

    Object* obj = m_pMaxNode->GetObjectRef();
    if(!obj) return NULL;

    while(obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
    {
        IDerivedObject* derivedObj = static_cast<IDerivedObject*>(obj);
        int numMod = derivedObj->NumModifiers();

        // モディファイアを探し、そのインデックスを取得
        for(int i = 0; i < numMod; i++)
        {
            if(m_pSkinMod == derivedObj->GetModifier(i)) skinModFound = true;
            if(m_pMorphMod == derivedObj->GetModifier(i)) morphModFound = true;

            // skinとmorphが見つかった場合
            if(skinModFound && morphModFound)
            {
                // 下のモディファイアのインデックス
                int nextIdx = i + 1;
                Object* initialObj = NULL;
                if(nextIdx < numMod) // 下にモディファイアがある場合？
                {
                    ObjectState os = derivedObj->Eval(0, i + 1);
                    initialObj = os.obj;
                }
                else // モディファイアが一番下の場合？
                {
                    initialObj = derivedObj->GetObjRef();
                }

                // Objectの種類によって
                SClass_ID sid = initialObj->SuperClassID();
                if (sid == WSM_DERIVOB_CLASS_ID || sid == DERIVOB_CLASS_ID || sid == GEN_DERIVOB_CLASS_ID)
                {
                    IDerivedObject* initDerivedObject = (IDerivedObject*) initialObj;
                    if (initDerivedObject->NumModifiers() > 0)
                    {
                        ObjectState os = initDerivedObject->Eval(0, 0);
                        initialObj = os.obj;
                    }
                    else
                    {
                        initialObj = initDerivedObject->GetObjRef();
                    }
                }
                return initialObj;
            }
        }
        obj = derivedObj->GetObjRef();
    }
    return NULL;
}

// repair biped matrix
void RemoveNonUniformScale(Matrix3 &mtx)
{
    AffineParts parts;
    decomp_affine(mtx, &parts);
    parts.q.MakeMatrix(mtx);
    mtx.SetRow(3,  parts.t);
}


// スキニング情報を取得する
bool CNodeTree::GetSkinInfoFromModifier(Modifier* mod)
{
    if(!mod || !m_pMaxNode) return false;
    m_SkinVtx.clear();
    m_MaxBones.Init();
    m_BonesInitMtx.clear();

    // Physique
    if(mod->ClassID() == Class_ID(PHYSIQUE_CLASS_ID_A, PHYSIQUE_CLASS_ID_B))
    {
        IPhysiqueExport* phyExport = static_cast<IPhysiqueExport*>(mod->GetInterface(I_PHYINTERFACE));
        IPhyContextExport* phyData = static_cast<IPhyContextExport*>(phyExport->GetContextInterface(m_pMaxNode));
        phyData->ConvertToRigid(TRUE);
        //phyData->AllowBlending(TRUE);

        int numVtx = phyData->GetNumberVertices();
        IPhyBlendedRigidVertex* rb_vtx;
        IPhyRigidVertex* r_vtx;
        //IPhyFloatingVertex* f_vtx;

        // Physiqueはボーンの関連付けはIDではなく参照で管理されている。
        // なのでボーンリストは頂点毎に列挙しながら作成する。

        // 頂点毎に割り当てられたボーンとウェイトを取得
        for (int i = 0; i< numVtx; i++)
        {
            float totalWeight = 0.0f, weight = 0.0f;
            TSTR nodeName;
            skinVtxInfo vtxInfo;

            int boneId;
            INode* bone;

            IPhyVertexExport* vi = phyData->GetVertexInterface(i);

            if(vi)
            {
                int vtxType = vi->GetVertexType();
                switch(vtxType)
                {
                    case RIGID_NON_BLENDED_TYPE: // ボーン一つのみに割り当てられる頂点
                        r_vtx = static_cast<IPhyRigidVertex*>(vi);
                        bone = r_vtx->GetNode();
                        // INodeTabはデフォルトでは重複を排除するのでボーンリストにボーンを追加し、
                        // そのインデックスをボーンIDにする
                        m_MaxBones.AppendNode(bone);
                        boneId = m_MaxBones.IndexOf(bone);
                        vtxInfo.push_back(skinWeight(boneId, 1.0f));
                        break;
                    case RIGID_BLENDED_TYPE:	// 複数のボーンに割り当てられる頂点
                    {
                        rb_vtx = static_cast<IPhyBlendedRigidVertex*>(vi);
                        int numNodes = rb_vtx->GetNumberNodes();
                        for (int j = 0; j < numNodes; j++)
                        {
                            bone = rb_vtx->GetNode(j);
                            weight = rb_vtx->GetWeight(j);
                            // ボーンウェイトが0の時は追加しない。
                            if(weight > Dcc::R_SAME_TOLERANCE_F)
                            {
                                m_MaxBones.AppendNode(bone);
                                boneId = m_MaxBones.IndexOf(bone);
                                // INodeTabはデフォルトでは重複を排除するのでボーンリストにボーンを追加し、
                                // そのインデックスをボーンIDにする
                                vtxInfo.push_back(skinWeight(boneId, weight));
                            }
                        }

                        break;
                    }
                    default:
                    {
                        DebugPrint(_M_TEXT("unknown vertextype %d\n"), vtxType);
                    }
                    break;
                }
                phyData->ReleaseVertexInterface(vi);
            }
            else
            {
                DebugPrint(_M_TEXT("Can't get VertexInterface\n"));
            }
            m_SkinVtx.push_back(vtxInfo);
        }

        // Physique適用時の初期行列を取得
        for(int i = 0; i < m_MaxBones.Count(); i++)
        {
            Matrix3 mtx;
            phyExport->GetInitNodeTM(m_MaxBones[i], mtx);

            Control* c = m_MaxBones[i]->GetTMController();
            if( (c->ClassID() == BIPSLAVE_CONTROL_CLASS_ID) ||
                (c->ClassID() == BIPBODY_CONTROL_CLASS_ID) ||
                (c->ClassID() == FOOTPRINT_CLASS_ID) )
 			{
                RemoveNonUniformScale(mtx);
            }
            Dcc::RMtx44 rmtx =  maxMtxToRMtx(mtx);
            m_BonesInitMtx.push_back(rmtx);
        }

        // スキンモデル自体のバインドポーズも登録しておく
        Matrix3 skinInitMtx = m_pMaxNode->GetNodeTM(0);
        phyExport->GetInitNodeTM(m_pMaxNode, skinInitMtx);
        //m_MaxBones.AppendNode(m_pMaxNode);
        //m_BonesInitMtx.push_back(maxMtxToRMtx(skinInitMtx));

        phyExport->ReleaseContextInterface(phyData);
        mod->ReleaseInterface(I_PHYINTERFACE, phyExport);
    }
    else if(mod->ClassID() == SKIN_CLASSID)
    {
        ISkin* skin = static_cast<ISkin*>(mod->GetInterface( I_SKIN ));
        ISkinContextData* skinData = skin->GetContextInterface(m_pMaxNode);
        int numVtx = skinData->GetNumPoints();

        // スキンに割り当てられているボーンを列挙
        for(int i = 0; i < skin->GetNumBones(); i++)
        {
            INode* bone = skin->GetBone(i);
            m_MaxBones.AppendNode(bone);
            assert(i == m_MaxBones.IndexOf(bone));
            // スキン適用時の初期行列を取得
            Matrix3 mtx;
            int ret = skin->GetBoneInitTM(bone, mtx);
            Dcc::RMtx44 rmtx = maxMtxToRMtx(mtx);
            /*
            RVec3 scale, rotate, trans;
            rmtx.GetTransform(scale, rotate, trans);

            DebugPrint("Bone(%s) bind pose\n", bone->GetName());
            DebugPrint("CafeScaleAfter x:%f, y:%f, z:%f\n", scale.x, scale.y, scale.z);
            DebugPrint("CafeRotAfter   x:%f, y:%f, z:%f\n", rotate.x, rotate.y, rotate.z);
            DebugPrint("CafeTransAfter x:%f, y:%f, z:%f\n", trans.x, trans.y, trans.z);
            DebugPrint("\n");

            */
            if(ret == SKIN_OK)
            {
                m_BonesInitMtx.push_back(rmtx);
            }
            else
            {
                // ボーンのバインドポーズがない場合とりあえず単位行列をセット
                rmtx.Ident();
                m_BonesInitMtx.push_back(rmtx);
                DebugPrint(_M_TEXT("bone() dosn't has bind pose\n"), bone->GetName());
            }
        }

        // スキンモデル自体のバインドポーズも登録しておく
        Matrix3 skinInitMtx;
        skin->GetSkinInitTM(m_pMaxNode, skinInitMtx);
        m_MaxBones.AppendNode(m_pMaxNode);
        m_BonesInitMtx.push_back(maxMtxToRMtx(skinInitMtx));

        // 頂点毎に割り当てられたボーンとウェイトを取得
        for(int i = 0; i < numVtx; i++)
        {
            int numBone = skinData->GetNumAssignedBones(i);
            skinVtxInfo vtxInfo;
            for(int j = 0; j < numBone; j++)
            {
                int boneId = skinData->GetAssignedBone(i, j);
                //int boneWeight = int(skinData->GetBoneWeight(i, j) * RVtxWeight::WEIGHT_MAX + 0.5f);
                float boneWeight = skinData->GetBoneWeight(i, j);
                // ボーンウェイトが0の時は追加しない。
                if(boneWeight > Dcc::R_SAME_TOLERANCE_F)
                {
                    vtxInfo.push_back(skinWeight(boneId, boneWeight));
                }
                else
                {
                    boneWeight = 0.0f;
                }
            }
            m_SkinVtx.push_back(vtxInfo);
        }
        mod->ReleaseInterface(I_SKIN, skin);
    }

#if 0
    // ボーン情報を表示
    for(int i = 0; i < maxBones.Count(); i++)
    {
        Matrix3 mtx;
        RMtx44 rmtx = bonesInitMtx[i];
        RVec3 s,r,t;
        rmtx.GetTransform(s, r, t);
        DebugPrint("bone(%s) scale(%6f, %6f, %6f), trans(%6f, %6f, %6f)\n", maxBones[i]->GetName(), s.x, s.y, s.z, t.x, t.y, t.z);
    }
#endif
    return true;
}



// Physiqueからノードのバインド行列を再帰的に取り出す
void CNodeTree::ResolvePhysiqueInitMtxRecursive(CNodeTree* node, IPhysiqueExport* phyExport)
{
    bool needRelease = false;
    if(!phyExport)
    {
        if(m_pSkinMod && m_pSkinMod->ClassID() == Class_ID(PHYSIQUE_CLASS_ID_A, PHYSIQUE_CLASS_ID_B))
        {
            phyExport = static_cast<IPhysiqueExport*>(m_pSkinMod->GetInterface(I_PHYINTERFACE));
            needRelease = true;
            //DebugPrint("skin mesh begin %s\n", name.c_str());
        }
        else
        {
            //DebugPrint("not skin mesh%s\n", name.c_str());
        }
    }

    if(phyExport)
    {
        nodeTreeList::iterator it;
        for(it = node->m_Children.begin(); it != node->m_Children.end(); it++)
        {
            CNodeTree* child = (*it);
            Dcc::RVec3 s,r,t;
            Matrix3 tm;
            Dcc::RMtx44 mtx;
            int ret = phyExport->GetInitNodeTM(child->m_pMaxNode, tm);
            mtx = maxMtxToRMtx(tm);
            if(child->m_IsBone)
            {
                //DebugPrint("\talready has m_Init mtx %s\n", child->name.c_str());
                if(ret == MATRIX_RETURNED)
                {
                    child->m_InitMtx.GetTransform(s, r, t);
                    //DebugPrint("\t\told: rotate(%6f, %6f, %6f), trans(%6f, %6f, %6f)\n",  r.x, r.y, r.z, t.x, t.y, t.z);
                    Dcc::RVec3 ns,nr,nt;
                    mtx.GetTransform(ns, nr, nt);
                    //DebugPrint("\t\tnew: rotate(%6f, %6f, %6f), trans(%6f, %6f, %6f)\n",  nr.x, nr.y, nr.z, nt.x, nt.y, nt.z);
                    ns -= s;
                    nr -= r;
                    nt -= t;
                    //DebugPrint("\t\tdiff: rotate(%6f), scale(%6f), trans(%6f)\n", nr.Length(), ns.Length(), nt.Length());
                }
            }
            else
            {
                if(ret == MATRIX_RETURNED)
                {
                    //DebugPrint("\tfind m_Init mtx %s(%d)\n", child->name.c_str(), ret);
                    child->m_InitMtx = maxMtxToRMtx(tm);
                    child->m_IsBone = true;
                    child->m_InitMtx.GetTransform(s, r, t);
                    //DebugPrint("\t\tnew: rotate(%6f, %6f, %6f), trans(%6f, %6f, %6f)\n",  r.x, r.y, r.z, t.x, t.y, t.z);
                }
                else
                {
                    //DebugPrint("\tnot find m_Init mtx %s(%d)\n", child->name.c_str(), ret);
                }
            }
            ResolvePhysiqueInitMtxRecursive(child, phyExport);
            //ResolvePhysiqueInitMtxRecursive(child);
        }
    }

    if(needRelease)
    {
        m_pSkinMod->ReleaseInterface(I_PHYINTERFACE, phyExport);
        //DebugPrint("skin mesh end %s\n", name.c_str());
    }
}

// Physiqueからノードのバインド行列を再帰的に取り出す
void CNodeTree::ResolvePhysiqueInitMtxRecursiveAll(CNodeTree* root)
{
    ResolvePhysiqueInitMtxRecursive(root);
    // 子も再帰的に処理
    nodeTreeList::iterator it;
    for(it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->ResolvePhysiqueInitMtxRecursiveAll(root);
    }
}


void CNodeTree::SetInitMtxFromCurrentTM()
{
    if(m_IsBone) // && !m_pSkinMod)
    {
        Dcc::RMtx44 mtx = maxMtxToRMtx(m_pMaxNode->GetNodeTM(0));
        m_pRNode->setBindMatrix(mtx);
    }
    else if(m_pSkinMod)
    {
        //m_InitMtx.Ident();
    }

    nodeTreeList::iterator it;
    for(it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->SetInitMtxFromCurrentTM();
    }
}


////////////////////////////////////////////////
// ノードが対応しているライトかどうか？
static bool isLightNode(INode* node)
{
    ObjectState os = node->EvalWorldState(0);
    if (os.obj)
    {
        if(os.obj->SuperClassID() == LIGHT_CLASS_ID)
        {
            GenLight* light = (GenLight*)os.obj;
            LightState ls;
            Interval valid = FOREVER;
            light->EvalLightState(0, valid, &ls);
            if	(    ls.type == OMNI_LGT
                  || ls.type == SPOT_LGT
                  || ls.type == DIRECT_LGT
                  //|| ls.type == AMBIENT_LGT
                )
            {
                //MCHAR* name = node->GetName();
                return true;
            }
        }
    }
    return false;
}


// ライトを再帰的に取り出す
void CNodeTree::setLightsRecursive(RSceneLights &lights, int start, int end, int frameTicks)
{
    ObjectState os = m_pMaxNode->EvalWorldState(0);
    //if (os.obj && !maxNode->IsHidden())
    // ライトのみ
    if (os.obj && os.obj->SuperClassID() == LIGHT_CLASS_ID)
    {
        GenLight* maxlit = (GenLight*)os.obj;
        LightState ls;
        Interval valid = FOREVER;
        // サポートするライトかどうか？
        maxlit->EvalLightState(0, valid, &ls);
        if(ls.type == OMNI_LGT || ls.type == SPOT_LGT || ls.type == DIRECT_LGT )
        {
            // RLightを作成して設定する。
            RLight* rlit = lights.GetRLight();

            // -----------------------------------------------
            // アニメーションしない値を設定
            rlit->SetName(m_Name.c_str());

            // 種類やスポットライトのパラメータなどを出力
            switch(ls.type)
            {
            case SPOT_LGT:
                {
                    rlit->SetLightKind(RLight::Spot);
                    //rlit->SetAngleAttenuation(true, ls.hotsize * 0.5f, ls.fallsize * 0.5f);
                }
                break;
            case DIRECT_LGT:
                rlit->SetLightKind(RLight::Directional);
                break;
            case OMNI_LGT:
            default:
                rlit->SetLightKind(RLight::Point);
                break;
            }

            // カラーがアニメーションしているかどうかチェック
            bool hasLightColAnim = false;
            Control* c = maxlit->GetColorControl();
            if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
            {
                hasLightColAnim = true;
            }

            // ----------------------------
            // アニメーションする値を設定
            int t;
            Dcc::RVec3 scale, rot, trans;
            Matrix3 tm;
            Dcc::RMtx44 mtx;
            Point3 dir, target;

            bool firstFrame = true;
            bool useDistAtten = maxlit->GetUseAtten() == TRUE;
            rlit->EnableDistAttenAnim(useDistAtten);
            for(int f = start; f <= end; f++)
            {
                t = f * frameTicks;
                maxlit->EvalLightState(t, valid, &ls);
                // 姿勢情報を取り出す
                Dcc::RVec3 scale, rot, trans;
                Matrix3 tm = m_pMaxNode->GetNodeTM(t);
                Dcc::RMtx44 mtx = maxMtxToRMtx(tm);
                mtx.GetTransform(scale, rot, trans);
                // 位置を設定
                rlit->SetTransAnim(trans);

                // ライトの方向、ターゲットを設定
                if(m_pMaxNode->GetTarget())
                {
                    // ターゲットが有ればその方向を使う
                    Matrix3 ttm;
                    m_pMaxNode->GetTargetTM(t, ttm);
                    target = ttm.GetTrans();
                    dir = ttm.GetTrans() - tm.GetTrans();
                }
                else
                {
                    // Z方向の単にベクトルをライトの行列で回転させる
                    Matrix3 rotTm(tm);
                    rotTm.NoTrans();
                    dir = rotTm.VectorTransform(Point3(0,0,-1));
                    target = tm.GetTrans() + dir;
                }

                dir = dir.Normalize();

                if(GetNintendoOptions()->IsEnableZYAxisConversion())
                {
                    rlit->SetDirectionAnim(Dcc::RVec3(dir.x, dir.z, -dir.y));
                    rlit->SetAimAnim(Dcc::RVec3(target.x, target.z, -target.y));
                }
                else
                {
                    rlit->SetDirectionAnim(Dcc::RVec3(dir.x, dir.y, dir.z));
                    rlit->SetAimAnim(Dcc::RVec3(target.x, target.y, target.z));
                }

                // 角度減衰のアニメーションを出力
                rlit->SetAngleAttenAnim(ls.hotsize * 0.5f, ls.fallsize * 0.5f);

                // 距離減衰のアニメーションを出力
                if(useDistAtten)
                {
                    rlit->SetDistAttenAnim(ls.attenStart, ls.attenEnd);
                }

                // 色や強さを出力
                // ライトがoffの時ls.color が黒にされてしまうので、GenLight
                // から取得する。
                // NW4Fでは強度をかけないで出力する
                Color col = maxlit->GetRGBColor(t);

                // maxの拡張効果でディフューズとスペキュラーにこのライトを使用するかどうか指定できる
                /*
                Dcc::RVec3 dif, spec;
                dif = Dcc::RVec3(col.r, col.g, col.b);
                if(maxlit->GetAffectDiffuse() == TRUE)
                {
                    dif = Dcc::RVec3(col.r, col.g, col.b);
                }
                if(maxlit->GetAffectSpecular() == TRUE)
                {
                    spec = Dcc::RVec3(col.r, col.g, col.b);
                }
                */

                Dcc::RVec3 dif(col.r, col.g, col.b);
                //if(hasLightColAnim)
                {
                    rlit->SetColor0Anim(dif);

                    bool lightOn = ls.on == TRUE;
                    if(lightOn)
                    {
                        lightOn = Dcc::RSnapToZero(ls.intens) > 0.0f;
                    }
                    rlit->SetVisibleAnim(lightOn);
                }

                if(firstFrame) firstFrame = false;
            }
        }
    }
    // 子も再帰的に処理
    nodeTreeList::iterator it;
    for(it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->setLightsRecursive(lights, start, end, frameTicks);
    }
}


static void getTransformZXYRotate(Matrix3 tm, Dcc::RVec3& scale, Dcc::RVec3& rotate, Dcc::RVec3& translate)
{
    Dcc::RMtx44 mtx = maxMtxToRMtx(tm);
    mtx.GetTransform(scale, rotate, translate);

    AffineParts ap;
    decomp_affine(tm, &ap);
    Matrix3 rTm;
    ap.q.MakeMatrix(rTm);

    if (GetNintendoOptions()->IsEnableZYAxisConversion())
    {
        rTm.PreRotateX(-HALFPI);
        Quat quat(rTm);

        float euler[3];
        QuatToEuler(quat, euler, EULERTYPE_YXZ);
        // 座標系変更
        rotate.x = euler[1];
        rotate.y = euler[2];
        rotate.z = -euler[0];
        rotate *= RAD_TO_DEG;
    }
    else
    {
        //rTm.PreRotateX(-HALFPI);
        Quat quat(rTm);

        float euler[3];
        QuatToEuler(quat, euler, EULERTYPE_ZXY);
        rotate.x = euler[1];
        rotate.y = euler[2];
        rotate.z = euler[0];
        rotate *= RAD_TO_DEG;
    }
}

// ボーンに挟まれたボーンでないノードを再帰的に調べて返す。
void CNodeTree::CheckNonBoneIsExistBetweenBonesRecursive(nodeTreeList& nodelist)
{
    if(IsNonBoneIsExistBetweenBones())
    {
        nodelist.push_back(this);
    }

    // 子も再帰的に処理
    nodeTreeList::iterator it;
    for(it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->CheckNonBoneIsExistBetweenBonesRecursive(nodelist);
    }
}


static bool isChildrenBone(CNodeTree* p)
{
    nodeTreeList::iterator it;

    for(it = p->m_Children.begin(); it != p->m_Children.end(); it++)
    {
        if((*it)->m_IsBone)
        {
            return true;
        }
        if(isChildrenBone(*it))
        {
            return true;
        }
    }
    return false;
}

// ボーンに挟まれたボーンでないノードか調べる。
bool CNodeTree::IsNonBoneIsExistBetweenBones()
{
    if(m_IsBone) return false;

    bool isParentBone = false;
    bool isChildBone = false;

    CNodeTree* p = m_pParent;

    while(p)
    {
        if(p->m_IsBone)
        {
            isParentBone = true;
            p = NULL;
            break;
        }
        p = p->m_pParent;
    }

    if(isParentBone)
    {
        if(isChildrenBone(this))
        {
            return true;
        }
    }

    return false;
}

static bool hasForceExportTrans(INode* node)
{
    if(node == NULL) return false;

    ICustAttribContainer* cc = node->GetCustAttribContainer();
    if(cc == NULL) return false;

    CustAttrib* attrib = NULL;
    // すでにNw4fNodeCustAttribが存在すればそれを返す
    for(int i = 0; i < cc->GetNumCustAttribs(); i++)
    {
        if(cc->GetCustAttrib(i)->ClassID() == NINTENDOEXPORT_NODE_ATTRIB_CLASS_ID)
        {
            attrib = cc->GetCustAttrib(i);
            break;
        }
    }
    // Nw4fNodeCustAttribが有る場合はフラグをチェック
    if(attrib == NULL) return false;

    IParamBlock2* pblock = attrib->GetParamBlock(0);
    if(pblock == NULL) return false;

    // TODO:
    //return (pblock->GetInt(nw4f_node_forceKeyTranslate) == TRUE);
    return false;
}

#define CAM_PB_FOV						0
#define CAM_PB_TDIST					1
#define CAM_PB_HITHER					2
#define CAM_PB_YON						3
#define CAM_PB_NRANGE					4
#define CAM_PB_FRANGE					5

// カメラを再帰的に取り出す
void CNodeTree::SetCamerasRecursive(RSceneCameras &cameras, int start, int end, int frameTicks)
{
    ObjectState os = m_pMaxNode->EvalWorldState(start);
    // カメラのみ
    if (os.obj && os.obj->SuperClassID() == CAMERA_CLASS_ID)
    {
        GenCamera* cam = (GenCamera*)os.obj;
        Interval valid = FOREVER;
        // サポートするカメラかどうか？
        RCamera* rcam = cameras.GetRCamera();
        if(rcam)
        {
            rcam->SetName(m_Name.c_str());
            //rcam->setForceExportKeyFlag(m_DoesExportScaleKey, m_DoesExportRotateKey, m_DoesExportTransKey);

            // レンダリング解像度からアスペクト比などを取得
            Interface* ip = GetCOREInterface();
            float w,h,asp;
            w = (float)ip->GetRendWidth();
            h = (float)ip->GetRendHeight();
            asp = w / h;

            if(cam->IsOrtho() == TRUE)
            {
                // 正射影
                rcam->SetProjectionMode(Dcc::RCamera::ORTHO);
            }
            else
            {
                // 透視変換
                rcam->SetProjectionMode(Dcc::RCamera::PERSP);
            }

            if(m_pMaxNode->GetTarget())
            {
                // ターゲットが有ればその位置を使う
                rcam->SetRotateMode(Dcc::RCamera::ROTATE_AIM);
            }
            else
            {
                rcam->SetRotateMode(Dcc::RCamera::ROTATE_EULER_ZXY);
            }

            int t;
            float hFov, vFov;
            float nearClip, farCilp;
            float tDist, oHeight;
            Dcc::RVec3 scale, rot, trans;
            Dcc::RVec3 tScale, tRot, tTrans;
            Matrix3 tm, ttm;
            Dcc::RMtx44 tmtx;

            // SRT以外のアニメーション要素を調査
            /*
            bool hasFovAnim, hasNearAnim, hasFarAnim, hasOrthoHAnim;
            hasFovAnim =  hasNearAnim = hasFarAnim = hasOrthoHAnim = false;
            {
                rcam->setTargetAnimForceExportKeyFlag(hasForceExportTrans(m_pMaxNode->GetTarget()));

                IParamBlock* pb = (IParamBlock*)(cam->SubAnim(0));
                Control* c;
                // FOV
                c = pb->GetController(CAM_PB_FOV);
                if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                {	hasFovAnim = true;}
                // ClipNear
                c = pb->GetController(CAM_PB_HITHER);
                if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                {	hasNearAnim = true;}
                //ClipFar
                c = pb->GetController(CAM_PB_YON);
                if(c && ((c->NumKeys() > 0) || c->IsAnimated()))
                {	hasFarAnim = true;}
                // OrthoHeight(ターゲットの距離が変わると高さも変わる・・はず) とりあえず無効に
                //c = pb->GetController(CAM_PB_TDIST);
                //if(c && c->NumKeys() > 0)
                //{	hasOrthoHAnim = true;}
            }
            */

            bool firstFrame = true;
            for(int f = start; f <= end; f++)
            {
                t = f * frameTicks;

                rcam->SetAspectAnim(asp);

                // fovを計算する。
                // maxのfovは幅方向なので、高さ方向に変換
                hFov = cam->GetFOV(t, valid);
                vFov = float(2.0*atan(tan(hFov/2.0f)/asp));
                rcam->SetFovYAnim(vFov * RAD_TO_DEG);

                // クリップの指定
                nearClip = cam->GetClipDist(t, CAM_HITHER_CLIP);
                farCilp = cam->GetClipDist(t, CAM_YON_CLIP);
                rcam->SetNearAnim(nearClip);
                rcam->SetFarAnim(farCilp);

                if(cam->IsOrtho() == TRUE)
                {
                    // ターゲットの距離とFovから幅と高さを割り出す
                    tDist = cam->GetTDist(t);
                    oHeight = tDist * tan(vFov/2.0f) * 2.0f;
                    rcam->SetOrthoHAnim(oHeight);
                }

                // 姿勢情報を取り出す
                tm = m_pMaxNode->GetObjTMBeforeWSM(t);

                getTransformZXYRotate(tm, scale, rot, trans);

                // 位置・回転・スケールを設定
                if(firstFrame)
                {
                    //rcam->SetScaling(scale);
                    //rcam->SetTranslation(trans);
                }
                //rcam->SetScaleAnim(scale);
                rcam->SetTransAnim(trans);

                if(m_pMaxNode->GetTarget())
                {
                    // ターゲットが有ればその位置を使う
                    m_pMaxNode->GetTargetTM(t, ttm);
                    tmtx = maxMtxToRMtx(ttm);
                    tmtx.GetTransform(tScale, tRot, tTrans);
                    //if(firstFrame) rcam->SetTargetPosition(tTrans);
                    rcam->SetTargetAnim(tTrans);

                    // ロール角を取得
                    Control* tmc = m_pMaxNode->GetTMController();
                    // コントローラがルックアップの時
                    if(tmc->ClassID() == Class_ID(LOOKAT_CONTROL_CLASS_ID, 0))
                    {
                        Control* roll = tmc->GetRollController();
                        if(roll)
                        {
                            float twist;
                            Interval valid = FOREVER;
                            roll->GetValue(t, &twist, valid);
                            //if(firstFrame) rcam->SetTwist(twist * RAD_TO_DEG);
                            //if(roll->NumKeys() > 0 || roll->IsAnimated())
                            {
                                rcam->SetTwistAnim(twist * RAD_TO_DEG);
                            }
                        }
                    }
                }
                else
                {
                    //if(firstFrame) rcam->SetRotation(rot);
                    rcam->SetRotateAnim(rot);
                }
                if(firstFrame) firstFrame = false;
            }

            // ターゲットがある場合は、アニメーションしていなければキーを消す
            /*
            if(m_pMaxNode->GetTarget())
            {
                bool isAnim = false;
                for(int iattr = RCamera::kAx; iattr <= RCamera::kAz; iattr++)
                {
                    RFloatArray& anim = rcam->mAnims[iattr].mFullValues;

                    float vmin, vmax, v;
                    vmin = vmax = anim.at(0);
                    for(UINT f = 1; f < anim.size(); f++)
                    {
                        v = anim.at(f);
                        vmin = (v < vmin)? v: vmin;
                        vmax = (v > vmax)? v: vmax;
                        if((vmax - vmin) > R_SAME_TOLERANCE_F)
                        {
                            isAnim = true;
                            break;
                        }
                    }
                }
                if(!isAnim)
                {
                    for(int iattr = RCamera::kAx; iattr <= RCamera::kAz; iattr++)
                    {
                        RFloatArray& anim = rcam->mAnims[iattr].mFullValues;

                        RFloatArray emptyArray;
                        anim.swap(emptyArray);
                    }
                }
            }
            */
        }
    }
    // 子も再帰的に処理
    nodeTreeList::iterator it;
    for(it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->SetCamerasRecursive(cameras, start, end, frameTicks);
    }
}

void CNodeTree::GetCamerasRecursive( INodeTab& camNodes )
{
    ObjectState os = m_pMaxNode->EvalWorldState(0);
    // カメラのみ
    if (os.obj && os.obj->SuperClassID() == CAMERA_CLASS_ID)
    {
        GenCamera* cam = static_cast<GenCamera*>(os.obj);
        Interval valid = FOREVER;
        // サポートするカメラかどうか？
        camNodes.AppendNode(m_pMaxNode);
    }

    // 子も再帰的に処理
    nodeTreeList::iterator it;
    for(it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->GetCamerasRecursive(camNodes);
    }
}

// ノードからSkinまたはPhysiqueを取得
bool CNodeTree::GetSkinModFromNode()
{
    assert(m_pMaxNode);

    Object* obj = m_pMaxNode->GetObjectRef();

    char buf[1024];

    while(obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
    {
        IDerivedObject* derivedObj = static_cast<IDerivedObject*>(obj);

        for(int i = 0; i < derivedObj->NumModifiers(); i++)
        {
            Modifier* mod = derivedObj->GetModifier(i);
            if(!mod || !mod->IsEnabled()) continue;

            if(mod->ClassID() == Class_ID(PHYSIQUE_CLASS_ID_A, PHYSIQUE_CLASS_ID_B))
            {
                IPhysiqueExport* phyExport = static_cast<IPhysiqueExport*>(mod->GetInterface(I_PHYINTERFACE));
                if(phyExport)
                {
                    IPhyContextExport* phyData = static_cast<IPhyContextExport*>(phyExport->GetContextInterface(m_pMaxNode));
                    if(phyData && phyData->GetNumberVertices() > 0)
                    {
                        sprintf_s(buf, "physique: %d\n", i);
                        OutputDebugStringA(buf);
                        phyExport->ReleaseContextInterface(phyData);
                        mod->ReleaseInterface(I_PHYINTERFACE, phyExport);
                        m_pSkinMod = mod;
                        return true;
                    }
                    mod->ReleaseInterface(I_PHYINTERFACE, phyExport);
                }
            }
            else if(mod->ClassID() == SKIN_CLASSID)
            {
                ISkin* skin = static_cast<ISkin*>(mod->GetInterface( I_SKIN ));
                if(skin)
                {
                    ISkinContextData* skinData = skin->GetContextInterface(m_pMaxNode);
                    if(skinData && skinData->GetNumPoints() > 0)
                    {
                        sprintf_s(buf, "skin: %d\n", i);
                        OutputDebugStringA(buf);
                        mod->ReleaseInterface(I_SKIN, skin);
                        m_pSkinMod = mod;
                        return true;
                    }
                    mod->ReleaseInterface(I_SKIN, skin);
                }
            }
        }

        obj = derivedObj->GetObjRef();
    }
    m_pSkinMod = NULL;
    return false;
}

// ノードからモーフモディファイアを取得
bool CNodeTree::GetMorphFromNode()
{
    assert(m_pMaxNode);

    m_pMorphMod = NULL;
    m_MorphCh.Init();
    m_MorphTargetNodes.Init();

    Object* obj = m_pMaxNode->GetObjectRef();

    //TCHAR buf[1024];

    while(obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
    {
        IDerivedObject* derivedObj = static_cast<IDerivedObject*>(obj);

        for(int i = 0; i < derivedObj->NumModifiers(); i++)
        {
            Modifier* mod = derivedObj->GetModifier(i);
            if(!mod || !mod->IsEnabled()) continue;

            if(mod->ClassID() == Class_ID(MR3_CLASS_ID))
            {
                m_pMorphMod = static_cast<MorphR3*>(mod);

                for(size_t ch = 0; ch < m_pMorphMod->chanBank.size(); ch++ )
                {
                    morphChannel const & chInfo = m_pMorphMod->chanBank[ch];
                    if( chInfo.mActive && chInfo.mConnection)
                    {
                        int ich = (int)ch;
                        m_MorphCh.Append(1, &ich);
                        m_MorphTargetNodes.AppendNode(chInfo.mConnection, true);
                    }
                }

                if(m_MorphCh.Count() == 0)
                {
                    // モーフターゲットがない
                    m_pMorphMod = NULL;
                    return false;
                }

                // morph の下にskinがないかチェックするためここでは抜けない
                //return true;
            }
            // モーフが存在した場合はそのままモディファイアスタックを下に探し続けて、
            // GetSkinModFromNode() で取得したモディファイアがないか調べる。
            if(m_pMorphMod && m_pSkinMod)
            {
                if(mod == m_pSkinMod)
                {
                    // モーフの下にスキンが見つかったので、モーフは無効にする
                    RLogger::LogMessagebyID(RLogger::kLogWRN_MorpherMustBeUnderSkinModifier, std::string(M_2_A(m_pMaxNode->GetName())));
                    m_pMorphMod = NULL;
                    return false;
                }
            }
        }

        obj = derivedObj->GetObjRef();
    }

    if(m_pMorphMod)
    {
        if(checkMorphTagetValidity())
        {
            return true;
        }
        m_pMorphMod = NULL;
        m_MorphCh.Init();
        m_MorphTargetNodes.Init();
    }
    return false;
}


// モーフターゲットの頂点数などが妥当かチェック
bool CNodeTree::checkMorphTagetValidity()
{
    assert(m_pMaxNode && m_pMorphMod && m_pTriObj);

    TriBoolPairVector objList;

    // モーフターゲットからメッシュを取り出す
    // 削除が必要かどうかのフラグも必要
    for(int i = 0; i < m_MorphTargetNodes.Count(); i++)
    {
        INode* target = m_MorphTargetNodes[i];
        assert(target);
        ObjectState os = target->EvalWorldState(0);
        Object* obj = os.obj;

        if(!obj) continue;

        // メッシュに変換できない物は出力しない
        if(		 (obj->SuperClassID() != GEOMOBJECT_CLASS_ID)
            ||	 (obj->ClassID() == Class_ID(TARGET_CLASS_ID, 0))
            ||	!(obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0))) )
        {
            continue;
        }

        TriObject* triobj = (TriObject*) obj->ConvertToType(0, Class_ID(TRIOBJ_CLASS_ID, 0));
        if (!triobj) continue;

        // objがTriobjectではない場合は明示的に削除が必要
        bool needDelTri = (obj != triobj);
        objList.push_back(TriBoolPairVector::value_type(triobj, needDelTri));
    }

    // ターゲットメッシュが一つも取得できない場合（多分ない）
    if(objList.size() == 0)
    {
        return false;
    }

    // ターゲットの頂点を調べる
    bool isValid = true;
    Mesh& mesh = m_pTriObj->GetMesh();

    //for(it = objList.begin(); it != objList.end(); it++)
    for(UINT iTarget = 0; iTarget < objList.size(); iTarget++)
    {
        INode* targetNode = m_MorphTargetNodes[iTarget];
        Mesh& targetMesh = objList[iTarget].first->GetMesh();
        //RLogger::RMessageDetail logID;
        // topology
        // 法線は頂点から計算するので、一緒に調べる
        if(m_ExportShapePosition || m_ExportShapeNormal)
        {
            if(  (mesh.getNumFaces() != targetMesh.getNumFaces())
                || (mesh.getNumVerts() != targetMesh.getNumVerts())
                )
            {
                isValid = false;
            }
        }

#if 0 // 頂点カラーの比較はここでは行わない
        if(m_ExportShapeColor)
        {
            if(	 (mesh.getNumMaps() != targetMesh.getNumMaps())
                || (mesh.getNumVertCol() != targetMesh.getNumVertCol())
                )
            {
                isValid = false;
            }

            // マップ
            for(int mp = 1; mp < MAX_MESHMAPS; mp++)
            {
                // マップの使用状態を比較
                if(mesh.mapSupport(mp) != targetMesh.mapSupport(mp))
                {
                    isValid = false;
                    break;
                }

                // マップ内の情報をチェック
                if(mesh.mapSupport(mp) && targetMesh.mapSupport(mp))
                {
                    MeshMap& map = mesh.Map(mp);
                    MeshMap& targetMap = targetMesh.Map(mp);

                    if(  (map.getNumFaces() != targetMap.getNumFaces())
                        || (map.getNumVerts() != targetMap.getNumVerts())
                  )
                    {
                        isValid = false;
                        break;
                    }
                }
            }
        }
#endif
        if(!isValid)
        {
            string mes = "Node: ";
            mes += M_2_A(m_pMaxNode->GetName());
            mes += ", Target: ";
            mes += M_2_A(targetNode->GetName());
            RLogger::LogMessagebyID(RLogger::kLogWRN_MorpherTargetTopologyIsDifferentFromOriginalMesh, mes);
            break;
        }
    }

    // ターゲットのtriobjを必要なら削除
    TriBoolPairVector::iterator it;
    for(it = objList.begin(); it != objList.end(); it++)
    {
        if(it->second) delete it->first;
    }

    return isValid;
}

///////////////////////////////////////////////////////////////////////////
// CSceneNode

// 登録されたノードからカメラを作成する。
void CSceneNode::SetCameras(RSceneCameras &cameras, int start, int end, int frameTicks)
{
    m_Root.SetCamerasRecursive(cameras, start, end, frameTicks);
}


// 登録されたノードからライトを作成する。
// また、レンダリング設定の環境光をアンビエントライトとして作成する。
void CSceneNode::setLights(RSceneLights &lights, int start, int end, int frameTicks)
{
    // ルートから再帰的にライトをセット
    m_Root.setLightsRecursive(lights, start, end, frameTicks);

    // 環境の周囲光をアンビエントライトとして出力
    RLight* rlit = lights.GetRLight();
    rlit->SetLightKind(RLight::Ambient);
    rlit->SetName("AmbientLight");

    Interval valid;
    int t;

    Color amb = GetCOREInterface()->GetAmbient(start* frameTicks, valid);
    //rlit->SetColor0(Dcc::RVec3(amb.r, amb.g, amb.b));

    Control* c = GetCOREInterface()->GetAmbientController();
    if(c) // && ((c->NumKeys() > 0) || c->IsAnimated()))
    {
        for(int f = start; f <= end; f++)
        {
            t = f * frameTicks;
            Color amb = GetCOREInterface()->GetAmbient(t, valid);
            rlit->SetColor0Anim(Dcc::RVec3(amb.r, amb.g, amb.b));
            rlit->SetVisibleAnim(true);
        }
    }
}

static int CompNodeByName(const void* item1, const void* item2)
{
    //INode* a = static_cast<INode*>(const_cast<void*>(item1));
    //INode* b = static_cast<INode*>(const_cast<void*>(item2));
    INode *a = *((INode**)item1);
    INode *b = *((INode**)item2);
    const MCHAR* an = a->GetName();
    const MCHAR* bn = b->GetName();
    return (_tcsicmp(an, bn));
}


#if !defined( NW4F_C3T_DELETE )
// フォグを作成
void CSceneNode::setFogs( FSceneFogs& sceneFog, int start, int end, int frameTicks )
{
    INodeTab camNodes;
    INode* node = NULL;
    Interval valid = FOREVER;

    Interface11* ip11 = GetCOREInterface11();
    int numAtmos = ip11->NumAtmospheric();


    if(numAtmos == 0)
    {
        return;
    }
    // ルートから再帰的にカメラを取得
    m_Root.GetCamerasRecursive(camNodes);

    if(camNodes.Count() == 0)
    {
        // alert no camera
        RLogger::LogMessage("Fog needs a Camera in scene.", RLogger::kWarning);
    }
    else if(camNodes.Count() > 1)
    {
        // alert many camera
        camNodes.Sort(CompNodeByName);
        node = camNodes[0];
        string msg = "Fog use enviroment range of camera(";
        msg += M_2_A(node->GetName());
        msg += ") ";
        RLogger::LogMessage(msg, RLogger::kWarning);
    }
    else // one camera
    {
        node = camNodes[0];
    }

    if(node)
    {

        ObjectState os = node->EvalWorldState(0);
        GenCamera* cam = (GenCamera*)os.obj;
        if(cam)
        {
            IParamBlock* pb = (IParamBlock*)(cam->SubAnim(0));
            //EnvNear
            //Control* envNearCnt = pb->GetController(CAM_PB_NRANGE);
            //EnvFar
            //Control* envFarCnt = pb->GetController(CAM_PB_FRANGE);

            //assert(envNearCnt);
            //assert(envFarCnt);

            // フォグの列挙
            int fogCount = 0;
            for (int i = 0; i < numAtmos; ++i)
            {
                Atmospheric* atmos = ip11->GetAtmospheric(i);
                //DebugPrint("atmos(%d): %s\n", i, atmos->GetName().data());
                if(atmos->ClassID() == Class_ID(FOG_CLASS_ID, 0))
                {
                    StdFog* maxfog = static_cast<StdFog*>(atmos);
                    FFog* ffog = sceneFog.GetFFog();
                    if(ffog)
                    {
                        string name = "fog_";
                        name += RGetNumberString(fogCount);
                        ffog->m_Name = name;
                        int t;
                        //float fogStart, fogEnd;

                        for(int f = start; f <= end; f++)
                        {
                            t = f * frameTicks;

                            // fog color
                            Color col = maxfog->GetColor(t);
                            ffog->SetFogColorAnim(Dcc::RVec3(col.r, col.g, col.b));


#if 1
                            // フォグの近接・遠方は考慮しない
                            float envNear = cam->GetEnvRange(t, ENV_NEAR_RANGE);
                            float envFar = cam->GetEnvRange(t, ENV_FAR_RANGE);
                            ffog->SetDistanceAttenAnim(envNear, envFar);
#else
                            //fog start, end
                            float envNear, envFar;
                            //envNearCnt->GetValue(t, &envNear, valid);
                            //envFarCnt->GetValue(t, &envFar, valid);
                            envNear = cam->GetEnvRange(t, ENV_NEAR_RANGE);
                            envFar = cam->GetEnvRange(t, ENV_FAR_RANGE);

                            float envLength = envFar - envNear;

                            float startPercent = maxfog->GetNear(t);
                            float endPercent = maxfog->GetFar(t);

                            float fogStart = envNear + envLength * startPercent;
                            float fogEnd = envNear + envLength * endPercent;

                            ffog->SetDistanceAttenAnim(fogStart, fogEnd);
#endif
                        }
                        fogCount++;
                    }
                }
            }
        }
    }
}
#endif

// ノードをツリーに追加する。
CNodeTree* CSceneNode::AddNodeInTree(INode* node, bool underRoot)
{
    if(!node) return NULL;

    // ノードが登録済みならそれを返す。
    CNodeTree* cNode = FindNodeInTree(node);
    if(cNode)
    {
        return cNode;
    }

    INode* parent = node->GetParentNode();
    if(!parent)
    {
        // ここに来る場合はROOTが適切に設定されていない
        //mprintf("ERROR: Internal error. m_Root is invalid in CSceneNode\n");
        RLogger::LogMessagebyID(RLogger::kLogERR_InternalError, "Root is invalid in CSceneNode");
        return NULL;
    }

    if(underRoot)
    {
        // シーンルート直下に追加（選択出力の場合）
        cNode = m_Root.AddChild(node);
    }
    else
    {

        // 親ノードが登録済みかどうか調べ、登録されていなければ
        // 親ノードを登録する。
        CNodeTree* cParent = FindNodeInTree(parent);
        if(!cParent)
        {
            cParent = AddNodeInTree(parent);
        }

        // 親ノードが登録されていない場合（多分あり得ない）
        assert(cParent);

        cNode = cParent->AddChild(node);
    }

    return cNode;
}

CNodeTree* CSceneNode::FindNodeInTree(INode* node)
{
    if(m_Root.m_pMaxNode == node) return &m_Root;
    return m_Root.FindChildRecursive(node);
}

void CSceneNode::RecursiveAddNodes(INode* node)
{
    if(!node) return;
    SetRootNode(node);
    //m_Root.m_DoesExportModel = false;
    m_Root.AddChildrenRecursive();
}

// ボーンをツリーに追加
void CSceneNode::SddBonesInTree(CNodeTree* cNode)
{
    for(int i= 0; i < cNode->m_MaxBones.Count(); i++)
    {
        CNodeTree* cBone = AddNodeInTree(cNode->m_MaxBones[i]);
        if(!cBone->m_IsBone)
        {
            cBone->m_InitMtx = cNode->m_BonesInitMtx[i];
            cBone->m_IsBone = true;
        }
    }

    // 子のボーンも再帰的に追加
    nodeTreeList::iterator it;
    for(it = cNode->m_Children.begin(); it != cNode->m_Children.end(); it++)
    {
        SddBonesInTree(*it);
    }
}

// 登録されたノードにスキンで必要なボーンが有るかどうか調べる。
bool CSceneNode::IsBonesExists(CNodeTree* cNode)
{
    for(int i= 0; i < cNode->m_MaxBones.Count(); i++)
    {
        INode* bone = cNode->m_MaxBones[i];
        CNodeTree* cBone = FindNodeInTree(bone);
        // ボーンがシーンツリーに登録されていない場合はエラー
        if(cBone == NULL)
        {
            return false;
        }
        // ボーンとして設定する。
        if(!cBone->m_IsBone)
        {
            cBone->m_InitMtx = cNode->m_BonesInitMtx[i];
            cBone->m_IsBone = true;
        }
    }

    // 子のボーンも再帰的に調べる
    bool ret;
    nodeTreeList::iterator it;
    for(it = cNode->m_Children.begin(); it != cNode->m_Children.end(); it++)
    {
        ret = IsBonesExists(*it);
        if(!ret)
        {
            return false;
        }
    }
    return true;
}

void flatternCNodes(nodeTreeList& nodeList, CNodeTree* parent)
{
    nodeList.push_back(parent);

    nodeTreeList::iterator it;
    for(it = parent->m_Children.begin(); it != parent->m_Children.end(); it++)
    {
        flatternCNodes(nodeList, *it);
    }
}

// 	ノード名の重複を解消する。
bool CSceneNode::SetNamesUnique()
{
    nodeTreeList flatNodes;
    flatternCNodes(flatNodes, &m_Root);

    nodeTreeList::iterator it = flatNodes.begin();
    it++;
    while(it != flatNodes.end())
    {
        int count = 1;
        nodeTreeList::iterator sit = flatNodes.begin();
        string iName = (*it)->GetName();

        //char orgName[256], newName[256];
        string orgName = iName;
        string newName = iName;
        //sprintf_s(orgName, "%s", iName);
        //sprintf_s(newName, "%s", iName);

        while(sit != it && sit != flatNodes.end())
        {
            string sName = (*sit)->GetName();
            //int cmp = _stricmp(newName, sName);
            // 同じ名前
            //if(cmp == 0)
            if(newName == sName)
            {
                CStr n;
                n.printf("%s_%02d", orgName.c_str(), count);
                newName = n.data();

                //sprintf_s(newName, "%s_%02d", orgName, count);
                (*it)->SetName(newName);
                //it->second.material->SetName(newName);
                sit = flatNodes.begin();
                count++;
                if(GetNintendoOptions()->m_Opt.m_WarnsNodeNameChanged)
                {
                    RLogger::LogMessagebyID(RLogger::kLogWRN_NodeNameIsChanged, orgName);
                }
            }
            else
            {
                sit++;
            }
        }
        it++;
    }

    return true;
}

// エクスポート用にRNode等を準備する。
bool CSceneNode::PrepareExport( void )
{
#if 0
    // 登録されたノードが参照しているボーンを追加
    addAllBones();
#else
    // 登録されたノードが参照しているボーンが追加されているか確認する。
    if(!IsAllBonesExists())
    {
        RLogger::LogMessagebyID(RLogger::kLogERR_InfluenceObj);
        return false;
    }
#endif

    SetNamesUnique();

    // シーンに存在するPhysiqueから未登録のバインド行列を取り出す
    m_Root.ResolvePhysiqueInitMtxRecursiveAll(&m_Root);

    return true;
}


//	シーン中のノードを処理する
bool CSceneNode::ProcessNodes( int f )
{
    m_INodeToRNode.clear();
    m_Root.CreateRNodeRecursive(NULL, m_INodeToRNode, f);

    return true;
}

//	シーン中のモデルを処理する
bool CSceneNode::ProcessModels( RSceneMaterials &sceneMaterials, std::vector<RNode*>& bones, Dcc::RExpOpt& rOpt )
{
    if(!m_Root.CreateRModelRecursive(NULL, sceneMaterials, m_INodeToRNode, bones, rOpt))
    {
        return false;
    }
    sceneMaterials.SetNamesUnique();

    return true;
}

// 階層内にボーンでないノードが含まれるか調べる。
void CSceneNode::CheckNonBoneIsExistBetweenBones(nodeTreeList& node)
{
    m_Root.CheckNonBoneIsExistBetweenBonesRecursive(node);
}

//全てのノードに指定された期間のアニメーションを設定する。
void CSceneNode::SetAnimations(int start, int end, int frameTicks)
{
    m_Root.SetAnimationRecursive(start, end, frameTicks);
}

//全てのノードに指定された期間のビジビリティアニメーションを設定する。
void CSceneNode::SetVisibilityAnimations(int start, int end, int frameTicks)
{
    m_Root.SetVisibilityAnimationRecursive(start, end, frameTicks);
}

//全てのノードに指定された期間のシェイプアニメーションを設定する。
void CSceneNode::SetShapeAnimations(int start, int end, int frameTicks)
{
    m_Root.SetShapeAnimationRecursive(start, end, frameTicks);
}

void CSceneNode::HideMorphTargets( std::vector<RNode*> Nodelist )
{
    INodeTab morphTargets;

    m_Root.CollectMorphTargetRecursive(morphTargets);
    m_Root.HideMorphTargetRecursive(morphTargets);
}

/////////////////////////////////////////////////////////////////

// 法線を計算するために使用するクラス
class VNormal
{
public:
    Point3 m_Norm;
    DWORD m_Smooth;
    VNormal* m_Next;
    BOOL m_Init;

    VNormal() {m_Smooth=0;m_Next=NULL;m_Init=FALSE;m_Norm=Point3(0,0,0);}
    VNormal(Point3 &n,DWORD s) {m_Next=NULL;m_Init=TRUE;m_Norm=n;m_Smooth=s;}
    ~VNormal()
    {
        if(m_Next)
        {
            delete m_Next;
            m_Next = NULL;
        }
    }
    void AddNormal(Point3 &n,DWORD s);
    Point3 &GetNormal(DWORD s);
    void Normalize();
};


void VNormal::AddNormal(Point3 &n,DWORD s)
{
    if (!(s&m_Smooth) && m_Init)
    {
        if (m_Next)
        {	m_Next->AddNormal(n,s);	}
        else
        {	m_Next = new VNormal(n,s); }
    }
    else
    {
        m_Norm += n;
        m_Smooth |= s;
        m_Init = TRUE;
    }
}

Point3 &VNormal::GetNormal(DWORD s)
{
    if (m_Smooth&s || !m_Next)
    {	return m_Norm; }
    else
    {	return m_Next->GetNormal(s);	}
}

// Normalize each normal in the list
void VNormal::Normalize()
{
    VNormal* ptr = m_Next;
    VNormal* prev = this;
    while (ptr)
    {
        if (ptr->m_Smooth&m_Smooth)
        {
            m_Norm += ptr->m_Norm;
            prev->m_Next = ptr->m_Next;
            ptr->m_Next = NULL;
            delete ptr;
            ptr = prev->m_Next;
        }
        else
        {
            prev = ptr;
            ptr = ptr->m_Next;
        }
    }
    m_Norm = ::Normalize(m_Norm);
    if (m_Next) m_Next->Normalize();
}

int getMapChFormat(INode* node, int ch, int& numElem, Dcc::RPrimVtx::ValueType& valueType)
{
    // 規定の要素数（頂点カラーはRGBA、それ以外はRGB）
    int userCh = -1;
    //numElem = (ch <= 0)? 4: 3;
    numElem = 3;
    valueType = Dcc::RPrimVtx::VALUE_FLOAT;
    MSTR key, name;
    key.printf(_M_TEXT("MapChannel:%d"), ch);
    name = _M_TEXT("");
    if (node->UserPropExists(key))
    {
        node->GetUserPropString(key, name);

        // ユーザー定義頂点カラーの場合は名前がnw4f_colorから始まる
        std::string sname = M_2_A(name.data());
        const char* namest = strstr(sname.c_str(), "nw4f_color");
        if(sname.c_str() == namest)
        {
            // 名前の後ろからアンダーバーを探す
            const char* dst = strrchr(sname.c_str(), '_');

            // アンダーバーが存在して、nw4f_colorに含まれるアンダーバーと違う場合
            if(dst && dst != strchr(sname.c_str(), '_'))
            {
                dst++;
                while(*dst != NULL)
                {
                    switch(*dst)
                    {
                    case '1':
                        numElem = 1;break;
                    case '2':
                        numElem = 2;break;
                    case '3':
                        numElem = 3;break;
                    case '4':
                        numElem = 4;break;
                    case 'f':
                        valueType = Dcc::RPrimVtx::VALUE_FLOAT; break;
                    case 'u':
                        valueType = Dcc::RPrimVtx::VALUE_UINT; break;
                    case 'i':
                        valueType = Dcc::RPrimVtx::VALUE_INT; break;
                    }
                    dst++;
                }
            }
        }
    }

    return userCh;
}


#define CheckNanMinus(X) (_isnan(X)? 0.0f : (((X) < 0.0f)? 0.0f : (X) ) )

// 頂点座標やテクスチャ座標の値をRModelに登録する。
bool CNodeTree::GetGeomVertex( Mesh* mesh, RModel &multiMateShape, cafeToMaxUVInfoVec& cafeUVToMaxUV, intintmap& cafeColToMaxCol )
{
    //int i,j;

    FOutShapeInfo &shapeInfo = multiMateShape.getShapeInfo();

    Matrix3 objTm = m_pMaxNode->GetObjectTM(0);
    Matrix3 nodeTm = m_pMaxNode->GetNodeTM(0);
    Matrix3 localObjTm = objTm * Inverse(m_pMaxNode->GetNodeTM(0));

    localObjTm.IdentityMatrix();
    localObjTm.PreTranslate(m_pMaxNode->GetObjOffsetPos());
    PreRotateMatrix(localObjTm, m_pMaxNode->GetObjOffsetRot());
    ApplyScaling(localObjTm, m_pMaxNode->GetObjOffsetScale());

    Point3 p = nodeTm.GetTrans();
    //shapeInfo.mCenterPos = RVec3(p.x, p.y, p.z);

    bool isNegativeScale = (DotProd(CrossProd(objTm.GetRow(0),objTm.GetRow(1)),objTm.GetRow(2)) < 0.0); //? true: false;

    // export shape animation
    bool doesExportShapeAnim = GetNintendoOptions()->m_Opt.ExportsShapeAnim() && m_MorphTargetNodes.Count() > 0;
    doesExportShapeAnim = doesExportShapeAnim && (m_ExportShapePosition || m_ExportShapeNormal || m_ExportShapeColor);

    // すべてのキーフレームで全ての頂点属性が同じか
    bool isAllShapeAnimSame = true;

    TriBoolPairVector triobjList;
    vector<Mesh*> targetMeshes;
    //vector<Mesh*>::iterator morphMeshIt;

    if(doesExportShapeAnim)
    {
        shapeInfo.m_KeyNames.push_back(GetName());

        // モーフターゲットからメッシュを取り出す
        // 削除が必要かどうかのフラグも必要
        CNodeTree* root = FindRootNode();

        for(int ti = 0; ti < m_MorphTargetNodes.Count(); ti++)
        {
            INode* target = m_MorphTargetNodes[ti];
            assert(target);
            ObjectState os = target->EvalWorldState(0);
            Object* obj = os.obj;

            if(!obj) continue;

            // メッシュに変換できない物は出力しない
            if(		 (obj->SuperClassID() != GEOMOBJECT_CLASS_ID)
                ||	 (obj->ClassID() == Class_ID(TARGET_CLASS_ID, 0))
                ||	!(obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0))) )
            {
                continue;
            }

            TriObject* triobj = (TriObject*) obj->ConvertToType(0, Class_ID(TRIOBJ_CLASS_ID, 0));
            if (!triobj) continue;

            // objがTriobjectではない場合は明示的に削除が必要
            bool needDelTri = (obj != triobj);
            triobjList.push_back(TriBoolPairVector::value_type(triobj, needDelTri));
            targetMeshes.push_back(&(triobj->GetMesh()));
            CNodeTree* targetnode = root->FindChildRecursive(target);
            shapeInfo.m_KeyNames.push_back(targetnode->GetName());
        }

        // ターゲットメッシュが一つも取得できない場合（多分ない）
        if(triobjList.size() == 0)
        {
            doesExportShapeAnim =  false;
        }
        assert(m_MorphTargetNodes.Count() == targetMeshes.size());
    }


    // 頂点座標
    if(mesh->getNumVerts() <= 0)
    {
        //mprintf("WARNING: No verticies in mesh(%s)\n", m_pMaxNode->GetName());
        string mes = "No verticies in mesh(";
        mes += M_2_A(m_pMaxNode->GetName());
        mes += ")";
        RLogger::LogMessage(mes, RLogger::kWarning);

        return false;
    }

    Matrix3 y_up;
    y_up.RotateX(PI * 0.5f);

    Dcc::RVec3Array* pos_array = new Dcc::RVec3Array(mesh->getNumVerts());

    if (GetNintendoOptions()->IsEnableZYAxisConversion())
    {
        for (int i = 0; i < mesh->getNumVerts(); i++)
        {
            Point3 v = localObjTm * mesh->verts[i];
            // y-upに変換
            (*pos_array)[i] = Dcc::RVec3(v.x, v.z, -v.y).SnapToZero();
        }
    }
    else
    {
        for (int i = 0; i < mesh->getNumVerts(); i++)
        {
            Point3 v = localObjTm * mesh->verts[i];
            (*pos_array)[i] = Dcc::RVec3(v.x, v.y, v.z).SnapToZero();
        }
    }


    shapeInfo.m_VtxInfos[Dcc::RPrimVtx::POS0] =
        Dcc::ROutShapeVtxInfo(3, Dcc::RPrimVtx::VALUE_FLOAT, pos_array);

    // position shape anim
    if(doesExportShapeAnim && m_ExportShapePosition)
    {
        // すべてのキーシェイプで頂点の位置が同じかどうか
        bool isAllKeyshapePosSame = true;

        for(UINT ti = 0; ti < targetMeshes.size(); ti++)
        {
            INode* targetNode = m_MorphTargetNodes[ti];
            Mesh* targetMesh = targetMeshes[ti];
            Dcc::RVec3Array* morphPos_array = new Dcc::RVec3Array(mesh->getNumVerts());
            Matrix3 targetObjTm = targetNode->GetObjectTM(0);
            Matrix3 targetLocalObjTm = targetObjTm * Inverse(targetNode->GetNodeTM(0));

            targetLocalObjTm.IdentityMatrix();
            targetLocalObjTm.PreTranslate(targetNode->GetObjOffsetPos());
            PreRotateMatrix(targetLocalObjTm, targetNode->GetObjOffsetRot());
            ApplyScaling(targetLocalObjTm, targetNode->GetObjOffsetScale());

            //bool isSame = true;
            Dcc::RVec3 pos;

            if (GetNintendoOptions()->IsEnableZYAxisConversion())
            {
                for (int i = 0; i < mesh->getNumVerts(); i++)
                {
                    Point3 v = targetLocalObjTm * targetMesh->verts[i];
                    // y-upに変換
                    pos = Dcc::RVec3(v.x, v.z, -v.y).SnapToZero();
                    if(isAllKeyshapePosSame)
                    {
                        isAllKeyshapePosSame = pos.IsEquivalent(pos_array->at(i));
                    }

                    (*morphPos_array)[i] = pos;
                }
            }
            else
            {
                for (int i = 0; i < mesh->getNumVerts(); i++)
                {
                    Point3 v = targetLocalObjTm * targetMesh->verts[i];
                    pos = Dcc::RVec3(v.x, v.y, v.z).SnapToZero();
                    if(isAllKeyshapePosSame)
                    {
                        isAllKeyshapePosSame = pos.IsEquivalent(pos_array->at(i));
                    }

                    (*morphPos_array)[i] = pos;
                }
            }


            shapeInfo.m_VtxInfos[Dcc::RPrimVtx::POS0].m_pArrays.push_back(morphPos_array);
        }
        // すべてのキーシェイプで頂点の位置が同じならキーシェイプを削除
        if(isAllKeyshapePosSame)
        {
            shapeInfo.m_VtxInfos[Dcc::RPrimVtx::POS0].m_pArrays.resize(1);
        }
        else
        {
            isAllShapeAnimSame = false;
        }
    }



    // 法線ベクトル
    // 3dsmaxの法線は他の属性と違いインデックスはないので、全ての面*3頂点分
    // 出力する
    Dcc::RVec3Array* nml_array = new Dcc::RVec3Array(mesh->getNumFaces() * 3);
    GetNormals(mesh, nml_array);
    shapeInfo.m_VtxInfos[Dcc::RPrimVtx::NRM0] =
        Dcc::ROutShapeVtxInfo(3, Dcc::RPrimVtx::VALUE_FLOAT, nml_array);

    // normal shape anim
    if(doesExportShapeAnim  && m_ExportShapeNormal)
    {
        // すべてのキーシェイプで頂点の法線が同じかどうか
        bool isAllKeyshapeNmlSame = true;
        for(UINT ti = 0; ti < targetMeshes.size(); ti++)
        {
            Mesh* targetMesh = targetMeshes[ti];
            Dcc::RVec3Array* targetNml_array = new Dcc::RVec3Array(mesh->getNumFaces() * 3);
            if(GetNormals(targetMesh, targetNml_array, nml_array))
            {
                isAllKeyshapeNmlSame = false;
            }
            shapeInfo.m_VtxInfos[Dcc::RPrimVtx::NRM0].m_pArrays.push_back(targetNml_array);
        }
        // すべてのキーシェイプで頂点の法線が同じならキーシェイプを削除
        if(isAllKeyshapeNmlSame)
        {
            shapeInfo.m_VtxInfos[Dcc::RPrimVtx::NRM0].m_pArrays.resize(1);
        }
        else
        {
            isAllShapeAnimSame = false;
        }
    }

    // ユーザ定義頂点カラーの出力
    assert(cafeColToMaxCol.size() == Dcc::RPrimVtx::VTX_COL_MAX);

    for(int colch = 0; colch < Dcc::RPrimVtx::VTX_COL_MAX; colch++)
    {
        int mapch = cafeColToMaxCol[colch];
        if(mapch >= MAP_ALPHA)
        {
            MeshMap& map = mesh->Map(mapch);
            UINT colID = Dcc::RPrimVtx::COL0 + colch;
            int numVtxCol = map.getNumVerts();
            int numElem;
            Dcc::RPrimVtx::ValueType vcVType = Dcc::RPrimVtx::VALUE_FLOAT;
            getMapChFormat(m_pMaxNode, mapch, numElem, vcVType);

            Dcc::RVec4Array* vcol_array = new Dcc::RVec4Array(numVtxCol);
            for (int i = 0; i < numVtxCol; i++)
            {
                UVVert vcol = map.tv[i];
                (*vcol_array)[i] = Dcc::RVec4(vcol.x, vcol.y, vcol.z, 1.0f).SnapToZero();
            }

            shapeInfo.m_VtxInfos[colID] =
                Dcc::ROutShapeVtxInfo(numElem, vcVType, vcol_array);

            // vtx color shape anim
            if(doesExportShapeAnim && m_ExportShapeColor)
            {
                bool doesAnimateThisVCol = true;
                vector<intintmap> keyshapeVcolMaps;

                // すべてのターゲットの頂点カラーチャンネルの出力情報を取得
                for(UINT ti = 0; ti < targetMeshes.size(); ti++)
                {
                    INode* targetNode = m_MorphTargetNodes[ti];
                    Mesh* targetMesh = targetMeshes[ti];
                    assert(targetNode && targetMesh);

                    intintmap vcolmap;
                    getCafeVColToMaxVColMap(targetNode, targetMesh, vcolmap);
                    keyshapeVcolMaps.push_back(vcolmap);
                }

                // すべてのターゲットがこの頂点カラーを持つことを確認
                for(UINT ti = 0; ti < targetMeshes.size(); ti++)
                {
                    Mesh* targetMesh = targetMeshes[ti];
                    int targetMapch = keyshapeVcolMaps[ti][colch];
                    // ターゲットに頂点カラーがなければこのチャンネルは出力しない
                    if(!targetMesh->mapSupport(targetMapch))
                    {
                        doesAnimateThisVCol = false;
                        break;
                    }
                    // ターゲットがベースと頂点数が違えばこのチャンネルは出力しない
                    else if(map.getNumVerts() != targetMesh->Map(targetMapch).getNumVerts())
                    {
                        doesAnimateThisVCol = false;
                        break;
                    }
                }

                // このチャンネルの頂点カラーをシェイプアニメーションとして出力
                if(doesAnimateThisVCol)
                {
                    // すべてのキーシェイプで頂点カラーが同じかどうか
                    bool isAllKeyshapeVColSame = true;

                    Dcc::RVec4 fvcol;
                    //bool isSame = true;
                    for(UINT ti = 0; ti < targetMeshes.size(); ti++)
                    {
                        Mesh* targetMesh = targetMeshes[ti];
                        int targetMapch = keyshapeVcolMaps[ti][colch];
                        MeshMap& targetMap = targetMesh->Map(targetMapch);
                        Dcc::RVec4Array* targetVCol_array = new Dcc::RVec4Array(numVtxCol);
                        for (int i = 0; i < numVtxCol; i++)
                        {
                            UVVert vcol = targetMap.tv[i];
                            fvcol = Dcc::RVec4(vcol.x, vcol.y, vcol.z, 1.0f).SnapToZero();
                            if(isAllKeyshapeVColSame)
                            {
                                isAllKeyshapeVColSame = fvcol.IsEquivalent(vcol_array->at(i));
                            }
                            (*targetVCol_array)[i] = fvcol;
                        }

                        shapeInfo.m_VtxInfos[colID].m_pArrays.push_back(targetVCol_array);
                    }

                    // すべてのキーシェイプで頂点カラーが同じならキーシェイプを削除
                    if(isAllKeyshapeVColSame)
                    {
                        shapeInfo.m_VtxInfos[colID].m_pArrays.resize(1);
                    }
                    else
                    {
                        isAllShapeAnimSame = false;
                    }
                }
            }
        }
    }

    // 頂点カラーの出力
    // maxの頂点カラー、頂点アルファ、頂点シェードから計算する。

    int numVcFaces = 0; // 頂点カラーで使う面の数
    int vcCh[3] = {0, MAP_SHADING, MAP_ALPHA};
    VertColor*	vcVerts[3] = {NULL, NULL, NULL};
    TVFace*		vcFaces[3] = {NULL, NULL, NULL};
    // 使用されているかどうか調査
    for(int chi = vc_color; chi <= vc_alpha; chi++)
    {
        int numVerts = mesh->Map(vcCh[chi]).getNumVerts();
        int numFaces = mesh->Map(vcCh[chi]).getNumFaces();
        if(numVcFaces == 0)
        {
            numVcFaces = numFaces;
        }

        if(numVerts > 0 && numFaces > 0 && numVcFaces == numFaces)
        {
            vcVerts[chi] = mesh->Map(vcCh[chi]).tv;
            vcFaces[chi] = mesh->Map(vcCh[chi]).tf;
        }
    }

    // 頂点カラー、アルファ、シェードのどれかが使用されていて
    // nw4f_color0が指定されていない場合、標準の頂点カラーをnw4f_color0に出力
    if(numVcFaces > 0 && cafeColToMaxCol[0] < MAP_ALPHA)
    {
        int numElem;
        Dcc::RPrimVtx::ValueType vcVType = Dcc::RPrimVtx::VALUE_FLOAT;
        // 頂点アルファが使われている場合はRGBA、無ければRGBとする。
        numElem = vcFaces[vc_alpha]? 4 : 3;
        //getMapChFormat(m_pMaxNode, 0, numElem, vcVType);
        Dcc::RVec4Array* vcol_array = new Dcc::RVec4Array(numVcFaces * 3);
        GetStandardVCol(numVcFaces, vcFaces, vcVerts, vcol_array);
        shapeInfo.m_VtxInfos[Dcc::RPrimVtx::COL0] =
            Dcc::ROutShapeVtxInfo(numElem, vcVType, vcol_array);

        // standrd vtx color shape anim
        if(doesExportShapeAnim && m_ExportShapeColor)
        {
            VertColor*	targetVcVerts[3] = {NULL, NULL, NULL};
            TVFace*		targetVcFaces[3] = {NULL, NULL, NULL};

            bool doesAnimateThisVCol = true;

            // すべてのターゲットがベースと同じ標準頂点カラーを持つことを確認
            for(UINT ti = 0; ti < targetMeshes.size(); ti++)
            {
                Mesh* targetMesh = targetMeshes[ti];
                for(int chi = vc_color; chi <= vc_alpha; chi++)
                {
                    if(vcFaces[chi] && vcVerts[chi])
                    {
                        if	(  !targetMesh->mapSupport(vcCh[chi])
                            || mesh->Map(vcCh[chi]).getNumVerts() != targetMesh->Map(vcCh[chi]).getNumVerts()
                            || mesh->Map(vcCh[chi]).getNumFaces() != targetMesh->Map(vcCh[chi]).getNumFaces()
                            )
                        {
                            doesAnimateThisVCol = false;
                            break;
                        }
                    }
                }

                if(!doesAnimateThisVCol)
                {
                    break;
                }
            }

            if(doesAnimateThisVCol)
            {
                // すべてのキーシェイプで頂点カラーが同じかどうか
                bool isAllKeyshapeVColSame = true;

                for(UINT ti = 0; ti < targetMeshes.size(); ti++)
                {
                    Dcc::RVec4Array* targetVCol_array = new Dcc::RVec4Array(numVcFaces * 3);
                    Mesh* targetMesh = targetMeshes[ti];

                    for(int chi = vc_color; chi <= vc_alpha; chi++)
                    {
                        int numTargetVerts = targetMesh->Map(vcCh[chi]).getNumVerts();
                        int numTargetFaces = targetMesh->Map(vcCh[chi]).getNumFaces();
                        if(numTargetVerts > 0 && numTargetFaces > 0 && numVcFaces == numTargetFaces)
                        {
                            targetVcVerts[chi] = targetMesh->Map(vcCh[chi]).tv;
                            targetVcFaces[chi] = targetMesh->Map(vcCh[chi]).tf;
                        }
                    }

                    if(GetStandardVCol(numVcFaces, targetVcFaces, targetVcVerts, targetVCol_array, vcol_array))
                    {
                        isAllKeyshapeVColSame = false;
                    }
                    shapeInfo.m_VtxInfos[Dcc::RPrimVtx::COL0].m_pArrays.push_back(targetVCol_array);
                }

                // すべてのキーシェイプで頂点カラーが同じならキーシェイプを削除
                if(isAllKeyshapeVColSame)
                {
                    shapeInfo.m_VtxInfos[Dcc::RPrimVtx::COL0].m_pArrays.resize(1);
                }
                else
                {
                    isAllShapeAnimSame = false;
                }

            }
        }
    }

    // すべてのキーフレームで全ての頂点属性が同じ場合
    // シェイプアニメーションの出力をしない
    if(doesExportShapeAnim && isAllShapeAnimSame)
    {
        // キーフレームの名前を削除
        shapeInfo.m_KeyNames.clear();
        m_pRNode->ClearShapeAnim();
    }

    // テクスチャ座標
    int mapCount = 0;

    // サポートするテクスチャ座標の最大値
    unsigned int texNumMax = Dcc::RPrimVtx::VTX_TEX_MAX;

    assert(cafeUVToMaxUV.size() <= Dcc::RPrimVtx::VTX_TEX_MAX);

    //Dcc::RIntArray* uvHintIdxs = new Dcc::RIntArray;
    //Dcc::RStringArray* uvAttribNames = new Dcc::RStringArray;

    Dcc::RIntArray& uvHintIdxs = shapeInfo.m_uvHintIdxs;
    Dcc::RStringArray& uvAttribNames = shapeInfo.m_uvAttribNames;
    //for(int i = 0; i < Dcc::RPrimVtx::VTX_TEX_MAX; i++)
    for(int i = 0; i < cafeUVToMaxUV.size(); i++)
    {
        int cafeUVCh = i;
        int maxUVch = cafeUVToMaxUV[i].maxUV;
        if (maxUVch > 0 && isMapChExist(mesh, maxUVch))
        {
            MeshMap& map = mesh->Map(maxUVch);
            UINT texid = Dcc::RPrimVtx::TEX0 + cafeUVCh;
            int numUVVtx = map.getNumVerts();

            Dcc::RVec2Array* uv_array = new Dcc::RVec2Array(numUVVtx);
            for (int i = 0; i < numUVVtx; i++)
            {
                UVVert  uv = map.tv[i];
                (*uv_array)[i] = Dcc::RVec2(uv.x, 1.0f - uv.y).SnapToZero();
            }

            shapeInfo.m_VtxInfos[texid] =
                Dcc::ROutShapeVtxInfo(2, Dcc::RPrimVtx::VALUE_FLOAT, uv_array);
        }
        uvHintIdxs.push_back(cafeUVToMaxUV[i].cafeUV);
        uvAttribNames.push_back(cafeUVToMaxUV[i].maxUVName);
    }
    //shapeInfo.m_pUvHintIdxs = uvHintIdxs;
    //shapeInfo.m_pUvAttribNames = uvAttribNames;

#if 0
    // マップチャンネル9以降が設定されていたら、警告を出す
    for (UINT mp = 9; mp < MAX_MESHMAPS; mp++)
    {
        if (mesh->mapSupport(mp) && mesh->Map(mp).getNumVerts() > 0)
        {
            string mes = "UV map channel over 9 is not export (";
            mes += node->GetName();
            mes += ")";
            RLogger::LogMessage(mes, RLogger::kWarning);
            break;
        }
    }
#endif


    // ユーザー定義頂点データ
#if 0
    for(int i = texNumMax + 1; i < MAX_MESHMAPS; i++)
    {
        if(mesh->mapSupport(i) == FALSE) continue;
        MeshMap& mmap = mesh->Map(i);
        // 頂点か面の数が0なら出力しない
        if(mmap.GetFlag(MESHMAP_USED) == FALSE) continue;
        if(mmap.getNumVerts() == 0) continue;
        if(mmap.getNumFaces() == 0) continue;


        int numElem, userCh;
        Point3 col;
        RVec2 val2;
        RVec3 val3;
        RVec4 val;
        Dcc::RPrimVtx::ValueType ucVType;

        userCh = getMapChFormat(node, i, numElem, ucVType);
        if(userCh >= 0)
        {
            // すでにチャンネルが使用されていたらエラー
            if(shapeInfo.m_VtxInfos[Dcc::RPrimVtx::USR0 + userCh].m_ArrayPtr != NULL)
            {
                RLogger::LogMessagebyID(RLogger::kLogERR_SameUserAttrib, node->GetName());
                return false;
            }
            int numFaces = mmap.getNumFaces();
#if 1
            RVec4Array* uc_array = new RVec4Array(numFaces * 3);
#else
            void* uc_array;
            switch(numElem)
            {
            case 1:
                uc_array = new RFloatArray(numFaces * 3); break;
            case 2:
                uc_array = new RVec2Array(numFaces * 3); break;
            case 3:
            default:
                uc_array = new RVec3Array(numFaces * 3); break;
            }
#endif
            int vi = 0;

            for(int f = 0; f < numFaces; f++)
            {
                for(int v = 0; v <= 2; v++)
                {
                    col = mmap.tv[mmap.tf[f].getTVert(v)];
#if 1
                    (*uc_array)[vi] = RVec4(col.x, col.y, col.z, 0);
#else
                    switch(numElem)
                    {
                    case 1:
                        (*((RFloatArray*)uc_array))[vi] = col.x;
                        break;
                    case 2:
                        (*((RVec2Array*)uc_array))[vi] = RVec2(col.x, col.y);
                        break;
                    case 3:
                    default:
                        (*((RVec3Array*)uc_array))[vi] = RVec3(col.x, col.y, col.z);
                        break;
                    }
#endif
                    vi++;
                }
            }
            shapeInfo.m_VtxInfos[Dcc::RPrimVtx::USR0 + userCh] =
                Dcc::ROutShapeVtxInfo(numElem, ucVType, uc_array);
        }
    }
#endif

    // ターゲットのtriobjを必要なら削除
    TriBoolPairVector::iterator it;
    for(it = triobjList.begin(); it != triobjList.end(); it++)
    {
        if(it->second) delete it->first;
    }

    return true;
}

bool CNodeTree::GetNormals( Mesh* mesh, Dcc::RVec3Array* nml_array, Dcc::RVec3Array* basenml_array /*= NULL*/ )
{
    MeshNormalSpec* spNormal = mesh->GetSpecifiedNormals();
    Point3 vn;  // Vertex normal
    Dcc::RVec3 nml;
    bool isSame = basenml_array != NULL;
    int idx;
    int zeroNormalCount = 0;
    if(spNormal && (spNormal->GetNumFaces() == mesh->getNumFaces()) )
    {
        // 「法線を編集」モディファイアで法線がカスタムされている場合
        for(int i = 0; i < spNormal->GetNumNormals(); i++)
        {
            vn = spNormal->GetNormalArray()[i];
            killNaN(vn);
            if(vn.FLength() < Dcc::R_SAME_TOLERANCE_F)
            {
                zeroNormalCount++;
            }
        }

        bool isEnableAxisConv = GetNintendoOptions()->IsEnableZYAxisConversion();
        for(int i = 0; i < spNormal->GetNumFaces(); i++)
        {
            for(int j = 0; j < 3; j++)
            {
                idx = i * 3 + j;
                vn = spNormal->GetNormal(i, j);
                killNaN(vn);
                nml = Dcc::RVec3(vn.x, vn.y, vn.z).SnapToZero();

                if(nml.Length() < Dcc::R_SAME_TOLERANCE_F)
                {
                    Point3 *verts = mesh->verts;
                    Point3 v0, v1, v2;
                    Face *f = &mesh->faces[i];
                    v0 = verts[f->v[0]];
                    v1 = verts[f->v[1]];
                    v2 = verts[f->v[2]];
                    Point3 fNrm = (v1-v0)^(v2-v1);
                    vn = fNrm.Normalize();
                    nml = Dcc::RVec3(vn.x, vn.y, vn.z).SnapToZero();
                }

                if (isEnableAxisConv)
                {
                    // y-upに変換
                    nml = Dcc::RVec3(nml.x, nml.z, -nml.y);
                }
                nml.Normalize();
                if(isSame)
                {
                    isSame = nml.IsEquivalent(basenml_array->at(idx));
                }
                (*nml_array)[idx] = nml;
            }
        }
    }
    else
    {
        // 通常のスムージンググループから法線を計算
        Face* f;
        Point3* verts;
        Point3 v0, v1, v2;
        Tab<VNormal> vNrms;
        Point3 fNrm;
        f = mesh->faces;
        verts = mesh->verts;
        vNrms.SetCount(mesh->getNumVerts());

        // Compute face and vertex surface normals
        for (int i = 0; i < mesh->getNumVerts(); i++)
        {	vNrms[i] = VNormal(); }

        for (int i = 0; i < mesh->getNumFaces(); i++)
        {
            // Calculate the surface normal
            f = &mesh->faces[i];
            v0 = verts[f->v[0]];
            v1 = verts[f->v[1]];
            v2 = verts[f->v[2]];
            fNrm = (v1-v0)^(v2-v1);
            //fNrm = fNrm.Normalize();
            for (int j = 0; j < 3; j++)
            {
                vNrms[f->v[j]].AddNormal(fNrm, f->smGroup);
            }
        }

        for (int i = 0; i < mesh->getNumVerts(); i++)
        {	vNrms[i].Normalize();	}

        for (int i = 0; i < mesh->getNumFaces(); i++)
        {
            f = &mesh->faces[i];
            v0 = verts[f->v[0]];
            v1 = verts[f->v[1]];
            v2 = verts[f->v[2]];
            fNrm = (v1-v0)^(v2-v1);
            fNrm = fNrm.Normalize();

            for(int j = 0; j < 3; j++)
            {
                idx = i * 3 + j;
                int vert = f->getVert(j);
                // no Smoothing Group
                if(f->smGroup == 0)
                {
                    vn = fNrm;
                }
                else
                {
                    vn = vNrms[vert].GetNormal(f->smGroup);
                }
                killNaN(vn);
                nml = Dcc::RVec3(vn.x, vn.y, vn.z).SnapToZero();

                if(nml.Length() < Dcc::R_SAME_TOLERANCE_F)
                {
                    zeroNormalCount++;
                    //RLogger::LogMessagebyID(RLogger::kLogWRN_ZeroNormalExist, std::string(M_2_A(m_pMaxNode->GetName())));
                    nml = Dcc::RVec3(fNrm.x, fNrm.y, fNrm.z).SnapToZero();
                }

                if (GetNintendoOptions()->IsEnableZYAxisConversion())
                {
                    // y-upに変換
                    nml = Dcc::RVec3(nml.x, nml.z, -nml.y);
                }

                if(isSame)
                {
                    isSame = nml.IsEquivalent(basenml_array->at(idx));
                }
                (*nml_array)[idx] = nml;
            }
        }
        vNrms.Resize(0);
    }

    if (zeroNormalCount > 0)
    {
        //std::string mes = M_2_A(m_pMaxNode->GetName());
        CStr mes;
        mes.printf("%s (%d vertex)", M_2_A(m_pMaxNode->GetName()), zeroNormalCount);

        RLogger::LogMessagebyID(RLogger::kLogWRN_ZeroNormalExist, mes.data());
    }

    // ベースメッシュの法線が指定されて、すべての法線が一致した場合はfalseを返す
    if(basenml_array && isSame)
    {
        return false;
    }
    return true;
}

bool CNodeTree::GetStandardVCol( int numVcFaces, TVFace** vcFaces, VertColor** vcVerts, Dcc::RVec4Array* col_array, Dcc::RVec4Array* basecol_array /*= NULL*/ )
{
    UINT idx;
    Point3 col, ac;
    float alpha;
    Dcc::RVec4 vCol;
    int vi = 0;

    bool isSame = basecol_array != NULL;

    for(int f = 0; f < numVcFaces; f++)
    {
        for(int v = 0; v <= 2; v++)
        {
            col.Set(1.0f,1.0f,1.0f);
            alpha = 1.0f;

            if(vcFaces[vc_color]) // 頂点カラー(RGBのみ)
            {
                idx = vcFaces[vc_color][f].getTVert(v);
                Point3 tc = vcVerts[vc_color][idx];
                col = vcVerts[vc_color][idx];
            }
            if(vcFaces[vc_shade]) // 頂点シェード
            {
                idx = vcFaces[vc_shade][f].getTVert(v);
                col *= vcVerts[vc_shade][idx];
            }
            if(vcFaces[vc_alpha]) // 頂点アルファ
            {
                idx = vcFaces[vc_alpha][f].getTVert(v);
                ac = vcVerts[vc_alpha][idx];
                alpha = (ac.x + ac.y + ac.z) / 3.0f; //明るさをアルファにする
            }

            vCol.x = CheckNanMinus(col.x);
            vCol.y = CheckNanMinus(col.y);
            vCol.z = CheckNanMinus(col.z);
            vCol.w = CheckNanMinus(alpha);
            vCol.SnapToZero();

            if(isSame)
            {
                isSame = vCol.IsEquivalent(basecol_array->at(vi));
            }
            (*col_array)[vi] = vCol;
            vi++;
        }
    }

    // ベースメッシュの頂点カラーが指定されて、すべての頂点カラーが一致した場合はfalseを返す
    if(basecol_array && isSame)
    {
        return false;
    }
    return true;
}

void CNodeTree::CollectMorphTargetRecursive( INodeTab& morphTargets )
{
    // モーフターゲットをリストに追加
    for(int iNode = 0; iNode < m_MorphTargetNodes.Count(); iNode++)
    {
        morphTargets.AppendNode(m_MorphTargetNodes[iNode]);
    }

    // 子ノードを再帰的に作成する。
    nodeTreeList::iterator it;
    for (it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->CollectMorphTargetRecursive(morphTargets);
    }

}

void CNodeTree::HideMorphTargetRecursive( INodeTab& morphTargets )
{
    // モーフターゲットリストにこのノードが追加されていたら、
    // メッシュを出力しない。
    int idx = morphTargets.IndexOf(m_pMaxNode);
    if(idx >= 0)
    {
        if(m_pTriObj && m_IsNeedDelTriObj)
        {
            delete m_pTriObj;
            m_IsNeedDelTriObj = false;
        }
        m_pTriObj = NULL;
    }

    // 子ノードを再帰的に作成する。
    nodeTreeList::iterator it;
    for (it = m_Children.begin(); it != m_Children.end(); it++)
    {
        (*it)->HideMorphTargetRecursive(morphTargets);
    }
}



// ポリゴン情報などをとりだし、RModel等に登録する。
bool CNodeTree::ExportGeomObject( Mesh* mesh, RModel &multiMateShape, std::vector<int>& VtxMtxIdxs, RSceneMaterials &sceneMaterials, MultiMtlInfo& multiMtl, cafeToMaxUVInfoVec& cafeUVToMaxUV, intintmap& cafeColToMaxCol )
{
    int i,j;

    // メッシュに面がなければ何もしない
    if(mesh->getNumFaces() <= 0) return true;

    // 頂点行列が渡されたらスキニングを有効にする
    Dcc::RShape::SkinningMode skinmode;
#if	0
    if(vertexBoneRefs)
    {
        skinmode = RPrimSet::SMOOTH;
        multiMateShape.setVertexBoneRefArray(vertexBoneRefs);
    }
    else
    {
        skinmode = RPrimSet::NONE;
    }
#else
        skinmode = Dcc::RShape::NO_SKINNING;
#endif

    // 頂点属性データ(頂点データ)を代入する FOutShapeInfo オブジェクトを取得
    FOutShapeInfo &shapeInfo = multiMateShape.getShapeInfo();

    // 頂点属性データから使用フラグを設定
    multiMateShape.setVtxAttrUsedFromShapeInfo();

    // 頂点属性が使われているかどうかのフラグを取得
    const bool useNormalAttribute =
        multiMateShape.getVertexAttributeUsed(Dcc::RPrimVtx::NRM0);

    bool useColorAttribute[Dcc::RPrimVtx::VTX_COL_MAX];
    for(int i = 0; i < Dcc::RPrimVtx::VTX_COL_MAX; ++i)
    {
        useColorAttribute[i] = multiMateShape.getVertexAttributeUsed((Dcc::RPrimVtx::VtxAttr)(Dcc::RPrimVtx::COL0 + i));
    }

    bool useUserAttribute = multiMateShape.getVertexAttributeUsed((Dcc::RPrimVtx::VtxAttr)(Dcc::RPrimVtx::USR0));

    Mtl* nodeMtl = m_pMaxNode->GetMtl();
    /*
    if(nodeMtl == NULL)
    {
        //mprintf("WARNING: node(%s) has no material\n", m_pMaxNode->GetName());
        string mes = "Node has no material(";
        mes += m_pMaxNode->GetName();
        mes += ")";
        RLogger::LogMessage(mes, RLogger::kWarning);
    }
    */

    Matrix3 objTm = m_pMaxNode->GetObjectTM(0);
    BOOL isNegativeScale = (DotProd(CrossProd(objTm.GetRow(0),objTm.GetRow(1)),objTm.GetRow(2)) < 0.0); //? true: false;

    // スケールでポリゴンが反転された場合に3dsmaxと同じように表示されるように
    // ポリゴンの指定する頂点の順番を反転する
    int vIdx[3] = {0, 1, 2};
    if (isNegativeScale)
    {
        vIdx[0] = 2;
        vIdx[1] = 1;
        vIdx[2] = 0;
    }

    // マテリアル情報の取得
    /*
    MultiMtlInfo multiMtl;
    bool ret = multiMtl.Init(m_pMaxNode->GetMtl(), sceneMaterials.GetDoesExportTexture());
    if(!ret) return false;
    */

    // 頂点アルファが使用されているかどうか調べる。
    bool useVtxAlpha = false;
    if(useColorAttribute[0])
    {
        // 頂点カラーがRGBAの時のみチェック
        if(shapeInfo.m_VtxInfos[Dcc::RPrimVtx::COL0].m_CompCount == 4)
        {
            //#pragma message(TODO("シェイプアニメーション用キーシェープのコードは未実装"))
            for(UINT ishape = 0; ishape < shapeInfo.m_VtxInfos[Dcc::RPrimVtx::COL0].m_pArrays.size(); ishape++)
            {
                const Dcc::RVec4Array* vcol = static_cast<const Dcc::RVec4Array*>(shapeInfo.m_VtxInfos[Dcc::RPrimVtx::COL0].m_pArrays[ishape]);
                if(vcol)
                {
                    Dcc::RVec4Array::const_iterator it;
                    for(it = vcol->begin(); it != vcol->end(); it++)
                    {
                        if(it->w < 0.9999f)
                        {
                            useVtxAlpha = true;
                            break;
                        }
                    }
                }
                if(useVtxAlpha) break;
            }
        }
    }

    // 全てのサブマテリアルで頂点アルファ使用フラグを設定
    //#pragma message(TODO("頂点アルファのフラグをマテリアルに設定していいのか？"))
    for(int i = 0; i < multiMtl.GetNumSubMtl(); i++)
    {
        MtlInfo* info = multiMtl.m_SubMtl[i];
        if(info)
        {
            info->m_HasVtxAlpha = useVtxAlpha;
        }
    }

    // マテリアルで使用するテクスチャ座標を調べる
#if 0
    bool usedTexMap[3] = {false,false,false};
    for(int i = 0; i < multiMtl.getNumSubMtl(); i++)
    {
        MtlInfo* info = multiMtl.subMtl[i];
        if(info)
        {
            for(UINT j = 0; j < info->getNumBitmap(); j++)
            {
                int ch = info->getBitmap(j)->mapch;
                ch = (ch > 3)? 1: ch;
                usedTexMap[ch-1] = true;
            }
        }
    }
#endif

    // テクスチャ座標を取得するためのMeshMapへのポインタを
    // あらかじめ列挙しておく
    vector<MeshMap*> maps;
    Dcc::RIntArray mapCh;
    Dcc::RIntArray hintCh;

    // サポートするテクスチャ座標の最大値
    unsigned int texNumMax = Dcc::RPrimVtx::VTX_TEX_MAX;//Dcc::RPrimVtx::TEX7 - Dcc::RPrimVtx::TEX0 + 1;

    //for (UINT maxUVCh = 1; maxUVCh <= texNumMax; maxUVCh++)
    //for(intintmap::iterator it = cafeUVToMaxUV.begin(); it != cafeUVToMaxUV.end(); it++)
    for(int i = 0; i < cafeUVToMaxUV.size(); i++)
    {
        int cafeUVCh = i;
        int maxUVch = cafeUVToMaxUV[i].maxUV;
        if (maxUVch > 0 && isMapChExist(mesh, maxUVch))
        {
            MeshMap& map = mesh->Map(maxUVch);
            maps.push_back(&map);
            mapCh.push_back(maxUVch);
            hintCh.push_back(cafeUVToMaxUV[i].cafeUV);
        }
        else
        {
//			maps.push_back(NULL);
//			mapCh.push_back(-1);
        }
        if(maps.size() >= texNumMax) break;
    }


    // 頂点カラーを取得するためのMeshMapへのポインタを
    // あらかじめ列挙しておく
    vector<MeshMap*> vcolMaps;
    vector<int> vcolMapCh;

    for(intintmap::iterator it = cafeColToMaxCol.begin(); it != cafeColToMaxCol.end(); it++)
    {
        int cafeColCh = it->first;
        int maxColch = it->second;
        if(maxColch >= MAP_ALPHA)
        {
            MeshMap& map = mesh->Map(maxColch);
            vcolMaps.push_back(&map);
            vcolMapCh.push_back(maxColch);
        }
        else
        {
            vcolMaps.push_back(NULL);
            vcolMapCh.push_back(MAP_ALPHA-1);
        }
    }

    // はじめに全faceをスキャンし、
    // - マテリアルデータ(FMaterial)を作成
    // - サブマテリアルID -> グローバルマテリアルIDのマップを作成
    std::map<int, int> subMateID2globalMateIDMap;
    {
        for (i = 0; i < mesh->getNumFaces(); i++)
        {
            // このポリゴンのサブマテリアルIDからMAXのマテリアルを参照し、
            // 対応するRMateriarlとそのグローバルマテリアルIDを作成する。
            UINT subMateID = mesh->faces[i].getMatID();
            int globalMateID;
            MtlInfo* info = multiMtl.GetSubMtl(subMateID);

            // infoがNULLとなることはありえないはず。
            if(!info) continue;
            // バンプマップのテクスチャ、UV座標情報を決定。
            info->DecideBumpTexInfo(mapCh);

            bool isExistMtl = sceneMaterials.isExistFMaterial((void*)info->m_pMaxMtl);
            globalMateID = sceneMaterials.getFMaterialIndex((void*)info->m_pMaxMtl);
            FMaterial* fmat = sceneMaterials.getFMaterialByIndex(globalMateID);

            // このマテリアルが始めて登場したら、FMaterialに情報をコピーする。
            if(!isExistMtl)
            {
                // 明示的に出力するUV座標のフラグを立てる
                for(int uvi = 0; uvi < Dcc::RPrimVtx::VTX_TEX_MAX; uvi++)
                {
                    fmat->m_UseCafeUV[uvi] = multiMtl.m_ForceUseCafeUV[uvi];
                }

                // マテリアルの情報をコピー
                info->CopyToFMaterial(sceneMaterials, fmat, mapCh, hintCh);
            }
            else
            {
#if	0
                //	TODO:
                // マテリアルが登録済みで、頂点アルファフラグが矛盾する場合は警告する
                if(		(subMateID2globalMateIDMap.size() <= subMateID)
                    ||	(subMateID2globalMateIDMap[subMateID] != globalMateID))
                {
                    bool missmatchVtxCol = false;
                    if(fmat->GetUseVtxColor() != useColorAttribute)
                    {	missmatchVtxCol = true;	}
                    else if (info->hasVtxAlpha && (fmat->GetBlendMode() == RMaterial::BLEND_MODE_OPAQUE))
                    {	missmatchVtxCol = true;	}

                    if(missmatchVtxCol)
                    {
                        string mes = "Can't combine shapes because of vertex-alpha settings(node: ";
                        mes += node->GetName();
                        mes += ", mtl: ";
                        mes += info->getName();
                        mes += ")";
                        RLogger::LogMessage(mes, RLogger::kWarning);
                    }
                }
#endif
            }

            // サブマテリアルID -> グローバルマテリアルIDのマップを作成
            subMateID2globalMateIDMap[subMateID] = globalMateID;
        }
    }

#if 0
    // マテリアルで使用されているマップチャンネルとジオメトリの持つマップチャンネルに矛盾はないか？
    {
        bool noWarnTexCoord = true;
        RMaterial* rmat;
        std::map<int, int>::iterator matit = subMateID2globalMateIDMap.begin();
        while(matit != subMateID2globalMateIDMap.end() && noWarnTexCoord)
        {
            rmat = sceneMaterials.getRMaterialByIndex(matit->second);
            if(rmat)
            {
                int numMap = rmat->GetNumTexCoordinator();
                numMap = (numMap <= 3)? numMap: 3;
                for(int c = 0; c < numMap ; c++)
                {
                    RTexCoordinator* coord = rmat->GetTexCoordinator(c);
                    if(coord)
                    {
                        int srcCoord = coord->mSourceCoordinate;
                        if(srcCoord < 0 || 2 < srcCoord || maps[srcCoord] == NULL)
                        {
                            coord->mSourceCoordinate = 0;
                            char buf[2048];
                            sprintf_s(buf, "node: %s, materila: %s, MapCh: %d.", node->GetName(), rmat->GetName(), srcCoord);
                            RLogger::LogMessagebyID(RLogger::kLogWRN_MapChannelDoesNotExist, buf);
                            break;
                        }
                    }
                }
            }
            matit++;
        }
    }
#endif

#if	0
    // 使用中のマップチャンネルを調べる
    bool usedTexMap[3] = {false,false,false};
    {
        FMaterial* fmat;
        std::map<int, int>::iterator matit = subMateID2globalMateIDMap.begin();
        while(matit != subMateID2globalMateIDMap.end())
        {
            fmat = sceneMaterials.getFMaterialByIndex(matit->second);
            if(fmat)
            {
                int numMap = fmat->GetNumTexCoordinator();
                numMap = (numMap <= 3)? numMap: 3;
                for(int c = 0; c < numMap ; c++)
                {
                    RTexCoordinator* coord = fmat->GetTexCoordinator(c);
                    if(coord)
                    {
                        int srcCoord = coord->mSourceCoordinate;
                        if(0 <= srcCoord && srcCoord <= 2)
                        {
                            usedTexMap[srcCoord] = true;
                        }
                    }
                }
            }
            matit++;
        }
    }
#endif

    // 異方性反射が使われていたら、UV座標を使用するフラグを立てる
    int anisoMapCh = -1;
    for(int i = 0; i < multiMtl.GetNumSubMtl(); i++)
    {
        MtlInfo* info = multiMtl.GetSubMtl(i);
        if(info && info->m_IsAnisotropic)
        {
            if(mapCh.size() > 0)
            {
                anisoMapCh = mapCh[0];
            }
            else
            {

            }

            // UV座標が見つからなかったら警告
            if(anisoMapCh < 0)
            {
                RLogger::LogMessagebyID(RLogger::kLogWRN_GeneratingTangentFailed, info->GetName());
            }
            break;
        }
    }

    // 全てのポリゴンを登録する
    for (i = 0; i < mesh->getNumFaces(); i++)
    {
        Dcc::RPrimitive rprim(3); //三角形ポリゴン
        for(j = 0; j < 3; j++)
        {
            int idx = vIdx[j];
            Dcc::RPrimVtx primv;

            // 頂点インデックス
            primv[Dcc::RPrimVtx::POS0] = mesh->faces[i].v[idx];
            //	TODO: 頂点座標配列と skinVtx の並びが同じと想定
            //	もしかすると違うかもしれないので確認する。
            primv.m_Mtx = VtxMtxIdxs[primv[Dcc::RPrimVtx::POS0]];

            // 法線インデックス
            primv[Dcc::RPrimVtx::NRM0] = useNormalAttribute? (i * 3 + idx): -1;


            // テクスチャ座標インデックス
            for(UINT mi = 0; mi < maps.size(); mi++)
            {
                if(maps[mi])
                {
                    primv[Dcc::RPrimVtx::TEX0 + mi] = maps[mi]->tf[i].t[idx];
                }
            }

            // 頂点カラーインデックス
            // COL0には頂点カラー・アルファー・イルミネーションが合成された値が入る場合がある
            // その時はインデックスが特殊なので特別な処理をする
            {
                UINT iCol = 0;

                if(useColorAttribute[iCol] && cafeColToMaxCol[iCol] < MAP_ALPHA)
                {
                    primv[(Dcc::RPrimVtx::VtxAttr)(Dcc::RPrimVtx::COL0)] = i * 3 + idx;
                    iCol = 1;
                }


                for(; iCol < Dcc::RPrimVtx::VTX_COL_MAX; ++iCol)
                {
                    Dcc::RPrimVtx::VtxAttr a = (Dcc::RPrimVtx::VtxAttr)(Dcc::RPrimVtx::COL0 + iCol);
                    if(useColorAttribute[iCol])
                    {
                        if(vcolMaps[iCol])
                        {
                            primv[a] = vcolMaps[iCol]->tf[i].t[idx];
                        }
                    }
                    else
                    {
                        primv[a] = -1;
                    }
                }
            }

            // ユーザー定義頂点データ
            primv[Dcc::RPrimVtx::USR0] = useUserAttribute? (i * 3 + idx): -1;

            // 頂点情報を登録する
            rprim.SetPrimVtx(j, primv);
        }

        // サブマテリアルIDをグローバルなマテリアルIDに変換してポリゴンを追加。
        int subMateID = mesh->faces[i].getMatID();
        int globalMateID = subMateID2globalMateIDMap[subMateID];
        if (!multiMateShape.appendPolygon(globalMateID, rprim, g_VtxMtxs, false))
        {
            // ボーン用のレジスタが足りない場合はエラー
            RLogger::LogMessagebyID(RLogger::kLogERR_PolygonBoneSizeIsOver, std::string(M_2_A(m_pMaxNode->GetName())));
            return false;
        }
    }

    // マテリアル情報を調べて、必要ならタンジェントの出力を行う
    if(multiMtl.GetNumSubMtl() > 0)
    {
        MtlInfo* info = NULL;
        for(int i = 0; i < multiMtl.GetNumSubMtl(); i++)
        {
            info = multiMtl.GetSubMtl(i);
            if(info)
            {
                //if(info->m_HasBump)
                if(info->m_HasBump || info->m_IsAnisotropic)
                {
                    break;
                }
            }
            info = NULL;
        }

        if(info)
        {
            int tangentBaseCh = -1;
            if(info->m_HasBump && info->m_BumpTexCh >= 0)
            {
                tangentBaseCh = info->m_BumpTexCh;
            }
            else if(info->m_IsAnisotropic && anisoMapCh >= 0)
            {
                tangentBaseCh = anisoMapCh;
            }
            if(tangentBaseCh >= 0)
            {
//				for(intintmap::iterator it = cafeUVToMaxUV.begin(); it != cafeUVToMaxUV.end(); it++)
//				{
                for(unsigned int i = 0; i < texNumMax; i++)
                {
                    //int cafeUVCh = it->first;
                    //int maxUVch = it->second;
                    int cafeUVCh = i;
                    int maxUVch = cafeUVToMaxUV[i].maxUV;

                    if(maxUVch == tangentBaseCh)
                    {
                        // バンプ用のタンジェント情報を生成する
                        multiMateShape.calcTangent(Dcc::RPrimVtx::VtxAttr(Dcc::RPrimVtx::TEX0 + cafeUVCh), true);
                        break;
                    }
                }
            }
        }
    }

    return true;
}

int CNodeTree::findValidCh( cafeToMaxUVInfoVec &cafeUVToMaxUV, int maxCh )
{
    int cafeCh = -1;

    for(int i = 0; i <= Dcc::RVtxAttrib::HINT_INDEX_MAX; i++)
    {
        if(cafeUVToMaxUV[i].maxUV == maxCh)
        {
            cafeCh = i;
            break;
        }
    }
    if(cafeCh < 0)
    {
        // 空きCHを探す
        for(int i = 0; i <= Dcc::RVtxAttrib::HINT_INDEX_MAX; i++)
        {
            if(cafeUVToMaxUV[i].maxUV < 0)
            {
                cafeCh = i;
                break;
            }
        }
    }
    return cafeCh;
}

#if 0
// 面と頂点を指定して、法線を取り出す
// （主にスムージンググループから計算する）
Point3 GetVertexNormal(Mesh* mesh, int faceNo, RVertex* rv)
{
    Face* f = &mesh->faces[faceNo];
    DWORD smGroup = f->smGroup;
    int numNormals = 0;
    Point3 vertexNormal;

    MeshNormalSpec* spNormal = mesh->GetSpecifiedNormals();

    if (rv->rFlags & SPECIFIED_NORMAL)
    {
        // SPECIFIED_NORMALの場合
        // SPECIFIED_NORMALは今は使われていない？
        vertexNormal = rv->rn.getNormal();
    }
    else if ((numNormals = rv->rFlags & NORCT_MASK) != 0 && smGroup)
    {
        // スムージンググループから計算する
        if (numNormals == 1)
        {
            vertexNormal = rv->rn.getNormal();
        }
        else
        {
            for (int i = 0; i < numNormals; i++)
            {
                if (rv->ern[i].getSmGroup() & smGroup)
                {
                    vertexNormal = rv->ern[i].getNormal();
                }
            }
        }
    }
    else
    {
        // 面の法線をそのまま返す（多分ここには来ない？）
        vertexNormal = mesh->getFaceNormal(faceNo);
    }

    return vertexNormal.Normalize();
}
#endif
