﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

///////////////////////////////////////////////////////////
// version definition
#define NINTENDOEXPORT_VERSION (400) // 100 -> 1.0.0
#define NINTENDOEXPORT_VERSION_MINOR_STR ""
#define NINTENDOEXPORT_DISPVERSION_MAJOR (4)
#define NINTENDOEXPORT_DISPVERSION_MINOR (0)
#define NINTENDOEXPORT_DISPVERSION_PATCH (0)

#include "DccCommon.h"
#include "DccOutput.h"
#include "DccImage.h"
#include "DccMaterial.h"

// max headers

#include "3dsmaxsdk_preinclude.h"
#pragma warning(disable : 4819) // disable unicode warning
#pragma warning(disable : 4244) // disable double->float warning
#include "Max.h"
#include "resource.h"
#include "istdplug.h"
#include "iparamb2.h"
#include "iparamm2.h"
#include "MeshNormalSpec.h"
#include "iskin.h"
#include "iSkinPose.h"
#include "CS/BipedApi.h"
#include "cs\Phyexp.h"
#include "imenuman.h"
#include "stdmat.h"
#include "Path.h"
#include "IPathConfigMgr.h"
#include "imtl.h"
#include "CustAttrib.h"
#include "ICustAttribContainer.h"
#include "maxscript/util/listener.h"
#if (MAX_VERSION_MAJOR >= 14) // max2012
#include "maxscript/maxscript.h"
#else
#include "maxscrpt/maxscrpt.h"
#endif
#if (MAX_VERSION_MAJOR >= 12) // max2010
#include "IFileResolutionManager.h"
#endif
#pragma warning(default: 4819) // recover unicode warning
#pragma warning(default: 4244) // recover double->float warning


// stdlib headers
#include <iostream>
#include <iomanip>
#include <fstream>
#include <strstream>
#include <string>

#include "DccUtilityNode.h"
#include "DccUtilitySceneMaterials.h"
#include "DccUtilityModel.h"
#include "DccUtilityLogger.h"



// util headers
#include "MaterialInfo.h"
#include "NodeTree.h"

namespace Dcc = nn::gfx::tool::dcc;

#ifdef UNICODE
#define M_2_A(p) (Dcc::RGetShiftJisFromUnicode(std::wstring(p)).c_str())
#define M_2_S(p) (Dcc::RGetShiftJisFromUnicode(std::wstring(p)))
#define A_2_M(p) (Dcc::RGetUnicodeFromShiftJis(std::string(p)).c_str())
//#define M_2_A(p) (Dcc::RGetShiftJisFromUnicode(std::wstring(p)))
//#define A_2_M(p) (Dcc::RGetUnicodeFromShiftJis(std::string(p)))
#else
#define M_2_A(p) (p)
#define M_2_S(p) (std::string(p))
#define A_2_M(p) (p)
#endif


extern bool SetUserDataFromString(std::string str, Dcc::utility::FMaterial* fmat, Dcc::utility::RNode* rnode);

//SIMPLE TYPE

extern TCHAR* GetString(int id);
extern HINSTANCE hInstance;

extern ClassDesc2* GetNintendoExportDesc();
extern ClassDesc2* GetNintendoAttribDesc();
extern ClassDesc2* GetNintendoNodeAttribDesc();
extern ClassDesc2* GetNintendoMtlAttribDesc();
extern ClassDesc2* GetNintendoSamplerAttribDesc();
extern ClassDesc2* GetNintendoMultiExportAttribDesc();

class export_opt;

#define NINTENDOEXPORT_CLASS_ID				Class_ID(0x53bb1bfc, 0x25a94b06)
#define NINTENDOEXPORT_INTERCFACE_ID				Interface_ID(0x103d77af, 0xaa73082)

#define NINTENDOEXPORT_ATTRIB_CLASS_ID			Class_ID(0x72ca1ef3, 0x46b1f99)
#define NINTENDOEXPORT_ATTRIB_NAME				_M_TEXT("NW4F Custom Attribute")

#define NINTENDOEXPORT_NODE_ATTRIB_CLASS_ID	Class_ID(0x78706629, 0x6af069b6)
#define NINTENDOEXPORT_NODE_ATTRIB_NAME		_M_TEXT("NW4F Node Custom Attribute")

#define NINTENDOEXPORT_MTL_ATTRIB_CLASS_ID		Class_ID(0x5885346e, 0x85c7d85)
#define NINTENDOEXPORT_MTL_ATTRIB_NAME			_M_TEXT("NW4F Material Custom Attribute")

#define NINTENDOEXPORT_SAMPLER_ATTRIB_CLASS_ID	Class_ID(0x5c7103cb, 0x67a45a6)
#define NINTENDOEXPORT_SAMPLER_ATTRIB_NAME		_M_TEXT("NW4F Sampler Custom Attribute")

#define NINTENDOEXPORT_MULTIEXPORT_ATTRIB_CLASS_ID	Class_ID(0x9ab7380, 0x78963587)
#define NINTENDOEXPORT_MULTIEXPORT_ATTRIB_NAME		_M_TEXT("NW4F MultiExport Custom Attribute")

class NintendoExport : public SceneExport
{
public:
    typedef std::pair<string, string> strStrPair;
    typedef std::vector<strStrPair> strStrList;

    static HWND m_hParams_;
    Interface* m_pIp;

    CStr m_OutFolder;
    string m_TmpOutFolderPath;

    CSceneNode m_SceneNode;
    Dcc::utility::RNodeProcessor	m_NodeProcessor;
    Dcc::utility::RSceneMaterials	m_SceneMaterials;
    Dcc::utility::RSceneCameras		m_SceneCameras;
    Dcc::utility::RSceneLights		m_SceneLights;

    vector<Dcc::utility::FMaterial*>	m_pOutMaterials;
    export_opt* m_pExp_options;
    Dcc::RFileMoveArray m_TmpFileMoves;
    Dcc::RImagePtrArray m_pOutTexImgs;

    // FigureMode,SkinPoseにするノードを記録
    INodeTab m_FigureModeNodes; // FigureModeにするノードを記録
    INodeTab m_FigureModeNoRestoreNodes; //endFigireModeで元に戻さなくて良いノードを記録(はじめからFigureModeになっている)
    INodeTab m_SkinPoseNodes;	// SkinPoseにするノードを記録
    INodeTab m_SkinPoseNoRestoreNodes; //endFigireModeで元に戻さなくて良いノードを記録(はじめからSkinPoseになっている)

    // エラー・警告ログ
    string errorLog;
    clock_t m_ModelClocks; //!< モデル出力のクロック数です。
    clock_t m_TexClocks; //!< テクスチャ出力のクロック数です。
    clock_t m_AnimClocks; //!< アニメーション出力のクロック数です。

    //Constructor/Destructor
private:
    NintendoExport()
    {
        Init();
    };
    static NintendoExport*	m_pInstance_;
public:
    ~NintendoExport(){};
    // Singleton
    static NintendoExport* Get(void);
    int DoPreExportScript();

    // SceneExport
    int				ExtCount() override { return 1; };				// Number of extensions supported
    const TCHAR* 	Ext(int n) override { return (n == 0)?_T("cmdl"): _T("");};		// Extension #n (i.e. "3DS")
    const TCHAR* 	LongDesc() override { return _T("Nintendo");};			// Long ASCII description (i.e. "Autodesk 3D Studio File")
    const TCHAR* 	ShortDesc() override { return _T("Nintendo Data");};	// Short ASCII description (i.e. "3D Studio")
    const TCHAR* 	AuthorName() override { return _T("Nintendo");};	// ASCII Author name
    const TCHAR* 	CopyrightMessage() override
                        { return _T("(c)Nintendo");};	// ASCII Copyright message
    const TCHAR* 	OtherMessage1() override { return _T("");};		// Other message #1
    const TCHAR* 	OtherMessage2() override { return _T("");};		// Other message #2
    void			ShowAbout(HWND hWnd) override {};					// Show DLL's "About..." box
    BOOL			SupportsOptions(int ext, DWORD options) override
                        {return TRUE;};
    unsigned int	Version() override { return NINTENDOEXPORT_VERSION; };				// Version number * 100 (i.e. v3.01 = 301)

    int	 DoExport(const TCHAR* name,ExpInterface* ei,Interface* i, BOOL suppressPrompts=FALSE, DWORD options=0) override ;

    void FinishExport(bool succeed);

    // エラー・警告ログ
    // エラーログバッファを初期化
    void ClearErrorLog()
    {
        errorLog.clear();
    }
    // エラーログを追加
    void AddErrorLog(string& err)
    {
        errorLog += err;
    }
    // エラーログを取得
    string& GetErrorLog()
    {
        return errorLog;
    }

    //! @brief テンポラリフォルダに出力する中間ファイルのパスを返します。
    //!
    //! @param[in] fileType 中間ファイルタイプです。
    //!
    //! @return テンポラリフォルダに出力する中間ファイルのパスを返します。
    //!
    std::string GetTmpFilePath(const Dcc::RExpOpt::FileType fileType) const;

    //
    //BOOL NodeEnum(INode* maxNode, RNode* parent);
    //Object* GetObjBeforeModifier(INode* node, Modifier* mod);

    void BeginFigureMode(INode* node = nullptr);
    void EndFigureMode(INode* node = nullptr);

    //! @brief テクスチャパターンアニメーションが設定されたマテリアルの数を返します。
    int GetTexPatAnimatedMaterialCount(void);

    //! @brief テクスチャパターンアニメーションデータを作成します。
    //!        全テクスチャを取得してから処理するので ftp ファイル出力時に呼ばれます。
    bool CreateTexPatAnimData(Dcc::RStringArray& texPatRefs, const Dcc::RExpOpt& rOpt);

    //! @brief 全サブフレームにおけるテクスチャパターンアニメーション値を分析します。
    //!        参照されるテクスチャイメージの texImgIdxs に 0 を設定します。
    //!        全テクスチャを取得してから処理するので ftp ファイル出力時に呼ばれます。
    bool AnalyzeTexPatFullAnim(
        Dcc::utility::TexPatAnim& texPatAnim,
        Dcc::RIntArray& texImgIdxs,
        const Dcc::RExpOpt& rOpt );

    //! @brief テクスチャパターンのキーアニメーションを取得します。
    //!        全テクスチャを取得してから処理するので ftp ファイル出力時に呼ばれます。
    void GetTexPatKeyAnim(
        Dcc::utility::TexPatAnim& texPatAnim,
        const Dcc::RIntArray& texImgIdxs,
        const Dcc::RExpOpt& rOpt
    );

    //! @brief テクスチャパターンアニメーションを 1 つ出力します。
    void OutputTexPatAnims(basic_ostream<char>& os, vector<Dcc::RDataStream>& dataStreams, const bool isFma, const int tc, const Dcc::utility::FMaterial& mat);

    //! @brief カラーアニメーションが設定されたマテリアルの数を返します。
    int GetColorAnimatedMaterialCount(void);

    //! @brief カラーアニメーションを 1 つ出力します。
    void OutputColorAnim( std::ostream& os, Dcc::RDataStreamArray& dataStreams, const int tc, const int matAnimIdx, const Dcc::utility::FMaterial& mat );

    //	マテリアルを名前でソートします。
    void SortMaterialByName( void );
    //	出力テクスチャリストを作成します。
    void SetOutTexImgPtrs( void );

    bool OutputFmdFile(ostream& os, Dcc::RDataStreamArray& dataStreams);
    bool OutputFskFile(ostream& os, Dcc::RDataStreamArray& dataStreams);
    bool OutputFtpFile(ostream& os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt);
    bool OutputFclFile(ostream& os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt );
    bool OutputFmaFile(ostream& os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt, const bool outputsColor, const bool outputsTexSrt, const bool outputsTexPat );
    bool OutputFvbFile(ostream& os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt);
    bool OutputFshFile(ostream& os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt);
    bool OutputFsnFile(ostream& os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt );

    //	@brief	テクスチャ中間ファイル郡を出力します。
    bool OutputTextureFiles(const std::string tmpFolder, const Dcc::RExpOpt& rOpt);

    //
    void OutputHeader(std::ostream& os, const int tc, const Dcc::RExpOpt::FileType fileType, const Dcc::RExpOpt& rOpt);

    const char* GetPluginVersionString();

    const char* GetMaxVersionString();

    void OutputFooter(std::ostream& os, const int tc, const Dcc::RExpOpt::FileType fileType, const Dcc::RExpOpt& rOpt);

    bool OutputFiles(int& fileCount, const Dcc::RExpOpt& rOpt);
    bool OutputOneMatAnimFile( std::ostream& os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt, const Dcc::RExpOpt::FileType fileType );
    bool OutputOneFile( const Dcc::RExpOpt& rOpt, const std::string& filePath, const Dcc::RExpOpt::FileType fileType );
    //void OutRenderSets(std::ostream& os, const int tc, RNodeProcessor& np);
    void OutputMaterials(std::ostream& os, const int tc);
    void OutputOriginalMaterials(std::ostream& os, const int tc);

    ///	@brief	テクスチャ出力後に設定可能なマテリアル属性を設定します。
    void SetMaterialAttrAfterOutTex(void);

    void OutputSkeletalModelTag(ostream& os, std::string name);
    void OutputDummyMtl(ostream& os, Mtl* mtl, vector<int> mapCh);
    void OutputMaterialAnimTag(ostream& osMat, CStr loop, int start, int end, int length);
    int GetTexSrtAnimatedMaterialCount(void);
    bool OutputFtsFile( std::ostream& os, Dcc::RDataStreamArray& dataStreams, const Dcc::RExpOpt& rOpt );
    bool OutputTexPatterns(ostream& os, const Dcc::RExpOpt& rOpt, const int tc);
    void OutputTexSrtAnims(basic_ostream<char>& os, vector<Dcc::RDataStream>& dataStreams, const int tc, const Dcc::utility::FMaterial& mat);

    // コンフィグファイルを読み込む
    bool loadConfigFile(void);
    void DisplayProfileInfo( std::string name, const clock_t clocks );
    void SetOptionEnvVarForPrePostExportScript(bool isPost, bool exportSucceeded);
    void Init();
};

class NintendoExportClassDesc : public ClassDesc2
{
public:
    //virtual int IsPublic() 							{ return TRUE; }
    virtual int IsPublic() 							{ return FALSE; }
    virtual void* Create(BOOL /*loading = FALSE*/)	{ return NintendoExport::Get(); }
    virtual const TCHAR* 	ClassName() 			{ return GetString(IDS_CLASS_NAME); }
    virtual SClass_ID SuperClassID() 				{ return SCENE_EXPORT_CLASS_ID; }
    virtual Class_ID ClassID() 						{ return NINTENDOEXPORT_CLASS_ID; }
    virtual const TCHAR* Category() 				{ return GetString(IDS_CATEGORY); }

    virtual const TCHAR* InternalName() 			{ return _T("NintendoExport"); }	// returns fixed parsable name (scripter-visible name)
    virtual HINSTANCE HInstance() 					{ return hInstance; }					// returns owning module handle
};


class NintendoOptions
{
public:

    int m_OptionVersion; // file or attribute option version

    Dcc::RExpOpt m_Opt;

    // スキンのバインドポーズにフィギュアモードやスキンポーズを使う
    bool m_UseFigureMode;

    BOOL m_UseUnifiedQuantizeScale;

    // maxのz-uoからy-upへの変換を行う
    bool m_DisableZYAxisConversion;

    // preset ui
    BOOL m_showPresetUI;

    // lock info
    BOOL m_Lock_export_target;
    BOOL m_Lock_output_file_name;
    BOOL m_Lock_output_folder;
    BOOL m_Lock_use_figure_mode;
    BOOL m_Lock_merge_fmd;
    BOOL m_Lock_merge_ftx;
    BOOL m_Lock_merge_anim;
    BOOL m_Lock_merge_anim_name;
    BOOL m_Lock_magnify;
    BOOL m_Lock_texsrt_mode;
    BOOL m_Lock_remove_namespace;
    BOOL m_Lock_comment;
    BOOL m_Lock_frame_range;
    BOOL m_Lock_loop_anim;
    BOOL m_Lock_bake_all_anim;
    BOOL m_Lock_frame_precision;
    BOOL m_Lock_output_fmd;
    BOOL m_Lock_output_ftx;
    BOOL m_Lock_output_fsk;
    BOOL m_Lock_output_fvb;
    BOOL m_Lock_output_fcl;
    BOOL m_Lock_output_fts;
    BOOL m_Lock_output_ftp;
    BOOL m_Lock_output_fsh;
    BOOL m_Lock_output_fsn;
    BOOL m_Lock_compress_bone;
    BOOL m_Lock_unite_child;
    BOOL m_Lock_compress_material;
    BOOL m_Lock_compress_shape;
    BOOL m_Lock_tolerance_scale;
    BOOL m_Lock_tolerance_rotate;
    BOOL m_Lock_tolerance_translate;
    BOOL m_Lock_tolerance_color;
    BOOL m_Lock_tolerance_tex_scale;
    BOOL m_Lock_tolerance_tex_rotate;
    BOOL m_Lock_tolerance_tex_translate;
    BOOL m_Lock_quantize_tolerance_scale;
    BOOL m_Lock_quantize_tolerance_rotate;
    BOOL m_Lock_quantize_tolerance_translate;
    BOOL m_Lock_quantize_tolerance_tex_scale;
    BOOL m_Lock_quantize_tolerance_tex_rotate;
    BOOL m_Lock_quantize_tolerance_tex_translate;
    BOOL m_Lock_pre_export_script;
    BOOL m_Lock_post_export_script;

    NintendoOptions()
    {
        Init();
    }

    void Init(void);

    void InitConfigOptions();

    inline bool IsEnableZYAxisConversion()
    {
        return !m_DisableZYAxisConversion;
    }

    IParamBlock2* GetNw4fCustAttribParam(void);
    void LoadFromAtribute(void);
    void SaveToAttribute(void);
};

////////////////////////////////////////////////////////////////////
// Custom attribute
class NintendoCustAttrib : public CustAttrib
{
public:
    IParamBlock2* pblock;
    //HWND m_hPanel;

    NintendoCustAttrib();

#if (MAX_PRODUCT_YEAR_NUMBER >= 2015) // max2015以降
    virtual RefResult NotifyRefChanged(const Interval& changeInt, RefTargetHandle hTarget,
        PartID& partID,  RefMessage message, BOOL propagate){return REF_SUCCEED;}
#else
    virtual RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget,
                               PartID& partID,  RefMessage message){return REF_SUCCEED;}
#endif

    int	NumParamBlocks() { return 1; }					// return number of ParamBlocks in this instance
    IParamBlock2* GetParamBlock(int i) { return pblock; } // return i'th ParamBlock
    IParamBlock2* GetParamBlockByID(BlockID id) { return (pblock->ID() == id) ? pblock : NULL; } // return id'd ParamBlock

    int NumRefs() { return 1;}
    virtual RefTargetHandle GetReference(int i) { return (i == 0)? pblock : NULL; }
    virtual void SetReference(int i, RefTargetHandle rtarg) { if(i == 0) pblock = (IParamBlock2*)rtarg;}

    virtual	int NumSubs()  { return 1; }
    virtual	Animatable* SubAnim(int i) { return pblock; }
    virtual TSTR SubAnimName(int i){ return GetString(IDS_NW4F_CA);}

    //void BeginEditParams(IObjParam* m_pIp,ULONG flags,Animatable* prev);
    //void EndEditParams(IObjParam* m_pIp, ULONG flags, Animatable* next);
    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_ATTRIB_CLASS_ID;}

    ReferenceTarget* Clone(RemapDir &remap = DefaultRemapDir());
    virtual bool CheckCopyAttribTo(ICustAttribContainer* to) { return true; }

#if (MAX_VERSION_MAJOR >= 12) // max2010-
    virtual const MCHAR* GetName(){ return NINTENDOEXPORT_ATTRIB_NAME;}
#else
    virtual MCHAR* GetName(){ return NINTENDOEXPORT_ATTRIB_NAME;}
#endif
    void DeleteThis() { delete this;}
};

class NintendoAttribClassDesc:public ClassDesc2 {
    public:
    int 			IsPublic() {return 1;}
    void*			Create(BOOL loading) {return new NintendoCustAttrib;}
    const TCHAR*	ClassName() { return GetString(IDS_NW4F_CA); }
    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_ATTRIB_CLASS_ID;}
    const TCHAR* 	Category() {return _T("Nintendo");}
    const TCHAR*	InternalName() { return _T("Nw4fCustAttrib"); }	// returns fixed parsable name (scripter-visible name)
    HINSTANCE		HInstance() { return hInstance; }			// returns owning module handle
};

// カスタムアトリビュートのパラメータブロックのID
enum { nw4f_params};
enum {
    nw4f_p_optionVersion,

    nw4f_p_exportAllNode,
    nw4f_p_outFolder,		nw4f_p_outName,

    nw4f_p_obsolete_use3DEditor, //パラメータブロックの互換性のため、削除しない

    nw4f_p_mergeFmdFlag,	nw4f_p_mergeFmdPath,	nw4f_p_mergeFmdAuto,


    nw4f_p_magnify,
    nw4f_p_exportAllFrames,
    nw4f_p_startFrame,		nw4f_p_endFrame,
    nw4f_p_useFigureMode,

    nw4f_p_binaryFormat,
    nw4f_p_exportModel,		nw4f_p_exportSkeletalAnim,
    nw4f_p_exportBoneVisAnim,
    nw4f_p_exportMtlColAnim,nw4f_p_exportMtlTexSRTAnim, nw4f_p_exportMtlTexPatAnim,
    nw4f_p_exportShapeAnim,

    nw4f_p_exportCam,		nw4f_p_exportLight,
    nw4f_p_exportSceneAnim,
    nw4f_p_exportEnv,

    nw4f_p_exportTex,

    nw4f_p_compressBoneMode,

    nw4f_p_uniteChild, 		nw4f_p_compressMtl,
    nw4f_p_compressShape,	nw4f_p_optimizePrim,
    nw4f_p_quantizationAnalysis,

    nw4f_p_quantPos,		nw4f_p_quantNormal,		nw4f_p_quantTex,

    nw4f_p_framePrecision,	nw4f_p_isLoop,

    nw4f_p_toleranceTrans,	nw4f_p_toleranceRotate,	nw4f_p_toleranceScale,
    nw4f_p_toleranceUVTrans,	nw4f_p_toleranceUVRotate,	nw4f_p_toleranceUVScale,
    nw4f_p_toleranceColor,

    nw4f_p_quantToleranceTrans,		nw4f_p_quantToleranceRotate,	nw4f_p_quantToleranceScale,
    nw4f_p_quantToleranceUVTrans,	nw4f_p_quantToleranceUVRotate,	nw4f_p_quantToleranceUVScale,
    nw4f_p_quantToleranceFrame,

    nw4f_p_mergeFtxFlag,
    nw4f_p_mergeAnimFlag,	nw4f_p_mergeAnimFolder,	nw4f_p_mergeAnimName,

    nw4f_p_disableZYAxisConversion,

    nw4f_p_commnetText,

    // ver 5
    nw4f_p_presetName, nw4f_p_presetFile, nw4f_p_showPresetUI,

    nw4f_p_lock_export_target,
    nw4f_p_lock_output_file_name,nw4f_p_lock_output_folder,
    nw4f_p_obsolete_lock_use_3deditor,//パラメータブロックの互換性のため、削除しない
    nw4f_p_lock_use_figure_mode,
    nw4f_p_lock_merge_fmd,nw4f_p_lock_merge_ftx,
    nw4f_p_lock_merge_anim,nw4f_p_lock_merge_anim_name,
    nw4f_p_lock_magnify,
    nw4f_p_lock_comment,
    nw4f_p_lock_frame_range,
    nw4f_p_lock_loop_anim,
    nw4f_p_lock_bake_all_anim,
    nw4f_p_lock_frame_precision,
    nw4f_p_lock_output_fmd,nw4f_p_lock_output_ftx,
    nw4f_p_lock_output_fsk,nw4f_p_lock_output_fvb,
    nw4f_p_lock_output_fcl,nw4f_p_lock_output_fts,
    nw4f_p_lock_output_ftp,nw4f_p_lock_output_fsh,
    nw4f_p_lock_output_fsn,
    nw4f_p_lock_compress_bone,
    nw4f_p_lock_unite_child,
    nw4f_p_lock_compress_material,nw4f_p_lock_compress_shape,
    nw4f_p_lock_tolerance_scale,nw4f_p_lock_tolerance_rotate,nw4f_p_lock_tolerance_translate,
    nw4f_p_lock_tolerance_color,
    nw4f_p_lock_tolerance_tex_scale,nw4f_p_lock_tolerance_tex_rotate,nw4f_p_lock_tolerance_tex_translate,
    nw4f_p_lock_quantize_tolerance_scale,nw4f_p_lock_quantize_tolerance_rotate,nw4f_p_lock_quantize_tolerance_translate,
    nw4f_p_lock_quantize_tolerance_tex_scale,nw4f_p_lock_quantize_tolerance_tex_rotate,nw4f_p_lock_quantize_tolerance_tex_translate,
    nw4f_p_lock_pre_export_script,nw4f_p_lock_post_export_script,

    nw4f_p_preExportScript,nw4f_p_postExportScript,

};

#if 1
////////////////////////////////////////////////////////////////////
// Node Custom attribute
class NintendoNodeCustAttrib : public CustAttrib
{
public:
    IParamBlock2* pblock;

    NintendoNodeCustAttrib();

#if (MAX_PRODUCT_YEAR_NUMBER >= 2015) // max2015以降
    virtual RefResult NotifyRefChanged(const Interval& changeInt, RefTargetHandle hTarget,
        PartID& partID,  RefMessage message, BOOL propagate){return REF_SUCCEED;}
#else
    virtual RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget,
        PartID& partID,  RefMessage message){return REF_SUCCEED;}
#endif
    int	NumParamBlocks() { return 1; }					// return number of ParamBlocks in this instance
    IParamBlock2* GetParamBlock(int i) { return (i == 0)?pblock : NULL; } // return i'th ParamBlock
    IParamBlock2* GetParamBlockByID(BlockID id) { return (pblock->ID() == id) ? pblock : NULL; } // return id'd ParamBlock
    int NumRefs() { return 1;}
    virtual RefTargetHandle GetReference(int i) { if(i == 0) return pblock; else return NULL; }
    virtual void SetReference(int i, RefTargetHandle rtarg) { if(i == 0) pblock = (IParamBlock2*)rtarg;}

    virtual	int NumSubs()  { return 1; }
    virtual	Animatable* SubAnim(int i) { return (i == 0)?pblock : NULL; }
    virtual TSTR SubAnimName(int i){ return (i == 0)? GetString(IDS_NW4F_NODE_CA): _T("");}

    //void BeginEditParams(IObjParam* m_pIp,ULONG flags,Animatable* prev);
    //void EndEditParams(IObjParam* m_pIp, ULONG flags, Animatable* next);
    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_NODE_ATTRIB_CLASS_ID;}

    ReferenceTarget* Clone(RemapDir &remap = DefaultRemapDir());
    virtual bool CheckCopyAttribTo(ICustAttribContainer* to) { return true; }

#if (MAX_VERSION_MAJOR >= 12) // max2010-
    virtual const MCHAR* GetName(){ return NINTENDOEXPORT_NODE_ATTRIB_NAME;}
#else
    virtual MCHAR* GetName(){ return NINTENDOEXPORT_NODE_ATTRIB_NAME;}
#endif
    void DeleteThis() { delete this;}
};

class NintendoNodeAttribClassDesc:public ClassDesc2 {
    public:
    int 			IsPublic() {return 1;}
    void*			Create(BOOL loading) {return new NintendoNodeCustAttrib;}
    const TCHAR*	ClassName() { return GetString(IDS_NW4F_NODE_CA); }
    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_NODE_ATTRIB_CLASS_ID;}
    const TCHAR* 	Category() {return _T("Nintendo");}
    const TCHAR*	InternalName() { return _T("Nw4fNodeCustAttrib"); }	// returns fixed parsable name (scripter-visible name)
    HINSTANCE		HInstance() { return hInstance; }			// returns owning module handle
};

// パラメータのインデックス
enum {
    nw4f_node_noCompress,
    nw4f_node_shapeExportPosition,
    nw4f_node_shapeExportNormal,
    nw4f_node_shapeExportColor,
    nw4f_node_userSettingString,
/*
    nw4f_node_billboardType,
    nw4f_node_combineGroup,
    nw4f_node_forceKeyScale,
    nw4f_node_forceKeyRotate,
    nw4f_node_forceKeyTranslate,
    nw4f_node_forceKeyVisibility,
*/
};


////////////////////////////////////////////////////////////////////
// Material Custom attribute
class NintendoMtlCustAttrib : public CustAttrib
{
public:
    IParamBlock2* pblock;
    HWND m_hPanel;

    NintendoMtlCustAttrib();

    virtual ParamDlg* CreateParamDlg(HWND hwMtlEdit, IMtlParams* imp);

#if (MAX_PRODUCT_YEAR_NUMBER >= 2015) // max2015以降
    virtual RefResult NotifyRefChanged(const Interval& changeInt, RefTargetHandle hTarget,
        PartID& partID,  RefMessage message, BOOL propagate){return REF_SUCCEED;}
#else
    virtual RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget,
        PartID& partID,  RefMessage message){return REF_SUCCEED;}
#endif
    int	NumParamBlocks() { return 1; }					// return number of ParamBlocks in this instance
    IParamBlock2* GetParamBlock(int i) { return pblock; } // return i'th ParamBlock
    IParamBlock2* GetParamBlockByID(BlockID id) { return (pblock->ID() == id) ? pblock : NULL; } // return id'd ParamBlock
    int NumRefs() { return 1;}
    virtual RefTargetHandle GetReference(int i) { if(i == 0) return pblock; else return NULL; }
    virtual void SetReference(int i, RefTargetHandle rtarg) { if(i == 0) pblock = (IParamBlock2*)rtarg;}

    virtual	int NumSubs()  { return 1; }
    virtual	Animatable* SubAnim(int i) { return pblock; }
    virtual TSTR SubAnimName(int i){ return GetString(IDS_NW4F_MTL_CA);}

    //void BeginEditParams(IObjParam* m_pIp,ULONG flags,Animatable* prev);
    //void EndEditParams(IObjParam* m_pIp, ULONG flags, Animatable* next);
    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_MTL_ATTRIB_CLASS_ID;}

    ReferenceTarget* Clone(RemapDir &remap = DefaultRemapDir());
    virtual bool CheckCopyAttribTo(ICustAttribContainer* to) { return true; }

#if (MAX_VERSION_MAJOR >= 12) // max2010-
    virtual const MCHAR* GetName(){ return NINTENDOEXPORT_MTL_ATTRIB_NAME;}
#else
    virtual MCHAR* GetName(){ return NINTENDOEXPORT_MTL_ATTRIB_NAME;}
#endif
    void DeleteThis() { delete this;}
};

class NintendoMtlAttribClassDesc : public ClassDesc2 {
    public:
    int 			IsPublic() {return 1;}
    void* 			Create(BOOL loading) {return new NintendoMtlCustAttrib;}
    const TCHAR* 	ClassName() { return GetString(IDS_NW4F_MTL_CA); }
    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_MTL_ATTRIB_CLASS_ID;}
    const TCHAR* 	Category() {return _T("Nintendo");}
    const TCHAR*	InternalName() { return _T("Nw4fMtlCustAttrib"); }	// returns fixed parsable name (scripter-visible name)
    HINSTANCE		HInstance() { return hInstance; }			// returns owning module handle
};

// パラメータのインデックス
enum {
    nw4f_mtl_compress,
    nw4f_mtl_userSettingString,
/*
    nw4f_mtl_faceCulling,
    nw4f_mtl_fragmentLighting,
    nw4f_mtl_vertexLighting,
    nw4f_mtl_renderPriorityType,
    nw4f_mtl_renderPriority,
    nw4f_mtl_blendMode,
*/
};
#endif

////////////////////////////////////////////////////////////////////
// Material Custom attribute
class NintendoSamplerCustAttrib : public CustAttrib
{
public:
    IParamBlock2* pblock;
    HWND m_hPanel;

    NintendoSamplerCustAttrib()
    {
        pblock = NULL;
        GetNintendoSamplerAttribDesc()->MakeAutoParamBlocks(this);
    }

#if (MAX_PRODUCT_YEAR_NUMBER >= 2015) // max2015以降
    virtual RefResult NotifyRefChanged(const Interval& changeInt, RefTargetHandle hTarget,
        PartID& partID,  RefMessage message, BOOL propagate){return REF_SUCCEED;}
#else
    virtual RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget,
        PartID& partID,  RefMessage message){return REF_SUCCEED;}
#endif
    int	NumParamBlocks() { return 1; }					// return number of ParamBlocks in this instance
    IParamBlock2* GetParamBlock(int i) { return pblock; } // return i'th ParamBlock
    IParamBlock2* GetParamBlockByID(BlockID id) { return (pblock->ID() == id) ? pblock : NULL; } // return id'd ParamBlock
    int NumRefs() { return 1;}
    virtual RefTargetHandle GetReference(int i) { if(i == 0) return pblock; else return NULL; }
    virtual void SetReference(int i, RefTargetHandle rtarg) { if(i == 0) pblock = (IParamBlock2*)rtarg;}

    virtual	int NumSubs()  { return 1; }
    virtual	Animatable* SubAnim(int i) { return pblock; }
    virtual TSTR SubAnimName(int i){ return GetString(IDS_NW4F_SAMPLER_CA);}

    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_SAMPLER_ATTRIB_CLASS_ID;}

    ReferenceTarget* Clone(RemapDir &remap = DefaultRemapDir())
    {
        NintendoSamplerCustAttrib* pnew = new NintendoSamplerCustAttrib;
        pnew->ReplaceReference(0,remap.CloneRef(pblock));
        BaseClone(this, pnew, remap);
        return pnew;
    }

    virtual bool CheckCopyAttribTo(ICustAttribContainer* to) { return true; }

#if (MAX_VERSION_MAJOR >= 12) // max2010-
    virtual const MCHAR* GetName(){ return NINTENDOEXPORT_SAMPLER_ATTRIB_NAME;}
#else
    virtual MCHAR* GetName(){ return NINTENDOEXPORT_SAMPLER_ATTRIB_NAME;}
#endif
    void DeleteThis() { delete this;}
};

class NintendoSamplerAttribClassDesc : public ClassDesc2 {
    public:
    int 			IsPublic() {return 1;}
    void* 			Create(BOOL loading) {return new NintendoSamplerCustAttrib;}
    const TCHAR* 	ClassName() { return GetString(IDS_NW4F_SAMPLER_CA); }
    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_SAMPLER_ATTRIB_CLASS_ID;}
    const TCHAR* 	Category() {return _T("Nintendo");}
    const TCHAR*	InternalName() { return _T("Nw4fSamplerCustAttrib"); }	// returns fixed parsable name (scripter-visible name)
    HINSTANCE		HInstance() { return hInstance; }			// returns owning module handle
};

// パラメータのインデックス
enum {
    nw4f_sampler_name,
    nw4f_sampler_hint,
};


////////////////////////////////////////////////////////////////////
// MultiExport Custom attribute
class NintendoMultiExportCustAttrib : public CustAttrib
{
public:
    IParamBlock2* pblock;
    HWND m_hPanel;

    NintendoMultiExportCustAttrib()
    {
        pblock = NULL;
        GetNintendoMultiExportAttribDesc()->MakeAutoParamBlocks(this);
    }

#if (MAX_PRODUCT_YEAR_NUMBER >= 2015) // max2015以降
    virtual RefResult NotifyRefChanged(const Interval& changeInt, RefTargetHandle hTarget,
        PartID& partID,  RefMessage message, BOOL propagate){return REF_SUCCEED;}
#else
    virtual RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget,
        PartID& partID,  RefMessage message){return REF_SUCCEED;}
#endif
    int	NumParamBlocks() { return 1; }					// return number of ParamBlocks in this instance
    IParamBlock2* GetParamBlock(int i) { return pblock; } // return i'th ParamBlock
    IParamBlock2* GetParamBlockByID(BlockID id) { return (pblock->ID() == id) ? pblock : NULL; } // return id'd ParamBlock
    int NumRefs() { return 1;}
    virtual RefTargetHandle GetReference(int i) { if(i == 0) return pblock; else return NULL; }
    virtual void SetReference(int i, RefTargetHandle rtarg) { if(i == 0) pblock = (IParamBlock2*)rtarg;}

    virtual	int NumSubs()  { return 1; }
    virtual	Animatable* SubAnim(int i) { return pblock; }
    virtual TSTR SubAnimName(int i){ return GetString(IDS_NW4F_MULTIEXPORT_CA);}

    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_MULTIEXPORT_ATTRIB_CLASS_ID;}

    ReferenceTarget* Clone(RemapDir &remap = DefaultRemapDir())
    {
        NintendoMultiExportCustAttrib* pnew = new NintendoMultiExportCustAttrib;
        pnew->ReplaceReference(0,remap.CloneRef(pblock));
        BaseClone(this, pnew, remap);
        return pnew;
    }

    virtual bool CheckCopyAttribTo(ICustAttribContainer* to) { return true; }

#if (MAX_VERSION_MAJOR >= 12) // max2010-
    virtual const MCHAR* GetName(){ return NINTENDOEXPORT_MULTIEXPORT_ATTRIB_NAME;}
#else
    virtual MCHAR* GetName(){ return NINTENDOEXPORT_MULTIEXPORT_ATTRIB_NAME;}
#endif
    void DeleteThis() { delete this;}
};

class NintendoMultiExportAttribClassDesc : public ClassDesc2 {
public:
    int 			IsPublic() {return 1;}
    void* 			Create(BOOL loading) {return new NintendoMultiExportCustAttrib;}
    const TCHAR* 	ClassName() { return GetString(IDS_NW4F_MULTIEXPORT_CA); }
    SClass_ID		SuperClassID() {return CUST_ATTRIB_CLASS_ID;}
    Class_ID 		ClassID() {return NINTENDOEXPORT_MULTIEXPORT_ATTRIB_CLASS_ID;}
    const TCHAR* 	Category() {return _T("Nintendo");}
    const TCHAR*	InternalName() { return _T("Nw4fMultiExportCustAttrib"); }	// returns fixed parsable name (scripter-visible name)
    HINSTANCE		HInstance() { return hInstance; }			// returns owning module handle
};

// パラメータのインデックス
enum {
    nw4f_mrender_filename,
    nw4f_mrender_folder,
    nw4f_mrender_doesMergeFmd,
    nw4f_mrender_mergeFmdFile,
    nw4f_mrender_fdesFile,
    nw4f_mrender_nodes,
    nw4f_mrender_dispLayers,
    nw4f_mrender_animLayers,
    nw4f_mrender_comment,
    nw4f_mrender_doesExportAllFrames,
    nw4f_mrender_startFrame,
    nw4f_mrender_endFrame,
    nw4f_mrender_isLoop,

    nw4f_mrender_exportModel,
    nw4f_mrender_exportTex,
    nw4f_mrender_exportSkeletalAnim,
    nw4f_mrender_exportBoneVisAnim,
    nw4f_mrender_exportMtlColAnim,
    nw4f_mrender_exportMtlTexSRTAnim,
    nw4f_mrender_exportMtlTexPatAnim,
    nw4f_mrender_exportShapeAnim,
    nw4f_mrender_exportSceneAnim,

    nw4f_mrender_options,
};


/////////////////////////////////////////////////////////////////
// maxscript用Function Publishing

//extern int QuantTypeToInt(RQuantize q);
//extern RQuantize IntToQuantType(int q);
extern int MaxIdxToIdx(int max_idx);
extern int IdxToMaxIdx(int idx);

extern NintendoOptions* GetNintendoOptions();

#define myMinMax(val, min, max) ((val < min)? min: ((max < val)? max:val))

class NintendoExport_Imp : public FPStaticInterface
{
public:
    // function IDs
    enum {
        // function
        nw4f_startExport,
        nw4f_searchAnimations,
        nw4f_getDateTimeString,
        nw4f_getBuildDateTimeString,
        nw4f_resetSetting,
        nw4f_loadSetting,	nw4f_saveSetting,
        nw4f_getVersion,	nw4f_getVersionString,
        nw4f_getMtlEditorMenu,
        nw4f_getErrorLog,
        nw4f_setOptionEnvVar,
        nw4f_LayerTest,
        nw4f_useFclFtsFtp,
        nw4f_separateMaterialAnim,

        // properties
        nw4f_getPresetName,		nw4f_setPresetName,
        nw4f_getPresetFile,		nw4f_setPresetFile,
        nw4f_getShowPresetUI,	nw4f_setShowPresetUI,

        nw4f_getDoesExportSelected,	nw4f_setDoesExportSelected,
        nw4f_getOutFolder,			nw4f_setOutFolder,
        nw4f_getOutName,			nw4f_setOutName,

        nw4f_getUseMerge,			nw4f_setUseMerge,
        nw4f_getMergeFile,			nw4f_setMergeFile,
        nw4f_getMergeAuto,			nw4f_setMergeAuto,

        nw4f_getMagnify,				nw4f_setMagnify,
        nw4f_getDoesExportAllFrames,	nw4f_setDoesExportAllFrames,
        nw4f_getStartFrame,				nw4f_setStartFrame,
        nw4f_getEndFrame,				nw4f_setEndFrame,
        nw4f_getUseFigureMode,			nw4f_setUseFigureMode,
        nw4f_getCommentText,			nw4f_setCommentText,

        nw4f_getUseBinaryFormat,		nw4f_setUseBinaryFormat,

        nw4f_getDoesExportModel,			nw4f_setDoesExportModel,
        nw4f_getDoesExportSkelAnim,			nw4f_setDoesExportSkelAnim,
        nw4f_getDoesExportVisBoneAnim,		nw4f_setDoesExportVisBoneAnim,
        nw4f_getDoesExportMtlColAnim,		nw4f_setDoesExportMtlColAnim,
        nw4f_getDoesExportMtlTexSRTAnim,	nw4f_setDoesExportMtlTexSRTAnim,
        nw4f_getDoesExportMtlTexPatAnim,	nw4f_setDoesExportMtlTexPatAnim,
        nw4f_getDoesExportShapeAnim,		nw4f_setDoesExportShapeAnim,

        nw4f_getDoesExportCam,				nw4f_setDoesExportCam,
        nw4f_getDoesExportLight,			nw4f_setDoesExportLight,
        nw4f_getDoesExportSceneAnim,		nw4f_setDoesExportSceneAnim,
        nw4f_getDoesExportEnv,				nw4f_setDoesExportEnv,

        nw4f_getDoesExportTex,		nw4f_setDoesExportTex,

        nw4f_getCompressBone,		nw4f_setCompressBone,
        nw4f_getDoesUniteChild,		nw4f_setDoesUniteChild,
        nw4f_getDoesCompressMtl,	nw4f_setDoesCompressMtl,
        nw4f_getDoesCompressShape,	nw4f_setDoesCompressShape,
        nw4f_getDoesOptimizePrim,	nw4f_setDoesOptimizePrim,
        nw4f_getDoesQuantizationAnalize,	nw4f_setDoesQuantizationAnalize,

        nw4f_getQuantPos,		nw4f_setQuantPos,
        nw4f_getQuantNormal,	nw4f_setQuantNormal,
        nw4f_getQuantTex,		nw4f_setQuantTex,

        nw4f_getFramePrecision, nw4f_setFramePrecision,
        nw4f_getIsLoop, nw4f_setIsLoop,

        nw4f_getToleranceTrans,		nw4f_setToleranceTrans,
        nw4f_getToleranceRotate,	nw4f_setToleranceRotate,
        nw4f_getToleranceScale,		nw4f_setToleranceScale,
        nw4f_getToleranceUVTrans,	nw4f_setToleranceUVTrans,
        nw4f_getToleranceUVRotate,	nw4f_setToleranceUVRotate,
        nw4f_getToleranceUVScale,	nw4f_setToleranceUVScale,
        nw4f_getToleranceColor,		nw4f_setToleranceColor,

        nw4f_getQuantToleranceTrans,	nw4f_setQuantToleranceTrans,
        nw4f_getQuantToleranceRotate,	nw4f_setQuantToleranceRotate,
        nw4f_getQuantToleranceScale,	nw4f_setQuantToleranceScale,
        nw4f_getQuantToleranceUVTrans,	nw4f_setQuantToleranceUVTrans,
        nw4f_getQuantToleranceUVRotate,	nw4f_setQuantToleranceUVRotate,
        nw4f_getQuantToleranceUVScale,	nw4f_setQuantToleranceUVScale,
        nw4f_getQuantToleranceFrame,	nw4f_setQuantToleranceFrame,

        nw4f_getUseMergeFtx,		nw4f_setUseMergeFtx,
        nw4f_getUseMergeAnim,		nw4f_setUseMergeAnim,
        nw4f_getMergeAnimFolder,	nw4f_setMergeAnimFolder,
        nw4f_getMergeAnimName,		nw4f_setMergeAnimName,

        nw4f_getDisableZYAxisConversion,	nw4f_setDisableZYAxisConversion,

        //scirpt
        nw4f_getPreExportScript,	nw4f_setPreExportScript,
        nw4f_getPostExportScript,	nw4f_setPostExportScript,

        // lock
        nw4f_getLock_export_target, nw4f_setLock_export_target,
        nw4f_getLock_output_file_name, nw4f_setLock_output_file_name,
        nw4f_getLock_output_folder, nw4f_setLock_output_folder,
        nw4f_getLock_use_figure_mode, nw4f_setLock_use_figure_mode,
        nw4f_getLock_merge_fmd, nw4f_setLock_merge_fmd,
        nw4f_getLock_merge_ftx, nw4f_setLock_merge_ftx,
        nw4f_getLock_merge_anim, nw4f_setLock_merge_anim,
        nw4f_getLock_merge_anim_name, nw4f_setLock_merge_anim_name,
        nw4f_getLock_magnify, nw4f_setLock_magnify,
        nw4f_getLock_texsrt_mode, nw4f_setLock_texsrt_mode,
        nw4f_getLock_remove_namespace, nw4f_setLock_remove_namespace,
        nw4f_getLock_comment, nw4f_setLock_comment,
        nw4f_getLock_frame_range, nw4f_setLock_frame_range,
        nw4f_getLock_loop_anim, nw4f_setLock_loop_anim,
        nw4f_getLock_bake_all_anim, nw4f_setLock_bake_all_anim,
        nw4f_getLock_frame_precision, nw4f_setLock_frame_precision,
        nw4f_getLock_output_fmd, nw4f_setLock_output_fmd,
        nw4f_getLock_output_ftx, nw4f_setLock_output_ftx,
        nw4f_getLock_output_fsk, nw4f_setLock_output_fsk,
        nw4f_getLock_output_fvb, nw4f_setLock_output_fvb,
        nw4f_getLock_output_fcl, nw4f_setLock_output_fcl,
        nw4f_getLock_output_fts, nw4f_setLock_output_fts,
        nw4f_getLock_output_ftp, nw4f_setLock_output_ftp,
        nw4f_getLock_output_fsh, nw4f_setLock_output_fsh,
        nw4f_getLock_output_fsn, nw4f_setLock_output_fsn,
        nw4f_getLock_compress_bone, nw4f_setLock_compress_bone,
        nw4f_getLock_unite_child, nw4f_setLock_unite_child,
        nw4f_getLock_compress_material, nw4f_setLock_compress_material,
        nw4f_getLock_compress_shape, nw4f_setLock_compress_shape,
        nw4f_getLock_tolerance_scale, nw4f_setLock_tolerance_scale,
        nw4f_getLock_tolerance_rotate, nw4f_setLock_tolerance_rotate,
        nw4f_getLock_tolerance_translate, nw4f_setLock_tolerance_translate,
        nw4f_getLock_tolerance_color, nw4f_setLock_tolerance_color,
        nw4f_getLock_tolerance_tex_scale, nw4f_setLock_tolerance_tex_scale,
        nw4f_getLock_tolerance_tex_rotate, nw4f_setLock_tolerance_tex_rotate,
        nw4f_getLock_tolerance_tex_translate, nw4f_setLock_tolerance_tex_translate,
        nw4f_getLock_quantize_tolerance_scale, nw4f_setLock_quantize_tolerance_scale,
        nw4f_getLock_quantize_tolerance_rotate, nw4f_setLock_quantize_tolerance_rotate,
        nw4f_getLock_quantize_tolerance_translate, nw4f_setLock_quantize_tolerance_translate,
        nw4f_getLock_quantize_tolerance_tex_scale, nw4f_setLock_quantize_tolerance_tex_scale,
        nw4f_getLock_quantize_tolerance_tex_rotate, nw4f_setLock_quantize_tolerance_tex_rotate,
        nw4f_getLock_quantize_tolerance_tex_translate, nw4f_setLock_quantize_tolerance_tex_translate,
        nw4f_getLock_pre_export_script, nw4f_setLock_pre_export_script,
        nw4f_getLock_post_export_script, nw4f_setLock_post_export_script,

    };
    // enum IDs
    enum { nw4f_type };

    // unicode
    TCHAR* returnString(std::string& str);

    // preset name
    virtual const TCHAR* GetPresetName(void)
    {	return returnString(GetNintendoOptions()->m_Opt.m_PresetName);	}
    virtual void SetPresetName(const TCHAR* name)
    {	GetNintendoOptions()->m_Opt.m_PresetName  = M_2_A(name);	}
    // preset folder
    virtual const TCHAR* GetPresetFile(void)
    {	return returnString(GetNintendoOptions()->m_Opt.m_PresetFilePath);	}
    virtual void SetPresetFile(const TCHAR* file)
    {	GetNintendoOptions()->m_Opt.m_PresetFilePath = M_2_A(file);	}
    // preset UI
    virtual BOOL GetShowPresetUI(void)
    {	return GetNintendoOptions()->m_showPresetUI;	}
    virtual void SetShowPresetUI(BOOL flag)
    {	GetNintendoOptions()->m_showPresetUI = flag;	}
    // export selection
    virtual BOOL GetDoesExportSelected(void)
    {	return GetNintendoOptions()->m_Opt.m_Target == Dcc::RExpOpt::EXPORT_TARGET_SELECTION;	}
    virtual void SetDoesExportSelected(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_Target
                = flag ? Dcc::RExpOpt::EXPORT_TARGET_SELECTION : Dcc::RExpOpt::EXPORT_TARGET_ALL;	}
    // out folder
    virtual const TCHAR* GetOutFolder(void)
    {	return returnString(Dcc::RGetUnixFilePath(GetNintendoOptions()->m_Opt.m_OutFolderPath));	}
    virtual void SetOutFolder(const TCHAR* folder)
    {
        std::string str = Dcc::RGetUnixFilePath(M_2_S(folder));
        size_t len = str.length();
        if(len > 0 && str[len - 1] != '/') str += "/";
        GetNintendoOptions()->m_Opt.m_OutFolderPath = str;
    }
    // filename
    virtual const TCHAR* GetOutFilename(void)
    {		return returnString(GetNintendoOptions()->m_Opt.m_OutFileName); }
    virtual void SetOutFilename(const TCHAR* filename)
    {	GetNintendoOptions()->m_Opt.m_OutFileName = M_2_S(filename);	}

    ////////////////////////////////////////////////////////
    virtual BOOL GetUseMerge(void)
    {	return GetNintendoOptions()->m_Opt.m_MergeFmdFlag; }
    virtual void SetUseMerge(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_MergeFmdFlag = (flag == TRUE); }
    // merge filename
    virtual const TCHAR* GetMergeFilename(void)
    {	return returnString(Dcc::RGetUnixFilePath(GetNintendoOptions()->m_Opt.m_MergeFmdPath));	}
    virtual void SetMergeFilename(const TCHAR* filename)
    {	GetNintendoOptions()->m_Opt.m_MergeFmdPath = Dcc::RGetUnixFilePath(M_2_A(filename));	}
    virtual BOOL GetDoesMergeAuto(void)
    {	return TRUE; }
    virtual void SetDoesMergeAuto(BOOL flag)
    {	/*GetNintendoOptions()->m_Opt.m_AutoMerge = (flag == TRUE);*/ }
    ////////////////////////////////////////////////////////
    virtual BOOL GetUseMergeFtx(void)
    {	return GetNintendoOptions()->m_Opt.m_MergeFtxFlag; }
    virtual void SetUseMergeFtx(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_MergeFtxFlag = (flag == TRUE); }
    // merge animation
    virtual BOOL GetUseMergeAnim(void)
    {	return GetNintendoOptions()->m_Opt.m_MergeAnimFlag; }
    virtual void SetUseMergeAnim(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_MergeAnimFlag = (flag == TRUE); }
    virtual const TCHAR* GetMergeAnimFolder(void)
    {	return returnString(Dcc::RGetUnixFilePath(GetNintendoOptions()->m_Opt.m_MergeAnimFolder));	}
    virtual void SetMergeAnimFolder(const TCHAR* folder)
    {
        std::string str = Dcc::RGetUnixFilePath(M_2_S(folder));
        size_t len = str.length();
        if(len > 0 && str[len - 1] != '/') str += "/";
        GetNintendoOptions()->m_Opt.m_MergeAnimFolder = str;
    }
    virtual const TCHAR* GetMergeAnimName(void)
    {	return returnString(GetNintendoOptions()->m_Opt.m_MergeAnimName);	}
    virtual void SetMergeAnimName(const TCHAR* filename)
    {	GetNintendoOptions()->m_Opt.m_MergeAnimName = M_2_A(filename);	}
    ////////////////////////////////////////////////////////
    virtual float GetMagnify(void)
    {	return static_cast<float>(GetNintendoOptions()->m_Opt.m_Magnify);	}
    virtual void SetMagnify(float val)
    {	GetNintendoOptions()->m_Opt.m_Magnify = val;	}
    ////////////////////////////////////////////////////////
    virtual BOOL GetUseFigureMode(void)
    {	return GetNintendoOptions()->m_UseFigureMode; }
    virtual void SetUseFigureMode(BOOL flag)
    {	GetNintendoOptions()->m_UseFigureMode = (flag == TRUE); }
    ////////////////////////////////////////////////////////
    virtual const TCHAR* GetCommentText(void)
    {	return returnString(GetNintendoOptions()->m_Opt.m_CommentText);	}
    virtual void SetCommentText(const TCHAR* text)
    {	GetNintendoOptions()->m_Opt.m_CommentText = M_2_A(text);	}
    ////////////////////////////////////////////////////////
    virtual BOOL GetUseBinaryFormat(void)
    {	return GetNintendoOptions()->m_Opt.m_IsBinaryFormat;	}
    virtual void SetUseBinaryFormat(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_IsBinaryFormat = (flag == TRUE);	}
    ////////////////////////////////////////////////////////
    // export model
    virtual BOOL GetDoesExportModel(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FMD];	}
    virtual void SetDoesExportModel(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FMD]= (flag == TRUE);	}
    // export anim
    virtual BOOL GetDoesExportSkelAnim(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSK];	}
    virtual void SetDoesExportSkelAnim(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSK]= (flag == TRUE);	}
    // export model anim(visibility)
    virtual BOOL GetDoesExportVisBoneAnim(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FVB];	}
    virtual void SetDoesExportVisBoneAnim(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FVB] = (flag == TRUE);	}
    // export mtl color
    virtual BOOL GetDoesExportMtlColorAnim(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FCL];	}
    virtual void SetDoesExportMtlColorAnim(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FCL] = (flag == TRUE);	}
    // export mtl tex srt
    virtual BOOL GetDoesExportMtlTexSRTAnim(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FTS];	}
    virtual void SetDoesExportMtlTexSRTAnim(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FTS] = (flag == TRUE);	}
    // export mtl tex pattern
    virtual BOOL GetDoesExportMtlTexPatAnim(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FTP];	}
    virtual void SetDoesExportMtlTexPatAnim(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FTP] = (flag == TRUE);	}
    // export shape anim
    virtual BOOL GetDoesExportShapeAnim(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSH];	}
    virtual void SetDoesExportShapeAnim(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSH] = (flag == TRUE);	}
    ////////////////////////////////////////////////////////
    // export camera
    virtual BOOL GetDoesExportCam(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSN];	}
    virtual void SetDoesExportCam(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSN]= (flag == TRUE);	}
    // export light
    virtual BOOL GetDoesExportLight(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSN];	}
    virtual void SetDoesExportLight(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSN] = (flag == TRUE);	}
    // export Scene
    virtual BOOL GetDoesExportSceneAnim(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSN];	}
    virtual void SetDoesExportSceneAnim(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSN]= (flag == TRUE);	}
    // export Env
    virtual BOOL GetDoesExportEnv(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSN];	}
    virtual void SetDoesExportEnv(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFileFlag[Dcc::RExpOpt::FSN]= (flag == TRUE);	}
    ////////////////////////////////////////////////////////
    // export tex
    virtual BOOL GetDoesExportTex(void)
    {	return GetNintendoOptions()->m_Opt.m_OutFtxFlag;	}
    virtual void SetDoesExportTex(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OutFtxFlag = (flag == TRUE);	}
    ////////////////////////////////////////////////////////
    virtual int GetCompressBone(void)
    {	return IdxToMaxIdx(GetNintendoOptions()->m_Opt.m_CompressBone); }
    virtual void SetCompressBone(int val)
    {	GetNintendoOptions()->m_Opt.m_CompressBone = static_cast<Dcc::RExpOpt::CompressBone>(MaxIdxToIdx(val)); }
    virtual BOOL GetDoesUniteChild(void)
    {	return GetNintendoOptions()->m_Opt.m_UniteChild; }
    virtual void SetDoesUniteChild(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_UniteChild= (flag == TRUE); }
    virtual BOOL GetDoesCompressMtl(void)
    {	return GetNintendoOptions()->m_Opt.m_CompressMaterial; }
    virtual void SetDoesCompressMtl(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_CompressMaterial = (flag == TRUE); }
    virtual BOOL GetDoesCompressShape(void)
    {	return GetNintendoOptions()->m_Opt.m_CompressShape; }
    virtual void SetDoesCompressShape(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_CompressShape = (flag == TRUE); }
    virtual BOOL GetDoesOptimizePrim(void)
    {	return TRUE; }
    virtual void SetDoesOptimizePrim(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_OptimizePrimitive = Dcc::RExpOpt::OPTIMIZE_PRIMITIVE_FORSYTH; }
    virtual BOOL GetDoesQuantizationAnalize(void)
    {	return GetNintendoOptions()->m_Opt.m_QuantizationAnalysis; }
    virtual void SetDoesQuantizationAnalize(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_QuantizationAnalysis = (flag == TRUE); }
    ////////////////////////////////////////////////////////
    /*
    virtual int GetQuantPos(void)
    {	return IdxToMaxIdx(GetNintendoOptions()->m_Opt.m_QuantPos);	}
    virtual void SetQuantPos(int val)
    {	GetNintendoOptions()->m_Opt.m_QuantPos = (Dcc::RExpOpt::Quantize)MaxIdxToIdx(val);	}
    virtual int GetQuantNormal(void)
    {	return IdxToMaxIdx(GetNintendoOptions()->m_Opt.m_QuantNrm);	}
    virtual void SetQuantNormal(int val)
    {	GetNintendoOptions()->m_Opt.m_QuantNrm = (Dcc::RExpOpt::Quantize)MaxIdxToIdx(val);	}
    virtual int GetQuantTex(void)
    {	return IdxToMaxIdx(GetNintendoOptions()->m_Opt.m_QuantTex);	}
    virtual void SetQuantTex(int val)
    {	GetNintendoOptions()->m_Opt.m_QuantTex = (Dcc::RExpOpt::Quantize)MaxIdxToIdx(val);	}
    */
    ////////////////////////////////////////////////////////
    virtual BOOL GetDoesExportAllFrames(void)
    {	return GetNintendoOptions()->m_Opt.m_FrameRange == Dcc::RExpOpt::FRAME_RANGE_ALL; }
    virtual void SetDoesExportAllFrames(BOOL flag)
    {	GetNintendoOptions()->m_Opt.m_FrameRange
    = flag? Dcc::RExpOpt::FRAME_RANGE_ALL : Dcc::RExpOpt::FRAME_RANGE_SPECIFY; }
    virtual int GetStartFrame(void)
    {	return GetNintendoOptions()->m_Opt.m_StartFrame; }
    virtual void SetStartFrame(int f)
    {	GetNintendoOptions()->m_Opt.m_StartFrame = f; }
    virtual int GetEndFrame(void)
    {	return GetNintendoOptions()->m_Opt.m_EndFrame; }
    virtual void SetEndFrame(int f)
    {	GetNintendoOptions()->m_Opt.m_EndFrame = f; }
    virtual int GetFramePrecision(void)
        {	return GetNintendoOptions()->m_Opt.m_FramePrecision; }
    virtual void SetFramePrecision(int p)
        {	GetNintendoOptions()->m_Opt.m_FramePrecision = p; }
    virtual BOOL GetIsLoop(void)
        {	return GetNintendoOptions()->m_Opt.m_LoopAnim; }
    virtual void SetIsLoop(BOOL flag)
        {	GetNintendoOptions()->m_Opt.m_LoopAnim = (flag == TRUE); }
    ////////////////////////////////////////////////////////
    // Bake Tolerance
    ////////////////////////////////////////////////////////
    virtual float GetToleranceTrans(void)
        {	return GetNintendoOptions()->m_Opt.m_TolT;	}
    virtual void SetToleranceTrans(float val)
        {	GetNintendoOptions()->m_Opt.m_TolT = val;	}
    virtual float GetToleranceRotate(void)
        {	return GetNintendoOptions()->m_Opt.m_TolR;	}
    virtual void SetToleranceRotate(float val)
        {	GetNintendoOptions()->m_Opt.m_TolR = val;	}
    virtual float GetToleranceScale(void)
        {	return GetNintendoOptions()->m_Opt.m_TolS;	}
    virtual void SetToleranceScale(float val)
        {	GetNintendoOptions()->m_Opt.m_TolS = val;	}
    //
    virtual float GetToleranceUVTrans(void)
        {	return GetNintendoOptions()->m_Opt.m_TolTexT;	}
    virtual void SetToleranceUVTrans(float val)
        {	GetNintendoOptions()->m_Opt.m_TolTexT = val;	}
    virtual float GetToleranceUVRotate(void)
        {	return GetNintendoOptions()->m_Opt.m_TolTexR;	}
    virtual void SetToleranceUVRotate(float val)
        {	GetNintendoOptions()->m_Opt.m_TolTexR = val;	}
    virtual float GetToleranceUVScale(void)
        {	return GetNintendoOptions()->m_Opt.m_TolTexS;	}
    virtual void SetToleranceUVScale(float val)
        {	GetNintendoOptions()->m_Opt.m_TolTexS = val;	}
    //
    virtual float GetToleranceColor(void)
        {	return GetNintendoOptions()->m_Opt.m_TolC;	}
    virtual void SetToleranceColor(float val)
        {	GetNintendoOptions()->m_Opt.m_TolC = val;	}
    ////////////////////////////////////////////////////////
    // Quantization Tolerance
    ////////////////////////////////////////////////////////
    virtual float GetQuantToleranceTrans(void)
    {	return GetNintendoOptions()->m_Opt.m_QuantTolT;	}
    virtual void SetQuantToleranceTrans(float val)
    {	GetNintendoOptions()->m_Opt.m_QuantTolT = val;	}
    virtual float GetQuantToleranceRotate(void)
    {	return GetNintendoOptions()->m_Opt.m_QuantTolR;	}
    virtual void SetQuantToleranceRotate(float val)
    {	GetNintendoOptions()->m_Opt.m_QuantTolR = val;	}
    virtual float GetQuantToleranceScale(void)
    {	return GetNintendoOptions()->m_Opt.m_QuantTolS;	}
    virtual void SetQuantToleranceScale(float val)
    {	GetNintendoOptions()->m_Opt.m_QuantTolS = val;	}
    //
    virtual float GetQuantToleranceUVTrans(void)
    {	return GetNintendoOptions()->m_Opt.m_QuantTolTexT;	}
    virtual void SetQuantToleranceUVTrans(float val)
    {	GetNintendoOptions()->m_Opt.m_QuantTolTexT = val;	}
    virtual float GetQuantToleranceUVRotate(void)
    {	return GetNintendoOptions()->m_Opt.m_QuantTolTexR;	}
    virtual void SetQuantToleranceUVRotate(float val)
    {	GetNintendoOptions()->m_Opt.m_QuantTolTexR = val;	}
    virtual float GetQuantToleranceUVScale(void)
    {	return GetNintendoOptions()->m_Opt.m_QuantTolTexS;	}
    virtual void SetQuantToleranceUVScale(float val)
    {	GetNintendoOptions()->m_Opt.m_QuantTolTexS = val;	}
    //
    virtual float GetQuantToleranceFrame(void)
    {	return 0.5f;  /*GetNintendoOptions()->m_Opt.m_QuantTolF;*/	}
    virtual void SetQuantToleranceFrame(float val)
    {	/*GetNintendoOptions()->m_Opt.m_QuantTolF = val;*/	}

    // Don`t convert to y-up
    virtual BOOL GetDisableZYAxisConversion(void)
    {	return GetNintendoOptions()->m_DisableZYAxisConversion; }
    virtual void SetDisableZYAxisConversion(BOOL flag)
    {	GetNintendoOptions()->m_DisableZYAxisConversion = (flag == TRUE); }

    ////////////////////////////////////////////////////////
    virtual const TCHAR* GetPreExportScript(void)
    {	return returnString(GetNintendoOptions()->m_Opt.m_PreExpScript);	}
    virtual void SetPreExportScript(const TCHAR* text)
    {	GetNintendoOptions()->m_Opt.m_PreExpScript = M_2_A(text);	}
    ////////////////////////////////////////////////////////
    virtual const TCHAR* GetPostExportScript(void)
    {	return returnString(GetNintendoOptions()->m_Opt.m_PostExpScript);	}
    virtual void SetPostExportScript(const TCHAR* text)
    {	GetNintendoOptions()->m_Opt.m_PostExpScript = M_2_A(text);	}

    ///////////////////////
    // lock ui
    virtual BOOL GetLock_export_target(void){return GetNintendoOptions()->m_Lock_export_target;}
    virtual void SetLock_export_target(BOOL flag){	GetNintendoOptions()->m_Lock_export_target = flag;}
    virtual BOOL GetLock_output_file_name(void){return GetNintendoOptions()->m_Lock_output_file_name;}
    virtual void SetLock_output_file_name(BOOL flag){	GetNintendoOptions()->m_Lock_output_file_name = flag;}
    virtual BOOL GetLock_output_folder(void){return GetNintendoOptions()->m_Lock_output_folder;}
    virtual void SetLock_output_folder(BOOL flag){	GetNintendoOptions()->m_Lock_output_folder = flag;}
    virtual BOOL GetLock_use_figure_mode(void){return GetNintendoOptions()->m_Lock_use_figure_mode;}
    virtual void SetLock_use_figure_mode(BOOL flag){	GetNintendoOptions()->m_Lock_use_figure_mode = flag;}
    virtual BOOL GetLock_merge_fmd(void){return GetNintendoOptions()->m_Lock_merge_fmd;}
    virtual void SetLock_merge_fmd(BOOL flag){	GetNintendoOptions()->m_Lock_merge_fmd = flag;}
    virtual BOOL GetLock_merge_ftx(void){return GetNintendoOptions()->m_Lock_merge_ftx;}
    virtual void SetLock_merge_ftx(BOOL flag){	GetNintendoOptions()->m_Lock_merge_ftx = flag;}
    virtual BOOL GetLock_merge_anim(void){return GetNintendoOptions()->m_Lock_merge_anim;}
    virtual void SetLock_merge_anim(BOOL flag){	GetNintendoOptions()->m_Lock_merge_anim = flag;}
    virtual BOOL GetLock_merge_anim_name(void){return GetNintendoOptions()->m_Lock_merge_anim_name;}
    virtual void SetLock_merge_anim_name(BOOL flag){	GetNintendoOptions()->m_Lock_merge_anim_name = flag;}
    virtual BOOL GetLock_magnify(void){return GetNintendoOptions()->m_Lock_magnify;}
    virtual void SetLock_magnify(BOOL flag){	GetNintendoOptions()->m_Lock_magnify = flag;}
    virtual BOOL GetLock_texsrt_mode(void){return GetNintendoOptions()->m_Lock_texsrt_mode;}
    virtual void SetLock_texsrt_mode(BOOL flag){	GetNintendoOptions()->m_Lock_texsrt_mode = flag;}
    virtual BOOL GetLock_remove_namespace(void){return GetNintendoOptions()->m_Lock_remove_namespace;}
    virtual void SetLock_remove_namespace(BOOL flag){	GetNintendoOptions()->m_Lock_remove_namespace = flag;}
    virtual BOOL GetLock_comment(void){return GetNintendoOptions()->m_Lock_comment;}
    virtual void SetLock_comment(BOOL flag){	GetNintendoOptions()->m_Lock_comment = flag;}
    virtual BOOL GetLock_frame_range(void){return GetNintendoOptions()->m_Lock_frame_range;}
    virtual void SetLock_frame_range(BOOL flag){	GetNintendoOptions()->m_Lock_frame_range = flag;}
    virtual BOOL GetLock_loop_anim(void){return GetNintendoOptions()->m_Lock_loop_anim;}
    virtual void SetLock_loop_anim(BOOL flag){	GetNintendoOptions()->m_Lock_loop_anim = flag;}
    virtual BOOL GetLock_bake_all_anim(void){return GetNintendoOptions()->m_Lock_bake_all_anim;}
    virtual void SetLock_bake_all_anim(BOOL flag){	GetNintendoOptions()->m_Lock_bake_all_anim = flag;}
    virtual BOOL GetLock_frame_precision(void){return GetNintendoOptions()->m_Lock_frame_precision;}
    virtual void SetLock_frame_precision(BOOL flag){	GetNintendoOptions()->m_Lock_frame_precision = flag;}
    virtual BOOL GetLock_output_fmd(void){return GetNintendoOptions()->m_Lock_output_fmd;}
    virtual void SetLock_output_fmd(BOOL flag){	GetNintendoOptions()->m_Lock_output_fmd = flag;}
    virtual BOOL GetLock_output_ftx(void){return GetNintendoOptions()->m_Lock_output_ftx;}
    virtual void SetLock_output_ftx(BOOL flag){	GetNintendoOptions()->m_Lock_output_ftx = flag;}
    virtual BOOL GetLock_output_fsk(void){return GetNintendoOptions()->m_Lock_output_fsk;}
    virtual void SetLock_output_fsk(BOOL flag){	GetNintendoOptions()->m_Lock_output_fsk = flag;}
    virtual BOOL GetLock_output_fvb(void){return GetNintendoOptions()->m_Lock_output_fvb;}
    virtual void SetLock_output_fvb(BOOL flag){	GetNintendoOptions()->m_Lock_output_fvb = flag;}
    virtual BOOL GetLock_output_fcl(void){return GetNintendoOptions()->m_Lock_output_fcl;}
    virtual void SetLock_output_fcl(BOOL flag){	GetNintendoOptions()->m_Lock_output_fcl = flag;}
    virtual BOOL GetLock_output_fts(void){return GetNintendoOptions()->m_Lock_output_fts;}
    virtual void SetLock_output_fts(BOOL flag){	GetNintendoOptions()->m_Lock_output_fts = flag;}
    virtual BOOL GetLock_output_ftp(void){return GetNintendoOptions()->m_Lock_output_ftp;}
    virtual void SetLock_output_ftp(BOOL flag){	GetNintendoOptions()->m_Lock_output_ftp = flag;}
    virtual BOOL GetLock_output_fsh(void){return GetNintendoOptions()->m_Lock_output_fsh;}
    virtual void SetLock_output_fsh(BOOL flag){	GetNintendoOptions()->m_Lock_output_fsh = flag;}
    virtual BOOL GetLock_output_fsn(void){return GetNintendoOptions()->m_Lock_output_fsn;}
    virtual void SetLock_output_fsn(BOOL flag){	GetNintendoOptions()->m_Lock_output_fsn = flag;}
    virtual BOOL GetLock_compress_bone(void){return GetNintendoOptions()->m_Lock_compress_bone;}
    virtual void SetLock_compress_bone(BOOL flag){	GetNintendoOptions()->m_Lock_compress_bone = flag;}
    virtual BOOL GetLock_unite_child(void){return GetNintendoOptions()->m_Lock_unite_child;}
    virtual void SetLock_unite_child(BOOL flag){	GetNintendoOptions()->m_Lock_unite_child = flag;}
    virtual BOOL GetLock_compress_material(void){return GetNintendoOptions()->m_Lock_compress_material;}
    virtual void SetLock_compress_material(BOOL flag){	GetNintendoOptions()->m_Lock_compress_material = flag;}
    virtual BOOL GetLock_compress_shape(void){return GetNintendoOptions()->m_Lock_compress_shape;}
    virtual void SetLock_compress_shape(BOOL flag){	GetNintendoOptions()->m_Lock_compress_shape = flag;}
    virtual BOOL GetLock_tolerance_scale(void){return GetNintendoOptions()->m_Lock_tolerance_scale;}
    virtual void SetLock_tolerance_scale(BOOL flag){	GetNintendoOptions()->m_Lock_tolerance_scale = flag;}
    virtual BOOL GetLock_tolerance_rotate(void){return GetNintendoOptions()->m_Lock_tolerance_rotate;}
    virtual void SetLock_tolerance_rotate(BOOL flag){	GetNintendoOptions()->m_Lock_tolerance_rotate = flag;}
    virtual BOOL GetLock_tolerance_translate(void){return GetNintendoOptions()->m_Lock_tolerance_translate;}
    virtual void SetLock_tolerance_translate(BOOL flag){	GetNintendoOptions()->m_Lock_tolerance_translate = flag;}
    virtual BOOL GetLock_tolerance_color(void){return GetNintendoOptions()->m_Lock_tolerance_color;}
    virtual void SetLock_tolerance_color(BOOL flag){	GetNintendoOptions()->m_Lock_tolerance_color = flag;}
    virtual BOOL GetLock_tolerance_tex_scale(void){return GetNintendoOptions()->m_Lock_tolerance_tex_scale;}
    virtual void SetLock_tolerance_tex_scale(BOOL flag){	GetNintendoOptions()->m_Lock_tolerance_tex_scale = flag;}
    virtual BOOL GetLock_tolerance_tex_rotate(void){return GetNintendoOptions()->m_Lock_tolerance_tex_rotate;}
    virtual void SetLock_tolerance_tex_rotate(BOOL flag){	GetNintendoOptions()->m_Lock_tolerance_tex_rotate = flag;}
    virtual BOOL GetLock_tolerance_tex_translate(void){return GetNintendoOptions()->m_Lock_tolerance_tex_translate;}
    virtual void SetLock_tolerance_tex_translate(BOOL flag){	GetNintendoOptions()->m_Lock_tolerance_tex_translate = flag;}
    virtual BOOL GetLock_quantize_tolerance_scale(void){return GetNintendoOptions()->m_Lock_quantize_tolerance_scale;}
    virtual void SetLock_quantize_tolerance_scale(BOOL flag){	GetNintendoOptions()->m_Lock_quantize_tolerance_scale = flag;}
    virtual BOOL GetLock_quantize_tolerance_rotate(void){return GetNintendoOptions()->m_Lock_quantize_tolerance_rotate;}
    virtual void SetLock_quantize_tolerance_rotate(BOOL flag){	GetNintendoOptions()->m_Lock_quantize_tolerance_rotate = flag;}
    virtual BOOL GetLock_quantize_tolerance_translate(void){return GetNintendoOptions()->m_Lock_quantize_tolerance_translate;}
    virtual void SetLock_quantize_tolerance_translate(BOOL flag){	GetNintendoOptions()->m_Lock_quantize_tolerance_translate = flag;}
    virtual BOOL GetLock_quantize_tolerance_tex_scale(void){return GetNintendoOptions()->m_Lock_quantize_tolerance_tex_scale;}
    virtual void SetLock_quantize_tolerance_tex_scale(BOOL flag){	GetNintendoOptions()->m_Lock_quantize_tolerance_tex_scale = flag;}
    virtual BOOL GetLock_quantize_tolerance_tex_rotate(void){return GetNintendoOptions()->m_Lock_quantize_tolerance_tex_rotate;}
    virtual void SetLock_quantize_tolerance_tex_rotate(BOOL flag){	GetNintendoOptions()->m_Lock_quantize_tolerance_tex_rotate = flag;}
    virtual BOOL GetLock_quantize_tolerance_tex_translate(void){return GetNintendoOptions()->m_Lock_quantize_tolerance_tex_translate;}
    virtual void SetLock_quantize_tolerance_tex_translate(BOOL flag){GetNintendoOptions()->m_Lock_quantize_tolerance_tex_translate = flag;}
    virtual BOOL GetLock_pre_export_script(void){return GetNintendoOptions()->m_Lock_pre_export_script;}
    virtual void SetLock_pre_export_script(BOOL flag){GetNintendoOptions()->m_Lock_pre_export_script = flag;}
    virtual BOOL GetLock_post_export_script(void){return GetNintendoOptions()->m_Lock_post_export_script;}
    virtual void SetLock_post_export_script(BOOL flag){GetNintendoOptions()->m_Lock_post_export_script = flag;}

    ///////////////////////
    // functions
    BOOL DoExport(BOOL selected);
    //MCHAR* SearchAnimations(BOOL selected);
    TSTR SearchAnimations(BOOL selected);

    void ResetSetting(void)
    {	GetNintendoOptions()->Init(); }

    void LoadSetting(void)
    {	GetNintendoOptions()->LoadFromAtribute(); }

    void SaveSetting(void)
    {	GetNintendoOptions()->SaveToAttribute(); }

    int GetVersion(void)
    {	return NINTENDOEXPORT_VERSION;	}

    void SetOptionEnvVar(BOOL isPost, BOOL isSucceedExport)
    {
        NintendoExport::Get()->SetOptionEnvVarForPrePostExportScript(isPost==TRUE, isSucceedExport==TRUE);
    }

    MCHAR* GetVersionString(void)
    {
#if 1
        const char* buf = NintendoExport::Get()->GetPluginVersionString();
        char* sVer = new char[128];
        strcpy_s(sVer, 128, buf);

#else
        char buf[64];
        sprintf_s(buf, "%.3d", NINTENDOEXPORT_VERSION);
        char* sVer = new char[64];
        sprintf_s(sVer, 64, "%c.%c.%c%s", buf[0], buf[1], buf[2], NINTENDOEXPORT_VERSION_MINOR_STR);
#endif

#ifndef UNICODE
        return sVer;
#else
        std::wstring wVer = Dcc::RGetUnicodeFromShiftJis(sVer);
        MCHAR* retChar = new MCHAR[wVer.size() + 1];
        wcscpy_s(retChar, wVer.size() + 1,wVer.c_str());
        return retChar;
#endif
    }

    MCHAR* GetDateTimeString(void)
    {
        string datetime = Dcc::RGetDateTimeString();
#ifndef UNICODE
        char* retChar = new char[datetime.size() + 1];
        strcpy_s(retChar, datetime.size() + 1,datetime.c_str());
        return retChar;
#else
        std::wstring wdatetime = Dcc::RGetUnicodeFromShiftJis(datetime);
        MCHAR* retChar = new MCHAR[wdatetime.size() + 1];
        wcscpy_s(retChar, wdatetime.size() + 1,wdatetime.c_str());
        return retChar;
#endif
    }

    MCHAR* GetBuildDateTimeString(void)
    {
        string datetime;
        datetime += __DATE__;
        datetime += "_";
        datetime += __TIME__;

        string::iterator it;
        for(it = datetime.begin(); it != datetime.end(); ++it)
        {
            if(*it == ' ' || *it == ':')
            {
                *it = '-';
            }
        }


#ifndef UNICODE
        char* retChar = new char[datetime.size() + 1];
        strcpy_s(retChar, datetime.size() + 1,datetime.c_str());
        return retChar;
#else
        std::wstring wdatetime = Dcc::RGetUnicodeFromShiftJis(datetime);
        MCHAR* retChar = new MCHAR[wdatetime.size() + 1];
        wcscpy_s(retChar, wdatetime.size() + 1,wdatetime.c_str());
        return retChar;
#endif
    }


    IMenu* GetMtlEditorMenu(void)
    {
        IMenuManager* menuMan = GetCOREInterface()->GetMenuManager();
        if(!menuMan) return nullptr;
        IMenuBarContext* menuBarContext = static_cast<IMenuBarContext*>(menuMan->GetContext(kMaterialEditorMenuBar));
        if(!menuBarContext) return nullptr;
        return menuBarContext->GetMenu();
    }

    TSTR GetErrorLog(void)
    {
        TSTR log = A_2_M(NintendoExport::Get()->GetErrorLog().c_str());
        return log;
    }

    void LayerTest(void);
    BOOL GetUseFclFtsFtp(void)
    {
        return GetNintendoOptions()->m_Opt.m_UsesFclFtsFtp;
    }
    BOOL GetSeparateMaterialAnim(void)
    {
        return GetNintendoOptions()->m_Opt.m_SeparatesFma;
    }


    DECLARE_DESCRIPTOR(NintendoExport_Imp)
#pragma warning(disable : 4238) // disable nonstandard extension  warning
    // dispatch map
    BEGIN_FUNCTION_MAP
        //
        FN_1	(nw4f_startExport, TYPE_BOOL, DoExport, TYPE_BOOL);
        FN_1	(nw4f_searchAnimations, TYPE_TSTR_BV, SearchAnimations, TYPE_BOOL);
        VFN_2	(nw4f_setOptionEnvVar, SetOptionEnvVar, TYPE_BOOL, TYPE_BOOL);
        VFN_0	(nw4f_resetSetting, ResetSetting);
        VFN_0	(nw4f_loadSetting, LoadSetting);
        VFN_0	(nw4f_saveSetting, SaveSetting);
        FN_0	(nw4f_getVersion, TYPE_INT, GetVersion);
        FN_0	(nw4f_getVersionString, TYPE_STRING, GetVersionString);
        FN_0	(nw4f_getDateTimeString, TYPE_STRING, GetDateTimeString);
        FN_0	(nw4f_getBuildDateTimeString, TYPE_STRING, GetBuildDateTimeString);
        FN_0	(nw4f_getMtlEditorMenu, TYPE_INTERFACE, GetMtlEditorMenu);
        FN_0	(nw4f_getErrorLog, TYPE_TSTR_BV, GetErrorLog);
        VFN_0	(nw4f_LayerTest, LayerTest);
        FN_0	(nw4f_useFclFtsFtp, TYPE_BOOL, GetUseFclFtsFtp);
        FN_0	(nw4f_separateMaterialAnim, TYPE_BOOL, GetSeparateMaterialAnim);
        //
        PROP_FNS(nw4f_getPresetName, GetPresetName, nw4f_setPresetName, SetPresetName, TYPE_STRING);
        PROP_FNS(nw4f_getPresetFile, GetPresetFile, nw4f_setPresetFile, SetPresetFile, TYPE_STRING);
        PROP_FNS(nw4f_getShowPresetUI, GetShowPresetUI, nw4f_setShowPresetUI, SetShowPresetUI, TYPE_BOOL);
        //
        PROP_FNS(nw4f_getDoesExportSelected, GetDoesExportSelected, nw4f_setDoesExportSelected, SetDoesExportSelected, TYPE_BOOL);
        PROP_FNS(nw4f_getOutFolder, GetOutFolder, nw4f_setOutFolder, SetOutFolder, TYPE_STRING);
        PROP_FNS(nw4f_getOutName, GetOutFilename, nw4f_setOutName, SetOutFilename, TYPE_STRING);
        //
        PROP_FNS(nw4f_getUseMerge, GetUseMerge, nw4f_setUseMerge, SetUseMerge, TYPE_BOOL);
        PROP_FNS(nw4f_getMergeFile, GetMergeFilename, nw4f_setMergeFile, SetMergeFilename, TYPE_STRING);
        PROP_FNS(nw4f_getMergeAuto, GetDoesMergeAuto, nw4f_setMergeAuto, SetDoesMergeAuto, TYPE_BOOL);
        //
        PROP_FNS(nw4f_getMagnify, GetMagnify, nw4f_setMagnify, SetMagnify, TYPE_FLOAT);
        PROP_FNS(nw4f_getUseFigureMode, GetUseFigureMode, nw4f_setUseFigureMode, SetUseFigureMode, TYPE_BOOL);
        PROP_FNS(nw4f_getCommentText, GetCommentText, nw4f_setCommentText, SetCommentText, TYPE_STRING);
        //
        PROP_FNS(nw4f_getUseBinaryFormat, GetUseBinaryFormat, nw4f_setUseBinaryFormat, SetUseBinaryFormat, TYPE_BOOL);
        //
        PROP_FNS(nw4f_getDoesExportModel, GetDoesExportModel, nw4f_setDoesExportModel, SetDoesExportModel, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesExportSkelAnim, GetDoesExportSkelAnim, nw4f_setDoesExportSkelAnim, SetDoesExportSkelAnim, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesExportVisBoneAnim, GetDoesExportVisBoneAnim, nw4f_setDoesExportVisBoneAnim, SetDoesExportVisBoneAnim, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesExportMtlColAnim, GetDoesExportMtlColorAnim, nw4f_setDoesExportMtlColAnim, SetDoesExportMtlColorAnim, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesExportMtlTexPatAnim, GetDoesExportMtlTexPatAnim, nw4f_setDoesExportMtlTexPatAnim, SetDoesExportMtlTexPatAnim, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesExportMtlTexSRTAnim, GetDoesExportMtlTexSRTAnim, nw4f_setDoesExportMtlTexSRTAnim, SetDoesExportMtlTexSRTAnim, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesExportShapeAnim, GetDoesExportShapeAnim,nw4f_setDoesExportShapeAnim, SetDoesExportShapeAnim, TYPE_BOOL);

        PROP_FNS(nw4f_getDoesExportCam, GetDoesExportCam, nw4f_setDoesExportCam, SetDoesExportCam, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesExportLight, GetDoesExportLight, nw4f_setDoesExportLight, SetDoesExportLight, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesExportSceneAnim, GetDoesExportSceneAnim, nw4f_setDoesExportSceneAnim, SetDoesExportSceneAnim, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesExportEnv, GetDoesExportEnv, nw4f_setDoesExportEnv, SetDoesExportEnv, TYPE_BOOL);
        //
        PROP_FNS(nw4f_getDoesExportTex, GetDoesExportTex, nw4f_setDoesExportTex, SetDoesExportTex, TYPE_BOOL);
        //
        //
        PROP_FNS(nw4f_getCompressBone, GetCompressBone, nw4f_setCompressBone, SetCompressBone, TYPE_INT);
        PROP_FNS(nw4f_getDoesUniteChild, GetDoesUniteChild, nw4f_setDoesUniteChild, SetDoesUniteChild, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesCompressMtl, GetDoesCompressMtl, nw4f_setDoesCompressMtl, SetDoesCompressMtl, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesCompressShape, GetDoesCompressShape, nw4f_setDoesCompressShape, SetDoesCompressShape, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesOptimizePrim, GetDoesOptimizePrim, nw4f_setDoesOptimizePrim, SetDoesOptimizePrim, TYPE_BOOL);
        PROP_FNS(nw4f_getDoesQuantizationAnalize, GetDoesQuantizationAnalize, nw4f_setDoesQuantizationAnalize, SetDoesQuantizationAnalize, TYPE_BOOL);
        //
        /*
        PROP_FNS(nw4f_getQuantPos, GetQuantPos, nw4f_setQuantPos, SetQuantPos, TYPE_INT);
        PROP_FNS(nw4f_getQuantNormal	, GetQuantNormal	, nw4f_setQuantNormal, SetQuantNormal, TYPE_INT);
        PROP_FNS(nw4f_getQuantTex, GetQuantTex, nw4f_setQuantTex, SetQuantTex, TYPE_INT);
        */
        //
        PROP_FNS(nw4f_getDoesExportAllFrames, GetDoesExportAllFrames, nw4f_setDoesExportAllFrames, SetDoesExportAllFrames, TYPE_BOOL);
        PROP_FNS(nw4f_getStartFrame, GetStartFrame, nw4f_setStartFrame, SetStartFrame, TYPE_INT);
        PROP_FNS(nw4f_getEndFrame, GetEndFrame, nw4f_setEndFrame, SetEndFrame, TYPE_INT);
        PROP_FNS(nw4f_getFramePrecision, GetFramePrecision, nw4f_setFramePrecision, SetFramePrecision, TYPE_INT);
        PROP_FNS(nw4f_getIsLoop, GetIsLoop, nw4f_setIsLoop, SetIsLoop, TYPE_BOOL);
        //
        PROP_FNS(nw4f_getToleranceTrans, GetToleranceTrans, nw4f_setToleranceTrans, SetToleranceTrans, TYPE_FLOAT);
        PROP_FNS(nw4f_getToleranceRotate, GetToleranceRotate, nw4f_setToleranceRotate, SetToleranceRotate, TYPE_FLOAT);
        PROP_FNS(nw4f_getToleranceScale, GetToleranceScale, nw4f_setToleranceScale, SetToleranceScale, TYPE_FLOAT);
        //
        PROP_FNS(nw4f_getToleranceUVTrans, GetToleranceUVTrans, nw4f_setToleranceUVTrans, SetToleranceUVTrans, TYPE_FLOAT);
        PROP_FNS(nw4f_getToleranceUVRotate, GetToleranceUVRotate, nw4f_setToleranceUVRotate, SetToleranceUVRotate, TYPE_FLOAT);
        PROP_FNS(nw4f_getToleranceUVScale, GetToleranceUVScale, nw4f_setToleranceUVScale, SetToleranceUVScale, TYPE_FLOAT);
        PROP_FNS(nw4f_getToleranceColor, GetToleranceColor, nw4f_setToleranceColor, SetToleranceColor, TYPE_FLOAT);
        //
        PROP_FNS(nw4f_getQuantToleranceTrans, GetQuantToleranceTrans, nw4f_setQuantToleranceTrans, SetQuantToleranceTrans, TYPE_FLOAT);
        PROP_FNS(nw4f_getQuantToleranceRotate, GetQuantToleranceRotate, nw4f_setQuantToleranceRotate, SetQuantToleranceRotate, TYPE_FLOAT);
        PROP_FNS(nw4f_getQuantToleranceScale, GetQuantToleranceScale, nw4f_setQuantToleranceScale, SetQuantToleranceScale, TYPE_FLOAT);
        //
        PROP_FNS(nw4f_getQuantToleranceUVTrans, GetQuantToleranceUVTrans, nw4f_setQuantToleranceUVTrans, SetQuantToleranceUVTrans, TYPE_FLOAT);
        PROP_FNS(nw4f_getQuantToleranceUVRotate, GetQuantToleranceUVRotate, nw4f_setQuantToleranceUVRotate, SetQuantToleranceUVRotate, TYPE_FLOAT);
        PROP_FNS(nw4f_getQuantToleranceUVScale, GetQuantToleranceUVScale, nw4f_setQuantToleranceUVScale, SetQuantToleranceUVScale, TYPE_FLOAT);
        PROP_FNS(nw4f_getQuantToleranceFrame, GetQuantToleranceFrame, nw4f_setQuantToleranceFrame, SetQuantToleranceFrame, TYPE_FLOAT);
        //
        PROP_FNS(nw4f_getUseMergeFtx, GetUseMergeFtx, nw4f_setUseMergeFtx, SetUseMergeFtx, TYPE_BOOL);
        PROP_FNS(nw4f_getUseMergeAnim, GetUseMergeAnim, nw4f_setUseMergeAnim, SetUseMergeAnim, TYPE_BOOL);
        PROP_FNS(nw4f_getMergeAnimFolder, GetMergeAnimFolder, nw4f_setMergeAnimFolder, SetMergeAnimFolder, TYPE_STRING);
        PROP_FNS(nw4f_getMergeAnimName, GetMergeAnimName, nw4f_setMergeAnimName, SetMergeAnimName, TYPE_STRING);

        PROP_FNS(nw4f_getDisableZYAxisConversion, GetDisableZYAxisConversion, nw4f_setDisableZYAxisConversion, SetDisableZYAxisConversion, TYPE_BOOL);

        //script
        PROP_FNS(nw4f_getPreExportScript,	GetPreExportScript,		nw4f_setPreExportScript,	SetPreExportScript,		TYPE_STRING);
        PROP_FNS(nw4f_getPostExportScript,	GetPostExportScript,	nw4f_setPostExportScript,	SetPostExportScript,	TYPE_STRING);

        // lock
        PROP_FNS(nw4f_getLock_export_target, GetLock_export_target, nw4f_setLock_export_target, SetLock_export_target, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_file_name, GetLock_output_file_name, nw4f_setLock_output_file_name, SetLock_output_file_name, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_folder, GetLock_output_folder, nw4f_setLock_output_folder, SetLock_output_folder, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_use_figure_mode, GetLock_use_figure_mode, nw4f_setLock_use_figure_mode, SetLock_use_figure_mode, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_merge_fmd, GetLock_merge_fmd, nw4f_setLock_merge_fmd, SetLock_merge_fmd, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_merge_ftx, GetLock_merge_ftx, nw4f_setLock_merge_ftx, SetLock_merge_ftx, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_merge_anim, GetLock_merge_anim, nw4f_setLock_merge_anim, SetLock_merge_anim, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_merge_anim_name, GetLock_merge_anim_name, nw4f_setLock_merge_anim_name, SetLock_merge_anim_name, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_magnify, GetLock_magnify, nw4f_setLock_magnify, SetLock_magnify, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_texsrt_mode, GetLock_texsrt_mode, nw4f_setLock_texsrt_mode, SetLock_texsrt_mode, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_remove_namespace, GetLock_remove_namespace, nw4f_setLock_remove_namespace, SetLock_remove_namespace, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_comment, GetLock_comment, nw4f_setLock_comment, SetLock_comment, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_frame_range, GetLock_frame_range, nw4f_setLock_frame_range, SetLock_frame_range, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_loop_anim, GetLock_loop_anim, nw4f_setLock_loop_anim, SetLock_loop_anim, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_bake_all_anim, GetLock_bake_all_anim, nw4f_setLock_bake_all_anim, SetLock_bake_all_anim, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_frame_precision, GetLock_frame_precision, nw4f_setLock_frame_precision, SetLock_frame_precision, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_fmd, GetLock_output_fmd, nw4f_setLock_output_fmd, SetLock_output_fmd, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_ftx, GetLock_output_ftx, nw4f_setLock_output_ftx, SetLock_output_ftx, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_fsk, GetLock_output_fsk, nw4f_setLock_output_fsk, SetLock_output_fsk, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_fvb, GetLock_output_fvb, nw4f_setLock_output_fvb, SetLock_output_fvb, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_fcl, GetLock_output_fcl, nw4f_setLock_output_fcl, SetLock_output_fcl, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_fts, GetLock_output_fts, nw4f_setLock_output_fts, SetLock_output_fts, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_ftp, GetLock_output_ftp, nw4f_setLock_output_ftp, SetLock_output_ftp, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_fsh, GetLock_output_fsh, nw4f_setLock_output_fsh, SetLock_output_fsh, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_output_fsn, GetLock_output_fsn, nw4f_setLock_output_fsn, SetLock_output_fsn, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_compress_bone, GetLock_compress_bone, nw4f_setLock_compress_bone, SetLock_compress_bone, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_unite_child, GetLock_unite_child, nw4f_setLock_unite_child, SetLock_unite_child, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_compress_material, GetLock_compress_material, nw4f_setLock_compress_material, SetLock_compress_material, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_compress_shape, GetLock_compress_shape, nw4f_setLock_compress_shape, SetLock_compress_shape, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_tolerance_scale, GetLock_tolerance_scale, nw4f_setLock_tolerance_scale, SetLock_tolerance_scale, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_tolerance_rotate, GetLock_tolerance_rotate, nw4f_setLock_tolerance_rotate, SetLock_tolerance_rotate, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_tolerance_translate, GetLock_tolerance_translate, nw4f_setLock_tolerance_translate, SetLock_tolerance_translate, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_tolerance_color, GetLock_tolerance_color, nw4f_setLock_tolerance_color, SetLock_tolerance_color, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_tolerance_tex_scale, GetLock_tolerance_tex_scale, nw4f_setLock_tolerance_tex_scale, SetLock_tolerance_tex_scale, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_tolerance_tex_rotate, GetLock_tolerance_tex_rotate, nw4f_setLock_tolerance_tex_rotate, SetLock_tolerance_tex_rotate, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_tolerance_tex_translate, GetLock_tolerance_tex_translate, nw4f_setLock_tolerance_tex_translate, SetLock_tolerance_tex_translate, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_quantize_tolerance_scale, GetLock_quantize_tolerance_scale, nw4f_setLock_quantize_tolerance_scale, SetLock_quantize_tolerance_scale, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_quantize_tolerance_rotate, GetLock_quantize_tolerance_rotate, nw4f_setLock_quantize_tolerance_rotate, SetLock_quantize_tolerance_rotate, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_quantize_tolerance_translate, GetLock_quantize_tolerance_translate, nw4f_setLock_quantize_tolerance_translate, SetLock_quantize_tolerance_translate, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_quantize_tolerance_tex_scale, GetLock_quantize_tolerance_tex_scale, nw4f_setLock_quantize_tolerance_tex_scale, SetLock_quantize_tolerance_tex_scale, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_quantize_tolerance_tex_rotate, GetLock_quantize_tolerance_tex_rotate, nw4f_setLock_quantize_tolerance_tex_rotate, SetLock_quantize_tolerance_tex_rotate, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_quantize_tolerance_tex_translate, GetLock_quantize_tolerance_tex_translate, nw4f_setLock_quantize_tolerance_tex_translate, SetLock_quantize_tolerance_tex_translate, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_pre_export_script, GetLock_pre_export_script, nw4f_setLock_pre_export_script, SetLock_pre_export_script, TYPE_BOOL);
        PROP_FNS(nw4f_getLock_post_export_script, GetLock_post_export_script, nw4f_setLock_post_export_script, SetLock_post_export_script, TYPE_BOOL);

    END_FUNCTION_MAP
#pragma warning(default: 4238) // recover nonstandard extension  warning
};

