﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#if defined( NN_BUILD_CONFIG_OS_WIN )

#include <new>
#include <sstream>
#include <zlib.h>
#include <nn/fs.h>
#include "detail/util_FileSystem.h"
#include "detail/util_PngIO.h"

namespace detail {

#define NNT_GFX_PLACEMENT_NEW(addr)     new(addr)
//------------------------------------------------------------------------------
//  PNGファイル読み込み
//------------------------------------------------------------------------------
bool VfxViewerUtilPngIO::ReadPng( uint8_t** pImage, const char* pPath, nn::lmem::HeapHandle pHeapHandle, PNG_IHDR& ihdr )
{
    // ファイルからデータを入力
    uint8_t* pData;
    size_t size;
    bool resRead = VfxViewerUtilFileSystem::GetInstance().Read( &pData, &size, pPath );

    uint8_t* dataHead = pData;

    uint8_t pixelDepth;  // ピクセル深度
    uint32_t rowBytes;

    // エラーチェック
    if (!resRead)
    { return false; }

    // PNGヘッダを入力
    {
        uint8_t header[8];
        InputBytes( &pData, &header, sizeof( header ) );
    }

    // IHDRチャンクを出力
    {
        // チャンク長を入力
        Input32( &pData );

        // チャンクデータを入力
        uint8_t ihdrHeader[4];
        InputBytes( &pData, ihdrHeader, sizeof( ihdrHeader ) );
        ihdr.width         = Input32( &pData );
        ihdr.height        = Input32( &pData );
        ihdr.bitDepth      = Input8( &pData );
        ihdr.colorType     = Input8( &pData );
        ihdr.compressionType = Input8( &pData );
        ihdr.filterType    = Input8( &pData );
        ihdr.interlaceType = Input8( &pData );

        // CRCを入力
        Input32( &pData );

        switch (ihdr.colorType )
        {
        case PNG_COLOR_TYPE_GRAY:
        case PNG_COLOR_TYPE_PALETTE:
            ihdr.channels = 1;
            break;
        case PNG_COLOR_TYPE_RGB:
            ihdr.channels = 3;
            break;
        case PNG_COLOR_TYPE_GRAY_ALPHA:
            ihdr.channels = 2;
            break;
        case PNG_COLOR_TYPE_RGB_ALPHA:
            ihdr.channels = 4;
            break;
        default:
            ihdr.channels = 3;
            break;
        }

        pixelDepth = ihdr.bitDepth * ihdr.channels;
        rowBytes = ((ihdr.width * static_cast<uint32_t>(pixelDepth) + 7) >> 3);

    }

    // 圧縮データのバッファを作成
    uint32_t decompressBufferSize = (rowBytes) * ihdr.height;
    *pImage = static_cast<uint8_t*>( nn::lmem::AllocateFromExpHeap(pHeapHandle, decompressBufferSize) );

    // IDATチャンクを入力
    {
        // チャンク長を入力
        uint32_t length = Input32( &pData );

        // チャンクデータを入力
        uint8_t  idatHeader[4];
        InputBytes( &pData, idatHeader, sizeof( idatHeader ) );

        uint8_t* pCompressBuffer = NNT_GFX_PLACEMENT_NEW( nn::lmem::AllocateFromExpHeap(pHeapHandle, length - 4) ) uint8_t;
        InputBytes( &pData, pCompressBuffer, length - 4 );

        // zstreamを作成
        z_stream zstream;

        zstream.zalloc = Z_NULL;
        zstream.zfree = Z_NULL;
        zstream.opaque = Z_NULL;

        uint32_t filtedImageSize = (1 + rowBytes) * ihdr.height;
        uint8_t* pFilteredImage = NNT_GFX_PLACEMENT_NEW( nn::lmem::AllocateFromExpHeap(pHeapHandle, filtedImageSize) ) uint8_t;

        zstream.next_in   = pCompressBuffer;
        zstream.avail_in  = length - 4;
        zstream.next_out  = pFilteredImage;
        zstream.avail_out = filtedImageSize;

        int res = inflateInit2(&zstream, MAX_WBITS);
        NN_SDK_ASSERT(res == Z_OK);

        // データを解凍
        res = inflate(&zstream, Z_FINISH);
        NN_SDK_ASSERT(res == Z_STREAM_END);

        // zstreamを破棄
        res = inflateEnd(&zstream);
        NN_SDK_ASSERT(res == Z_OK);

        // フィルターを展開
        DecompressFilter( pFilteredImage, ihdr.width, ihdr.height, pixelDepth / 8, *pImage );

        nn::lmem::FreeToExpHeap( pHeapHandle, pCompressBuffer );
        nn::lmem::FreeToExpHeap( pHeapHandle, pFilteredImage );

        // CRCを入力
        Input32( &pData );
    }

    // IENDチャンクを入力
    {
        // チャンク長を入力
        Input32( &pData );

        // チャンクデータを入力
        uint8_t  iendHeader[4];
        InputBytes( &pData, iendHeader, sizeof( iendHeader ) );

        // CRCを入力
        Input32( &pData );
    }

    VfxViewerUtilFileSystem::GetInstance().ReleaseData(dataHead);

    return resRead;
}

//------------------------------------------------------------------------------
//  PNGファイル読み込み
//------------------------------------------------------------------------------
bool VfxViewerUtilPngIO::ReadPng( uint8_t** pImage, const char* pPath, nn::lmem::HeapHandle pHeapHandle )
{
    PNG_IHDR ihdr;
    return ReadPng( pImage, pPath, pHeapHandle, ihdr );
}

//------------------------------------------------------------------------------
//  PNGファイル書き込み
//------------------------------------------------------------------------------
bool VfxViewerUtilPngIO::WritePng( const char* pPath, const uint8_t* pImage, nn::lmem::HeapHandle pHeapHandle, int width, int height, uint8_t bitDepth, uint8_t colorType, uint8_t compressType, uint8_t filterType, uint8_t interlaceType )
{
    // メモリーストリーム
    std::ostringstream stream;

    uint8_t channels = 4; // チャンネル数

    // PNGヘッダを出力
    {
        uint8_t header[] = { 0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a };
        OutputBytes( &stream, header, sizeof( header ) );
    }

    switch (colorType )
    {
    case PNG_COLOR_TYPE_GRAY:
    case PNG_COLOR_TYPE_PALETTE:
        channels = 1;
        break;
    case PNG_COLOR_TYPE_RGB:
        channels = 3;
        break;
    case PNG_COLOR_TYPE_GRAY_ALPHA:
        channels = 2;
        break;
    case PNG_COLOR_TYPE_RGB_ALPHA:
        channels = 4;
        break;
    default:
        channels = 3;
        break;
    }

    uint8_t pixelDepth = bitDepth * channels;
    uint32_t rowBytes = ((width * static_cast<uint32_t>(pixelDepth) + 7) >> 3);

    // IHDRチャンクを出力
    {
        uint32_t ihdrLength        = 13;
        uint8_t  ihdrHeader[]      = { 'I', 'H', 'D', 'R' };

        // CRCを初期化
        uint32_t crc = crc32( 0L, Z_NULL, 0 );

        // チャンク長を出力
        Output32( &stream, &ihdrLength );

        // チャンクデータを出力
        OutputBytes( &stream, ihdrHeader, sizeof( ihdrHeader ), &crc );
        Output32( &stream, &width, &crc );
        Output32( &stream, &height, &crc );
        Output8( &stream, &bitDepth, &crc );
        Output8( &stream, &colorType, &crc );
        Output8( &stream, &compressType, &crc );
        Output8( &stream, &filterType, &crc );
        Output8( &stream, &interlaceType, &crc );

        // CRCを出力
        Output32( &stream, &crc );
    }

    // 圧縮データのバッファを作成
    uint32_t compressBufferSize = (1 + rowBytes) * height + 128 * 1024;  // 念のため少し大きめのバッファサイズを確保
    uint8_t* pCompressBuffer = NNT_GFX_PLACEMENT_NEW( nn::lmem::AllocateFromExpHeap(pHeapHandle, compressBufferSize) ) uint8_t;

    uint32_t compressedSize;

    // ピクセルデータを圧縮
    {
        // zstreamを作成
        z_stream zstream;

        zstream.zalloc = Z_NULL;
        zstream.zfree = Z_NULL;
        zstream.opaque = Z_NULL;

        int res = deflateInit(&zstream, Z_DEFAULT_COMPRESSION);
        NN_SDK_ASSERT(res == Z_OK);

        // 圧縮率を上げるためピクセルデータにフィルターをかける
        uint32_t filteredSize = (1 + rowBytes) * height;
        uint8_t* pFilteredImage = NNT_GFX_PLACEMENT_NEW( nn::lmem::AllocateFromExpHeap(pHeapHandle, filteredSize) ) uint8_t;

        CompressFilter( pImage, width, height, pixelDepth / 8, pFilteredImage );

        zstream.next_in   = pFilteredImage;
        zstream.avail_in  = filteredSize;
        zstream.next_out  = pCompressBuffer;
        zstream.avail_out = compressBufferSize;

        // データを圧縮
        res = deflate(&zstream, Z_NO_FLUSH);
        NN_SDK_ASSERT(res == Z_OK);

        res = deflate(&zstream, Z_FINISH);
        NN_SDK_ASSERT(res == Z_STREAM_END);

        compressedSize = zstream.total_out;

        // zstreamを破棄
        res = deflateEnd(&zstream);
        NN_SDK_ASSERT(res == Z_OK);

        nn::lmem::FreeToExpHeap( pHeapHandle, pFilteredImage );
    }

    // IDATチャンクを出力
    {
        uint32_t idatLength = 4 + compressedSize;
        uint8_t  idatHeader[] = { 'I', 'D', 'A', 'T' };

        // CRCを初期化
        uint32_t crc = crc32( 0L, Z_NULL, 0 );

        // チャンク長を出力
        Output32( &stream, &idatLength );

        // チャンクデータを出力
        OutputBytes( &stream, idatHeader, sizeof( idatHeader ), &crc );
        OutputBytes( &stream, pCompressBuffer, compressedSize, &crc );

        // CRCを出力
        Output32( &stream, &crc );
    }

    // 圧縮データを破棄
    nn::lmem::FreeToExpHeap( pHeapHandle, pCompressBuffer );

    // IENDチャンクを出力
    {
        uint32_t iendLength = 0;
        uint8_t  iendHeader[] = { 'I', 'E', 'N', 'D' };

        // CRCを初期化
        uint32_t crc = crc32( 0L, Z_NULL, 0 );

        // チャンク長を出力
        Output32( &stream, &iendLength );

        // チャンクデータを出力
        OutputBytes( &stream, iendHeader, sizeof( iendHeader ), &crc );

        // CRCを出力
        Output32( &stream, &crc );
    }

    // ファイルにデータを保存
    bool ret = VfxViewerUtilFileSystem::GetInstance().Write( reinterpret_cast<const uint8_t*>( stream.str().c_str() ), stream.str().size(), pPath );

    return ret;
}

//------------------------------------------------------------------------------
//  PNGファイル書き込み
//------------------------------------------------------------------------------
bool VfxViewerUtilPngIO::WritePng( const char* pPath, const uint8_t* pImage, PNG_IHDR ihdr, nn::lmem::HeapHandle pHeapHandle  )
{
    return WritePng( pPath, pImage, pHeapHandle, ihdr.width, ihdr.height, ihdr.bitDepth, ihdr.colorType, ihdr.compressionType, ihdr.filterType, ihdr.interlaceType );
}

//------------------------------------------------------------------------------
//  8bitデータをストリームに出力
//------------------------------------------------------------------------------
void VfxViewerUtilPngIO::Output8( std::ostream* pStream, const void* pData, uint32_t* pCrc )
{
    pStream->write( static_cast<const char*>( pData ), 1 );

    if (pCrc != nullptr)
    {
        *pCrc = crc32( *pCrc, static_cast<const Bytef*>( pData ), 1 );
    }
}

//------------------------------------------------------------------------------
//  バイト列データをストリームに出力
//------------------------------------------------------------------------------
void VfxViewerUtilPngIO::OutputBytes( std::ostream* pStream, const void* pData, size_t size, uint32_t* pCrc )
{
    pStream->write( static_cast<const char*>( pData ), size );

    if (pCrc != nullptr)
    {
        *pCrc = crc32( *pCrc, static_cast<const Bytef*>( pData ), static_cast<uInt>(size) );
    }
}

//!------------------------------------------------------------------------------
//! 8bitデータをストリームから入力
//!------------------------------------------------------------------------------
uint8_t VfxViewerUtilPngIO::Input8( uint8_t** pStream )
{
    uint8_t data = *(*pStream);
    *pStream = *pStream + 1;
    return data;
}

//!------------------------------------------------------------------------------
//! バイト列データをストリームから入力
//!------------------------------------------------------------------------------
void VfxViewerUtilPngIO::InputBytes( uint8_t** pStream, void* pData, size_t size )
{
    memcpy( pData, *pStream, size );
    *pStream = *pStream + size;
}

}

#endif
