﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>
#include <nn/gfx/util/gfx_TransientDescriptorAllocatorHolder.h>

namespace detail {


const int g_HistoryLengthMax  = 2; // SwapChainのバッファ数だけ用意(ダブルバッファリングであれば 2、トリプルバッファリングであれば 3 )

class IDescriptorIndexAllocator
{
public:
    IDescriptorIndexAllocator() {}

    //---------------------------------------------------------------------------
    //! @brief       デスクリプタスロットを初期化
    //! @param[int]  descriptorPool 　  デスクリプタプール
    //! @param[int]  baseIndex          ディスクリプタスロットベースインデックス
    //! @param[int]  staticSlotCount    固定ディスクリプタスロット数
    //! @param[int]  transientSlotCount トランジェントディスクリプタスロット数
    //---------------------------------------------------------------------------
    void Initialize( nn::gfx::DescriptorPool*  descriptorPool, int baseIndex, int staticSlotCount, int transientSlotCount )
    {
        m_pDescPool = descriptorPool;

        // 固定デスクリプタインデックス アロケータの初期化
        m_baseIndex = baseIndex;
        m_StaticIndex = baseIndex;
        m_StaticIndexMaxCount = staticSlotCount;

        // ダイナミックデスクリプタインデックス アロケータの初期化
        m_TransientDescriptor.Initialize( m_pDescPool, baseIndex + staticSlotCount, transientSlotCount );
        m_TransientDescriptor.Get()->FillHistory();
    }

    //---------------------------------------------------------------------------
    //! @brief      デスクリプタスロット確保を開始
    //---------------------------------------------------------------------------
    void BeginAllocateTransientDescriptorSlot()
    {
        nn::gfx::util::TransientDescriptorAllocator* pTransientDescAllocator = m_TransientDescriptor.Get();

        if ( !pTransientDescAllocator->IsRecording() )
        {
            pTransientDescAllocator->Free();
            pTransientDescAllocator->Begin();
        }
    }

    //---------------------------------------------------------------------------
    //! @brief      デスクリプタスロット確保を終了
    //---------------------------------------------------------------------------
    void EndAllocateTransientDescriptorSlot()
    {
        nn::gfx::util::TransientDescriptorAllocator* pTransientDescAllocator = m_TransientDescriptor.Get();
        pTransientDescAllocator->End();
    }

    //---------------------------------------------------------------------------
    //! @brief     　固定デスクリプタスロットインデックスを取得
    //! @param[int]  requireCount   要求インデックス数
    //---------------------------------------------------------------------------
    int GetStaticDescriptorPoolIndex( int requireCount )
    {
        return GetDescriptorPoolIndex( requireCount );
    }

    //---------------------------------------------------------------------------
    //! @brief     　デスクリプタスプールを取得
    //---------------------------------------------------------------------------
    nn::gfx::DescriptorPool* GetDescriptorPool()
    {
        return m_pDescPool;
    }

    //---------------------------------------------------------------------------
    //! @brief     　トランジェントデスクリプタアロケータを取得
    //---------------------------------------------------------------------------
    nn::gfx::util::TransientDescriptorAllocator* GetTransientDescriptorAllocator()
    {
        return m_TransientDescriptor.Get();
    }

    //---------------------------------------------------------------------------
    //! @brief      デスクリプタプールのインデックスを取得
    //! @param[in]  requireCount    TBD
    //---------------------------------------------------------------------------
    int GetDescriptorPoolIndex( int requireCount )
    {
        int ret = m_StaticIndex;
        m_StaticIndex += requireCount;
        NN_SDK_ASSERT( m_StaticIndexMaxCount > m_StaticIndex - m_baseIndex );
        return ret;
    }

private:
    nn::gfx::util::TransientDescriptorAllocatorHolder<g_HistoryLengthMax> m_TransientDescriptor;
    nn::gfx::DescriptorPool*                    m_pDescPool;
    int                                         m_baseIndex;
    int                                         m_StaticIndex;
    int                                         m_StaticIndexMaxCount;
};


class TextureDescriptorIndexAllocator : public IDescriptorIndexAllocator
{
public:
    TextureDescriptorIndexAllocator() {}

    //---------------------------------------------------------------------------
    //! @brief     　固定デスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  textureView    テクスチャビュー
    //! @param[int]  pUserData      ユーザデータ
    //---------------------------------------------------------------------------
    bool AllocateStaticTextureDescriptorSlot(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData )
    {
        NN_UNUSED(pUserData);

        // 固定ディスクリプタインデックス取得
        int slot = GetDescriptorPoolIndex( 1 );

        GetDescriptorPool()->BeginUpdate();
        {
            GetDescriptorPool()->SetTextureView( slot, &textureView);
        }
        GetDescriptorPool()->EndUpdate();
        GetDescriptorPool()->GetDescriptorSlot( dstSlot, slot );
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief     　トランジェントデスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  textureView    テクスチャビュー
    //! @param[int]  pUserData      ユーザデータ
    //---------------------------------------------------------------------------
    bool AllocateTransientTextureDescriptorSlot(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData )
    {
        NN_UNUSED(pUserData);

        // トランジェントディスクリプタインデックス取得
        int slot = GetTransientDescriptorAllocator()->Allocate();

        GetDescriptorPool()->BeginUpdate();
        {
            GetDescriptorPool()->SetTextureView( slot, &textureView);
        }
        GetDescriptorPool()->EndUpdate();
        GetDescriptorPool()->GetDescriptorSlot( dstSlot, slot );
        return true;
    }
};

class SamplerDescriptorIndexAllocator : public IDescriptorIndexAllocator
{
public:

    //---------------------------------------------------------------------------
    //! @brief      固定デスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  sampler    　　サンプラビュー
    //! @param[int]  pUserData      ユーザデータ
    //---------------------------------------------------------------------------
    bool AllocateStaticSamplerDescriptorSlot(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData )
    {
        NN_UNUSED(pUserData);

        // 固定ディスクリプタインデックス取得
        int slot = GetDescriptorPoolIndex( 1 );

        GetDescriptorPool()->BeginUpdate();
        {
            GetDescriptorPool()->SetSampler( slot, &sampler);
        }
        GetDescriptorPool()->EndUpdate();
        GetDescriptorPool()->GetDescriptorSlot( dstSlot, slot );
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief       トランジェントデスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  sampler    　　サンプラビュー
    //! @param[int]  pUserData      ユーザデータ
    //---------------------------------------------------------------------------
    bool AllocateTransientSamplerDescriptorSlot(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData )
    {
        NN_UNUSED(pUserData);

        // トランジェントディスクリプタインデックス取得
        int slot = GetTransientDescriptorAllocator()->Allocate();

        GetDescriptorPool()->BeginUpdate();
        {
            GetDescriptorPool()->SetSampler( slot, &sampler);
        }
        GetDescriptorPool()->EndUpdate();
        GetDescriptorPool()->GetDescriptorSlot( dstSlot, slot );
        return true;
    }
};

} // namespace detail

