﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>

namespace nw { namespace g3d { namespace tool {
namespace util {

template <typename T>
class TStringRef
{
public:
    typedef TStringRef<T> SelfType;
    typedef T CharType;
    typedef std::basic_string<T> StringType;

    TStringRef() : str(nullptr), len(0) {}
    TStringRef(const CharType* str) : str(str), len(str ? StringType(str).length() : len) {}
    TStringRef(const CharType* str, size_t len) : str(str), len(len) {}
    TStringRef(const StringType& str) : str(str.data()), len(str.length()) {}

    void Clear() { str = nullptr; len = 0; }
    void Set(const CharType* str) { this->str = str; len = str ? StringType(str).length() : len; }
    void Set(const CharType* str, size_t len) { this->str = str; this->len = len; }
    void Set(const StringType& str) { this->str = str.data; len = str.length(); }

    const CharType* Data() const { return str; }
    size_t Len() const { return len; }
    StringType Str() const { return str ? StringType(str, len) : StringType(); }

    bool Empty() const { return len == 0; }

    const CharType* Begin() const { return str; }
    const CharType* End() const { return str + len; }
    const CharType* RBegin() const { return str + len - 1; }
    const CharType* REnd() const { return str - 1; }

    bool operator==(const SelfType& rhs) const
    {
        return Str() == rhs.Str();
    }
    bool operator==(const CharType* rhs) const
    {
        return Str() == StringType(rhs);
    }
    bool operator==(const StringType& rhs) const
    {
        return Str() == rhs;
    }

    bool operator!=(const SelfType& rhs) const
    {
        return !(*this == rhs);
    }
    bool operator!=(const CharType* rhs) const
    {
        return !(*this == rhs);
    }
    bool operator!=(const StringType& rhs) const
    {
        return !(*this == rhs);
    }

private:
    const CharType* str;
    size_t len;
};

// operator==

template <typename T>
inline
bool operator==(const typename TStringRef<T>::CharType* lhs, const TStringRef<T>& rhs)
{
    return rhs == lhs;
}

template <typename T>
inline
bool operator==(const typename TStringRef<T>::StringType& lhs, const TStringRef<T>& rhs)
{
    return rhs == lhs;
}

template <typename T>
inline
bool operator!=(const typename TStringRef<T>::CharType* lhs, const TStringRef<T>& rhs)
{
    return rhs != lhs;
}

template <typename T>
inline
bool operator!=(const typename TStringRef<T>::StringType& lhs, const TStringRef<T>& rhs)
{
    return rhs != lhs;
}

// operator<<

template <typename T>
inline
std::basic_ostream<T>& operator<<(std::basic_ostream<T>& lhs, const TStringRef<T>& rhs)
{
    lhs.write(rhs.Data(), rhs.Len());
    return lhs;
}

// 特殊化

typedef TStringRef<char> StringRef;
typedef TStringRef<wchar_t> WStringRef;

template<>
inline
StringRef::TStringRef(const CharType* string) : str(string), len(string ? strlen(string) : 0) {}

template<>
inline
WStringRef::TStringRef(const CharType* string) : str(string), len(string ? wcslen(string) : 0) {}

template<>
inline
void StringRef::Set(const CharType* string) { this->str = string; len = string ? strlen(string) : 0; }

template<>
inline
void WStringRef::Set(const CharType* string) { this->str = string; len = string ? wcslen(string) : 0; }

template<>
inline
bool StringRef::operator==(const StringRef& rhs) const
{
    return len == rhs.len && 0 == strncmp(str, rhs.str, len);
}

template<>
inline
bool WStringRef::operator==(const WStringRef& rhs) const
{
    return len == rhs.len && 0 == wcsncmp(str, rhs.str, len);
}

template<>
inline
bool StringRef::operator==(const char* rhs) const
{
    return 0 == strncmp(str, rhs, len) && 0 == rhs[len];
}

template<>
inline
bool WStringRef::operator==(const wchar_t* rhs) const
{
    return 0 == wcsncmp(str, rhs, len) && 0 == rhs[len];
}

template<>
inline
bool StringRef::operator==(const std::string& rhs) const
{
    return len == rhs.length() && 0 == strncmp(str, rhs.data(), rhs.length());
}

template<>
inline
bool WStringRef::operator==(const std::wstring& rhs) const
{
    return len == rhs.length() && 0 == wcsncmp(str, rhs.data(), rhs.length());
}

} // namespace util

} // namespace tool
} // namespace g3d
} // namespace nw
