﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/Serialization.h>
#include <regex>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

void elem_nw4f_3dif::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        version << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

int CastToInt(std::string& str)
{
    char* e;

    int i = std::strtol(str.c_str(), &e, 10);

    if (errno != ERANGE)
    {
        if (*e != '\0')
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_OPTION_CHOICE, "Invalid option choice: %hs.", str.c_str());
        }
    }
    else if (i == LONG_MAX)
    {
        THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_OPTION_CHOICE, "Invalid option choice: %hs.", str.c_str());
    }
    else if (i == LONG_MIN)
    {
        THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_OPTION_CHOICE, "Invalid option choice: %hs.", str.c_str());
    }

    return i;
}

std::shared_ptr<void> AnalizeAndCopyData(const char* textData, int count, StreamType type)
{
    // 中間ファイルからデータを読み込みます。
    // アニメーションデータは中間ファイルでは常に float として値を持っています。
    // これはフレームが小数フレームに対応するためです。
    // float として読み込んだデータは BinAnimCurve で変換しながらバイナリ化します。
    void* pData = nullptr;
    if (type == StreamTypeInt)
    {
        pData = ReadArray<int>(textData, count);
    }
    else if (type == StreamTypeFloat)
    {
        pData = ReadArray<float>(textData, count);
    }
    else if (type == StreamTypeByte)
    {
        pData = ReadArray<unsigned char>(textData, count);
    }
    else if (type == StreamTypeString)
    {
        size_t len = strlen(textData) + 1;
        pData = static_cast<char*>(malloc(len * sizeof(char)));
        memcpy(pData, textData, len);
    }
    else if (type == StreamTypeWString)
    {
        int wLen = MultiByteToWideChar(CP_UTF8,
            0,
            textData,
            -1,
            NULL,
            0);
        pData = static_cast<wchar_t*>(malloc(wLen * sizeof(wchar_t)));
        MultiByteToWideChar(CP_UTF8,
            0,
            textData,
            -1,
            static_cast<wchar_t*>(pData),
            wLen);
    }
    else
    {
        THROW_ERROR(ERRCODE_UNSUPPORTED, "Unsupported stream type.");
    }

    return std::shared_ptr<void>(pData, free);
}

void ExpandChoice(std::vector<std::pair<std::string, std::string>>& choiceArray, const std::string& choice)
{
    choiceArray.clear();
    static const std::regex minmax("\\[\\s*(-?\\s*\\d+)\\s*,\\s*(-?\\s*\\d+)\\s*\\]");
    std::smatch results;

    // 指定なし or bool
    if (choice.empty() || choice == "bool")
    {
        // 必ず 0, 1 の順に push する。
        choiceArray.push_back(std::make_pair("0", ""));
        choiceArray.push_back(std::make_pair("1", ""));
    }
    else if (std::regex_match(choice, results, minmax))
    {
        std::string minStr = results[1];
        std::string maxStr = results[2];

        int min = CastToInt(minStr);
        int max = CastToInt(maxStr);

        for (int i = min; i <= max; ++i)
        {
            std::stringstream ss;
            ss << i;
            choiceArray.push_back(std::make_pair(ss.str(), ""));
        }
    }
    else
    {
        std::vector<std::string> sep;
        util::Split(sep, choice, ",");

        for (auto str = sep.cbegin(); str != sep.cend(); ++str)
        {
            // ':' は複数記述できる。最初の ':' 以降がラベルとなる。
            std::string select = *str;
            std::string alias;
            auto found = str->find_first_of(':');
            if (found != std::string::npos)
            {
                select = select.substr(0, found);
                alias = str->substr(found + 1, str->length() - found);
            }

            choiceArray.push_back(std::make_pair(util::Trim(select), util::Trim(alias)));
        }
    }
}

void ExpandChoiceUint(std::vector<uint32_t>& choiceUintArray, const std::vector<std::pair<std::string, std::string>>& choiceArray)
{
    // choce の文字列を uint に変換する
    for (auto choice = choiceArray.cbegin(); choice != choiceArray.cend(); ++choice)
    {
        std::istringstream choiceString(choice->first);
        uint32_t choiceUint = 0;
        if (choiceString.str()[0] == '-')
        {
            // マイナス値の string は uint への変換が未定義なので、一度 int に変換して uint に代入する
            int32_t choiceInt = 0;
            choiceString >> choiceInt;
            choiceUint = static_cast<uint32_t>(choiceInt);
        }
        else
        {
            choiceString >> choiceUint;
        }

        choiceUintArray.emplace_back(choiceUint);
    }
}

int IncludeValue(const std::vector<std::pair<std::string, std::string>>& choiceArray, const std::string& value)
{
    int valueIndex = 0;
    // value が choiceArray に含まれるかどうか確認する。
    if (!value.empty())
    {
        valueIndex = -1;
        // choice の中から探索します。
        int choiceIndex = 0;
        for (auto c = choiceArray.cbegin(); c != choiceArray.cend(); ++c, ++choiceIndex)
        {
            if (c->first == value)
            {
                valueIndex = choiceIndex;
                break;
            }
        }
    }
    else
    {
        if (choiceArray.size() != 0)
        {
            valueIndex = 0;
        }
    }

    return valueIndex;
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
