﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/SceneAnim.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

G3DIF_DEFINE_ENUM_TABLE(
    camera_anim, rotate_mode,
    S11N_DEFINE_ENUM_ID(aim),
    S11N_DEFINE_ENUM_ID(euler_zxy)
    );

G3DIF_DEFINE_ENUM_TABLE(
    camera_anim, projection_mode,
    S11N_DEFINE_ENUM_ID(ortho),
    S11N_DEFINE_ENUM_ID(persp)
    );

G3DIF_DEFINE_ENUM_TABLE(
    camera_anim_target, target,
    S11N_DEFINE_ENUM_ID(position_x),
    S11N_DEFINE_ENUM_ID(position_y),
    S11N_DEFINE_ENUM_ID(position_z),
    S11N_DEFINE_ENUM_ID(aim_x),
    S11N_DEFINE_ENUM_ID(aim_y),
    S11N_DEFINE_ENUM_ID(aim_z),
    S11N_DEFINE_ENUM_ID(twist),
    S11N_DEFINE_ENUM_ID(rotate_x),
    S11N_DEFINE_ENUM_ID(rotate_y),
    S11N_DEFINE_ENUM_ID(rotate_z),
    S11N_DEFINE_ENUM_ID(aspect),
    S11N_DEFINE_ENUM_ID(near),
    S11N_DEFINE_ENUM_ID(far),
    S11N_DEFINE_ENUM_ID(ortho_height),
    S11N_DEFINE_ENUM_ID(persp_fovy)
    );

G3DIF_DEFINE_ENUM_TABLE(
    light_anim_target, target,
    S11N_DEFINE_ENUM_ID(enable),
    S11N_DEFINE_ENUM_ID(position_x),
    S11N_DEFINE_ENUM_ID(position_y),
    S11N_DEFINE_ENUM_ID(position_z),
    S11N_DEFINE_ENUM_ID(direction_x),
    S11N_DEFINE_ENUM_ID(direction_y),
    S11N_DEFINE_ENUM_ID(direction_z),
    S11N_DEFINE_ENUM_ID(aim_x),
    S11N_DEFINE_ENUM_ID(aim_y),
    S11N_DEFINE_ENUM_ID(aim_z),
    S11N_DEFINE_ENUM_ID(dist_attn_start),
    S11N_DEFINE_ENUM_ID(dist_attn_end),
    S11N_DEFINE_ENUM_ID(angle_attn_start),
    S11N_DEFINE_ENUM_ID(angle_attn_end),
    S11N_DEFINE_ENUM_ID(color0_r),
    S11N_DEFINE_ENUM_ID(color0_g),
    S11N_DEFINE_ENUM_ID(color0_b),
    S11N_DEFINE_ENUM_ID(color1_r),
    S11N_DEFINE_ENUM_ID(color1_g),
    S11N_DEFINE_ENUM_ID(color1_b)
    );

G3DIF_DEFINE_ENUM_TABLE(
    fog_anim_target, target,
    S11N_DEFINE_ENUM_ID(dist_attn_start),
    S11N_DEFINE_ENUM_ID(dist_attn_end),
    S11N_DEFINE_ENUM_ID(color_r),
    S11N_DEFINE_ENUM_ID(color_g),
    S11N_DEFINE_ENUM_ID(color_b)
    );

void elem_camera_anim_target::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        target << pElem;
        base_value << pElem;

        curve.Validate();
        if (!curve->Read(pElem))
        {
            curve.Invalidate();
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_camera_anim::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        camera_name << pElem;
        frame_count << pElem;
        loop << pElem;
        rotate_mode << pElem;
        projection_mode << pElem;

        for (const util::XMLElement* pTarget = pElem->Child(elem_camera_anim_target::Id());
            pTarget; pTarget = pTarget->NextSibling(elem_camera_anim_target::Id()))
        {
            camera_anim_target_array.push_back(elem_camera_anim_target());
            camera_anim_target_array.back() << pTarget;
            user_data_array << pElem->Child(elem_user_data::IdArray());
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_light_anim_target::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        target << pElem;
        base_value << pElem;

        curve.Validate();
        if (!curve->Read(pElem))
        {
            curve.Invalidate();
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_light_anim::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        light_name << pElem;
        frame_count << pElem;
        loop << pElem;
        type << pElem;
        dist_attn_func << pElem;
        angle_attn_func << pElem;

        for (const util::XMLElement* pTarget = pElem->Child(elem_light_anim_target::Id());
            pTarget; pTarget = pTarget->NextSibling(elem_light_anim_target::Id()))
        {
            light_anim_target_array.push_back(elem_light_anim_target());
            light_anim_target_array.back() << pTarget;
            user_data_array << pElem->Child(elem_user_data::IdArray());
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_fog_anim_target::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        target << pElem;
        base_value << pElem;

        curve.Validate();
        if (!curve->Read(pElem))
        {
            curve.Invalidate();
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_fog_anim::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        fog_name << pElem;
        frame_count << pElem;
        loop << pElem;
        dist_attn_func << pElem;

        for (const util::XMLElement* pTarget = pElem->Child(elem_fog_anim_target::Id());
            pTarget; pTarget = pTarget->NextSibling(elem_fog_anim_target::Id()))
        {
            fog_anim_target_array.push_back(elem_fog_anim_target());
            fog_anim_target_array.back() << pTarget;
            user_data_array << pElem->Child(elem_user_data::IdArray());
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_scene_anim_info::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
    }
    CATCH_THROW_XML_ERROR()
}

void elem_scene_anim::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        version << pElem;
        scene_anim_info << pElem->Child(elem_scene_anim_info::Id());
        camera_anim_array << pElem->Child(elem_camera_anim::IdArray());
        light_anim_array << pElem->Child(elem_light_anim::IdArray());
        fog_anim_array << pElem->Child(elem_fog_anim::IdArray());
        stream_array << pElem->Child(elem_stream::IdArray());
        user_data_array << pElem->Child(elem_user_data::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}


void elem_scene_anim::PostProcess()
{
    // 参照の解決。
    for (auto iter = camera_anim_array.begin(); iter != camera_anim_array.end(); ++iter)
    {
        for (auto iterTarget = iter->camera_anim_target_array.begin(); iterTarget != iter->camera_anim_target_array.end(); ++ iterTarget)
        {
            if (iterTarget->curve)
            {
                elem_anim_curve& curve = iterTarget->curve.Get();
                try
                {
                    int stream_array_size = static_cast<int>(stream_array.size());

                    if (stream_array_size <= curve.stream_index.value)
                    {
                        THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                            "Out of range. stream_array: %d camera_anim.stream_index: %d",
                            stream_array_size, curve.stream_index.value);
                    }
                    // テキストデータを解析します。
                    elem_stream& xmlStream = stream_array[curve.stream_index.value];
                    curve.stream.rawdata = AnalizeAndCopyData(xmlStream.textData, xmlStream.count.value, xmlStream.GetStreamType());
                    curve.stream.count = xmlStream.count.value;
                    curve.stream.type = static_cast<StreamType>(xmlStream.type.value);
                }
                CATCH_THROW_XML_ERROR()
            }
        }

        // ユーザーデータの解析
        for (auto iterUserData = iter->user_data_array.begin(); iterUserData != iter->user_data_array.end(); ++iterUserData)
        {
            iterUserData->PostProcess(stream_array);
        }
    }

    for (auto iter = light_anim_array.begin(); iter != light_anim_array.end(); ++iter)
    {
        for (auto iterTarget = iter->light_anim_target_array.begin(); iterTarget != iter->light_anim_target_array.end(); ++ iterTarget)
        {
            if (iterTarget->curve)
            {
                elem_anim_curve& curve = iterTarget->curve.Get();
                try
                {
                    int stream_array_size = static_cast<int>(stream_array.size());

                    if (stream_array_size <= curve.stream_index.value)
                    {
                        THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                            "Out of range. stream_array: %d light_anim.stream_index: %d",
                            stream_array_size, curve.stream_index.value);
                    }
                    // テキストデータを解析します。
                    elem_stream& xmlStream = stream_array[curve.stream_index.value];
                    curve.stream.rawdata = AnalizeAndCopyData(xmlStream.textData, xmlStream.count.value, xmlStream.GetStreamType());
                    curve.stream.count = xmlStream.count.value;
                    curve.stream.type = static_cast<StreamType>(xmlStream.type.value);
                }
                CATCH_THROW_XML_ERROR()
            }
        }

        // ユーザーデータの解析
        for (auto iterUserData = iter->user_data_array.begin(); iterUserData != iter->user_data_array.end(); ++iterUserData)
        {
            iterUserData->PostProcess(stream_array);
        }
    }

    for (auto iter = fog_anim_array.begin(); iter != fog_anim_array.end(); ++iter)
    {
        for (auto iterTarget = iter->fog_anim_target_array.begin(); iterTarget != iter->fog_anim_target_array.end(); ++ iterTarget)
        {
            if (iterTarget->curve)
            {
                elem_anim_curve& curve = iterTarget->curve.Get();
                try
                {
                    int stream_array_size = static_cast<int>(stream_array.size());

                    if (stream_array_size <= curve.stream_index.value)
                    {
                        THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                            "Out of range. stream_array: %d fog_anim.stream_index: %d",
                            stream_array_size, curve.stream_index.value);
                    }
                    // テキストデータを解析します。
                    elem_stream& xmlStream = stream_array[curve.stream_index.value];
                    curve.stream.rawdata = AnalizeAndCopyData(xmlStream.textData, xmlStream.count.value, xmlStream.GetStreamType());
                    curve.stream.count = xmlStream.count.value;
                    curve.stream.type = static_cast<StreamType>(xmlStream.type.value);
                }
                CATCH_THROW_XML_ERROR()
            }
        }

        // ユーザーデータの解析
        for (auto iterUserData = iter->user_data_array.begin(); iterUserData != iter->user_data_array.end(); ++iterUserData)
        {
            iterUserData->PostProcess(stream_array);
        }
    }

    // ユーザーデータの解析
    for (auto iter = user_data_array.begin(); iter != user_data_array.end(); ++iter)
    {
        iter->PostProcess(stream_array);
    }
}

void elem_scene_anim::PostBinaryProcess(void* data)
{
    try
    {
        StreamArray streamArray;
        AnalizeBinaryData(streamArray, data);

        int stream_array_size = static_cast<int>(streamArray.streamChunk.size());

        for (auto iter = camera_anim_array.begin(); iter != camera_anim_array.end(); ++iter)
        {
            for (auto iterTarget = iter->camera_anim_target_array.begin(); iterTarget != iter->camera_anim_target_array.end(); ++ iterTarget)
            {
                if (iterTarget->curve)
                {
                    elem_anim_curve& curve = iterTarget->curve.Get();

                    if (stream_array_size <= curve.stream_index.value)
                    {
                        THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                            "Out of range. stream_array: %d camera_anim.stream_index: %d",
                            stream_array_size, curve.stream_index.value);
                    }
                    void* rawdata = CopyRawData(streamArray.streamChunk[curve.stream_index.value]);
                    curve.stream.rawdata.reset(rawdata, free);
                    curve.stream.count = streamArray.streamChunk[curve.stream_index.value].count;
                    curve.stream.type = streamArray.streamChunk[curve.stream_index.value].type;
                }
            }
            for (auto iterUserData = iter->user_data_array.begin(); iterUserData != iter->user_data_array.end(); ++iterUserData)
            {
                iterUserData->PostBinaryProcess(streamArray);
            }
        }

        for (auto iter = light_anim_array.begin(); iter != light_anim_array.end(); ++iter)
        {
            for (auto iterTarget = iter->light_anim_target_array.begin(); iterTarget != iter->light_anim_target_array.end(); ++ iterTarget)
            {
                if (iterTarget->curve)
                {
                    elem_anim_curve& curve = iterTarget->curve.Get();

                    if (stream_array_size <= curve.stream_index.value)
                    {
                        THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                            "Out of range. stream_array: %d light_anim.stream_index: %d",
                            stream_array_size, curve.stream_index.value);
                    }
                    void* rawdata = CopyRawData(streamArray.streamChunk[curve.stream_index.value]);
                    curve.stream.rawdata.reset(rawdata, free);
                    curve.stream.count = streamArray.streamChunk[curve.stream_index.value].count;
                    curve.stream.type = streamArray.streamChunk[curve.stream_index.value].type;
                }
            }
            for (auto iterUserData = iter->user_data_array.begin(); iterUserData != iter->user_data_array.end(); ++iterUserData)
            {
                iterUserData->PostBinaryProcess(streamArray);
            }
        }

        for (auto iter = fog_anim_array.begin(); iter != fog_anim_array.end(); ++iter)
        {
            for (auto iterTarget = iter->fog_anim_target_array.begin(); iterTarget != iter->fog_anim_target_array.end(); ++ iterTarget)
            {
                if (iterTarget->curve)
                {
                    elem_anim_curve& curve = iterTarget->curve.Get();

                    if (stream_array_size <= curve.stream_index.value)
                    {
                        THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                            "Out of range. stream_array: %d fog_anim.stream_index: %d",
                            stream_array_size, curve.stream_index.value);
                    }
                    void* rawdata = CopyRawData(streamArray.streamChunk[curve.stream_index.value]);
                    curve.stream.rawdata.reset(rawdata, free);
                    curve.stream.count = streamArray.streamChunk[curve.stream_index.value].count;
                    curve.stream.type = streamArray.streamChunk[curve.stream_index.value].type;
                }
            }
            for (auto iterUserData = iter->user_data_array.begin(); iterUserData != iter->user_data_array.end(); ++iterUserData)
            {
                iterUserData->PostBinaryProcess(streamArray);
            }
        }

        for (auto iter = user_data_array.begin(); iter != user_data_array.end(); ++iter)
        {
            iter->PostBinaryProcess(streamArray);
        }
    }
    CATCH_THROW_XML_ERROR()
}

class CameraTargetFinder
{
public:
    CameraTargetFinder(elem_camera_anim_target::enum_target target)
        : m_Target(target) {}

    bool operator()(const elem_camera_anim_target& target)
    {
        return target.target.value == m_Target;
    }
private:
    elem_camera_anim_target::enum_target m_Target;
};

void
elem_scene_anim::CheckData(int /*flag*/)
{
    // camera_amim
    for (auto iter = camera_anim_array.cbegin(); iter != camera_anim_array.cend(); ++iter)
    {
        bool targets[elem_camera_anim_target::num_target] = { false };
        for (auto iterTarget = iter->camera_anim_target_array.cbegin();
            iterTarget != iter->camera_anim_target_array.cend();
            ++iterTarget)
        {
            targets[iterTarget->target.value] = true;
        }

        // camera_anim_target 以下の通り各要素が必須
        // rotate_mode      : aim   : aim_x, aim_y, aim_z, twist
        //                    euler : rotate_x, rotate_y, rotate_z
        // projection_mode  : ortho : ortho_height
        //                  : persp : persp_fovy
        // 残りの要素は必須
        if (iter->rotate_mode.value == elem_camera_anim::aim)
        {
            if (!targets[elem_camera_anim_target::target_aim_x] ||
                !targets[elem_camera_anim_target::target_aim_y] ||
                !targets[elem_camera_anim_target::target_aim_z] ||
                !targets[elem_camera_anim_target::target_twist])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_CAMERA_ANIM_TARGET, "Invalid camera_anim_target.");
            }
        }
        else
        {
            if (!targets[elem_camera_anim_target::target_rotate_x] ||
                !targets[elem_camera_anim_target::target_rotate_y] ||
                !targets[elem_camera_anim_target::target_rotate_z])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_CAMERA_ANIM_TARGET, "Invalid camera_anim_target.");
            }
        }

        if (iter->projection_mode.value == elem_camera_anim::ortho)
        {
            if (!targets[elem_camera_anim_target::target_ortho_height])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_CAMERA_ANIM_TARGET, "Invalid camera_anim_target.");
            }
        }
        else
        {
            if (!targets[elem_camera_anim_target::target_persp_fovy])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_CAMERA_ANIM_TARGET, "Invalid camera_anim_target.");
            }
        }

        if (!targets[elem_camera_anim_target::target_position_x] ||
            !targets[elem_camera_anim_target::target_position_y] ||
            !targets[elem_camera_anim_target::target_position_z] ||
            !targets[elem_camera_anim_target::target_aspect]     ||
            !targets[elem_camera_anim_target::target_near]       ||
            !targets[elem_camera_anim_target::target_far])
        {
            THROW_ERROR(ERROCDE_XML_INVALID_CAMERA_ANIM_TARGET, "Invalid camera_anim_target.");
        }
    }

    // light_amim
    for (auto iter = light_anim_array.cbegin(); iter != light_anim_array.cend(); ++iter)
    {
        bool targets[elem_light_anim_target::num_target] = { false };
        for (auto iterTarget = iter->light_anim_target_array.cbegin();
            iterTarget != iter->light_anim_target_array.cend();
            ++iterTarget)
        {
            targets[iterTarget->target.value] = true;
        }

        // position, direction, aim, dist_attn, angle_attn, color0, color1 はセットでなければならない
        if (targets[elem_light_anim_target::target_position_x] ||
            targets[elem_light_anim_target::target_position_y] ||
            targets[elem_light_anim_target::target_position_z])
        {
            if (!targets[elem_light_anim_target::target_position_x] ||
                !targets[elem_light_anim_target::target_position_y] ||
                !targets[elem_light_anim_target::target_position_z])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_LIGHT_ANIM_TARGET, "Invalid light_anim_target.");
            }
        }

        if (targets[elem_light_anim_target::target_direction_x] ||
            targets[elem_light_anim_target::target_direction_y] ||
            targets[elem_light_anim_target::target_direction_z])
        {
            if (!targets[elem_light_anim_target::target_direction_x] ||
                !targets[elem_light_anim_target::target_direction_y] ||
                !targets[elem_light_anim_target::target_direction_z])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_LIGHT_ANIM_TARGET, "Invalid light_anim_target.");
            }
        }

        if (targets[elem_light_anim_target::target_aim_x] ||
            targets[elem_light_anim_target::target_aim_y] ||
            targets[elem_light_anim_target::target_aim_z])
        {
            if (!targets[elem_light_anim_target::target_aim_x] ||
                !targets[elem_light_anim_target::target_aim_y] ||
                !targets[elem_light_anim_target::target_aim_z])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_LIGHT_ANIM_TARGET, "Invalid light_anim_target.");
            }
        }

        if (targets[elem_light_anim_target::target_dist_attn_start] ||
            targets[elem_light_anim_target::target_dist_attn_end])
        {
            if (!targets[elem_light_anim_target::target_dist_attn_start] ||
                !targets[elem_light_anim_target::target_dist_attn_end])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_LIGHT_ANIM_TARGET, "Invalid light_anim_target.");
            }
        }

        if (targets[elem_light_anim_target::target_angle_attn_start] ||
            targets[elem_light_anim_target::target_angle_attn_end])
        {
            if (!targets[elem_light_anim_target::target_angle_attn_start] ||
                !targets[elem_light_anim_target::target_angle_attn_end])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_LIGHT_ANIM_TARGET, "Invalid light_anim_target.");
            }
        }

        if (targets[elem_light_anim_target::target_color0_r] ||
            targets[elem_light_anim_target::target_color0_g] ||
            targets[elem_light_anim_target::target_color0_b])
        {
            if (!targets[elem_light_anim_target::target_color0_r] ||
                !targets[elem_light_anim_target::target_color0_g] ||
                !targets[elem_light_anim_target::target_color0_b])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_LIGHT_ANIM_TARGET, "Invalid light_anim_target.");
            }
        }

        if (targets[elem_light_anim_target::target_color1_r] ||
            targets[elem_light_anim_target::target_color1_g] ||
            targets[elem_light_anim_target::target_color1_b])
        {
            if (!targets[elem_light_anim_target::target_color1_r] ||
                !targets[elem_light_anim_target::target_color1_g] ||
                !targets[elem_light_anim_target::target_color1_b])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_LIGHT_ANIM_TARGET, "Invalid light_anim_target.");
            }
        }
    }

    // fog_amim
    for (auto iter = fog_anim_array.cbegin(); iter != fog_anim_array.cend(); ++iter)
    {
        bool targets[elem_fog_anim_target::num_target] = { false };
        for (auto iterTarget = iter->fog_anim_target_array.cbegin();
            iterTarget != iter->fog_anim_target_array.cend();
            ++iterTarget)
        {
            targets[iterTarget->target.value] = true;
        }

        // dist_attn, color はセットでなければならない
        if (targets[elem_fog_anim_target::target_dist_attn_start] ||
            targets[elem_fog_anim_target::target_dist_attn_end])
        {
            if (!targets[elem_fog_anim_target::target_dist_attn_start] ||
                !targets[elem_fog_anim_target::target_dist_attn_end])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_FOG_ANIM_TARGET, "Invalid fog_anim_target.");
            }
        }

        if (targets[elem_fog_anim_target::target_color_r] ||
            targets[elem_fog_anim_target::target_color_g] ||
            targets[elem_fog_anim_target::target_color_b])
        {
            if (!targets[elem_fog_anim_target::target_color_r] ||
                !targets[elem_fog_anim_target::target_color_g] ||
                !targets[elem_fog_anim_target::target_color_b])
            {
                THROW_ERROR(ERROCDE_XML_INVALID_FOG_ANIM_TARGET, "Invalid fog_anim_target.");
            }
        }
    }
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
