﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include <Sources/Argument.h>

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
// DLL の関数型の定義です。
//=============================================================================

//! @brief コンバーターのバージョンを整数値で返します。
typedef int (*GetCvtrVersionFunc)();

//! コンバーターの使用方法を表示します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] cvtrName コンバーター名です。
//! @param[in] cmdName コンバーターのコマンド名です。
//! @param[in] isVersionOnly バージョンのみ表示するなら true を指定します。
//!
typedef void (*ShowUsageFunc)(
    std::ostream& os,
    const wchar_t* cvtrName,
    const wchar_t* cmdName,
    const bool isVersionOnly
);

//! @brief 現在の環境で GPU によるエンコーディングが可能か判定します。
//!
//! @return GPU によるエンコーディングが可能なら true を返します。
//!
typedef bool (*CheckGpuEncodingFunc)();

//! @brief COM のモードを設定します。
//!        コンバーター内で COM を初期化する場合、
//!        ReadInputFile を呼ぶ前に呼ぶ必要があります。
//!
//! @param[in] initializesCom 必要になった際にコンバーター内で COM を初期化するなら true、
//!                           外部で初期化するなら false を指定します。
//!                           デフォルトでは false の動作です。
//! @param[in] coInit CoInitializeEx で指定する COM の初期化オプションです。
//!
typedef void (*SetComModeFunc)(const bool initializesCom, const int coInit);

//! @brief グローバルオプションを設定します。
//!
//! @param[in] グローバルオプションのユニコード文字列配列です。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*SetOptionsFunc)(const wchar_t* options[]);

//! @brief メモリーをクリアし、ジョブオプションを初期化します。
//!        元画像、出力データ、エラー文字列などをすべてクリアします。
//!
typedef void (*ClearFunc)();

//! @brief 入力ファイルをリードします。
//!
//! @param[in] paths 入力ファイルのパス配列です。
//!                  キューブマップの各面のファイルのパスは options で指定します。
//! @param[in] options ジョブオプション文字列配列です。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*ReadInputFileFunc)(const wchar_t* paths[], const wchar_t* options[]);

//! @brief メモリー上の ftx ファイルのデータをリードします。
//!
//! @param[in] pData ftx ファイルのデータです。
//! @param[in] dataSize ftx ファイルのデータサイズです。
//! @param[in] path ftx ファイルのパスです。
//!                 このパスの拡張子からバイナリー形式かアスキー形式か判別します。
//! @param[in] options ジョブオプション文字列配列です。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*ReadFtxDataFunc)(
    const void* pData,
    const size_t dataSize,
    const wchar_t* path,
    const wchar_t* options[]
);

//! @brief メモリー上のビットマップデータをリードします。
//!
//! @param[in] pData ビットマップデータです。
//!                  1 ピクセルあたり 4 または 16 バイトで、RGBA の順に格納します。画像の左上が先頭です。
//!                  キューブマップ（6 面個別指定）の場合、+X、-X、+Y、-Y、+Z、-Z 面の順に格納します。
//!                  ビットマップデータのサイズは width * height * depth * (4 または 16) となります。
//! @param[in] width ビットマップの幅です。
//! @param[in] height ビットマップの高さです。
//! @param[in] depth ビットマップの深さです。
//!                  水平十字／垂直十字キューブマップの場合は 1 を指定します。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
//! @param[in] paths 入力ファイルのパス配列です。
//! @param[in] originalPaths オリジナルファイルのパス配列です。
//!                          キューブマップ（6 面個別指定）の場合、+X、-X、+Y、-Y、+Z、-Z 面の順に指定します。
//!                          入力ファイルが ftx 以外の場合、paths と同じ値を指定します。
//! @param[in] options ジョブオプション文字列配列です。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*ReadBitmapDataFunc)(
    const void* pData,
    const int width,
    const int height,
    const int depth,
    const bool hasAlpha,
    const bool isFloat,
    const wchar_t* paths[],
    const wchar_t* originalPaths[],
    const wchar_t* options[]
);

//! @brief 変換して出力ファイルをセーブします。
//!        リード時のジョブオプションで指定した出力パスにセーブします。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*ConvertToFileFunc)();

//! @brief 変換して出力ファイルをメモリーに格納します。
//!        リード時のジョブオプションで指定した出力パスによって
//!        出力ファイル形式が決定されます。
//!
//! @param[out] ppData 出力ファイルを格納したメモリーのアドレスを格納します。
//! @param[out] pDataSize 出力ファイルのサイズを格納します。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*ConvertToDataFunc)(const void** ppData, size_t* pDataSize);

//! @brief エラーのユニコード文字列を返します。
typedef const wchar_t* (*GetErrorStringFunc)();

//! @brief エラーの Shift JIS 文字列を返します。
typedef const char* (*GetErrorShiftJisStringFunc)();

//! @brief 作成情報を設定します。
//!        ビットマップデータをリードした後に必要なら呼びます。
//!
//! @param[in] createInfo 作成情報の XML 文字列です。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*SetCreateInfoFunc)(const wchar_t* createInfo);

//! @brief ユーザーデータを設定します。
//!        ビットマップデータをリードした後に必要なら呼びます。
//!
//! @param[in] pPackedUserData パックされたデータへのポインタです。
//! @param[in] packedUserDataSize パックされたデータのサイズです。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*SetUserDataFunc)(const void* pPackedUserData, const size_t packedUserDataSize);

//! @brief 編集用コメントラベルを設定します。
//!        ビットマップデータをリードした後に必要なら呼びます。
//!
//! @param[in] pCommentLabel 編集用コメントラベルへのポインタです。文字コードは Shift-JIS です。
//! @param[in] commentLabelSize 編集用コメントラベルのサイズです。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*SetCommentLabelFunc)(const void* pCommentLabel, const size_t commentLabelSize);

//! @brief 編集用コメントカラー文字列を設定します。
//!        ビットマップデータをリードした後に必要なら呼びます。
//!
//! @param[in] pCommentCol 編集用コメントカラー文字列へのポインタです。文字コードは Shift-JIS です。
//! @param[in] commentColSize 編集用コメントカラー文字列のサイズです。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*SetCommentColorFunc)(const void* pCommentCol, const size_t commentColSize);

//! @brief ツールデータを設定します。
//!        ビットマップデータをリードした後に必要なら呼びます。
//!
//! @param[in] pToolData ツールデータ文字列へのポインタです。文字コードは Shift-JIS です。
//! @param[in] toolDataSize ツールデータ文字列のサイズです。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*SetToolDataFunc)(const void* pToolData, const size_t toolDataSize);

//! @brief ユーザーツールデータを設定します。
//!        ビットマップデータをリードした後に必要なら呼びます。
//!
//! @param[in] pUserToolData ユーザーツールデータ文字列へのポインタです。文字コードは Shift-JIS です。
//! @param[in] userToolDataSize ユーザーツールデータ文字列のサイズです。
//!
//! @return 処理成功なら true を返します。
//!
typedef bool (*SetUserToolDataFunc)(const void* pUserToolData, const size_t userToolDataSize);

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

