﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <shdrdefs.h>
#include <util/UtilStringRef.h>
#include <g3dif/ShaderDefinition.h>

using nw::g3d::tool::util::StringRef;

namespace nw { namespace g3d { namespace tool { namespace util {
class XMLElement;
}}}}

namespace nn { namespace g3dTool {

// 抄出したアノテーションは内部でxmlで管理する。そこで使われる要素名。
// fsdはそのxmlから下記の要素名を引いて生成する。
const StringRef id_shader = "shader";
const StringRef id_stage = "stage";
const StringRef id_path = "path";
const StringRef id_option = "option";
const StringRef id_attrib = "attrib";
const StringRef id_sampler = "sampler";
const StringRef id_uniform = "uniform";
const StringRef id_buffer = "buffer";
const StringRef id_block = "block";
const StringRef id_ssbo_block = "ssbo_block";
const StringRef id_ssbo_block_info = "ssbo_block_info";
const StringRef id_option_id = "option_id";
const StringRef id_attrib_id = "attrib_id";
const StringRef id_sampler_id = "sampler_id";
const StringRef id_block_id = "block_id";
const StringRef id_uniform_id = "uniform_id";
const StringRef id_at = "at";			// アノテーションの遠隔操作の情報を保持する要素
const StringRef id_in = "in";
const StringRef id_out = "out";
const StringRef id_triangles_adjacency = "triangles_adjacency";

const StringRef id_block_info = "block_info";
const StringRef id_name = "name";
const StringRef id_enable = "enable";
const StringRef id_id = "id";
const StringRef id_type = "type";
const StringRef id_alt = "alt";
const StringRef id_length = "length";
const StringRef id_hint = "hint";
const StringRef id_default = "default";
const StringRef id_label = "label";
const StringRef id_comment = "comment";
const StringRef id_order = "order";
const StringRef id_item = "item";
const StringRef id_visible = "visible";
const StringRef id_min = "min";
const StringRef id_max = "max";
const StringRef id_choice = "choice";
const StringRef id_nostrip = "nostrip";
const StringRef id_renderinfo = "renderinfo";
const StringRef id_textblock = "textblock";
const StringRef id_text = "text";
const StringRef id_count = "count";
const StringRef id_interleave = "interleave";
const StringRef id_group = "group";
const StringRef id_group_id = "group_id";
const StringRef id_streamout = "streamout";
const StringRef id_revision = "revision";
const StringRef id_binarize = "binarize";
const StringRef id_condition = "condition";
const StringRef id_editable = "editable";
const StringRef id_anim_editable = "anim_editable";
const StringRef id_layout = "layout";
const StringRef id_qualifier = "qualifier";
const StringRef id_converter = "converter";
const StringRef id_depend = "depend";
const StringRef id_page_name = "page_name";
const StringRef id_page = "page";
const StringRef id_file_info = "file_info";
const StringRef id_line_info = "line_info";
const StringRef id_var = "var";
const StringRef id_optional = "optional";
const StringRef id_default_min = "default_min";
const StringRef id_default_max = "default_max";

static const StringRef g_AnnotationOptionIdList[] =
{
    id_name,
    id_enable,
    id_id,
    id_choice,
    id_default,
    id_type,
    id_label,
    id_comment,
    id_group,
    id_order,
    id_visible,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationAttribIdList[] =
{
    id_name,
    id_id,
    id_hint,
    id_type,
    id_label,
    id_comment,
    id_group,
    id_editable,
    id_order,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationSamplerIdList[] =
{
    id_name,
    id_id,
    id_hint,
    id_type,
    id_alt,
    id_nostrip,
    id_label,
    id_comment,
    id_group,
    id_order,
    id_visible,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationUniformIdList[] =
{
    id_name,
    id_id,
    id_hint,
    id_type,
    id_length,
    id_default,
    id_nostrip,
    id_label,
    id_comment,
    id_group,
    id_order,
    id_item,
    id_min,
    id_max,
    id_visible,
    id_default_min,
    id_default_max,
    id_converter,
    id_depend,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationBlockIdList[] =
{
    id_name,
    id_var,
    id_id,
    id_type,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationRenderInfoIdList[] =
{
    id_renderinfo,
    id_type,
    id_count,
    id_choice,
    id_default,
    id_label,
    id_comment,
    id_group,
    id_order,
    id_item,
    id_visible,
    id_optional,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationTextBlockIdList[] =
{
    id_textblock,
    id_text,
    id_group,
    id_order,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationInterleaveIdList[] =
{
    id_interleave,
    id_binarize,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationGroupIdList[] =
{
    id_group_id,
    id_label,
    id_comment,
    id_order,
    id_condition,
    id_page,
    id_group,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationStreamoutIdList[] =
{
    id_streamout,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationRevisionIdList[] =
{
    id_revision,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationLayoutIdList[] =
{
    id_layout,
    id_qualifier,
    id_file_info,
    id_line_info
};
static const StringRef g_AnnotationPageIdList[] =
{
    id_page,
    id_label,
    id_comment,
    id_order,
    id_file_info,
    id_line_info
};

static int GetArrayIndex(const StringRef& id, const StringRef* pIdArray, int idCount)
{
    for (int index = 0; index < idCount; ++index)
    {
        if (pIdArray[index] == id)
        {
            return index;
        }
    }
    THROW_ERROR(ERRCODE_INTERNAL, "Internal error");
    return 0;
}

class AnnotationBase
{
public:
    AnnotationBase::AnnotationBase(const StringRef* pIdArray, int idCount, const XMLElement* pElem)
    {
        m_IdCount = idCount;
        m_IdArray = pIdArray;
        m_ValueArray.reset(new StringRef[m_IdCount]);
        for (int i = 0; i < m_IdCount; ++i)
        {
            m_ValueArray[i] = pElem->Attribute(m_IdArray[i]);
        }
    }
    AnnotationBase::AnnotationBase(const StringRef* pIdArray, int idCount, const XMLElement* pElem, const StringRef& name)
    {
        m_IdCount = idCount;
        m_IdArray = pIdArray;
        m_ValueArray.reset(new StringRef[m_IdCount]);
        for (int i = 0; i < m_IdCount; ++i)
        {
            if (m_IdArray[i] == id_name)
            {
                m_ValueArray[i] = name;
            }
            else
            {
                m_ValueArray[i] = pElem->Attribute(m_IdArray[i]);
            }
        }
    }

    virtual void MergeAttr(const XMLElement* pElem);

    void SetAttributeValue(const StringRef& id, const StringRef& value)
    {
        m_ValueArray[GetArrayIndex(id, m_IdArray, m_IdCount)] = value;
    }

    const StringRef& GetAttributeValue(const StringRef& id) const
    {
        return m_ValueArray[GetArrayIndex(id, m_IdArray, m_IdCount)];
    }

    int GetIdCount() const
    {
        return m_IdCount;
    }
    const StringRef* GetIdArray() const
    {
        return m_IdArray;
    }
    const StringRef* GetValueArray() const
    {
        return m_ValueArray.get();
    }

protected:
    static const char* GetInvalidXmlAnnotationName(const StringRef* pIdArray, int idCount, const XMLElement* pElem);

private:
    int m_IdCount;
    const StringRef* m_IdArray;
    std::unique_ptr<StringRef[]> m_ValueArray;
};

class AnnotOption : public AnnotationBase
{
public:
    AnnotOption::AnnotOption(const XMLElement* pElem, const StringRef& name)
        : AnnotationBase(g_AnnotationOptionIdList, sizeof(g_AnnotationOptionIdList) / sizeof(StringRef), pElem, name) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationOptionIdList, sizeof(g_AnnotationOptionIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotAttrib : public AnnotationBase
{
public:
    AnnotAttrib::AnnotAttrib(const XMLElement* pElem, const StringRef& name)
        : AnnotationBase(g_AnnotationAttribIdList, sizeof(g_AnnotationAttribIdList) / sizeof(StringRef), pElem, name) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationAttribIdList, sizeof(g_AnnotationAttribIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotSampler : public AnnotationBase
{
public:
    AnnotSampler::AnnotSampler(const XMLElement* pElem, const StringRef& name)
        : AnnotationBase(g_AnnotationSamplerIdList, sizeof(g_AnnotationSamplerIdList) / sizeof(StringRef), pElem, name) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationSamplerIdList, sizeof(g_AnnotationSamplerIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotUniform : public AnnotationBase
{
public:
    AnnotUniform::AnnotUniform(const XMLElement* pElem, const StringRef& name)
        : AnnotationBase(g_AnnotationUniformIdList, sizeof(g_AnnotationUniformIdList) / sizeof(StringRef), pElem, name) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationUniformIdList, sizeof(g_AnnotationUniformIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotBlock : public AnnotationBase
{
public:
    AnnotBlock::AnnotBlock(const XMLElement* pElem, const StringRef& name, const StringRef& storageInfoName)
        : AnnotationBase(g_AnnotationBlockIdList, sizeof(g_AnnotationBlockIdList) / sizeof(StringRef), pElem, name)
    {
        // id と type を上書き
        if (const char* attr = pElem->Child(storageInfoName)->Attribute(id_id))
        {
            SetAttributeValue(id_id, attr);
        }
        if (const char* attr = pElem->Child(storageInfoName)->Attribute(id_type))
        {
            SetAttributeValue(id_type, attr);
        }
    }

    virtual void MergeAttr(const XMLElement* pElem);
    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem);
    std::vector<AnnotUniform> uniforms;
};

// 使用するメンバは同じだがssboではtypeのエラーチェックを入れるため継承してコンストラクタの挙動を変える。
class AnnotSsboBlock : public AnnotBlock
{
public:
    AnnotSsboBlock(const XMLElement* pElem, const StringRef& name, const StringRef& storageInfoName) : AnnotBlock(pElem, name, storageInfoName)
    {
        if (GetAttributeValue(id_type).Data() != nullptr)
        {
            auto enumType = nw::g3d::tool::g3dif::elem_ssbo_block_var::GetType(std::string(GetAttributeValue(id_type).Data()));
            if (!nw::g3d::tool::g3dif::elem_ssbo_block_var::IsValidType(enumType))
            {
                THROW_ERROR(ERRCODE_XML_INVALID_SHADER_STORAGE_BLOCK_TYPE,
                    "Invalid type. Check if available type. shader_storage_block_var: %hs", GetAttributeValue(id_id).Data());
            }
        }
    }
};

class AnnotRenderInfo : public AnnotationBase
{
public:
    AnnotRenderInfo::AnnotRenderInfo(const XMLElement* pElem)
        : AnnotationBase(g_AnnotationRenderInfoIdList, sizeof(g_AnnotationRenderInfoIdList) / sizeof(StringRef), pElem) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationRenderInfoIdList, sizeof(g_AnnotationRenderInfoIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotTextBlock : public AnnotationBase
{
public:
    AnnotTextBlock::AnnotTextBlock(const XMLElement* pElem)
        : AnnotationBase(g_AnnotationTextBlockIdList, sizeof(g_AnnotationTextBlockIdList) / sizeof(StringRef), pElem) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationTextBlockIdList, sizeof(g_AnnotationTextBlockIdList) / sizeof(StringRef), pElem);
    }

    virtual void MergeAttr(const XMLElement* pElem);
};

class AnnotInterleave : public AnnotationBase
{
public:
    AnnotInterleave::AnnotInterleave(const XMLElement* pElem)
        : AnnotationBase(g_AnnotationInterleaveIdList, sizeof(g_AnnotationInterleaveIdList) / sizeof(StringRef), pElem) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationInterleaveIdList, sizeof(g_AnnotationInterleaveIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotGroup : public AnnotationBase
{
public:
    AnnotGroup::AnnotGroup(const XMLElement* pElem)
        : AnnotationBase(g_AnnotationGroupIdList, sizeof(g_AnnotationGroupIdList) / sizeof(StringRef), pElem) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationGroupIdList, sizeof(g_AnnotationGroupIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotStreamout : public AnnotationBase
{
public:
    AnnotStreamout::AnnotStreamout(const XMLElement* pElem)
        : AnnotationBase(g_AnnotationStreamoutIdList, sizeof(g_AnnotationStreamoutIdList) / sizeof(StringRef), pElem) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationStreamoutIdList, sizeof(g_AnnotationStreamoutIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotRevision : public AnnotationBase
{
public:
    AnnotRevision::AnnotRevision(const XMLElement* pElem)
        : AnnotationBase(g_AnnotationRevisionIdList, sizeof(g_AnnotationRevisionIdList) / sizeof(StringRef), pElem) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationRevisionIdList, sizeof(g_AnnotationRevisionIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotLayout : public AnnotationBase
{
public:
    AnnotLayout::AnnotLayout(const XMLElement* pElem)
        : AnnotationBase(g_AnnotationLayoutIdList, sizeof(g_AnnotationLayoutIdList) / sizeof(StringRef), pElem) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationLayoutIdList, sizeof(g_AnnotationLayoutIdList) / sizeof(StringRef), pElem);
    }
};

class AnnotPage : public AnnotationBase
{
public:
    AnnotPage::AnnotPage(const XMLElement* pElem)
        : AnnotationBase(g_AnnotationPageIdList, sizeof(g_AnnotationPageIdList) / sizeof(StringRef), pElem) {}

    static const char* GetInvalidXmlAnnotationName(const XMLElement* pElem)
    {
        return AnnotationBase::GetInvalidXmlAnnotationName(g_AnnotationPageIdList, sizeof(g_AnnotationPageIdList) / sizeof(StringRef), pElem);
    }
};

//--------------------------------------------------------------------------------------------------

class SymbolBase
{
public:
    SymbolBase::SymbolBase(const AnnotationBase& annot)
    {
        // AnnotationBase の内容をコピー
        m_IdArray = annot.GetIdArray();
        m_IdCount = annot.GetIdCount();
        m_ValueArray.reset(new std::string[m_IdCount]);
        for (int index = 0; index < m_IdCount; ++index)
        {
            const StringRef& id = annot.GetIdArray()[index];
            if (id == id_name || id == id_var)
            {
                // name と var はステージ別に保持するため別管理
                continue;
            }
            m_ValueArray[index] = annot.GetValueArray()[index].Str();
        }
    }

    virtual bool CompareAttr(const AnnotationBase& rhs) const;
    virtual void Override(const AnnotationBase& annot);

    const std::string& GetSymbolValue(const StringRef& id) const
    {
        return m_ValueArray[GetArrayIndex(id, m_IdArray, m_IdCount)];
    }

private:
    int m_IdCount;
    const StringRef* m_IdArray;
    std::unique_ptr<std::string[]> m_ValueArray;
};

class SymbolOption : public SymbolBase
{
public:
    SymbolOption(ShaderStage stage, const AnnotOption& annot) : SymbolBase(annot)
    {
        strName[stage].assign(annot.GetAttributeValue(id_name).Str());
    }
    std::string strName[ShaderStage_StageCount];
};

class SymbolAttrib : public SymbolBase
{
public:
    SymbolAttrib(ShaderStage stage, const AnnotAttrib& annot) : SymbolBase(annot)
    {
        strName[stage].assign(annot.GetAttributeValue(id_name).Str());
    }
    std::string strName[ShaderStage_StageCount];
};

class SymbolSampler : public SymbolBase
{
public:
    SymbolSampler(ShaderStage stage, const AnnotSampler& annot) : SymbolBase(annot)
    {
        strName[stage].assign(annot.GetAttributeValue(id_name).Str());
    }
    std::string strName[ShaderStage_StageCount];
};

class SymbolUniform : public SymbolBase
{
public:
    SymbolUniform(ShaderStage stage, const AnnotUniform& annot) : SymbolBase(annot)
    {
        strName[stage].assign(annot.GetAttributeValue(id_name).Str());
    }
    std::string strName[ShaderStage_StageCount];
};

class SymbolBlock : public SymbolBase
{
public:
    SymbolBlock(ShaderStage stage, const AnnotBlock& annot) : SymbolBase(annot)
    {
        strName[stage].assign(annot.GetAttributeValue(id_name).Str());
        strVar[stage].assign(annot.GetAttributeValue(id_var).Str());
        for (auto uniform = annot.uniforms.cbegin(); uniform != annot.uniforms.cend(); ++uniform)
        {
            uniforms.emplace_back(SymbolUniform(stage, *uniform));
        }
    }

    virtual bool CompareAttr(const AnnotBlock& rhs) const;
    virtual void Override(const AnnotBlock& annot);

    std::string strName[ShaderStage_StageCount];
    std::string strVar[ShaderStage_StageCount];

    std::vector<SymbolUniform> uniforms;
};

typedef SymbolBlock SymbolSsboBlock;

// 以下のアノテーションは name が存在しないために、Annot から Sysmbol へ変換する必要はない。
// ただし、他のアノテーションと同様に処理するために Symbol を定義して処理する。
class SymbolRenderInfo : public SymbolBase
{
public:
    SymbolRenderInfo(const AnnotRenderInfo& annot) : SymbolBase(annot) {};
};

class SymbolTextBlock : public SymbolBase
{
public:
    SymbolTextBlock(const AnnotTextBlock& annot) : SymbolBase(annot) {};
};

class SymbolInterleave : public SymbolBase
{
public:
    SymbolInterleave(const AnnotInterleave& annot) : SymbolBase(annot) {};
};

class SymbolStreamout : public SymbolBase
{
public:
    SymbolStreamout(const AnnotStreamout& annot) : SymbolBase(annot) {};
};

class SymbolRevision : public SymbolBase
{
public:
    SymbolRevision(const AnnotRevision& annot) : SymbolBase(annot) {};
};

class SymbolLayout : public SymbolBase
{
public:
    SymbolLayout(const AnnotLayout& annot) : SymbolBase(annot) {};
};

class SymbolGroup : public SymbolBase
{
public:
    SymbolGroup(const AnnotGroup& annot) : SymbolBase(annot) {};
};

class SymbolPage : public SymbolBase
{
public:
    SymbolPage(const AnnotPage& annot) : SymbolBase(annot) {};
};

} // namespace g3d
} // namespace nw
