﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @file
* @brief バッファに関する API の宣言
*/

#pragma once

#include <memory>
#include <util/types.h>
#include <Context.h>

namespace nn {
namespace g3dTool {

/**
* @brief メモリブロックに含まれるチャンクに関する情報を表します。
*/
struct Chunk
{
    size_t		size;
    ptrdiff_t	offset;
    size_t		alignment;
};


/**
* @brief 各メモリブロックに含まれるバイナリブロックを表すクラスです。全てのバイナリブロックは本クラスを継承します。
*/
class BinaryBlock
{
public:

    //! @brief 自身のサイズを計算します。
    virtual void CalculateSize()
    {
    }

    //! @brief メモリブロック内での自身のオフセットを計算します。
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );

    //! @brief メモリブロックに値を埋めていきます。
    virtual void Convert( std::shared_ptr<Context> pCtx )
    {
        NN_UNUSED( pCtx );
    }

    //! @brief コンバート後に行う処理を実装します。
    virtual void Adjust( std::shared_ptr<Context> pCtx )
    {
        NN_UNUSED( pCtx );
    }

    //! @brief 親から構造体分のメモリを割り当てられる場合に使用します。
    void SetStructOffset(ptrdiff_t offset)
    {
        m_StructOffset = offset;
    }

    //! @brief メインメモリブロックでの自身のポインタを取得します。
    void* GetPtr(void* pBuffer) const
    {
        if (m_StructOffset >= 0)
        {
            // 親から割り当てられている場合はそちらを利用する。
            return nw::g3d::tool::util::AddOffset(pBuffer, m_StructOffset);
        }

        if (m_Block[Context::MemBlockType_Main].size > 0)
        {
            // 自身の MAIN 領域から割り当てる。
            return nw::g3d::tool::util::AddOffset(pBuffer, m_Block[Context::MemBlockType_Main].offset);
        }
        return nullptr;
    }

    /**
    * @brief メインメモリブロック内での自身のアドレスを返します。
    *
    * @param[in] pBuffer メインメモリブロックのベースポインタ
    *
    */
    template <typename T>
    T* GetPtr(void* pBuffer) const
    {
        return static_cast<T*>(GetPtr(pBuffer));
    }

protected:
    //! @brief コンストラクタです。
    BinaryBlock() : m_StructOffset(-1)
    {
        for (int index = 0; index < Context::MemBlockType_Count; ++index)
        {
            m_Block[index].size = 0;
            m_Block[index].offset = -1;
            m_Block[index].alignment = 0;
        }
    }

    //! @brief チャンクのオフセット、サイズを計算します。
    static size_t CalcChunk(Chunk* pChunk, int count)
    {
        int idx = 0;
        pChunk[idx].offset = 0;
        for (; idx < count - 1; ++idx)
        {
            pChunk[idx + 1].offset = pChunk[idx].offset + pChunk[idx].size;
        }
        return pChunk[idx].offset + pChunk[idx].size;
    }

    //! @brief チャンクのオフセット、サイズを計算します。
    static size_t CalcChunk(Chunk* pChunk, int count, size_t alignment )
    {
        for( int chunkIdx = 0; chunkIdx < count; ++chunkIdx )
        {
            pChunk[ chunkIdx ].alignment = alignment;
            if( !nn::util::is_aligned( pChunk[ chunkIdx ].size, pChunk[ chunkIdx ].alignment ) )
            {
                pChunk[ chunkIdx ].size = nn::util::align_up( pChunk[ chunkIdx ].size, pChunk[ chunkIdx ].alignment );
            }
        }

        int idx = 0;
        pChunk[idx].offset = 0;
        for (; idx < count - 1; ++idx)
        {
            pChunk[idx + 1].offset = pChunk[idx].offset + pChunk[idx].size;
        }
        return pChunk[idx].offset + pChunk[idx].size;
    }

    //! @brief 自身が使用するメモリブロック内でのサイズを書き込みます。
    void SetBlockSize(Context::MemBlockType category, size_t size, size_t alignment = 8)
    {
        size_t alignedSize = size;
        if( !nn::util::is_aligned( size, alignment ) )
        {
            alignedSize = nn::util::align_up( size, alignment );
        }
        m_Block[category].size = alignedSize;
        m_Block[category].alignment = alignment;
    }

    //! @brief 自身が使用するメモリブロック内でのサイズを取得します。
    size_t GetBlockSize(Context::MemBlockType category) const
    {
        return m_Block[category].size;
    }

    //! @brief 自身が使用するメモリブロック内でのオフセットを設定します。
    void SetBlockOffset(Context::MemBlockType category, ptrdiff_t offset)
    {
        m_Block[category].offset = offset;
    }

    //! @brief 自身が使用するメモリブロック内でのオフセットを取得します。
    ptrdiff_t GetBlockOffset(Context::MemBlockType category) const
    {
        return m_Block[category].offset;
    }

    //! @brief 指定カテゴリ内の指定オフセット位置のポインタを取得する。
    void* GetPtr( std::shared_ptr<Context> pCtx, Context::MemBlockType category, ptrdiff_t offset );

    //! @brief 自身が使用する category で示されるメモリブロック内のポインタを取得します。
    template <typename T>
    T* GetPtr(std::shared_ptr<Context> pCtx, Context::MemBlockType category, ptrdiff_t offset)
    {
        return static_cast<T*>(GetPtr( pCtx, category, offset));
    }

private:
    Chunk m_Block[ Context::MemBlockType_Count ];	//!< 自身がいるメモリブロック内のオフセット、サイズを保持します。
    ptrdiff_t m_StructOffset;
};


//--------------------------------------------------------------------------------------------------

template <typename DstT, typename SrcT>
inline
void BuildArray(
    std::shared_ptr<Context> pCtx,
    std::vector<DstT>& dstArray,
    const std::vector<SrcT>& srcArray)
{
    auto size = srcArray.size();
    if (dstArray.size() != size)
    {
        // あらかじめ resize をしてある場合は呼び出さない。
        dstArray.resize(size);
    }

    auto iterDst = dstArray.begin();
    auto iterSrc = srcArray.cbegin();
    while (iterDst != dstArray.end())
    {
        iterDst->Build( pCtx, *iterSrc);
        ++iterDst;
        ++iterSrc;
    }
}

// ポインタ配列用の特殊化
template <typename DstT, typename SrcT>
inline
void BuildArray(
    std::shared_ptr<Context> pCtx,
    std::vector<DstT>& dstArray,
    const std::vector<SrcT*>& srcArray)
{
    auto size = srcArray.size();
    if (dstArray.size() != size)
    {
        // あらかじめ resize をしてある場合は呼び出さない。
        dstArray.resize(size);
    }

    auto iterDst = dstArray.begin();
    auto iterSrc = srcArray.cbegin();
    while (iterDst != dstArray.end())
    {
        iterDst->Build( pCtx, **iterSrc );
        ++iterDst;
        ++iterSrc;
    }
}

}
}
